1  --
   2  -- Vehicle
   3  -- Base class for all vehicles
   4  -- Functions of specializations called by vehicle:
   5  --   bool validateAttacherJoint(implement, jointDesc, dt), return true if joint should be revalidated
   6  --
   7  -- @author  Stefan Geiger
   8  -- @date  08/04/07
   9  --
  10  -- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
  11  
  12  Vehicle = {};
  13  
  14  source("dataS/scripts/vehicles/specializations/VehicleSetBeaconLightEvent.lua");
  15  
  16  InitStaticObjectClass(Vehicle, "Vehicle", ObjectIds.OBJECT_VEHICLE);
  17  
  18  Vehicle.springScale = 10;
  19  
  20  Vehicle.NUM_JOINTTYPES = 0;
  21  Vehicle.jointTypeNameToInt = {}
  22  
  23  Vehicle.defaultWidth = 8;
  24  Vehicle.defaultLength = 8;
  25  
  26  Vehicle.debugRendering = false;
  27  
  28  function Vehicle.registerJointType(name)
  29      local key = "JOINTTYPE_"..string.upper(name);
  30      if Vehicle[key] == nil then
  31          Vehicle.NUM_JOINTTYPES = Vehicle.NUM_JOINTTYPES+1;
  32          Vehicle[key] = Vehicle.NUM_JOINTTYPES;
  33          Vehicle.jointTypeNameToInt[name] = Vehicle.NUM_JOINTTYPES;
  34      end;
  35  end;
  36  
  37  Vehicle.registerJointType("implement");
  38  Vehicle.registerJointType("trailer");
  39  Vehicle.registerJointType("trailerLow");
  40  Vehicle.registerJointType("telehandler");
  41  Vehicle.registerJointType("frontloader");
  42  
  43  function Vehicle:new(isServer, isClient, customMt)
  44      --print("new vehicle");
  45      if Vehicle_mt == nil then
  46         Vehicle_mt = Class(Vehicle, Object);
  47      end;
  48  
  49      local mt = customMt;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



  50      if mt == nil then
  51          mt = Vehicle_mt;
  52      end;
  53  
  54      local instance = Object:new(isServer, isClient, mt);
  55      instance.className = "Vehicle";
  56      instance.isAddedToMission = false;
  57      return instance;
  58  end;
  59  
  60  function Vehicle:load(configFile, positionX, offsetY, positionZ, yRot, typeName)
  61  
  62      local instance = self;
  63  
  64      local modName, baseDirectory = getModNameAndBaseDirectory(configFile);
  65  
  66      instance.configFileName = configFile;
  67      instance.baseDirectory = baseDirectory;
  68      instance.customEnvironment = modName;
  69  
  70      local xmlFile = loadXMLFile("TempConfig", configFile);
  71  
  72      local i3dNode = Utils.loadSharedI3DFile(getXMLString(xmlFile, "vehicle.filename"), baseDirectory);
  73      -- todo remove this as soon as the changes are done
  74      instance.rootNode = getChildAt(i3dNode, 0)
  75      --link(getRootNode(), instance.rootNode);
  76  
  77      local tempRootNode = createTransformGroup("tempRootNode");
  78  
  79      instance.components = {};
  80      local numComponents = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.components#count"), 1);
  81  
  82      local rootX, rootY, rootZ;
  83      instance.vehicleNodes = {};
  84      for i=1, numComponents do
  85          table.insert(instance.components, {node=getChildAt(i3dNode, 0)});
  86          if not self.isServer then
  87              setRigidBodyType(instance.components[i].node, "Kinematic");
  88          end;
  89          link(tempRootNode, instance.components[i].node);
  90          if i == 1 then
  91              rootX, rootY, rootZ = getTranslation(instance.components[i].node);
  92          end;
  93          -- the position of the first component is the zero
  94          translate(instance.components[i].node, -rootX, -rootY, -rootZ);
  95          instance.components[i].originalTranslation = {getTranslation(instance.components[i].node)};
  96          instance.components[i].originalRotation = {getRotation(instance.components[i].node)};
  97          instance.vehicleNodes[instance.components[i].node] = instance.components[i].node;
  98      end;
  99      instance.interpolationAlpha = 0;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 100      instance.positionIsDirty = false;
 101  
 102      delete(i3dNode);
 103  
 104      -- position the vehicle
 105      local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, positionX, 300, positionZ);
 106  
 107      setTranslation(tempRootNode, positionX, terrainHeight+offsetY, positionZ);
 108      setRotation(tempRootNode, 0, yRot, 0);
 109  
 110      -- now move the objects to the scene root node
 111      for i=1, numComponents do
 112          local x,y,z = getWorldTranslation(instance.components[i].node);
 113          local rx,ry,rz = getWorldRotation(instance.components[i].node);
 114          local qx, qy, qz, qw = getWorldQuaternion(instance.components[i].node);
 115          setTranslation(instance.components[i].node, x,y,z);
 116          setRotation(instance.components[i].node, rx,ry,rz);
 117          link(getRootNode(), instance.components[i].node);
 118  
 119          instance.components[i].sentTranslation = {x,y,z};
 120          instance.components[i].sentRotation = {qx,qy,qz,qw};
 121          instance.components[i].lastTranslation = {x,y,z};
 122          instance.components[i].lastRotation = {qx,qy,qz,qw};
 123          instance.components[i].targetTranslation = {x,y,z};
 124          instance.components[i].targetRotation = {qx,qy,qz,qw};
 125          instance.components[i].curTranslation = {x,y,z};
 126          instance.components[i].curRotation = {qx,qy,qz,qw};
 127      end;
 128      delete(tempRootNode);
 129  
 130      -- load wheels
 131      instance.maxRotTime = 0;
 132      instance.minRotTime = 0;
 133      instance.autoRotateBackSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.wheels#autoRotateBackSpeed"), 1.0);
 134      instance.wheels = {};
 135      local i = 0;
 136      while true do
 137          local wheelnamei = string.format("vehicle.wheels.wheel(%d)", i);
 138          local wheel = {};
 139          local reprStr = getXMLString(xmlFile, wheelnamei .. "#repr");
 140          if reprStr == nil then
 141              break;
 142          end;
 143          wheel.repr = Utils.indexToObject(instance.components, reprStr);
 144          if wheel.repr == nil then
 145              print("Error: invalid wheel repr " .. reprStr);
 146          else
 147              wheel.rotSpeed = Utils.degToRad(getXMLFloat(xmlFile, wheelnamei .. "#rotSpeed"));
 148              wheel.rotMax = Utils.degToRad(getXMLFloat(xmlFile, wheelnamei .. "#rotMax"));
 149              wheel.rotMin = Utils.degToRad(getXMLFloat(xmlFile, wheelnamei .. "#rotMin"));Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 150              wheel.driveMode = Utils.getNoNil(getXMLInt(xmlFile, wheelnamei .. "#driveMode"), 0);
 151              wheel.driveNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, wheelnamei .. "#driveNode"));
 152              if wheel.driveNode == nil then
 153                  wheel.driveNode = wheel.repr;
 154              end;
 155              wheel.showSteeringAngle = Utils.getNoNil(getXMLBool(xmlFile, wheelnamei .. "#showSteeringAngle"), true);
 156              local radius = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#radius"), 1);
 157              local positionX, positionY, positionZ = getTranslation(wheel.repr);
 158              wheel.deltaY = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#deltaY"), 0.0);
 159              positionY = positionY+wheel.deltaY;
 160              local suspTravel = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#suspTravel"), 0);
 161              local spring = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#spring"), 0)*Vehicle.springScale;
 162              local damper = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#damper"), 0);
 163              local mass = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#mass"), 0.01);
 164              wheel.radius = radius;
 165              wheel.steeringAxleScale = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#steeringAxleScale"), 0);
 166              wheel.steeringAxleRotMax = Utils.degToRad(Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#steeringAxleRotMax"), 20));
 167              wheel.steeringAxleRotMin = Utils.degToRad(Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#steeringAxleRotMin"), -20));
 168              wheel.lateralStiffness = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralStiffness"), 19);
 169              wheel.longitudalStiffness = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalStiffness"), 1);
 170              wheel.steeringAngle = 0;
 171              wheel.hasGroundContact = false;
 172              wheel.axleSpeed = 0;
 173              wheel.hasHandbrake = true;
 174              wheel.node = getParent(wheel.repr);
 175              if self.isServer then
 176                  wheel.wheelShape = createWheelShape(wheel.node, positionX, positionY, positionZ, radius, suspTravel, spring, damper, mass);
 177  
 178                  setWheelShapeTireFunction(wheel.node, wheel.wheelShape, false, 1000000*wheel.lateralStiffness);
 179                  setWheelShapeTireFunction(wheel.node, wheel.wheelShape, true, 1000000*wheel.longitudalStiffness);
 180              end;
 181  
 182              wheel.netInfo = {};
 183              wheel.netInfo.xDrive = 0;
 184              wheel.netInfo.x = positionX;
 185              wheel.netInfo.y = positionY;
 186              wheel.netInfo.z = positionZ;
 187  
 188              wheel.netInfo.yMin = positionY-suspTravel;
 189              wheel.netInfo.yRange = math.ceil(2*suspTravel);
 190              if wheel.netInfo.yRange < 1 then
 191                  -- avoid division by 0
 192                  wheel.netInfo.yRange = 1;
 193              end;
 194  
 195              local maxRotTime = wheel.rotMax/wheel.rotSpeed;
 196              local minRotTime = wheel.rotMin/wheel.rotSpeed;
 197              if minRotTime > maxRotTime then
 198                  local temp = minRotTime;
 199                  minRotTime = maxRotTime;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 200                  maxRotTime = temp;
 201              end;
 202              if maxRotTime > instance.maxRotTime then
 203                  instance.maxRotTime = maxRotTime;
 204              end;
 205              if minRotTime < instance.minRotTime then
 206                  instance.minRotTime = minRotTime;
 207              end;
 208              table.insert(instance.wheels, wheel);
 209          end;
 210          i = i+1;
 211      end;
 212      instance.wheelFrictionScale = 1;
 213  
 214  
 215      instance.dynamicWheelFrictionCosAngleMax = math.cos(math.rad(Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.dynamicWheelFriction#minAngle"), 40)));
 216      instance.dynamicWheelFrictionCosAngleMin = math.cos(math.rad(Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.dynamicWheelFriction#maxAngle"), 50)));
 217      instance.dynamicWheelFrictionMinScale = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.dynamicWheelFriction#minScale"), 0.001);
 218  
 219      instance.lastWheelRpm = 0;
 220      instance.movingDirection = 0;
 221  
 222      instance.steeringAxleNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, "vehicle.steeringAxleNode#index"));
 223      if instance.steeringAxleNode == nil then
 224          instance.steeringAxleNode = instance.components[1].node;
 225      end;
 226  
 227      instance.downForce = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.downForce"), 0);
 228  
 229  
 230      instance.sizeWidth = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.size#width"), Vehicle.defaultWidth);
 231      instance.sizeLength = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.size#length"), Vehicle.defaultLength);
 232      instance.widthOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.size#widthOffset"), 0);
 233      instance.lengthOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.size#lengthOffset"), 0);
 234  
 235      instance.typeDesc = Utils.getXMLI18N(xmlFile, "vehicle.typeDesc", "", "TypeDescription", instance.customEnvironment);
 236  
 237      local numLights = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.lights#count"), 0);
 238      instance.lights = {};
 239      for i=1, numLights do
 240          local lightnamei = string.format("vehicle.lights.light%d", i);
 241          local node = Utils.indexToObject(instance.components, getXMLString(xmlFile, lightnamei .. "#index"));
 242          if node ~= nil then
 243              setVisibility(node, false);
 244              table.insert(instance.lights, node);
 245          end;
 246      end;
 247  
 248      self.lightCoronas = {};
 249      local i = 0;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 250      while true do
 251          local key = string.format("vehicle.lightCoronas.lightCorona(%d)", i);
 252          if not hasXMLProperty(xmlFile, key) then
 253              break;
 254          end;
 255          local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#index"));
 256          if node ~= nil then
 257              setVisibility(node, false);
 258              table.insert(self.lightCoronas, node);
 259          end;
 260          i = i + 1;
 261      end;
 262  
 263      self.lightCones = {};
 264      local i = 0;
 265      while true do
 266          local key = string.format("vehicle.lightCones.lightCone(%d)", i);
 267          if not hasXMLProperty(xmlFile, key) then
 268              break;
 269          end;
 270          local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#index"));
 271          if node ~= nil then
 272              setVisibility(node, false);
 273              table.insert(self.lightCones, node);
 274          end;
 275          i = i + 1;
 276      end;
 277  
 278      self.beaconLights = {};
 279      local i = 0;
 280      while true do
 281          local key = string.format("vehicle.beaconLights.beaconLight(%d)", i);
 282          if not hasXMLProperty(xmlFile, key) then
 283              break;
 284          end;
 285          local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key.."#index"));
 286          local speed = Utils.getNoNil(getXMLFloat(xmlFile, key.."#speed"), 0.02);
 287          if node ~= nil then
 288              setVisibility(node, false);
 289              table.insert(self.beaconLights, {node=node, speed=speed});
 290          end;
 291          i = i + 1;
 292      end;
 293  
 294      local numCuttingAreas = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.cuttingAreas#count"), 0);
 295      instance.cuttingAreas = {}
 296      for i=1, numCuttingAreas do
 297          instance.cuttingAreas[i] = {};
 298          local areanamei = string.format("vehicle.cuttingAreas.cuttingArea%d", i);
 299          instance.cuttingAreas[i].start = Utils.indexToObject(instance.components, getXMLString(xmlFile, areanamei .. "#startIndex"));Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 300          instance.cuttingAreas[i].width = Utils.indexToObject(instance.components, getXMLString(xmlFile, areanamei .. "#widthIndex"));
 301          instance.cuttingAreas[i].height = Utils.indexToObject(instance.components, getXMLString(xmlFile, areanamei .. "#heightIndex"));
 302      end;
 303  
 304      local attachSound = getXMLString(xmlFile, "vehicle.attachSound#file");
 305      if attachSound ~= nil and attachSound ~= "" then
 306          attachSound = Utils.getFilename(attachSound, self.baseDirectory);
 307          instance.attachSound = createSample("attachSound");
 308          loadSample(instance.attachSound, attachSound, false);
 309          instance.attachSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.attachSound#pitchOffset"), 0);
 310      end;
 311  
 312      for i=1, numComponents do
 313          local namei = string.format("vehicle.components.component%d", i);
 314          local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, namei .. "#centerOfMass"));
 315          if x ~= nil and y ~= nil and z ~= nil then
 316              setCenterOfMass(instance.components[i].node, x, y, z);
 317              instance.components[i].centerOfMass = { x, y, z };
 318          end;
 319          local count = getXMLInt(xmlFile, namei .. "#solverIterationCount");
 320          if count ~= nil then
 321              setSolverIterationCount(instance.components[i].node, count);
 322              instance.components[i].solverIterationCount = count;
 323          end;
 324      end;
 325  
 326      instance.componentJoints = {};
 327  
 328      local componentJointI = 0;
 329      while true do
 330          local key = string.format("vehicle.components.joint(%d)", componentJointI);
 331          local index1 = getXMLInt(xmlFile, key.."#component1");
 332          local index2 = getXMLInt(xmlFile, key.."#component2");
 333          local jointIndexStr = getXMLString(xmlFile, key.."#index");
 334          if index1 == nil or index2 == nil or jointIndexStr == nil then
 335              break;
 336          end;
 337          local jointNode = Utils.indexToObject(instance.components, jointIndexStr);
 338          if jointNode ~= nil and jointNode ~= 0 then
 339              local jointDesc = {};
 340              jointDesc.componentIndices = {index1+1, index2+1};
 341              jointDesc.jointNode = jointNode;
 342              if self.isServer then
 343                  local constr = JointConstructor:new();
 344                  if instance.components[index1+1] == nil or instance.components[index2+1] == nil then
 345                      print("Error: invalid joint indices (".. index1.. ", "..index2..") for component joint "..componentJointI.." in '"..self.configFileName.."'");
 346                      break;
 347                  end;
 348                  constr:setActors(instance.components[index1+1].node, instance.components[index2+1].node);
 349                  constr:setJointTransforms(jointNode, jointNode);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 350  
 351                  local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile,  key.."#rotLimit"));
 352                  local rotLimits = {};
 353                  rotLimits[1] = math.rad(Utils.getNoNil(x, 0));
 354                  rotLimits[2] = math.rad(Utils.getNoNil(y, 0));
 355                  rotLimits[3] = math.rad(Utils.getNoNil(z, 0));
 356  
 357                  local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile,  key.."#transLimit"));
 358                  local transLimits = {};
 359                  transLimits[1] = Utils.getNoNil(x, 0);
 360                  transLimits[2] = Utils.getNoNil(y, 0);
 361                  transLimits[3] = Utils.getNoNil(z, 0);
 362  
 363                  for i=1, 3 do
 364                      local rotLimit = rotLimits[i];
 365                      if rotLimit >= 0 then
 366                          constr:setRotationLimit(i-1, -rotLimit, rotLimit);
 367                      end;
 368  
 369                      local transLimit = transLimits[i];
 370                      if transLimit >= 0 then
 371                          constr:setTranslationLimit(i-1, true, -transLimit, transLimit);
 372                      else
 373                          constr:setTranslationLimit(i-1, false, 0, 0);
 374                      end;
 375                  end;
 376  
 377                  if Utils.getNoNil(getXMLBool(xmlFile, key.."#breakable"), false) then
 378                      local force = Utils.getNoNil(getXMLFloat(xmlFile, key.."#breakForce"), 10.);
 379                      local torque = Utils.getNoNil(getXMLFloat(xmlFile, key.."#breakTorque"), 10);
 380                      constr:setBreakable(force, torque);
 381                  end;
 382  
 383                  jointDesc.jointIndex = constr:finalize();
 384              end;
 385              table.insert(instance.componentJoints, jointDesc);
 386          end;
 387          componentJointI = componentJointI +1;
 388      end;
 389  
 390      local collisionPairI = 0;
 391      while true do
 392          local key = string.format("vehicle.components.collisionPair(%d)", collisionPairI);
 393          if not hasXMLProperty(xmlFile, key) then
 394              break;
 395          end;
 396          local enabled = getXMLBool(xmlFile, key.."#enabled");
 397          local index1 = getXMLInt(xmlFile, key.."#component1");
 398          local index2 = getXMLInt(xmlFile, key.."#component2");
 399          if index1 ~= nil and index2 ~= nil and enabled ~= nil thenCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 400              local component1 = instance.components[index1+1];
 401              local component2 = instance.components[index2+1];
 402              if component1 ~= nil and component2 ~= nil then
 403                  if not enabled then
 404                      setPairCollision(component1.node, component2.node, false);
 405                  end;
 406              end;
 407          end;
 408          collisionPairI = collisionPairI +1;
 409      end;
 410  
 411      instance.attacherJoints = {};
 412  
 413      local i=0;
 414      while true do
 415          local baseName = string.format("vehicle.attacherJoints.attacherJoint(%d)", i);
 416          local index = getXMLString(xmlFile, baseName.. "#index");
 417          if index == nil then
 418              break;
 419          end;
 420          local object = Utils.indexToObject(instance.components, index);
 421          if object ~= nil then
 422              local entry = {};
 423              entry.jointTransform = object;
 424  
 425              local jointTypeStr = getXMLString(xmlFile, baseName.. "#jointType");
 426              local jointType;
 427              if jointTypeStr ~= nil then
 428                  jointType = Vehicle.jointTypeNameToInt[jointTypeStr];
 429                  if jointType == nil then
 430                      print("Warning: invalid jointType " .. jointTypeStr);
 431                  end;
 432              end;
 433              if jointType == nil then
 434                  jointType = Vehicle.JOINTTYPE_IMPLEMENT;
 435              end;
 436              entry.jointType = jointType;
 437              entry.allowsJointLimitMovement = Utils.getNoNil(getXMLBool(xmlFile, baseName.."#allowsJointLimitMovement"), true);
 438              entry.allowsLowering = Utils.getNoNil(getXMLBool(xmlFile, baseName.."#allowsLowering"), true);
 439  
 440              local x, y, z;
 441              local rotationNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. "#rotationNode"));
 442              if rotationNode ~= nil then
 443                  entry.rotationNode = rotationNode;
 444                  x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#maxRot"));
 445                  entry.maxRot = {};
 446                  entry.maxRot[1] = math.rad(Utils.getNoNil(x, 0));
 447                  entry.maxRot[2] = math.rad(Utils.getNoNil(y, 0));
 448                  entry.maxRot[3] = math.rad(Utils.getNoNil(z, 0));
 449  Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 450                  x, y, z = getRotation(rotationNode);
 451                  entry.minRot = {x,y,z};
 452              end;
 453              local rotationNode2 = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. "#rotationNode2"));
 454              if rotationNode2 ~= nil then
 455                  entry.rotationNode2 = rotationNode2;
 456                  x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#maxRot2"));
 457                  entry.maxRot2 = {};
 458                  entry.maxRot2[1] = math.rad(Utils.getNoNil(x, 0));
 459                  entry.maxRot2[2] = math.rad(Utils.getNoNil(y, 0));
 460                  entry.maxRot2[3] = math.rad(Utils.getNoNil(z, 0));
 461  
 462                  x, y, z = getRotation(rotationNode2);
 463                  entry.minRot2 = {x,y,z};
 464              end;
 465  
 466  
 467              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#maxRotLimit"));
 468              entry.maxRotLimit = {};
 469              entry.maxRotLimit[1] = math.rad(math.abs(Utils.getNoNil(x, 0)));
 470              entry.maxRotLimit[2] = math.rad(math.abs(Utils.getNoNil(y, 0)));
 471              entry.maxRotLimit[3] = math.rad(math.abs(Utils.getNoNil(z, 0)));
 472  
 473              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#minRotLimit"));
 474              entry.minRotLimit = {};
 475              entry.minRotLimit[1] = math.rad(math.abs(Utils.getNoNil(x, 0)));
 476              entry.minRotLimit[2] = math.rad(math.abs(Utils.getNoNil(y, 0)));
 477              entry.minRotLimit[3] = math.rad(math.abs(Utils.getNoNil(z, 0)));
 478  
 479              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#maxTransLimit"));
 480              entry.maxTransLimit = {};
 481              entry.maxTransLimit[1] = math.abs(Utils.getNoNil(x, 0));
 482              entry.maxTransLimit[2] = math.abs(Utils.getNoNil(y, 0));
 483              entry.maxTransLimit[3] = math.abs(Utils.getNoNil(z, 0));
 484  
 485              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#minTransLimit"));
 486              entry.minTransLimit = {};
 487              entry.minTransLimit[1] = math.abs(Utils.getNoNil(x, 0));
 488              entry.minTransLimit[2] = math.abs(Utils.getNoNil(y, 0));
 489              entry.minTransLimit[3] = math.abs(Utils.getNoNil(z, 0));
 490  
 491              entry.moveTime = Utils.getNoNil(getXMLFloat(xmlFile, baseName.."#moveTime"), 0.5)*1000;
 492  
 493              local rotationNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. ".topArm#rotationNode"));
 494              local translationNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. ".topArm#translationNode"));
 495              local referenceNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. ".topArm#referenceNode"));
 496              if rotationNode ~= nil then
 497                  local topArm = {};
 498                  topArm.rotationNode = rotationNode;
 499                  topArm.rotX, topArm.rotY, topArm.rotZ = getRotation(rotationNode);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 500                  if translationNode ~= nil and referenceNode ~= nil then
 501                      topArm.translationNode = translationNode;
 502  
 503                      local x,y,z = getTranslation(translationNode);
 504                      if math.abs(x) >= 0.0001 or math.abs(y) >= 0.0001 or math.abs(z) >= 0.0001 then
 505                          print("Warning: translation of topArm of attacherJoint "..i.." is not 0/0/0 in '"..self.configFileName.."'");
 506                      end;
 507                      local ax, ay, az = getWorldTranslation(referenceNode);
 508                      local bx, by, bz = getWorldTranslation(translationNode);
 509                      topArm.referenceDistance = Utils.vector3Length(ax-bx, ay-by, az-bz);
 510                  end;
 511                  topArm.zScale = Utils.sign(Utils.getNoNil(getXMLFloat(xmlFile, baseName.. ".topArm#zScale"), 1));
 512                  entry.topArm = topArm;
 513              end;
 514              local rotationNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. ".bottomArm#rotationNode"));
 515              local translationNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. ".bottomArm#translationNode"));
 516              local referenceNode = Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.. ".bottomArm#referenceNode"));
 517              if rotationNode ~= nil then
 518                  local bottomArm = {};
 519                  bottomArm.rotationNode = rotationNode;
 520                  bottomArm.rotX, bottomArm.rotY, bottomArm.rotZ = getRotation(rotationNode);
 521                  if translationNode ~= nil and referenceNode ~= nil then
 522                      bottomArm.translationNode = translationNode;
 523  
 524                      local x,y,z = getTranslation(translationNode);
 525                      if math.abs(x) >= 0.0001 or math.abs(y) >= 0.0001 or math.abs(z) >= 0.0001 then
 526                          print("Warning: translation of bottomArm of attacherJoint "..i.." is not 0/0/0 in '"..self.configFileName.."'");
 527                      end;
 528                      local ax, ay, az = getWorldTranslation(referenceNode);
 529                      local bx, by, bz = getWorldTranslation(translationNode);
 530                      bottomArm.referenceDistance = Utils.vector3Length(ax-bx, ay-by, az-bz);
 531                  end;
 532                  bottomArm.zScale = Utils.sign(Utils.getNoNil(getXMLFloat(xmlFile, baseName.. ".bottomArm#zScale"), 1));
 533                  entry.bottomArm = bottomArm;
 534              end;
 535              entry.rootNode = Utils.getNoNil(Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.."#rootNode")), instance.components[1].node);
 536              entry.jointIndex = 0;
 537              table.insert(instance.attacherJoints, entry);
 538          end;
 539          i = i+1;
 540      end;
 541  
 542      local i=0;
 543      while true do
 544          local baseName = string.format("vehicle.trailerAttacherJoints.trailerAttacherJoint(%d)", i);
 545          local index = getXMLString(xmlFile, baseName.. "#index");
 546          if index == nil then
 547              break;
 548          end;
 549          local object = Utils.indexToObject(instance.components, index);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 550          if object ~= nil then
 551              local entry = {};
 552              entry.jointTransform = object;
 553              entry.jointIndex = 0;
 554              local isLow = Utils.getNoNil(getXMLBool(xmlFile, baseName.."#low"), false);
 555  
 556              if isLow then
 557                  entry.jointType = Vehicle.JOINTTYPE_TRAILERLOW;
 558              else
 559                  entry.jointType = Vehicle.JOINTTYPE_TRAILER;
 560              end;
 561  
 562              entry.allowsJointLimitMovement = Utils.getNoNil(getXMLBool(xmlFile, baseName.."#allowsJointLimitMovement"), false);
 563              entry.allowsLowering = false;
 564  
 565              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#maxRotLimit"));
 566              entry.maxRotLimit = {};
 567              entry.maxRotLimit[1] = Utils.degToRad(math.abs(Utils.getNoNil(x, 10)));
 568              entry.maxRotLimit[2] = Utils.degToRad(math.abs(Utils.getNoNil(y, 50)));
 569              entry.maxRotLimit[3] = Utils.degToRad(math.abs(Utils.getNoNil(z, 50)));
 570  
 571              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#minRotLimit"));
 572              entry.minRotLimit = {};
 573              entry.minRotLimit[1] = math.rad(math.abs(Utils.getNoNil(x, 0)));
 574              entry.minRotLimit[2] = math.rad(math.abs(Utils.getNoNil(y, 0)));
 575              entry.minRotLimit[3] = math.rad(math.abs(Utils.getNoNil(z, 0)));
 576  
 577              x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#maxTransLimit"));
 578              entry.maxTransLimit = {};
 579              entry.maxTransLimit[1] = math.abs(Utils.getNoNil(x, 0));
 580              entry.maxTransLimit[2] = math.abs(Utils.getNoNil(y, 0));
 581              entry.maxTransLimit[3] = math.abs(Utils.getNoNil(z, 0));
 582  
 583              local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.."#minTransLimit"));
 584              entry.minTransLimit = {};
 585              entry.minTransLimit[1] = math.abs(Utils.getNoNil(x, 0));
 586              entry.minTransLimit[2] = math.abs(Utils.getNoNil(y, 0));
 587              entry.minTransLimit[3] = math.abs(Utils.getNoNil(z, 0));
 588  
 589              entry.rootNode = Utils.getNoNil(Utils.indexToObject(instance.components, getXMLString(xmlFile, baseName.."#rootNode")), instance.components[1].node);
 590  
 591              table.insert(instance.attacherJoints, entry);
 592          end;
 593          i = i+1;
 594      end;
 595  
 596      instance.attachedImplements = {};
 597      instance.selectedImplement = 0;
 598  
 599  Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 600      instance.requiredDriveMode = 1;
 601      instance.steeringAxleAngle = 0;
 602      instance.rotatedTime = 0;
 603      instance.firstTimeRun = false;
 604  
 605      instance.lightsActive = false;
 606      instance.realLightsActive = false;
 607      instance.lightConesActive = false;
 608      instance.lastPosition = nil; -- = {0,0,0};
 609      instance.lastSpeed = 0;
 610      instance.lastSpeedReal = 0;
 611      instance.lastMovedDistance = 0;
 612      instance.speedDisplayDt = 0;
 613      instance.speedDisplayScale = 1;
 614      instance.isBroken = false;
 615  
 616      instance.isVehicleSaved = true;
 617  
 618      instance.checkSpeedLimit = true;
 619  
 620      instance.lastSoundSpeed = 0;
 621  
 622      instance.time = 0;
 623  
 624      instance.forceIsActive = false;
 625  
 626      instance.vehicleDirtyFlag = instance.nextDirtyFlag;
 627      instance.nextDirtyFlag = instance.vehicleDirtyFlag*2;
 628  
 629      instance.typeName = typeName;
 630      local typeDef = VehicleTypeUtil.vehicleTypes[typeName];
 631      instance.specializations = typeDef.specializations;
 632      for i=1, table.getn(instance.specializations) do
 633          instance.specializations[i].load(instance, xmlFile);
 634      end;
 635      for i=1, table.getn(instance.specializations) do
 636          if instance.specializations[i].postLoad ~= nil then
 637              instance.specializations[i].postLoad(instance, xmlFile);
 638          end;
 639      end;
 640  
 641      instance.componentsVisibility = true;
 642  
 643      delete(xmlFile);
 644  
 645      --return instance;
 646  end;
 647  
 648  function Vehicle:delete()
 649      for i=table.getn(self.attachedImplements), 1, -1 doCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 650          self:detachImplement(1, true);
 651      end;
 652  
 653      for i=table.getn(self.specializations), 1, -1 do
 654          self.specializations[i].delete(self);
 655      end;
 656  
 657      if self.attachSound ~= nil then
 658          delete(self.attachSound);
 659      end;
 660  
 661      if self.isServer then
 662          for k,v in pairs(self.componentJoints) do
 663              removeJoint(v.jointIndex);
 664          end;
 665      end;
 666  
 667      for k,v in pairs(self.components) do
 668          delete(v.node);
 669      end;
 670  
 671      Vehicle:superClass().delete(self);
 672  end;
 673  
 674  function Vehicle:readStream(streamId, connection)
 675      Vehicle:superClass().readStream(self, streamId);
 676      local configFile = Utils.convertFromNetworkFilename(streamReadString(streamId));
 677  
 678      local typeName = streamReadString(streamId);
 679  
 680      if self.configFileName == nil then
 681          self:load(configFile, 0,0,0,0,typeName);
 682      end;
 683      for i=1, table.getn(self.components) do
 684          local x=streamReadFloat32(streamId);
 685          local y=streamReadFloat32(streamId);
 686          local z=streamReadFloat32(streamId);
 687          local x_rot=streamReadFloat32(streamId)
 688          local y_rot=streamReadFloat32(streamId);
 689          local z_rot=streamReadFloat32(streamId);
 690          local w_rot=streamReadFloat32(streamId);
 691          self:setWorldPositionQuaternion(x,y,z, x_rot,y_rot,z_rot,w_rot, i);
 692  
 693          self.components[i].lastTranslation = {x,y,z};
 694          self.components[i].lastRotation = {x_rot,y_rot,z_rot,w_rot};
 695          self.components[i].targetTranslation = {x,y,z};
 696          self.components[i].targetRotation = {x_rot,y_rot,z_rot,w_rot};
 697          self.components[i].curTranslation = {x,y,z};
 698          self.components[i].curRotation = {x_rot,y_rot,z_rot,w_rot};
 699      end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 700      self.interpolationAlpha = 0;
 701      self.positionIsDirty = false;
 702      for i=1, table.getn(self.wheels) do
 703          local wheel = self.wheels[i];
 704          wheel.netInfo.x = streamReadFloat32(streamId);
 705          wheel.netInfo.y = streamReadFloat32(streamId);
 706          wheel.netInfo.z = streamReadFloat32(streamId);
 707          wheel.netInfo.xDrive = streamReadFloat32(streamId);
 708      end;
 709  
 710      local numImplements = streamReadInt8(streamId);
 711      for i=1, numImplements do
 712          local implementId = streamReadInt32(streamId);
 713          local jointDescIndex = streamReadInt8(streamId);
 714          local moveDown = streamReadBool(streamId);
 715          local object = networkGetObject(implementId)
 716          if object ~= nil then
 717              self:attachImplement(object, jointDescIndex, true);
 718              self:setJointMoveDown(jointDescIndex, moveDown, true);
 719          end;
 720      end;
 721      for k,v in pairs(self.specializations) do
 722          if v.readStream ~= nil then
 723              v.readStream(self, streamId, connection);
 724          end;
 725      end;
 726  end;
 727  
 728  function Vehicle:writeStream(streamId, connection)
 729      Vehicle:superClass().writeStream(self, streamId);
 730      streamWriteString(streamId, Utils.convertToNetworkFilename(self.configFileName));
 731      streamWriteString(streamId, self.typeName);
 732      for i=1, table.getn(self.components) do
 733          local x,y,z = getTranslation(self.components[i].node);
 734          local x_rot,y_rot,z_rot,w_rot=getQuaternion(self.components[i].node);
 735          streamWriteFloat32(streamId, x);
 736          streamWriteFloat32(streamId, y);
 737          streamWriteFloat32(streamId, z);
 738          streamWriteFloat32(streamId, x_rot);
 739          streamWriteFloat32(streamId, y_rot);
 740          streamWriteFloat32(streamId, z_rot);
 741          streamWriteFloat32(streamId, w_rot);
 742      end;
 743      for i=1, table.getn(self.wheels) do
 744          local wheel = self.wheels[i];
 745          streamWriteFloat32(streamId, wheel.netInfo.x);
 746          streamWriteFloat32(streamId, wheel.netInfo.y);
 747          streamWriteFloat32(streamId, wheel.netInfo.z);
 748          streamWriteFloat32(streamId, wheel.netInfo.xDrive);
 749      end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 750  
 751      -- write attached implements
 752      streamWriteInt8(streamId, table.getn(self.attachedImplements));
 753      for i=1, table.getn(self.attachedImplements) do
 754          local implement = self.attachedImplements[i];
 755          local jointDescIndex = implement.jointDescIndex;
 756          local jointDesc = self.attacherJoints[jointDescIndex];
 757          local moveDown = jointDesc.moveDown;
 758          streamWriteInt32(streamId, networkGetObjectId(implement.object));
 759          streamWriteInt8(streamId, jointDescIndex);
 760          streamWriteBool(streamId, moveDown);
 761      end;
 762      for k,v in pairs(self.specializations) do
 763          if v.writeStream ~= nil then
 764              v.writeStream(self, streamId, connection);
 765          end;
 766      end;
 767  end;
 768  
 769  function Vehicle:readUpdateStream(streamId, timestamp, connection)
 770      if connection.isServer then
 771          local hasUpdate = streamReadBool(streamId);
 772          if hasUpdate then
 773              for i=1, table.getn(self.components) do
 774                  local x=streamReadFloat32(streamId);
 775                  local y=streamReadFloat32(streamId);
 776                  local z=streamReadFloat32(streamId);
 777                  local x_rot = Utils.readCompressedAngle(streamId);
 778                  local y_rot = Utils.readCompressedAngle(streamId);
 779                  local z_rot = Utils.readCompressedAngle(streamId);
 780  
 781                  local x_rot,y_rot,z_rot,w_rot = mathEulerToQuaternion(x_rot,y_rot,z_rot);
 782                  --local x_rot=streamReadFloat32(streamId)
 783                  --local y_rot=streamReadFloat32(streamId);
 784                  --local z_rot=streamReadFloat32(streamId);
 785                  --local w_rot=streamReadFloat32(streamId);
 786                  self.components[i].targetTranslation = {x,y,z};
 787                  self.components[i].targetRotation = {x_rot,y_rot,z_rot,w_rot};
 788  
 789                  local trans = self.components[i].curTranslation;
 790                  local rot = self.components[i].curRotation;
 791                  self.components[i].lastTranslation = {trans[1], trans[2], trans[3]};
 792                  self.components[i].lastRotation = {rot[1], rot[2], rot[3], rot[4]};
 793                  --self:setWorldPosition(x,y,z, x_rot,y_rot,z_rot, i)
 794              end;
 795              self.interpolationAlpha = 0;
 796              self.positionIsDirty = true;
 797  
 798              -- read netinfo
 799              for i=1, table.getn(self.wheels) doCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 800                  local wheel = self.wheels[i];
 801                  local y = streamReadUInt8(streamId);
 802                  wheel.netInfo.y = y / 255 * wheel.netInfo.yRange + wheel.netInfo.yMin;
 803                  --wheel.netInfo.y = streamReadFloat32(streamId);
 804                  wheel.netInfo.xDrive = streamReadFloat32(streamId);
 805              end;
 806              self.rotatedTime = streamReadFloat32(streamId);
 807          end;
 808      end;
 809  
 810      for k,v in pairs(self.specializations) do
 811          if v.readUpdateStream ~= nil then
 812              v.readUpdateStream(self, streamId, timestamp, connection);
 813          end;
 814      end;
 815  end;
 816  
 817  function Vehicle:writeUpdateStream(streamId, connection, dirtyMask)
 818      if not connection.isServer then
 819          if bitAND(dirtyMask, self.vehicleDirtyFlag) ~= 0 then
 820              streamWriteBool(streamId, true);
 821              for i=1, table.getn(self.components) do
 822                  local x,y,z=getTranslation(self.components[i].node)
 823                  --local x_rot,y_rot,z_rot,w_rot=getQuaternion(self.components[i].node);
 824                  local x_rot,y_rot,z_rot = getRotation(self.components[i].node);
 825                  streamWriteFloat32(streamId, x);
 826                  streamWriteFloat32(streamId, y);
 827                  streamWriteFloat32(streamId, z);
 828                  Utils.writeCompressedAngle(streamId, x_rot);
 829                  Utils.writeCompressedAngle(streamId, y_rot);
 830                  Utils.writeCompressedAngle(streamId, z_rot);
 831                  --[[streamWriteFloat32(streamId, x_rot);
 832                  streamWriteFloat32(streamId, y_rot);
 833                  streamWriteFloat32(streamId, z_rot);
 834                  streamWriteFloat32(streamId, w_rot);]]
 835              end;
 836              for i=1, table.getn(self.wheels) do
 837                  local wheel = self.wheels[i];
 838                  streamWriteUInt8(streamId, Utils.clamp((wheel.netInfo.y - wheel.netInfo.yMin) / wheel.netInfo.yRange * 255, 0, 255));
 839                  --streamWriteFloat32(streamId, wheel.netInfo.y);
 840                  streamWriteFloat32(streamId, wheel.netInfo.xDrive);
 841              end;
 842              streamWriteFloat32(streamId, self.rotatedTime);
 843          else
 844              streamWriteBool(streamId, false);
 845          end;
 846      end;
 847  
 848      for k,v in pairs(self.specializations) do
 849          if v.writeUpdateStream ~= nil thenCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 850              v.writeUpdateStream(self, streamId, connection, dirtyMask);
 851          end;
 852      end;
 853  end;
 854  
 855  function Vehicle:testScope(x,y,z, coeff)
 856      local vx,vy,vz = getTranslation(self.components[1].node);
 857      local distanceSq = Utils.vector3LengthSq(vx-x, vy-y, vz-z);
 858      for k,v in pairs(self.specializations) do
 859          if v.testScope ~= nil then
 860              if not v.testScope(self, x,y,z, coeff, distanceSq) then
 861                  return false;
 862              end;
 863          end;
 864      end;
 865      --return distance < 10;
 866      return true;
 867  end;
 868  
 869  function Vehicle:getUpdatePriority(skipCount, x, y, z, coeff, connection)
 870      if self:getOwner() == connection then
 871          return 50;
 872      end;
 873      local x1, y1, z1 = getTranslation(self.components[1].node);
 874      local dist =  Utils.vector3Length(x1-x, y1-y, z1-z);
 875      local clipDist = getClipDistance(self.components[1].node)*coeff;
 876      return (1-dist/clipDist)* 0.8 + 0.5*skipCount * 0.2;
 877  end;
 878  
 879  function Vehicle:onGhostRemove()
 880      --print("remove ghost "..getName(self.components[1].node));
 881      --setVisibility(self.components[1].node, false);
 882      for k,v in pairs(self.specializations) do
 883          if v.onGhostRemove ~= nil then
 884              v.onGhostRemove(self);
 885          end;
 886      end;
 887  end;
 888  
 889  function Vehicle:onGhostAdd()
 890      --print("add ghost "..getName(self.components[1].node));
 891      --setVisibility(self.components[1].node, true);
 892      for k,v in pairs(self.specializations) do
 893          if v.onGhostAdd ~= nil then
 894              v.onGhostAdd(self);
 895          end;
 896      end;
 897  end;
 898  
 899  function Vehicle:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 900      local findPlace = resetVehicles;
 901      if not findPlace then
 902          local isAbsolute = Utils.getNoNil(getXMLBool(xmlFile, key.."#isAbsolute"), false);
 903          if isAbsolute then
 904              local pos = {};
 905              for i=1, table.getn(self.components) do
 906                  local componentKey = key..".component"..i;
 907                  local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, componentKey.."#position"));
 908                  local xRot,yRot,zRot = Utils.getVectorFromString(getXMLString(xmlFile, componentKey.."#rotation"));
 909                  if x == nil or y == nil or z == nil or xRot == nil or yRot == nil or zRot == nil then
 910                      findPlace = true;
 911                      break;
 912                  end;
 913                  pos[i] = {x=x, y=y, z=z, xRot=xRot, yRot=yRot, zRot=zRot};
 914              end;
 915              if not findPlace then
 916                  for i=1, table.getn(self.components) do
 917                      local p = pos[i];
 918                      self:setWorldPosition(p.x,p.y,p.z, p.xRot,p.yRot,p.zRot, i);
 919                  end;
 920              end;
 921          else
 922              local yOffset = getXMLFloat(xmlFile, key.."#yOffset");
 923              local xPosition = getXMLFloat(xmlFile, key.."#xPosition");
 924              local zPosition = getXMLFloat(xmlFile, key.."#zPosition");
 925              local yRotation = getXMLFloat(xmlFile, key.."#yRotation");
 926              if yOffset == nil or xPosition == nil or zPosition == nil or yRotation == nil then
 927                  findPlace = true;
 928              else
 929                  self:setRelativePosition(xPosition, yOffset, zPosition, math.rad(yRotation));
 930              end;
 931          end;
 932      end;
 933      if findPlace then
 934          if resetVehicles then
 935              local x, y, z, place, width, offset = PlacementUtil.getPlace(g_currentMission.loadSpawnPlaces, self.sizeWidth, self.sizeLength, self.widthOffset, self.lengthOffset, g_currentMission.usedLoadPlaces);
 936              if x ~= nil then
 937                  local yRot = Utils.getYRotationFromDirection(place.dirPerpX, place.dirPerpZ);
 938                  PlacementUtil.markPlaceUsed(g_currentMission.usedLoadPlaces, place, width);
 939                  self:setRelativePosition(x, offset, z, yRot);
 940              else
 941                  return BaseMission.VEHICLE_LOAD_ERROR;
 942              end;
 943          else
 944              return BaseMission.VEHICLE_LOAD_DELAYED;
 945          end;
 946      end;
 947      for k,v in pairs(self.specializations) do
 948          if v.loadFromAttributesAndNodes ~= nil then
 949              local r = v.loadFromAttributesAndNodes(self, xmlFile, key, resetVehicles)Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



 950              if r ~= BaseMission.VEHICLE_LOAD_OK then
 951                  return r;
 952              end;
 953          end;
 954      end;
 955      return BaseMission.VEHICLE_LOAD_OK;
 956  end;
 957  
 958  function Vehicle:getSaveAttributesAndNodes(nodeIdent)
 959  
 960      local attributes = 'isAbsolute="true"';
 961      local nodes = "";
 962      if not self.isBroken then
 963          for i=1, table.getn(self.components) do
 964              if i>1 then
 965                  nodes = nodes.."\n";
 966              end;
 967              local node = self.components[i].node;
 968              local x,y,z = getTranslation(node);
 969              local xRot,yRot,zRot = getRotation(node);
 970              nodes = nodes.. nodeIdent..'<component'..i..' position="'..x..' '..y..' '..z..'" rotation="'..xRot..' '..yRot..' '..zRot..'" />';
 971          end;
 972      end;
 973      for k,v in pairs(self.specializations) do
 974          if v.getSaveAttributesAndNodes ~= nil then
 975              local specAttributes, specNodes = v.getSaveAttributesAndNodes(self, nodeIdent);
 976              if specAttributes ~= nil and specAttributes ~= "" then
 977                  attributes = attributes.." "..specAttributes;
 978              end;
 979              if specNodes ~= nil and specNodes ~= "" then
 980                  nodes = nodes.."\n"..specNodes;
 981              end;
 982          end;
 983      end;
 984      return attributes, nodes;
 985  end;
 986  
 987  function Vehicle:setRelativePosition(positionX, offsetY, positionZ, yRot)
 988      local tempRootNode = createTransformGroup("tempRootNode");
 989  
 990      local numComponents = table.getn(self.components);
 991  
 992      for i=1, numComponents do
 993          link(tempRootNode, self.components[i].node);
 994          setTranslation(self.components[i].node, unpack(self.components[i].originalTranslation));
 995          setRotation(self.components[i].node, unpack(self.components[i].originalRotation));
 996      end;
 997  
 998      -- position the vehicle
 999      local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, positionX, 300, positionZ);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1000  
1001      setTranslation(tempRootNode, positionX, terrainHeight+offsetY, positionZ);
1002      setRotation(tempRootNode, 0, yRot, 0);
1003  
1004      -- now move the objects to the scene root node
1005      for i=1, numComponents do
1006          local x,y,z = getWorldTranslation(self.components[i].node);
1007          local rx,ry,rz = getWorldRotation(self.components[i].node);
1008          local qx, qy, qz, qw = getWorldQuaternion(self.components[i].node);
1009          setTranslation(self.components[i].node, x,y,z);
1010          setRotation(self.components[i].node, rx,ry,rz);
1011          link(getRootNode(), self.components[i].node);
1012  
1013          self.components[i].lastTranslation = {x,y,z};
1014          self.components[i].lastRotation = {qx,qy,qz,qw};
1015          self.components[i].targetTranslation = {x,y,z};
1016          self.components[i].targetRotation = {qx,qy,qz,qw};
1017          self.components[i].curTranslation = {x,y,z};
1018          self.components[i].curRotation = {qx,qy,qz,qw};
1019      end;
1020      self.interpolationAlpha = 0;
1021      self.positionIsDirty = false;
1022      delete(tempRootNode);
1023  
1024      for k,v in pairs(self.specializations) do
1025          if v.setRelativePosition ~= nil then
1026              v.setRelativePosition(self, positionX, offsetY, positionZ, yRot);
1027          end;
1028      end;
1029  end;
1030  
1031  function Vehicle:setWorldPosition(x,y,z, xRot,yRot,zRot, i)
1032      setTranslation(self.components[i].node, x,y,z);
1033      setRotation(self.components[i].node, xRot,yRot,zRot);
1034  end;
1035  
1036  function Vehicle:setWorldPositionQuaternion(x,y,z, xRot,yRot,zRot,wRot, i)
1037      setTranslation(self.components[i].node, x,y,z);
1038      setQuaternion(self.components[i].node, xRot,yRot,zRot,wRot);
1039  end;
1040  
1041  function Vehicle:addNodeVehicleMapping(list)
1042      for k,v in pairs(self.components) do
1043          list[v.node] = self;
1044      end;
1045  
1046      for k,v in pairs(self.specializations) do
1047          if v.addNodeVehicleMapping ~= nil then
1048              v.addNodeVehicleMapping(self, list);
1049          end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1050      end;
1051  end;
1052  
1053  function Vehicle:removeNodeVehicleMapping(list)
1054      for k,v in pairs(self.components) do
1055          list[v.node] = nil;
1056      end;
1057  
1058      for k,v in pairs(self.specializations) do
1059          if v.removeNodeVehicleMapping ~= nil then
1060              v.removeNodeVehicleMapping(self, list);
1061          end;
1062      end;
1063  end;
1064  
1065  function Vehicle:mouseEvent(posX, posY, isDown, isUp, button)
1066  
1067      for k,v in pairs(self.specializations) do
1068          v.mouseEvent(self, posX, posY, isDown, isUp, button);
1069      end;
1070  
1071      if self.selectedImplement ~= 0 then
1072          self.attachedImplements[self.selectedImplement].object:mouseEvent(posX, posY, isDown, isUp, button);
1073      end;
1074  end;
1075  
1076  function Vehicle:keyEvent(unicode, sym, modifier, isDown)
1077  
1078      for k,v in pairs(self.specializations) do
1079          v.keyEvent(self, unicode, sym, modifier, isDown);
1080      end;
1081  
1082      if self.selectedImplement ~= 0 then
1083          self.attachedImplements[self.selectedImplement].object:keyEvent(unicode, sym, modifier, isDown);
1084      end;
1085  end;
1086  
1087  function Vehicle:update(dt)
1088  
1089      if not self.isServer and self.positionIsDirty then
1090          self.interpolationAlpha = math.min(self.interpolationAlpha + dt/45, 1.2);
1091          if self.interpolationAlpha == 1.5 then
1092              self.positionIsDirty = false;
1093          end;
1094          for i=1, table.getn(self.components) do
1095              for c=1,3 do
1096                  self.components[i].curTranslation[c] = self.components[i].lastTranslation[c]*(1-self.interpolationAlpha) + self.components[i].targetTranslation[c]*self.interpolationAlpha;
1097              end;
1098              local rot1 = self.components[i].lastRotation;
1099              local rot2 = self.components[i].targetRotation;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1100              local x,y,z,w = Utils.nlerpQuaternionShortestPath(rot1[1], rot1[2], rot1[3], rot1[4], rot2[1], rot2[2], rot2[3], rot2[4], self.interpolationAlpha);
1101              self.components[i].curRotation = {x,y,z,w};
1102              local trans = self.components[i].curTranslation;
1103              self:setWorldPositionQuaternion(trans[1], trans[2], trans[3], x,y,z,w, i);
1104          end;
1105      end;
1106  
1107      self.time = self.time + dt;
1108  
1109      self.isActive = self:getIsActive();
1110  
1111      self.speedDisplayDt = self.speedDisplayDt + dt;
1112      self.lastMovedDistance = 0;
1113      if self.speedDisplayDt > 100 then
1114          local newX, newY, newZ = getWorldTranslation(self.components[1].node);
1115          if self.lastPosition == nil then
1116              self.lastPosition = {newX, newY, newZ};
1117          end;
1118          local dx, dy, dz = worldDirectionToLocal(self.components[1].node, newX-self.lastPosition[1], newY-self.lastPosition[2], newZ-self.lastPosition[3]);
1119          if dz > 0.01 then
1120              self.movingDirection = 1;
1121          elseif dz < -0.01 then
1122              self.movingDirection = -1;
1123          else
1124              self.movingDirection = 0;
1125          end;
1126          self.lastMovedDistance = Utils.vector3Length(dx, dy, dz);
1127          self.lastSpeedReal = (self.lastMovedDistance/100);
1128          self.lastSpeed = self.lastSpeed*0.85 + self.lastSpeedReal*0.15;
1129          self.lastPosition = {newX, newY, newZ};
1130          self.speedDisplayDt = self.speedDisplayDt - 100;
1131      end;
1132  
1133      if self.downForce ~= 0 and self.isServer then
1134          local worldX,worldY,worldZ = localDirectionToWorld(self.components[1].node, 0, -self.downForce*dt/1000, 0);
1135          addForce(self.components[1].node, worldX, worldY, worldZ, 0, 0, 0, true);
1136      end;
1137  
1138      if self.isActive then
1139          for k, implement in pairs(self.attachedImplements) do
1140              local jointDesc = self.attacherJoints[implement.jointDescIndex];
1141              local attacherJoint = implement.object.attacherJoint;
1142              if jointDesc.topArm ~= nil and attacherJoint.topReferenceNode ~= nil then
1143                  local ax, ay, az = getWorldTranslation(jointDesc.topArm.rotationNode);
1144                  local bx, by, bz = getWorldTranslation(attacherJoint.topReferenceNode);
1145  
1146                  local x, y, z = worldDirectionToLocal(getParent(jointDesc.topArm.rotationNode), bx-ax, by-ay, bz-az);
1147                  local upX, upY, upZ = 0,1,0;
1148                  if math.abs(y) > 0.99*Utils.vector3Length(x, y, z) then
1149                      -- direction and up is parallelCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1150                      upY = 0;
1151                      if y > 0 then
1152                          upZ = 1;
1153                      else
1154                          upZ = -1;
1155                      end;
1156                  end;
1157  
1158                  setDirection(jointDesc.topArm.rotationNode, x*jointDesc.topArm.zScale, y*jointDesc.topArm.zScale, z*jointDesc.topArm.zScale, upX, upY, upZ);
1159                  if jointDesc.topArm.translationNode ~= nil then
1160                      local distance = Utils.vector3Length(ax-bx, ay-by, az-bz);
1161                      setTranslation(jointDesc.topArm.translationNode, 0, 0, (distance-jointDesc.topArm.referenceDistance)*jointDesc.topArm.zScale);
1162                  end;
1163              end;
1164              if jointDesc.bottomArm ~= nil then
1165                  local ax, ay, az = getWorldTranslation(jointDesc.bottomArm.rotationNode);
1166                  local bx, by, bz = getWorldTranslation(attacherJoint.node);
1167  
1168                  local x, y, z = worldDirectionToLocal(getParent(jointDesc.bottomArm.rotationNode), bx-ax, by-ay, bz-az);
1169                  local upX, upY, upZ = 0,1,0;
1170                  if math.abs(y) > 0.99*Utils.vector3Length(x, y, z) then
1171                      -- direction and up is parallel
1172                      upY = 0;
1173                      if y > 0 then
1174                          upZ = 1;
1175                      else
1176                          upZ = -1;
1177                      end;
1178                  end;
1179                  setDirection(jointDesc.bottomArm.rotationNode, x*jointDesc.bottomArm.zScale, y*jointDesc.bottomArm.zScale, z*jointDesc.bottomArm.zScale, upX, upY, upZ);
1180                  if jointDesc.bottomArm.translationNode ~= nil then
1181                      local distance = Utils.vector3Length(ax-bx, ay-by, az-bz);
1182                      setTranslation(jointDesc.bottomArm.translationNode, 0, 0, (distance-jointDesc.bottomArm.referenceDistance)*jointDesc.bottomArm.zScale);
1183                  end;
1184              end;
1185          end;
1186  
1187          local realLightsActive = false;
1188          local lightConesActive = false;
1189          if self:getIsActiveForSound() then
1190              realLightsActive = self.lightsActive;
1191          else
1192              lightConesActive = self.lightsActive;
1193          end;
1194          if realLightsActive ~= self.realLightsActive then
1195              self.realLightsActive = realLightsActive;
1196              for _, light in pairs(self.lights) do
1197                  setVisibility(light, realLightsActive);
1198              end;
1199          end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1200          if lightConesActive ~= self.lightConesActive then
1201              self.lightConesActive = lightConesActive;
1202              for _, lightCone in pairs(self.lightCones) do
1203                  setVisibility(lightCone, lightConesActive);
1204              end;
1205          end;
1206  
1207          if self.beaconLightsActive then
1208              for _, beaconLight in pairs(self.beaconLights) do
1209                  rotate(beaconLight.node, 0, beaconLight.speed*dt, 0);
1210              end;
1211          end;
1212      end;
1213  
1214      if self.firstTimeRun then
1215          WheelsUtil.updateWheelsGraphics(self, dt);
1216      end;
1217  
1218      for k,v in pairs(self.specializations) do
1219          v.update(self, dt);
1220      end;
1221  
1222      for _,v in ipairs(self.specializations) do
1223          if v.postUpdate ~= nil then
1224              v.postUpdate(self, dt);
1225          end;
1226      end;
1227  
1228  
1229      self.firstTimeRun = true;
1230  end;
1231  
1232  function Vehicle:getAttachedTrailersFillLevelAndCapacity()
1233      local fillLevel = 0;
1234      local capacity = 0;
1235      local hasTrailer = false;
1236  
1237      if self.fillLevel ~= nil and self.capacity ~= nil then
1238          fillLevel = fillLevel + self.fillLevel;
1239          capacity = capacity + self.capacity;
1240          hasTrailer = true;
1241      end;
1242      if self.manureCapacity ~= nil and self.manureIsFilled ~= nil then
1243          if self.manureIsFilled then
1244              fillLevel = fillLevel + self.manureCapacity;
1245          end;
1246          capacity = capacity + self.manureCapacity;
1247          hasTrailer = true;
1248      end;
1249  Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1250      for k, implement in pairs(self.attachedImplements) do
1251          local f, c = implement.object:getAttachedTrailersFillLevelAndCapacity();
1252          if f ~= nil and c ~= nil then
1253              fillLevel = fillLevel + f;
1254              capacity = capacity + c;
1255              hasTrailer = true;
1256          end;
1257      end;
1258      if hasTrailer then
1259          return fillLevel, capacity;
1260      end;
1261      return nil;
1262  end;
1263  
1264  function Vehicle:draw()
1265  
1266      for k,v in pairs(self.specializations) do
1267          v.draw(self);
1268      end;
1269  
1270      if self.selectedImplement ~= 0 then
1271          self.attachedImplements[self.selectedImplement].object:draw();
1272      end;
1273  
1274      --[[local frictionScale = 1;
1275      if self:getIsActive() then
1276          local upX, cosAngle, upZ =localDirectionToWorld(self.components[1].node, 0, 1, 0);
1277          if cosAngle < self.dynamicWheelFrictionCosAngleMax then
1278              frictionScale = math.max((1-self.dynamicWheelFrictionMinScale)*(cosAngle-self.dynamicWheelFrictionCosAngleMin)/(self.dynamicWheelFrictionCosAngleMax-self.dynamicWheelFrictionCosAngleMin), self.dynamicWheelFrictionMinScale);
1279          end;
1280          renderText(0.5,0.04, 0.025, "Angle: ".. math.deg(math.acos(cosAngle)).." Friction scale: "..frictionScale);
1281      end;]]
1282  end;
1283  
1284  function Vehicle:updateTick(dt)
1285      self.tickDt = dt;
1286      if self.isServer then
1287          local hasOwner = self:getOwner() ~= nil;
1288          for i=1, table.getn(self.components) do
1289              local x,y,z = getTranslation(self.components[i].node);
1290              local x_rot,y_rot,z_rot,w_rot=getQuaternion(self.components[i].node);
1291              if hasOwner or
1292                 math.abs(x-self.components[i].sentTranslation[1]) > 0.005 or
1293                 math.abs(y-self.components[i].sentTranslation[2]) > 0.005 or
1294                 math.abs(z-self.components[i].sentTranslation[3]) > 0.005 or
1295                 math.abs(x_rot-self.components[i].sentRotation[1]) > 0.1 or
1296                 math.abs(y_rot-self.components[i].sentRotation[2]) > 0.1 or
1297                 math.abs(z_rot-self.components[i].sentRotation[3]) > 0.1
1298              then
1299                  self:raiseDirtyFlags(self.vehicleDirtyFlag);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1300                  self.components[i].sentTranslation = {x,y,z};
1301                  self.components[i].sentRotation = {x_rot,y_rot,z_rot, w_rot};
1302              end;
1303          end;
1304  
1305          if table.getn(self.wheels) > 0 then
1306              local frictionScale = 1;
1307              if self:getIsActive() then
1308                  local upX, cosAngle, upZ =localDirectionToWorld(self.components[1].node, 0, 1, 0);
1309                  if cosAngle < self.dynamicWheelFrictionCosAngleMax then
1310                      frictionScale = math.max((1-self.dynamicWheelFrictionMinScale)*(cosAngle-self.dynamicWheelFrictionCosAngleMin)/(self.dynamicWheelFrictionCosAngleMax-self.dynamicWheelFrictionCosAngleMin), self.dynamicWheelFrictionMinScale);
1311                  end;
1312              end;
1313              if math.abs(frictionScale - self.wheelFrictionScale) > 0.01 or (frictionScale == 1 and self.wheelFrictionScale ~= 1) then
1314                  for i=1, table.getn(self.wheels) do
1315                      local wheel = self.wheels[i];
1316                      setWheelShapeTireFunction(wheel.node, wheel.wheelShape, false, 1000000*wheel.lateralStiffness*frictionScale);
1317                      setWheelShapeTireFunction(wheel.node, wheel.wheelShape, true, 1000000*wheel.longitudalStiffness*frictionScale);
1318                  end;
1319                  self.wheelFrictionScale = frictionScale;
1320              end;
1321          end;
1322      end;
1323  
1324      if self.isActive then
1325          for k, implement in pairs(self.attachedImplements) do
1326              local jointDesc = self.attacherJoints[implement.jointDescIndex];
1327  
1328              local jointFrameInvalid = false;
1329              if jointDesc.rotationNode ~= nil then
1330                  local x, y, z = getRotation(jointDesc.rotationNode);
1331                  local rot = {x,y,z};
1332                  local newRot = Utils.getMovedLimitedValues(rot, jointDesc.maxRot, jointDesc.minRot, 3, jointDesc.moveTime, dt, not jointDesc.moveDown);
1333                  setRotation(jointDesc.rotationNode, unpack(newRot));
1334                  for i=1, 3 do
1335                      if math.abs(newRot[i] - rot[i]) > 0.0005 then
1336                          jointFrameInvalid = true;
1337                      end;
1338                  end;
1339              end;
1340              if jointDesc.rotationNode2 ~= nil then
1341                  local x, y, z = getRotation(jointDesc.rotationNode2);
1342                  local rot = {x,y,z};
1343                  local newRot = Utils.getMovedLimitedValues(rot, jointDesc.maxRot2, jointDesc.minRot2, 3, jointDesc.moveTime, dt, not jointDesc.moveDown);
1344                  setRotation(jointDesc.rotationNode2, unpack(newRot));
1345                  for i=1, 3 do
1346                      if math.abs(newRot[i] - rot[i]) > 0.0005 then
1347                          jointFrameInvalid = true;
1348                      end;
1349                  end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1350              end;
1351              for k,v in pairs(self.specializations) do
1352                  if v.validateAttacherJoint ~= nil then
1353                      jointFrameInvalid = jointFrameInvalid or v.validateAttacherJoint(self, implement, jointDesc, dt);
1354                  end;
1355              end;
1356              jointFrameInvalid = jointFrameInvalid or jointDesc.jointFrameInvalid;
1357              if jointFrameInvalid then
1358                  jointDesc.jointFrameInvalid = false;
1359                  if self.isServer then
1360                      setJointFrame(jointDesc.jointIndex, 0, jointDesc.jointTransform);
1361                  end;
1362              end;
1363  
1364              if self.isServer then
1365                  if jointDesc.allowsJointLimitMovement then
1366                      local attacherJoint = implement.object.attacherJoint;
1367                      if attacherJoint.allowsJointRotLimitMovement then
1368                          local newRotLimit = Utils.getMovedLimitedValues(implement.jointRotLimit, implement.maxRotLimit, implement.minRotLimit, 3, jointDesc.moveTime, dt, not jointDesc.moveDown);
1369                          for i=1, 3 do
1370                              if math.abs(newRotLimit[i] - implement.jointRotLimit[i]) > 0.0005 then
1371                                  setJointRotationLimit(jointDesc.jointIndex, i-1, true, -newRotLimit[i], newRotLimit[i]);
1372                              end;
1373                          end;
1374                          implement.jointRotLimit = newRotLimit;
1375                      end;
1376  
1377                      if attacherJoint.allowsJointTransLimitMovement then
1378                          local newTransLimit = Utils.getMovedLimitedValues(implement.jointTransLimit, implement.maxTransLimit, implement.minTransLimit, 3, jointDesc.moveTime, dt, not jointDesc.moveDown);
1379                          for i=1, 3 do
1380                              if math.abs(newTransLimit[i] - implement.jointTransLimit[i]) > 0.0005 then
1381                                  setJointTranslationLimit(jointDesc.jointIndex, i-1, true, -newTransLimit[i], newTransLimit[i]);
1382                              end;
1383                          end;
1384                          implement.jointTransLimit = newTransLimit;
1385                      end;
1386                  end;
1387              end;
1388          end;
1389      end;
1390  
1391      for k,v in pairs(self.specializations) do
1392          if v.updateTick ~= nil then
1393              v.updateTick(self, dt);
1394          end;
1395      end;
1396  
1397      for _,v in ipairs(self.specializations) do
1398          if v.postUpdateTick ~= nil then
1399              v.postUpdateTick(self, dt);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1400          end;
1401      end;
1402  end;
1403  
1404  function Vehicle:attachImplement(object, jointIndex, noEventSend, index)
1405      if noEventSend == nil or noEventSend == false then
1406          if g_server ~= nil then
1407              g_server:broadcastEvent(VehicleAttachEvent:new(self, object, jointIndex), nil, nil, self);
1408          else
1409              g_client:getServerConnection():sendEvent(VehicleAttachEvent:new(self, object, jointIndex));
1410          end;
1411      end
1412  
1413      local jointDesc = self.attacherJoints[jointIndex];
1414      local implement = {};
1415      implement.object = object;
1416      implement.object:onAttach(self);
1417      implement.jointDescIndex = jointIndex;
1418  
1419      if jointDesc.rotationNode ~= nil then
1420          setRotation(jointDesc.rotationNode, unpack(jointDesc.maxRot));
1421      end;
1422      if jointDesc.rotationNode2 ~= nil then
1423          setRotation(jointDesc.rotationNode2, unpack(jointDesc.maxRot2));
1424      end;
1425  
1426      if self.isServer then
1427          local constr = JointConstructor:new();
1428          constr:setActors(jointDesc.rootNode, implement.object.attacherJoint.rootNode);
1429          constr:setJointTransforms(jointDesc.jointTransform, implement.object.attacherJoint.node);
1430          --constr:setBreakable(20, 10);
1431  
1432          implement.jointRotLimit = {};
1433          implement.jointTransLimit = {};
1434          implement.maxRotLimit = {};
1435          implement.maxTransLimit = {};
1436          implement.minRotLimit = {};
1437          implement.minTransLimit = {};
1438          if jointDesc.minRotLimit == nil then
1439              print("Warning: jointDesc.minRotLimit of joint "..jointIndex.." was not set in '"..self.configFileName.."'");
1440              jointDesc.minRotLimit = {0,0,0};
1441          end;
1442          if jointDesc.minTransLimit == nil then
1443              print("Warning: jointDesc.minTransLimit of joint "..jointIndex.." was not set in '"..self.configFileName.."'");
1444              jointDesc.minTransLimit = {0,0,0};
1445          end;
1446          for i=1, 3 do
1447              local maxRotLimit = jointDesc.maxRotLimit[i]*implement.object.attacherJoint.rotLimitScale[i];
1448              local minRotLimit = jointDesc.minRotLimit[i]*implement.object.attacherJoint.rotLimitScale[i];
1449              if implement.object.attacherJoint.fixedRotation thenCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1450                  maxRotLimit = 0;
1451                  minRotLimit = 0;
1452              end;
1453              local maxTransLimit = jointDesc.maxTransLimit[i]*implement.object.attacherJoint.transLimitScale[i];
1454              local minTransLimit = jointDesc.minTransLimit[i]*implement.object.attacherJoint.transLimitScale[i];
1455              implement.maxRotLimit[i] = maxRotLimit;
1456              implement.minRotLimit[i] = minRotLimit;
1457  
1458              implement.maxTransLimit[i] = maxTransLimit;
1459              implement.minTransLimit[i] = minTransLimit;
1460  
1461              constr:setRotationLimit(i-1, -maxRotLimit, maxRotLimit);
1462              implement.jointRotLimit[i] = maxRotLimit;
1463  
1464              constr:setTranslationLimit(i-1, true, -maxTransLimit, maxTransLimit);
1465              implement.jointTransLimit[i] = maxTransLimit;
1466          end;
1467  
1468          jointDesc.jointIndex = constr:finalize();
1469      else
1470          jointDesc.jointIndex = -1;
1471      end;
1472      jointDesc.moveDown = implement.object.isDefaultLowered;
1473      if index == nil then
1474          table.insert(self.attachedImplements, implement);
1475      else
1476          -- make the table at least as big as the number of implements
1477          local numAdd = index-table.getn(self.attachedImplements);
1478          for i=1, numAdd do
1479             table.insert(self.attachedImplements, {});
1480          end;
1481          self.attachedImplements[index] = implement;
1482      end;
1483  
1484      if self.isClient then
1485  
1486  
1487          if self.selectedImplement == 0 then
1488              self.selectedImplement = 1;
1489              implement.object:onSelect();
1490          end;
1491      end;
1492  
1493      for k,v in pairs(self.specializations) do
1494          if v.attachImplement ~= nil then
1495              v.attachImplement(self, implement);
1496          end;
1497      end;
1498  
1499  end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1500  
1501  function Vehicle:detachImplement(implementIndex, noEventSend)
1502  
1503      if noEventSend == nil or noEventSend == false then
1504          if g_server ~= nil then
1505              g_server:broadcastEvent(VehicleDetachEvent:new(self, self.attachedImplements[implementIndex].object), nil, nil, self);
1506          else
1507              g_client:getServerConnection():sendEvent(VehicleDetachEvent:new(self, self.attachedImplements[implementIndex].object));
1508          end;
1509      end
1510      for k,v in pairs(self.specializations) do
1511          if v.detachImplement ~= nil then
1512              v.detachImplement(self, implementIndex);
1513          end;
1514      end;
1515  
1516      local implement = self.attachedImplements[implementIndex];
1517      local jointDesc = self.attacherJoints[implement.jointDescIndex];
1518  
1519      if self.isServer then
1520          removeJoint(jointDesc.jointIndex);
1521      end;
1522      jointDesc.jointIndex = 0;
1523  
1524      if self.isClient then
1525          if implementIndex == self.selectedImplement then
1526              implement.object:onDeselect();
1527          end;
1528      end;
1529      implement.object:onDetach();
1530      implement.object = nil;
1531      if self.isClient then
1532          if jointDesc.topArm ~= nil then
1533              setRotation(jointDesc.topArm.rotationNode, jointDesc.topArm.rotX, jointDesc.topArm.rotY, jointDesc.topArm.rotZ);
1534              if jointDesc.topArm.translationNode ~= nil then
1535                  setTranslation(jointDesc.topArm.translationNode, 0, 0, 0);
1536              end;
1537          end;
1538          if jointDesc.bottomArm ~= nil then
1539              setRotation(jointDesc.bottomArm.rotationNode, jointDesc.bottomArm.rotX, jointDesc.bottomArm.rotY, jointDesc.bottomArm.rotZ);
1540              if jointDesc.bottomArm.translationNode ~= nil then
1541                  setTranslation(jointDesc.bottomArm.translationNode, 0, 0, 0);
1542              end;
1543          end;
1544      end;
1545      if self.isServer then
1546          if jointDesc.rotationNode ~= nil then
1547              setRotation(jointDesc.rotationNode, unpack(jointDesc.minRot));
1548          end;
1549      end;Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1550  
1551      table.remove(self.attachedImplements, implementIndex);
1552      if self.isClient then
1553          self.selectedImplement = math.min(self.selectedImplement, table.getn(self.attachedImplements));
1554          if self.selectedImplement ~= 0 then
1555              self.attachedImplements[self.selectedImplement].object:onSelect();
1556          end;
1557      end;
1558  
1559  end;
1560  
1561  function Vehicle:detachImplementByObject(object, noEventSend)
1562  
1563      for i=1, table.getn(self.attachedImplements) do
1564          if self.attachedImplements[i].object == object then
1565              self:detachImplement(i, noEventSend);
1566              break;
1567          end;
1568      end;
1569  
1570  end;
1571  
1572  function Vehicle:getImplementByObject(object)
1573      for i=1, table.getn(self.attachedImplements) do
1574          if self.attachedImplements[i].object == object then
1575              return self.attachedImplements[i];
1576          end;
1577      end;
1578      return nil;
1579  end;
1580  
1581  function Vehicle:setSelectedImplement(selected)
1582      if self.selectedImplement ~= 0 then
1583          self.attachedImplements[self.selectedImplement].object:onDeselect();
1584      end;
1585      self.selectedImplement = selected;
1586      self.attachedImplements[selected].object:onSelect();
1587  end;
1588  
1589  function Vehicle:playAttachSound()
1590      if self.attachSound ~= nil then
1591          setSamplePitch(self.attachSound, self.attachSoundPitchOffset);
1592          playSample(self.attachSound, 1, 1, 0);
1593      end;
1594  end;
1595  
1596  function Vehicle:playDetachSound()
1597      if self.attachSound ~= nil then
1598          setSamplePitch(self.attachSound, self.attachSoundPitchOffset);
1599          playSample(self.attachSound, 1, 1, 0);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1600      end;
1601  end;
1602  
1603  function Vehicle:handleAttachEvent()
1604  
1605      if self == g_currentMission.controlledVehicle and g_currentMission.trailerInTipRange ~= nil then
1606  
1607          -- only toggle tipper if the station accepts the current fill type or if the tipper is empty (--> FruitUtil.FRUITTYPE_UNKNOWN)
1608          if g_currentMission.currentTipTrigger ~= nil then
1609              local fruitType = g_currentMission.trailerInTipRange:getCurrentFruitType();
1610              if fruitType == FruitUtil.FRUITTYPE_UNKNOWN or g_currentMission.currentTipTrigger.acceptedFruitTypes[fruitType] then
1611                  g_currentMission.trailerInTipRange:toggleTipState(g_currentMission.currentTipTrigger);
1612              end;
1613          end;
1614  
1615      elseif self:handleAttachAttachableEvent() then
1616          self:playAttachSound();
1617      elseif self:handleDetachAttachableEvent() then
1618          self:playDetachSound();
1619      end;
1620  end;
1621  
1622  function Vehicle:handleAttachAttachableEvent()
1623      if g_currentMission.attachableInMountRange ~= nil then
1624          if g_currentMission.attachableInMountRangeVehicle == self then
1625              if self.attacherJoints[g_currentMission.attachableInMountRangeIndex].jointIndex == 0 then
1626                  self:attachImplement(g_currentMission.attachableInMountRange, g_currentMission.attachableInMountRangeIndex);
1627                  return true;
1628              end;
1629          else
1630              for i=1, table.getn(self.attachedImplements) do
1631                  if self.attachedImplements[i].object:handleAttachAttachableEvent() then
1632                      return true;
1633                  end;
1634              end;
1635          end;
1636      end;
1637      return false;
1638  end;
1639  
1640  function Vehicle:handleDetachAttachableEvent()
1641  
1642      if self.selectedImplement ~= 0 then
1643          local attachable = self.attachedImplements[self.selectedImplement].object;
1644          if not Input.isKeyPressed(Input.KEY_shift) then
1645              if attachable:handleDetachAttachableEvent() then
1646                  return true;
1647              end;
1648          end;
1649          local implement = self.attachedImplements[self.selectedImplement];Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1650          if implement.object.allowsDetaching then
1651              self:detachImplement(self.selectedImplement);
1652              return true;
1653          end;
1654      end;
1655      return false;
1656  end;
1657  
1658  function Vehicle:handleLowerImplementEvent()
1659      if self.selectedImplement ~= 0 then
1660          local implement = self.attachedImplements[self.selectedImplement];
1661          if implement.object.allowsLowering then
1662              local jointDesc = self.attacherJoints[implement.jointDescIndex];
1663              if jointDesc.allowsLowering then
1664                  --jointDesc.moveDown = not jointDesc.moveDown;
1665                  self:setJointMoveDown(implement.jointDescIndex, not jointDesc.moveDown, false);
1666              end;
1667          end;
1668      end;
1669  end;
1670  
1671  function Vehicle:getAttachedIndexFromJointDescIndex(jointDescIndex)
1672      for i=1, table.getn(self.attachedImplements) do
1673          if self.attachedImplements[i].jointDescIndex == jointDescIndex then
1674              return i;
1675          end;
1676      end;
1677      return nil;
1678  end;
1679  
1680  function Vehicle:getImplementIndexByObject(implement)
1681      for i=1, table.getn(self.attachedImplements) do
1682          if self.attachedImplements[i].object == implement then
1683              return i;
1684          end;
1685      end;
1686      return nil;
1687  end;
1688  
1689  function Vehicle:setLightsVisibility(visibility, noEventSend)
1690      if visibility ~= self.lightsActive then
1691  
1692          if noEventSend == nil or noEventSend == false then
1693              if g_server ~= nil then
1694                  g_server:broadcastEvent(SteerableToggleLightEvent:new(self, visibility), nil, nil, self);
1695              else
1696                  g_client:getServerConnection():sendEvent(SteerableToggleLightEvent:new(self, visibility));
1697              end;
1698          end;
1699  Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1700          self.lightsActive = visibility;
1701  
1702  
1703          self.realLightsActive = false;
1704          if not visibility or self:getIsActiveForSound() then
1705              self.realLightsActive = visibility;
1706              for _, light in pairs(self.lights) do
1707                  setVisibility(light, visibility);
1708              end;
1709          end;
1710          for _, corona in pairs(self.lightCoronas) do
1711              setVisibility(corona, visibility);
1712          end;
1713          self.lightConesActive = false;
1714          for _, lightCone in pairs(self.lightCones) do
1715              setVisibility(lightCone, false);
1716          end;
1717          for _,v in pairs(self.attachedImplements) do
1718              v.object:setLightsVisibility(visibility, true);
1719          end;
1720          for _,v in pairs(self.specializations) do
1721              if v.setLightsVisibility ~= nil then
1722                  v.setLightsVisibility(self, visibility);
1723              end;
1724          end;
1725      end;
1726  end;
1727  
1728  
1729  function Vehicle:setBeaconLightsVisibility(visibility, noEventSend)
1730      if visibility ~= self.beaconLightsActive then
1731  
1732          if noEventSend == nil or noEventSend == false then
1733              if g_server ~= nil then
1734                  g_server:broadcastEvent(VehicleSetBeaconLightEvent:new(self, visibility), nil, nil, self);
1735              else
1736                  g_client:getServerConnection():sendEvent(VehicleSetBeaconLightEvent:new(self, visibility));
1737              end;
1738          end;
1739  
1740          self.beaconLightsActive = visibility;
1741  
1742          for _, beaconLight in pairs(self.beaconLights) do
1743              setVisibility(beaconLight.node, visibility);
1744          end;
1745          for _,v in pairs(self.attachedImplements) do
1746              v.object:setBeaconLightsVisibility(visibility, true);
1747          end;
1748          for _,v in pairs(self.specializations) do
1749              if v.setBeaconLightsVisibility ~= nil thenCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1750                  v.setBeaconLightsVisibility(self, visibility);
1751              end;
1752          end;
1753      end;
1754  end;
1755  
1756  function Vehicle:getIsActiveForInput()
1757      if g_gui.currentGui ~= nil then
1758          return false;
1759      end;
1760      if self.isEntered then
1761          return true;
1762      end;
1763      if self.attacherVehicle ~= nil then
1764          return self.isSelected and self.attacherVehicle:getIsActiveForInput();
1765      end;
1766      return false;
1767  end;
1768  
1769  function Vehicle:getIsActiveForSound()
1770      if self.isEntered then
1771          return true;
1772      end;
1773      if self.attacherVehicle ~= nil then
1774          return self.attacherVehicle:getIsActiveForSound();
1775      end;
1776      return false;
1777  end;
1778  
1779  function Vehicle:getIsActive()
1780      if self.isBroken then
1781          return false;
1782      end;
1783      if self.isEntered or self.isControlled then
1784          return true;
1785      end;
1786      if self.attacherVehicle ~= nil then
1787          return self.attacherVehicle:getIsActive();
1788      end;
1789      if self.forceIsActive then
1790          return true;
1791      end;
1792      return false;
1793  end;
1794  
1795  function Vehicle:getIsHired()
1796      if self.isHired then
1797          return true;
1798      elseif self.attacherVehicle ~= nil then
1799          return self.attacherVehicle:getIsHired();Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1800      end;
1801      return false;
1802  end;
1803  
1804  function Vehicle:getOwner()
1805      if self.owner ~= nil then
1806          return self.owner;
1807      end;
1808      if self.attacherVehicle ~= nil then
1809          return self.attacherVehicle:getOwner();
1810      end;
1811      return nil;
1812  end;
1813  
1814  function Vehicle:onDeactivateAttachements()
1815      for k,v in pairs(self.attachedImplements) do
1816          v.object:onDeactivate();
1817      end;
1818  end;
1819  
1820  function Vehicle:onActivateAttachements()
1821      for k,v in pairs(self.attachedImplements) do
1822          v.object:onActivate();
1823      end;
1824  end;
1825  
1826  function Vehicle:onDeactivateAttachementsSounds()
1827      for k,v in pairs(self.attachedImplements) do
1828          v.object:onDeactivateSounds();
1829      end;
1830  end;
1831  
1832  function Vehicle:onDeactivateAttachementsLights()
1833      for k,v in pairs(self.attachedImplements) do
1834          v.object:onDeactivateLights();
1835      end;
1836  end;
1837  
1838  function Vehicle:onActivate()
1839      self:onActivateAttachements();
1840      for k,v in pairs(self.specializations) do
1841          if v.onActivate ~= nil then
1842              v.onActivate(self);
1843          end;
1844      end;
1845  end;
1846  
1847  function Vehicle:onDeactivate()
1848      self:onDeactivateAttachements();
1849      for k,v in pairs(self.specializations) doCopyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1850          if v.onDeactivate ~= nil then
1851              v.onDeactivate(self);
1852          end;
1853      end;
1854  end;
1855  
1856  function Vehicle:onDeactivateSounds()
1857      self:onDeactivateAttachementsSounds();
1858      for k,v in pairs(self.specializations) do
1859          if v.onDeactivateSounds ~= nil then
1860              v.onDeactivateSounds(self);
1861          end;
1862      end;
1863  end;
1864  
1865  function Vehicle:onDeactivateLights()
1866      self:onDeactivateAttachementsLights();
1867      for k,v in pairs(self.specializations) do
1868          if v.onDeactivateLights ~= nil then
1869              v.onDeactivateLights(self);
1870          end;
1871      end;
1872  end;
1873  
1874  function Vehicle:doCheckSpeedLimit()
1875      if self.attacherVehicle ~= nil then
1876          return self.checkSpeedLimit and self.attacherVehicle:doCheckSpeedLimit();
1877      end
1878      return self.checkSpeedLimit;
1879  end;
1880  
1881  function Vehicle:isLowered(default)
1882      if self.attacherVehicle ~= nil then
1883          local implement = self.attacherVehicle:getImplementByObject(self);
1884          if implement ~= nil then
1885              local jointDesc = self.attacherVehicle.attacherJoints[implement.jointDescIndex];
1886              if jointDesc.allowsLowering then
1887                  return jointDesc.moveDown or self.attacherVehicle:isLowered(default);
1888              end;
1889          end;
1890      end;
1891      return default;
1892  end;
1893  
1894  function Vehicle:setJointMoveDown(jointDescIndex, moveDown, noEventSend)
1895      VehicleLowerImplementEvent.sendEvent(self, jointDescIndex, moveDown, noEventSend)
1896      local jointDesc = self.attacherJoints[jointDescIndex]
1897      jointDesc.moveDown = moveDown;
1898  
1899      local implementIndex = self:getAttachedIndexFromJointDescIndex(jointDescIndex);Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
This document is to be published solely by ls-mods.de



1900      if implementIndex ~= nil then
1901          local implement = self.attachedImplements[implementIndex];
1902          if implement.object.onSetLowered ~= nil then
1903              implement.object:onSetLowered(moveDown)
1904          end;
1905      end;
1906  end;
1907  
1908  function Vehicle:getIsVehicleNode(nodeId)
1909      return self.vehicleNodes[nodeId] ~= nil;
1910  end;
1911  
1912  function Vehicle:getIsAttachedVehicleNode(nodeId)
1913      if self.vehicleNodes[nodeId] ~= nil then
1914          return true;
1915      end;
1916      for _,v in pairs(self.attachedImplements) do
1917          if v.object:getIsAttachedVehicleNode(nodeId) then
1918              return true;
1919          end;
1920      end;
1921      return false;
1922  end;
1923  
1924  function Vehicle:setComponentsVisibility(visiblity)
1925      if self.componentsVisibility ~= visiblity then
1926          self.componentsVisibility = visiblity;
1927          for k,v in pairs(self.components) do
1928              setVisibility(v.node, visiblity);
1929              if visiblity then
1930                  addToPhysics(v.node);
1931              else
1932                  removeFromPhysics(v.node);
1933              end;
1934          end;
1935      end;
1936  end;
1937  
1938  function Vehicle:getIsAreaActive(area)
1939      return true;
1940  end;
1941  
1942  function Vehicle.consoleCommandToggleDebugRendering(unusedSelf)
1943      Vehicle.debugRendering = not Vehicle.debugRendering;
1944      return "VehicleDebugRendering = "..tostring(Vehicle.debugRendering);
1945  end;
1946  
1947  addConsoleCommand("gsVehicleToggleDebugRendering", "Toggles the debug rendering of the vehicles", "Vehicle.consoleCommandToggleDebugRendering", nil);