﻿--
--   Unit Switcher v2
--   Change the ingame speed & fuel units between Metric, Imperial and US
--
-- @original Knuuud and Daveaich (ls-uk.info)
-- @author   Decker    (ls-uk.info, Decker_MMIV)
-- @date     2011-09-05
--
-- @comment Found problems or bugs? Please post them in the www.ls-uk.info support-thread for this mod. Thank you.
--          This mod is placed in the section 'Other', which at time of writing is; http://www.ls-uk.info/downloads/6
--
-- @history         - Knuuud suggested I took over the UnitSwitcher
--          v2.00   - Cleaned up some code.
--                  - Gave each unit (speed, fuel, currency) their individual ALT+<key>, where the <key> can be configured.
--                  - Added load/save functionality, which is not bound to a single vehicle.
--          v2.01   - Misc. tweaks in the load/save functions.
--          v2.10   - Added possibility of custom speed/fuel unit-factors and other currencies, read from UnitSwitcher.XML file.
--                  - Added hide-HUD with two levels; hide only speed/fuel/load-display, and hide every HUD (disables mouse-movement and more.)
--

Units = {}
Units.modDir = g_currentModDirectory;
--
Units.settingsFile = getUserProfileAppPath().."/UnitSwitcher.xml";
--
Units.hideHudLevel  = 0;
Units.speedType     = 0;
Units.fuelType      = 0;
Units.currencyType  = 0;
--
Units.speedFactor = 1;
Units.fuelFactor  = 1;
-- Use original texts at startup.
Units.speedUnit = g_i18n:getText("speedometer");
Units.fuelUnit  = g_i18n:getText("fluid_unit_long");

function Units:loadMap(name)
    if self.initialized then
        -- Only initialize one time.
        return;
    end;
    
    -- Game defaults
    Units.speedConversions = {};
    table.insert(Units.speedConversions, {unitName="GameDefault", unitSymbol=g_i18n:getText("speedometer"),     unitFactor=1.00000 });
    Units.fuelConversions  = {};
    table.insert(Units.fuelConversions,  {unitName="GameDefault", unitSymbol=g_i18n:getText("fluid_unit_long"), unitFactor=1.00000 });
    Units.currencyArray = {};
    table.insert(Units.currencyArray,    {unitName="GameDefault", unitSymbol=g_i18n:getText("Currency_symbol") });
    
    -- Load additional from settings-file, if exist
    if (fileExist(Units.settingsFile)) then
        self:load();
    else
        -- If no settings-file exist, apply the precoded units (so something can be saved)
        table.insert(Units.speedConversions, {unitName="Kilometers",  unitSymbol="km/h", unitFactor=1.00000 });
        table.insert(Units.speedConversions, {unitName="Miles",       unitSymbol="mph",  unitFactor=0.62137 });
        --
        table.insert(Units.fuelConversions, {unitName="Metric",       unitSymbol="liter",      unitFactor=1.00000 });
        table.insert(Units.fuelConversions, {unitName="Imperial",     unitSymbol="gallon",     unitFactor=0.21996 });
        table.insert(Units.fuelConversions, {unitName="Imperial(US)", unitSymbol="gallon(US)", unitFactor=0.26417 });
        --
        table.insert(Units.currencyArray, {unitName="Euro",             unitSymbol="€"    });
        table.insert(Units.currencyArray, {unitName="Dollar",           unitSymbol="$"    });
        table.insert(Units.currencyArray, {unitName="Pound sterling",   unitSymbol="GBP"  });   -- The '£' character/font does not seem to be available. Take a look at "<installPath>\farming simulator 2011\shared\font\defaultFont.png"
        table.insert(Units.currencyArray, {unitName="Złoty",            unitSymbol="zł."  });
        table.insert(Units.currencyArray, {unitName="Kroner",           unitSymbol="kr."  });
      --table.insert(Units.currencyArray, {unitName="Pубль rubl",       unitSymbol="руб." });   -- Some of the character/font does not seem to be available. Take a look at "<installPath>\farming simulator 2011\shared\font\defaultFont.png"
        
    end;

    --
    g_gui:loadGui(Units.modDir.."/emptyGui.xml", "UnitsEmptyGui");
    --
    self.msgTimer = 0;
    --
    Units.inputbindingmodifierUnitsSpeed     = getKeyModifier(InputBinding.UnitsSpeed);
    Units.inputbindingmodifierUnitsFuel      = getKeyModifier(InputBinding.UnitsFuel);
    Units.inputbindingmodifierUnitsCurrency  = getKeyModifier(InputBinding.UnitsCurrency);
    Units.inputbindingmodifierUnitsHideHud   = getKeyModifier(InputBinding.UnitsHideHud);
    --
    self.initialized = true;
end;

function Units:deleteMap()
end;

function Units:load(xmlFile)
end;

--http://stackoverflow.com/questions/656199/search-for-an-item-in-a-lua-list
function makeSet(list)
    local set = {};
    for _,l in ipairs(list) do
        set[l]=true;
    end;
    return set;
end;

-- Find which of the modifier-keys (left/right shift/ctrl/alt) that may be assigned to the input-binding
function getKeyModifier(binding)
    local allowedModifiers = makeSet({
        Input.KEY_lshift,
        Input.KEY_rshift,
        Input.KEY_lctrl, 
        Input.KEY_rctrl, 
        Input.KEY_lalt,  
        Input.KEY_ralt,
        Input.KEY_shift
    });
    for _,k in pairs(InputBinding.digitalActions[binding].key1Modifiers) do
        if allowedModifiers[k] then
            return k;
        end;
    end;
    -- No modifier-key found.
    return nil;
end;

function hasKeyModifierPressed(binding)
    return ((binding == nil) or (Input.isKeyPressed(binding)));
end;

function fileExist(pathAndFilename)
    local rb = false;
    local fileHandle = io.open(pathAndFilename, "r");
    if (fileHandle ~= nil) then
        fileHandle:close();
        rb = true;
    end;
    return rb;
end;

function Units:delete()
end;

function Units:mouseEvent(posX, posY, isDown, isUp, button)
end;

function Units:keyEvent(unicode, sym, modifier, isDown)
end;

function Units:update(dt)
    if (self.msgTimer > 0) then
        self.msgTimer = self.msgTimer - dt;
    end;
    --
    if InputBinding.hasEvent(InputBinding.UnitsSpeed) then
        local unitName = self:updateSpeedUnit(1);
        Units:setInfo(string.format(g_i18n:getText("speedChanged"), unitName));
    elseif InputBinding.hasEvent(InputBinding.UnitsFuel) then
        local unitName = self:updateFuelUnit(1);
        Units:setInfo(string.format(g_i18n:getText("fuelChanged"), unitName));
    elseif InputBinding.hasEvent(InputBinding.UnitsCurrency) then
        local unitName = self:updateCurrencyUnit(1);
        Units:setInfo(string.format(g_i18n:getText("currencyChanged"), unitName));
    elseif InputBinding.hasEvent(InputBinding.UnitsHideHud) then
        Units.hideHudLevel = (Units.hideHudLevel + 1) % 4;
        if (Units.hideHudLevel == 0) then
            g_gui:showGui("");
        elseif (Units.hideHudLevel == 3) then
            g_gui:showGui("UnitsEmptyGui");
        end;
    end;
end;

function Units:updateSpeedUnit(incType)
    Units.speedType = (Units.speedType + incType) % #Units.speedConversions;
    Units.speedFactor = Units.speedConversions[Units.speedType+1].unitFactor;
    Units.speedUnit   = Units.speedConversions[Units.speedType+1].unitSymbol;
    return Units.speedConversions[Units.speedType+1].unitName;
end;

function Units:updateFuelUnit(incType)
    Units.fuelType = (Units.fuelType + incType) % #Units.fuelConversions;
    Units.fuelFactor = Units.fuelConversions[Units.fuelType+1].unitFactor;
    Units.fuelUnit   = Units.fuelConversions[Units.fuelType+1].unitSymbol;
    return Units.fuelConversions[Units.fuelType+1].unitName;
end;

function Units:updateCurrencyUnit(incType)
    Units.currencyType = (Units.currencyType + incType) % #Units.currencyArray;
    g_i18n.globalI18N.texts["Currency_symbol"] = Units.currencyArray[Units.currencyType+1].unitSymbol;
    g_i18n.globalI18N.texts["PricePerTon"]     = string.format("[%s/t]", Units.currencyArray[Units.currencyType+1].unitSymbol);
    return Units.currencyArray[Units.currencyType+1].unitName;
end;

function Units:setInfo(msg)
    self.msgTxt = msg;
    self.msgTimer = 3000;
end;

function Units:draw()
    -- Only add help-buttons when the help-box is visible. If not, the array 'g_currentMission.helpButtonTexts' will just grow-and-grow-and...(bad allocation?)
    -- Also this should reduce (for this mod) the "big helpbox huge-to-less flashing", when turning the helpbox on again with key F1.
    if g_currentMission.showHelpText then
        -- Only add help-buttons, if the modifier-key is pressed and the input-binding contains that modifier too.
        if hasKeyModifierPressed(Units.inputbindingmodifierUnitsSpeed) then
            g_currentMission:addHelpButtonText(g_i18n:getText("UnitsSpeed"), InputBinding.UnitsSpeed);
        end;
        if hasKeyModifierPressed(Units.inputbindingmodifierUnitsCurrency) then
            g_currentMission:addHelpButtonText(g_i18n:getText("UnitsCurrency"), InputBinding.UnitsCurrency);
        end;
        if hasKeyModifierPressed(Units.inputbindingmodifierUnitsFuel) then
            g_currentMission:addHelpButtonText(g_i18n:getText("UnitsFuel"), InputBinding.UnitsFuel);
        end;
        if hasKeyModifierPressed(Units.inputbindingmodifierUnitsHideHud) then
            if (Units.hideHudLevel >= 1) then
                g_currentMission:addHelpButtonText(g_i18n:getText("UnitsHideHudMore"), InputBinding.UnitsHideHud);
            else
                g_currentMission:addHelpButtonText(g_i18n:getText("UnitsHideHud"), InputBinding.UnitsHideHud);
            end;
        end;
    end;
    --
    if (self.msgTimer > 0) and (not Units.hideHud) then
        setTextAlignment(RenderText.ALIGN_CENTER);
        setTextBold(false);
        setTextColor(1,1,1, 1);
        renderText(0.50, 0.07, 0.03, self.msgTxt);
        --
        setTextAlignment(RenderText.ALIGN_LEFT);
    end;
    --
    if (Units.hideHudLevel == 2) then
        g_currentMission:addWarning(string.format(g_i18n:getText("hideHudWarning"), InputBinding.getKeyNamesOfDigitalAction(InputBinding.UnitsHideHud)), 0.05, 0.025+0.007);
    end;
end;

function Units:load()
    -- 'self' is in context of Units object
    local rootTag = "UnitSwitcher";
    local xmlFile = loadXMLFile(nil, Units.settingsFile);
    --
    local i=0;
    --
    i=0;
    while (true) do
        local tag = string.format(rootTag..".speedUnits.speed(%d)", i);
        local unitName      = getXMLString(xmlFile, tag.."#name");
        local unitSymbol    = getXMLString(xmlFile, tag.."#symbol");
        local unitFactor    = getXMLFloat( xmlFile, tag.."#factor");
        local selected      = getXMLBool(  xmlFile, tag.."#selected");
        if (unitName ~= nil and unitSymbol ~= nil and unitFactor ~= nil) then
            table.insert(Units.speedConversions, {unitName=unitName, unitSymbol=unitSymbol, unitFactor=unitFactor} );
            if (selected ~= nil and selected == true) then
                Units.speedType = (#Units.speedConversions)-1;
            end;
        else
            break;
        end;
        i=i+1;
    end;
    --
    i=0;
    while (true) do
        local tag = string.format(rootTag..".fuelUnits.fuel(%d)", i);
        local unitName      = getXMLString(xmlFile, tag.."#name");
        local unitSymbol    = getXMLString(xmlFile, tag.."#symbol");
        local unitFactor    = getXMLFloat( xmlFile, tag.."#factor");
        local selected      = getXMLBool(  xmlFile, tag.."#selected");
        if (unitName ~= nil and unitSymbol ~= nil and unitFactor ~= nil) then
            table.insert(Units.fuelConversions, {unitName=unitName, unitSymbol=unitSymbol, unitFactor=unitFactor} );
            if (selected ~= nil and selected == true) then
                Units.fuelType = (#Units.fuelConversions)-1;
            end;
        else
            break;
        end;
        i=i+1;
    end;
    --
    i=0;
    while (true) do
        local tag = string.format(rootTag..".currencyUnits.currency(%d)", i);
        local unitName      = getXMLString(xmlFile, tag.."#name");
        local unitSymbol    = getXMLString(xmlFile, tag.."#symbol");
        local selected      = getXMLBool(  xmlFile, tag.."#selected");
        if (unitName ~= nil and unitSymbol ~= nil) then
            table.insert(Units.currencyArray, {unitName=unitName, unitSymbol=unitSymbol} );
            if (selected ~= nil and selected == true) then
                Units.currencyType = (#Units.currencyArray)-1;
            end;
        else
            break;
        end;
        i=i+1;
    end;
    --
    delete(xmlFile);
    --
    self:updateSpeedUnit(0);
    self:updateFuelUnit(0);
    self:updateCurrencyUnit(0);
end;

function Units:save()
    -- 'self' is in context of the g_careerScreen object
    local rootTag = "UnitSwitcher";
    local xmlFile = createXMLFile(nil, Units.settingsFile, rootTag);
    --
    for i=2,#Units.speedConversions do  -- Skip first element, as it contains the "GameDefault"
        local tag = string.format(rootTag..".speedUnits.speed(%d)", i-2);
        setXMLString(xmlFile, tag.."#name",   Units.speedConversions[i].unitName);
        setXMLString(xmlFile, tag.."#symbol", Units.speedConversions[i].unitSymbol);
        setXMLFloat( xmlFile, tag.."#factor", Units.speedConversions[i].unitFactor);
        if (Units.speedType+1 == i) then
            setXMLBool(xmlFile, tag.."#selected", true);
        end;
    end;
    --
    for i=2,#Units.fuelConversions do  -- Skip first element, as it contains the "GameDefault"
        local tag = string.format(rootTag..".fuelUnits.fuel(%d)", i-2);
        setXMLString(xmlFile, tag.."#name",   Units.fuelConversions[i].unitName);
        setXMLString(xmlFile, tag.."#symbol", Units.fuelConversions[i].unitSymbol);
        setXMLFloat( xmlFile, tag.."#factor", Units.fuelConversions[i].unitFactor);
        if (Units.fuelType+1 == i) then
            setXMLBool(xmlFile, tag.."#selected", true);
        end;
    end;
    --
    for i=2,#Units.currencyArray do  -- Skip first element, as it contains the "GameDefault"
        local tag = string.format(rootTag..".currencyUnits.currency(%d)", i-2);
        setXMLString(xmlFile, tag.."#name",   Units.currencyArray[i].unitName);
        setXMLString(xmlFile, tag.."#symbol", Units.currencyArray[i].unitSymbol);
        if (Units.currencyType+1 == i) then
            setXMLBool(xmlFile, tag.."#selected", true);
        end;
    end;
    --
    saveXMLFile(xmlFile);   -- TODO - How to determine if the file was saved or not?
    delete(xmlFile);
end;

-- Re-implementation of Steerable.draw(), to alter the speed & fuel units and calculation.
function Units.steerableDraw(self)
    if (Units.hideHudLevel > 0) then
        return;
    end;
    --
    local kmh = math.min(100, math.max(0, self.lastSpeed*self.speedDisplayScale*3600));
    local maxSpeed = 80;

    self.hudBackgroundOverlay:render();
    setTextBold(true);
    setTextColor(1.0, 1.0, 1.0, 1.0);
    setTextAlignment(RenderText.ALIGN_CENTER);

    local speedInDisplayUnits = kmh * Units.speedFactor;

    setTextColor(0, 0, 0, 1);
    renderText(self.hudBasePosX + self.hudBaseWidth / 2 + 0.002, self.hudBasePosY + 0.005 + 3 * 0.0396, 0.024, string.format("%2d " .. Units.speedUnit, speedInDisplayUnits));
    setTextColor(1, 1, 1, 1);
    renderText(self.hudBasePosX + self.hudBaseWidth / 2 + 0.002, self.hudBasePosY + 0.008 + 3 * 0.0396, 0.024, string.format("%2d " .. Units.speedUnit, speedInDisplayUnits));

    self.hudBarRedOverlay.width = self.hudBarWidth * math.min(1, kmh / maxSpeed);
    setOverlayUVs(self.hudBarRedOverlay.overlayId, 0, 0.05, 0, 1, math.min(1, kmh / maxSpeed), 0.05, math.min(1, kmh / maxSpeed), 1);
    self.hudBarRedOverlay:render();

    setTextColor(0, 0, 0, 1);
    renderText(self.hudBasePosX + self.hudBaseWidth / 2 + 0.002, self.hudBasePosY + 0.005 + 2 * 0.0396, 0.024, string.format("%1.0f " .. g_i18n:getText("Currency_symbol"), g_currentMission.missionStats.money));
    setTextColor(1, 1, 1, 1);
    renderText(self.hudBasePosX + self.hudBaseWidth / 2 + 0.002, self.hudBasePosY + 0.008 + 2 * 0.0396, 0.024, string.format("%1.0f " .. g_i18n:getText("Currency_symbol"), g_currentMission.missionStats.money));

    local currentFuelPercentage = 0;
    local fuelWarnPercentage = 20;
    if self.fuelCapacity > 0 then
        currentFuelPercentage = (self.fuelFillLevel / self.fuelCapacity + 0.0001) * 100;
    end;

    local fuelInDisplayUnits = self.fuelFillLevel * Units.fuelFactor;

    setTextColor(0, 0, 0, 1);
    renderText(self.hudBasePosX + self.hudBaseWidth / 2 + 0.002, self.hudBasePosY + 0.005, 0.024, string.format("%d " .. Units.fuelUnit, fuelInDisplayUnits));
    if currentFuelPercentage < fuelWarnPercentage then
        setTextColor(1, 0, 0, 1);
    else
        setTextColor(1, 1, 1, 1);
    end;
    renderText(self.hudBasePosX + self.hudBaseWidth / 2 + 0.002, self.hudBasePosY + 0.008, 0.024, string.format("%d " .. Units.fuelUnit, fuelInDisplayUnits));

    if currentFuelPercentage < fuelWarnPercentage then
        setTextColor(1, 1, 1, 1);
    end;

    self.hudBarGreenOverlay.width = self.hudBarWidth * (self.fuelFillLevel / self.fuelCapacity); -- TODO: check for division by zero!
    setOverlayUVs(self.hudBarGreenOverlay.overlayId, 0, 0.05, 0, 1, self.fuelFillLevel / self.fuelCapacity, 0.05, self.fuelFillLevel / self.fuelCapacity, 1);
    self.hudBarGreenOverlay:render();

    local trailerFillLevel, trailerCapacity = self:getAttachedTrailersFillLevelAndCapacity();
    if trailerFillLevel ~= nil and trailerCapacity ~= nil and trailerCapacity > 0 then
        self:drawGrainLevel(trailerFillLevel, trailerCapacity, 101);
    else
        -- if we don't have a trailer attached and we're not in a combine, render the frames overlay now
        if self.grainTankFillLevel == nil then
            self.hudFramesOverlay:render();
        end;
    end;

    setTextAlignment(RenderText.ALIGN_LEFT);
    setTextBold(false);

    if self.showWaterWarning then
        g_currentMission:addWarning(g_i18n:getText("Dont_drive_to_depth_into_the_water"), 0.05, 0.025+0.007);
    end;

    if table.getn(self.attachedImplements) > 1 and g_currentMission.showHelpText then --
        g_currentMission:addHelpButtonText(g_i18n:getText("Change_tools"), InputBinding.SWITCH_IMPLEMENT);
    end;
end;

-- Re-implementation of Steerable.drawGrainLevel(), to be able to hide it.
function Units.steerableDrawGrainLevel(self, level, capacity, warnPercent)
    if (Units.hideHudLevel > 0) then
        return;
    end;
    Units.oldDrawGrainLevel(self, level, capacity, warnPercent);
end;

-- Overwrite the draw() functions of Steerable.
Steerable.draw              = Units.steerableDraw;
--
Units.oldDrawGrainLevel     = Steerable.drawGrainLevel;
Steerable.drawGrainLevel    = Units.steerableDrawGrainLevel;

-- Append the Units.save() function, to when saving game.
CareerScreen.saveSelectedGame = Utils.appendedFunction(CareerScreen.saveSelectedGame, Units.save);

--
addModEventListener(Units);
