--
-- ParallelMotion
-- Specialization for a real animation of different loaders kinematics
--
-- @author    PES 4ever
-- @date      11/07/11
-- @version   1.0
--
-- For support and documentation visit www.modding-society.de
-- Copyright (C) PES 4ever, All Rights Reserved.
--

ParallelMotion = {};

function ParallelMotion.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Cylindered, specializations);
end;

function ParallelMotion:load(xmlFile)

	-- Lower motor rpm function
	self.lowerMotorRpmActive = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.parallelMotion#lowerMotorRpm"), false);
	self.motorRpmLowered = false;

	-- Motion Parts
	self.animateParallelMotion = true;
	self.motionParts = {};
	local i = 0;
	while true do
		local baseName = string.format("vehicle.parallelMotion.motionPart(%d)", i);
		if not hasXMLProperty(xmlFile, baseName) then
			break;
		end;
		
		local node = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#index"));
		if node ~= nil then
			local entry = {};
			entry.node = node;
			entry.refPoint = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#refPoint"));
			entry.isCylinder = Utils.getNoNil(getXMLBool(xmlFile, baseName .. "#isTiltCylinder", false));
			entry.invert = Utils.getNoNil(getXMLBool(xmlFile, baseName .. "#invert"), false);
			
			entry.motionLink = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#motionLink"));
			entry.linkPoint = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#linkPoint"));

			if entry.motionLink ~= nil and entry.linkPoint ~= nil then
				
				if entry.refPoint == nil then
					if entry.isCylinder then
						print("Error: tilt cylinder in motion part ".. i+1 .." in parallel motion has no reference point in '".. self.configFileName .."'");
						self.animateParallelMotion = false;
						break;
					else
						entry.refPoint = entry.linkPoint;
					end;
				end;
				
				local ax, ay, az = getWorldTranslation(entry.node);
				local bx, by, bz = getWorldTranslation(entry.refPoint);
				entry.motionRadius = Utils.vector3Length(bx-ax, by-ay, bz-az);
				
				ax, ay, az = getWorldTranslation(entry.motionLink);
				bx, by, bz = getWorldTranslation(entry.linkPoint);
				entry.linkRadius = Utils.vector3Length(bx-ax, by-ay, bz-az);
			
				table.insert(self.motionParts, entry);
			else
				print("Error: inputs of motion part ".. i+1 .." in parallel motion are missing or wrong in '".. self.configFileName .."'");
				self.animateParallelMotion = false;
				break;
			end;
		end;
		
		i = i + 1;
	end;
	
	-- Load dependences to moving tools
	if self.animateParallelMotion then
		self.movingToolNumbers = {};
		local i = 0;
		while true do
			local baseName = string.format("vehicle.parallelMotion.movingTool(%d)", i);
			if not hasXMLProperty(xmlFile, baseName) then
				break;
			end;
		
			local number = getXMLInt(xmlFile, baseName.. "#indexNumber");
			if number ~= nil then
				if number >= 1 and number <= table.getn(self.movingTools) then
					table.insert(self.movingToolNumbers, number);
				else
					print("Warning: index number of moving tool ".. i+1 .." in parallel motion does not exist in '".. self.configFileName .."'");
					break;
				end;
			end;
			
			i = i + 1;
		end;
	end;
	
end;

function ParallelMotion:delete()
end;

function ParallelMotion:readStream(streamId, connection)
end;

function ParallelMotion:writeStream(streamId, connection)
end;

function ParallelMotion:mouseEvent(posX, posY, isDown, isUp, button)
end;

function ParallelMotion:keyEvent(unicode, sym, modifier, isDown)
end;

function ParallelMotion:update(dt)
end;

function ParallelMotion:updateTick(dt)
	
	if self:getIsActive() then
	
		if self.isClient and self.animateParallelMotion then
	
			-- Check if update is needed
			local updateParallelMotion = false;
			
			for _, number in pairs(self.movingToolNumbers) do
				local tool = self.movingTools[number];
				
				-- Check rotation of moving tool
				if tool.rotSpeed ~= nil then
					local curRot = tool.curRot[tool.rotationAxis];
					if tool.savedLastRot == nil then
						tool.savedLastRot = curRot;
					end;
					if curRot - tool.savedLastRot ~= 0 then
						updateParallelMotion = true;
					end;
					tool.savedLastRot = curRot;
				end;
				
				-- Check translation of moving tool
				if tool.transSpeed ~= nil then
					local curTrans = tool.curTrans[tool.translationAxis];
					if tool.savedLastTrans == nil then
						tool.savedLastTrans = curTrans;
					end;
					if curTrans - tool.savedLastTrans ~= 0 then
						updateParallelMotion = true;
					end;
					tool.savedLastTrans = curTrans;
				end;
			end;
			
			-- Update parallel motion
			if updateParallelMotion then
			
				for _, part in pairs(self.motionParts) do
					
					-- Get actual lenght of tilt cylinder
					if part.isCylinder then
						local ax, ay, az = getWorldTranslation(part.node);
						local bx, by, bz = getWorldTranslation(part.refPoint);
						part.motionRadius = Utils.vector3Length(bx-ax, by-ay, bz-az);
					end;
				
					-- Animate motionLink
					local ax, ay, az = getWorldTranslation(part.motionLink);
					local bx, by, bz = getWorldTranslation(part.node);
					local xm, ym, zm = worldDirectionToLocal(getParent(part.motionLink), bx-ax, by-ay, bz-az);
					if Utils.vector3Length(bx-ax, by-ay, bz-az) < part.linkRadius + part.motionRadius then
						local iy, iz, iy2, iz2 = Utils.getCircleCircleIntersection(0, 0, part.linkRadius, ym, zm, part.motionRadius);
						if iy ~= nil and iz ~= nil then
							local angle = -math.atan(iy / iz);
							if part.invert then
								angle = -angle;
							end;
							if iy > 0 and iz < 0 then
								angle = angle - math.pi;
							end;
							setRotation(part.motionLink, angle, 0, 0);
						end;
					end;
					
					-- Animate parallel arms or tilt cylinder
					local ax, ay, az = getWorldTranslation(part.node);
					local bx, by, bz = getWorldTranslation(part.linkPoint);
					local x, y, z = worldToLocal(part.node, bx, by, bz);
					local angle = math.atan(y/-z);
					local rx, ry, rz = getRotation(part.node);
					setRotation(part.node, rx + angle, ry, rz);
				end;
			end;
			
			-- Lower motor rpm
			if self:getIsActiveForInput() and self.lowerMotorRpmActive then
			
				-- Check moving tools
				local lowerMotorRpm = false;				
				for _, number in pairs(self.movingToolNumbers) do
					local tool = self.movingTools[number];
					if tool.lastRotSpeed ~= 0 or tool.lastTransSpeed ~= 0 then
						lowerMotorRpm = true;
						break;
					end;
				end;
				
				-- Change motor sound pitch offset
				if lowerMotorRpm then
					if self.motorSoundPitchOffset == self.motorSoundPitchOffsetSave then
						self.motorSoundPitchOffset = self.motorSoundPitchOffsetSave - 0.05;
						self.motorRpmLowered = true;					
					end;					
				elseif self.motorRpmLowered then
					if self.motorSoundPitchOffset ~= self.motorSoundPitchOffsetSave then
						self.motorSoundPitchOffset = self.motorSoundPitchOffsetSave;
						self.motorRpmLowered = false;
					end;
				end;
			end;
		end;
	end;
	
end;

function ParallelMotion:draw()
end;