--
-- ExtendedTipTrigger
-- Specialization for ExtendedTipTrigger
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date:		31/07/11
-- @version:	v1.0
-- @history:	v1.0 - initial implementation
--



-- local copy of directory
local modName = g_currentModName;
local modDir = g_currentModDirectory;


-- fix initialization bug
if g_gameVersionLocal == nil or g_gameVersionLocal <= 1.023 then
    InitObjectClass = function(classObject, className)
        if g_currentMission ~= nil then -- this is the difference to the original
            print("Error: Object initialization only allowed at compile time");
            printCallstack();
            return;
        end;
        if ObjectIds.objectClasses[className] ~= nil then
            print("Error: Same class name used multiple times "..className);
            printCallstack();
            return;
        end;
        ObjectIds.objectClasses[className] = classObject;
    end;    
end;


ExtendedTipTrigger = {};
ExtendedTipTrigger_mt = Class(ExtendedTipTrigger, Object);
InitObjectClass(ExtendedTipTrigger, modName..".ExtendedTipTrigger");


function ExtendedTipTrigger:new(isServer, isClient, customMt)
    local mt = customMt;
    if mt == nil then
        mt = ExtendedTipTrigger_mt;
    end;
	
    local self = Object:new(isServer, isClient, mt);

    self.triggerId = 0;
    self.className = modName..".ExtendedTipTrigger";
	self.extendedTipTriggerDirtyFlag = self:getNextDirtyFlag();
	
    return self;
end;

function ExtendedTipTrigger:load(id, trigger, fruitType)

    self.triggerId = id;
	g_currentMission:addTipTrigger(self);
	
    addTrigger(id, "triggerCallback", self);

    self.appearsOnPDA = false;
    self.isFarmTrigger = false;
    self.stationName = "AlternativeTip" .. Utils.getNoNil(getUserAttribute(id, "stationName"), "Station");
	self.isExtendedTrigger = true;
	self.shovelInRange = nil; -- custom shovel (Heady);
	self.currentShovel = nil; -- ls2011 shovel
	self.currentFillType = fruitType;
	self.fixpoint = trigger.fixpoint;
	self.transform = trigger.transform;
	self.planes = trigger.planes;
	self.isRemovalAllowed = true;
	self.capacity = Utils.getNoNil(getUserAttribute(id, "capacity"), 50000);
	self.fillLevel = 0;
	
    self.acceptedFruitTypes = {}; -- was fillTypes
    self.priceMultipliers = {};

    local fruitTypes = getUserAttribute(id, "fruitTypes");
    local priceMultipliersString = getUserAttribute(id, "priceMultipliers");
    if fruitTypes ~= nil then
        local types = Utils.splitString(" ", fruitTypes);
        
        local multipliers = {};
        if priceMultipliersString ~= nil then
            multipliers = Utils.splitString(" ", priceMultipliersString);
        elseif not self.isFarmTrigger then
            print("Error: Missing priceMultipliersString user attribute for TipTrigger "..getName(id));
        end;
        for k,v in pairs(types) do
            local desc = FruitUtil.fruitTypes[v];
            if desc ~= nil then
                self.acceptedFruitTypes[desc.index] = true;
                self.priceMultipliers[desc.index] = tonumber(multipliers[k]);
                if self.priceMultipliers[desc.index] == nil then
                    self.priceMultipliers[desc.index] = 0;
                    if not self.isFarmTrigger then
                        print("Error: "..k.."th priceMultiplier is invalid in TipTrigger "..getName(id));
                    end;
                end;
            end;
        end;
    end;
	
	local parent = getParent(id);
    local movingIndex = getUserAttribute(id, "movingIndex");
    if movingIndex ~= nil then
        self.movingId = Utils.indexToObject(parent, movingIndex);
        if self.movingId ~= nil then
            self.moveMinY = Utils.getNoNil(getUserAttribute(id, "moveMinY"), 0);
            self.moveMaxY = Utils.getNoNil(getUserAttribute(id, "moveMaxY"), 0);
            self.moveScale = Utils.getNoNil(getUserAttribute(id, "moveScale"), 0.001)*0.01;
            self.moveBackScale = (self.moveMaxY-self.moveMinY)/Utils.getNoNil(getUserAttribute(id, "moveBackTime"), 10000);
        end;
    end;
	
	
	if self.planes ~= nil then		
		local found = false;
		local defaultPlane = nil;
		for k,plane in pairs(self.planes) do
			local currentObj = Utils.indexToObject(id, plane.path);
			if k == 1 then
				defaultPlane = currentObj;
				self.movingId = getParent(defaultPlane);
			end;
			if plane.type == Fillable.fillTypeIntToName[self.currentFillType] then
				setVisibility(currentObj, true);
				found = true;
			else
				setVisibility(currentObj, false);
			end;				
		end;	
		if not found then
			setVisibility(defaultPlane, true);
			print("Warning: No plane defined for fruittpye " .. Fillable.fillTypeIntToName[self.currentFillType]);
		end;
	else
		print("Error: Could not set trigger plane. No planes defined");
	end;

    self.isEnabled = true;
	
    return self;
end;

function ExtendedTipTrigger:readStream(streamId, connection)
	ExtendedTipTrigger:superClass().readStream(self, streamId, connection);
end;

function ExtendedTipTrigger:writeStream(streamId, connection)
	ExtendedTipTrigger:superClass().writeStream(self, streamId, connection);
end;

function ExtendedTipTrigger:readUpdateStream(streamId, timestamp, connection)
	ExtendedTipTrigger:superClass().readUpdateStream(self, streamId, timestamp, connection);
	if connection:getIsServer() then
		self.fillLevel = streamReadFloat32(streamId);
		local x,y,z = getTranslation(self.movingId);
		y = self.moveMinY + (self.fillLevel / self.capacity)*(self.moveMaxY-self.moveMinY);
        setTranslation(self.movingId, x, y, z);
		if self.fillLevel <= 0 and self.isServer then
			g_currentMission.vehiclesToDelete[self] = self;
		end;
    end;
end;

function ExtendedTipTrigger:writeUpdateStream(streamId, connection, dirtyMask)
	ExtendedTipTrigger:superClass().writeUpdateStream(self, streamId, connection, dirtyMask);
    if not connection:getIsServer() then
		streamWriteFloat32(streamId, self.fillLevel);
    end;
end;

function ExtendedTipTrigger:isValidForLoad(node)
	local ax,ay,az = getWorldTranslation(node);
	local bx,by,bz = getWorldTranslation(self.movingId);
	local distance = Utils.vector2Length(bx-ax, bz-az);	
	local thresholdRadius = 9;
	local thresholdHeight = 1.5;
	
	if distance < 1 + thresholdRadius * (self.fillLevel / self.capacity) and ay < thresholdHeight + by then
		return true;
	end;
	return false;
end;

function ExtendedTipTrigger:update(dt)
	if self.shovelInRange ~= nil and self.isRemovalAllowed then
		if self.shovelInRange.fillTypes ~= nil and self.shovelInRange.setFillLevel ~= nil and self.shovelInRange.fillLevel ~= nil and not (self.shovelInRange.fallOut == nil or self.shovelInRange.fallOut) then
			if self.shovelInRange:allowFillType(self.currentFillType, true) then
				if self.shovelInRange.fillLevel < self.shovelInRange.capacity then
					if self:isValidForLoad(self.shovelInRange.rootNode) then
						local deltaFillLevel = dt*3;
						if self.shovelInRange.isSugarbeetCleanerLoader then
							deltaFillLevel =  self.shovelInRange.unloadingCapacity*dt/500;
						end
						deltaFillLevel = math.min(deltaFillLevel, self.shovelInRange.capacity - self.shovelInRange.fillLevel);
						self.shovelInRange:setFillLevel(self.shovelInRange.fillLevel + deltaFillLevel, self.currentFillType);
						self:updateMoving(-deltaFillLevel);
					end;
				end;
			end;
		end;
	end;
	if self.currentShovel ~= nil and self.currentFillType == Fillable.FILLTYPE_MANURE then
        local shovel = self.currentShovel;
        if not shovel.manureIsFilled and not shovel:getIsManureEmptying() then	
			local node = shovel.rootNode;
			if shovel.shovelTipReferenceNode ~= nil then
				node = shovel.shovelTipReferenceNode;
			end;
			if self:isValidForLoad(node) then
				local delta = math.max(math.min(shovel.manureCapacity, self.fillLevel), 0);
				if delta > 0 then
					shovel:setManureIsFilled(true);
					self:updateMoving(-delta);
				end;
			end;		
        end;
    end;
end;

function ExtendedTipTrigger:delete()
    g_currentMission:removeTipTrigger(self);
    removeTrigger(self.triggerId);
    delete(self.triggerId);
    ExtendedTipTrigger:superClass().delete(self);
	
	if g_currentMission ~= nil then	
		local i=1;
		for k,v in pairs(g_currentMission.trailerTipTriggers) do
			for key, tipTrigger in pairs(v) do
				if tipTrigger == self then
					v[key] = nil;
				end;
			end;
		end;
	end;
end;

function ExtendedTipTrigger:draw()
end;

function ExtendedTipTrigger:updateMoving(delta)	
	local oldFillLevel = self.fillLevel + delta;
	self.fillLevel = math.min(oldFillLevel, self.capacity);
	
	local refill = oldFillLevel - self.fillLevel;
	if refill > 0 then
		local trailer = g_currentMission.trailerInTipRange;
		if trailer ~= nil then
			trailer:setFillLevel(trailer.fillLevel + refill, trailer.currentFillType);
			trailer:onEndTip();
		end;
	end;		
	if self.movingId ~= nil then
		local x,y,z = getTranslation(self.movingId);
		y = self.moveMinY + (self.fillLevel / self.capacity)*(self.moveMaxY-self.moveMinY);
		setTranslation(self.movingId, x, y, z);
		if self.isServer then
			self:raiseDirtyFlags(self.extendedTipTriggerDirtyFlag);
		end;
	end;
	
	if self.fillLevel <= 0 and self.isServer then
		g_currentMission.vehiclesToDelete[self] = self;
	end;
end;

function ExtendedTipTrigger:triggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	
	if self.isEnabled then
		local vehicle = g_currentMission.objectToTrailer[otherId];		
		if onEnter then
			if vehicle ~= nil then
				if vehicle.isShovel then
					self.shovelInRange = vehicle;
				end;
				if vehicle.currentFillType == self.currentFillType then
					if vehicle ~= nil and vehicle.allowTipDischarge then
						if g_currentMission.trailerTipTriggers[vehicle] == nil then
							g_currentMission.trailerTipTriggers[vehicle] = {};
						end;
						table.insert(g_currentMission.trailerTipTriggers[vehicle], self);
					end;
				end;
			end;			
		elseif onLeave then
			if self.shovelInRange == vehicle then
				self.shovelInRange = nil;	
			end;
			if vehicle ~= nil and vehicle.allowTipDischarge then
				local triggers = g_currentMission.trailerTipTriggers[vehicle];
				if triggers ~= nil then
					for i=1, table.getn(triggers) do
						if triggers[i] == self then
							table.remove(triggers, i);
							if table.getn(triggers) == 0 then
								g_currentMission.trailerTipTriggers[vehicle] = nil;
							end;
							break;
						end;
					end;
					if table.getn(triggers) == 1 then
						if triggers[1] == self then
							g_currentMission.trailerTipTriggers[vehicle] = nil;
						end;
					end;
				end;
			end;
		end;
		
		-- check for ls2011 shovel	
		if (onEnter or onLeave) and self.currentFillType == Fillable.FILLTYPE_MANURE then
			local shovel = g_currentMission.nodeToVehicle[otherShapeId];
			if onLeave then
				if self.currentShovel == shovel then
					self.currentShovel = nil;
				end;
			else
				if shovel ~= nil and shovel.setManureIsFilled ~= nil and shovel.manureCapacity ~= nil then
					self.currentShovel = shovel;
				end;
			end;
		end;
	end;
end;

function ExtendedTipTrigger:onLoad(xmlFile, key)
end;

function ExtendedTipTrigger:onSave()
	local attributes = "";
	return attributes;
end;