--
-- AlternativeTipTrigger
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date:		01/08/11
-- @version:	v1.0
-- @history:	v1.0 - initial implementation
--


-- saves the mod path and name on sourcing
g_alternativeTipTriggerDir = g_currentModDirectory;
g_alternativeTipTriggerName = g_currentModName;
g_isAlternativeTipTriggerLoaded = false;

AlternativeTipTrigger = {};

function AlternativeTipTrigger:loadMap(name)
	
	if not g_isAlternativeTipTriggerLoaded then	
		g_currentMission.tipTriggerRangeThreshold = 4;
		g_currentMission.alternativeTipTrigger = self;
		
		self.baseDirectory = g_alternativeTipTriggerDir;
		local xmlPath = self.baseDirectory .. "AlternativeTipTrigger.xml"
		local xmlFile = loadXMLFile("TipTriggerXML", xmlPath);

		self.triggerNodePosition = nil;
		self.fixpointPosition = nil;
		self.adjustPosition = nil;
		
		self.triggers = {};
		local i=0;
		while true do
			local triggerName = string.format("alternativeTipTrigger.triggers.trigger(%d)", i);
			local trigger = {};
			trigger.fileName = getXMLString(xmlFile, triggerName .. "#fileName");
			if trigger.fileName == nil then
				break;
			end;
			trigger.xmlFile = xmlPath;
			trigger.key = triggerName;
			trigger.className = g_alternativeTipTriggerName .. "." .. Utils.getNoNil(getXMLString(xmlFile, triggerName .. "#className"),"ExtendedTipTrigger");
			trigger.fixpointStr = getXMLString(xmlFile, triggerName .. "#fixpoint");
			trigger.trigger = getXMLString(xmlFile, triggerName .. "#trigger");
			
			trigger.planes = {};
			local j=0;
			while true do
				local planeName = string.format(triggerName .. ".plane(%d)", j);
				local plane = {};
				plane.type = getXMLString(xmlFile, planeName .. "#type");
				if plane.type == nil then
					break;
				end;
				plane.path = getXMLString(xmlFile, planeName .. "#index");
				table.insert(trigger.planes, plane);
				j = j + 1;
			end;
			
			table.insert(self.triggers, trigger);		
			i = i + 1;
		end;
		self.offset = 0.1;

		self:loadTriggers();
		delete(xmlFile);
		g_isAlternativeTipTriggerLoaded = true;
	end;
end;

function AlternativeTipTrigger:deleteMap()
	g_isAlternativeTipTriggerLoaded = false;
end;

function AlternativeTipTrigger:mouseEvent(posX, posY, isDown, isUp, button)
end;

function AlternativeTipTrigger:keyEvent(unicode, sym, modifier, isDown)
end;

function AlternativeTipTrigger:update(dt)
end;

function AlternativeTipTrigger:draw()
	if g_currentMission ~= nil then
		for _,trigger in pairs(g_currentMission.tipTriggers) do
			if trigger.draw ~= nil then
				trigger:draw();
			end;
		end;
	end;
end;

function AlternativeTipTrigger:getTriggerByFilltype(fillType)
	
	local trigger = nil;
	for _, v in pairs(self.triggers) do
		for _, plane in pairs(v.planes) do
			if plane.type == fillType then
				trigger = v;
				break;
			end;
		end;
		if trigger ~= nil then
			break;
		end;
	end;	
	
	return trigger;
end;


function AlternativeTipTrigger:loadTriggers()
	if g_currentMission.missionInfo.isValid then
		local savegamePath = getUserProfileAppPath() .. "savegame".. g_currentMission.missionInfo.savegameIndex .. "/alternativeTipTrigger.xml";
		local fileExist = io.open (savegamePath, "r");
		if fileExist ~= nil then
			local xmlFile = loadXMLFile("AlternativeTipTrigger", savegamePath);
			local i = 0;
			while true do
				local triggerName = string.format("alternativeTipTrigger.triggers.trigger(%d)", i);
				local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, triggerName .. "#position"));
				if x == nil then
					break;
				end;
				local offset = getXMLFloat(xmlFile, triggerName .. "#offset");
				local fillType = getXMLString(xmlFile, triggerName.."#fruit");
				local rot = {Utils.getVectorFromString(getXMLString(xmlFile, triggerName .. "#rotation"))};
				self:createTrigger({x,y,z,offset}, rot, fillType, Utils.getNoNil(getXMLFloat(xmlFile, triggerName .. "#fillLevel"), 0));
				i = i + 1;
			end;
			delete(xmlFile);
		end;
	end;
end;

function AlternativeTipTrigger:createTrigger(trans, rot, fillType, fillLevel, positionNode, directionNode)
	local trigger = self:getTriggerByFilltype(fillType);
	if trigger ~= nil then
		local triggerFile = trigger.fileName;
		local callString = "triggerClass = " .. trigger.className;
		loadstring(callString)();
		if triggerClass ~= nil then
			local triggerRoot = getChildAt(Utils.loadSharedI3DFile(triggerFile, self.baseDirectory), 0);
			local transform = getChildAt(triggerRoot, 0);
			trigger.transform = transform;
			local triggerId = Utils.indexToObject(triggerRoot, trigger.trigger);
			trigger.fixpoint = Utils.indexToObject(triggerRoot, trigger.fixpointStr);
			link(getRootNode(), triggerRoot);			
			if trans == nil and rot == nil and positionNode ~= nil then
				setTranslation(triggerRoot, getWorldTranslation(positionNode));
				
				local xUp,yUp,zUp = localDirectionToWorld(directionNode,0,1,0);
				local dx,dy,dz = localDirectionToWorld(directionNode,0,0,1);
				setDirection(triggerRoot, dx, dy, dz, xUp, yUp, zUp);
				
				--setRotation(triggerRoot, getWorldRotation(positionNode));
				local ax,ay,az = getWorldTranslation(trigger.transform);
				local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, ax,ay,az);
				ay = math.max(ay, terrainHeight+1.5);
				raycastAll(ax, ay, az, 0, -1, 0, "triggerNodeCallback", 10, self);
				if self.triggerNodePosition ~= nil then
					ax,ay,az = unpack(self.triggerNodePosition);
				end;
				ay = math.max(ay, terrainHeight);
				setTranslation(transform, worldToLocal(triggerRoot, ax,ay,az));				
				local bx, by, bz = getWorldTranslation(trigger.fixpoint);
				terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, bx,by,bz);
				by = math.max(by, terrainHeight+1.5);
				raycastAll(bx, by, bz, 0, -1, 0, "fixpointCallback", 10, self);
				if self.fixpointPosition ~= nil then
					bx,by,bz = unpack(self.fixpointPosition);
				end;
				by = math.max(by, terrainHeight);
				local x, y, z = worldDirectionToLocal(getParent(transform), bx-ax, by-ay, bz-az);
				setDirection(transform, x, y, z, 0, 1, 0);					
			else
				setRotation(transform,  worldDirectionToLocal(triggerRoot, unpack(rot)));
				local x,y,z,offset = unpack(trans);
				setTranslation(transform, worldToLocal(triggerRoot, x,y,z));
				local a,b,c = getTranslation(getChildAt(triggerId, 0));
				setTranslation(getChildAt(triggerId, 0), 0, offset, 0);
			end;
				
			setRigidBodyType(triggerId, "Static");
			
			local triggerObj = triggerClass:new(g_server ~= nil, g_client ~= ni);
			triggerObj:load(triggerId, trigger, Fillable.fillTypeNameToInt[fillType]);			
			triggerObj:register(true);			
			if triggerObj.onLoad ~= nil then
				triggerObj:onLoad(xmlFile, triggerName);		
			end;
			triggerObj:updateMoving(Utils.getNoNil(fillLevel, 1));

			return triggerObj;
			
		else
			print("Error: Class not found");
		end;
	else	
		print("Error: Trigger could not be loaded! " .. fillType .. " not found");
	end;	
	return nil;	
end;

function AlternativeTipTrigger:addNewTrigger(vehicle, forceCreate, noEventSend)
	if g_server == nil and not forceCreate then
		-- request server to create trigger obj
		AddAlternativeTriggerEvent.sendEvent(vehicle, AddAlternativeTriggerEvent.TRIGGER_CREATE, -1, noEventSend);
	else
		if vehicle.tipState == Trailer.TIPSTATE_CLOSED and vehicle.fillLevel > 0 then
			local fillType = Fillable.fillTypeIntToName[vehicle.currentFillType];
			local triggerObj = self:createTrigger(nil, nil, fillType, 1, vehicle.triggerPlacement, vehicle.rootNode);
			if triggerObj ~= nil then
				self:adjustPlaneToGround(triggerObj);
				if g_currentMission.trailerTipTriggers[vehicle] == nil then
					g_currentMission.trailerTipTriggers[vehicle] = {};
				end;
				table.insert(g_currentMission.trailerTipTriggers[vehicle], triggerObj);
				g_currentMission.trailerInTipRange = vehicle;
				g_currentMission.currentTipTrigger = triggerObj;
					
				AddAlternativeTriggerEvent.sendEvent(vehicle, AddAlternativeTriggerEvent.TRIGGER_SEND, triggerObj.id, noEventSend);
				vehicle:onStartTip(g_currentMission.currentTipTrigger);
			end;
			
			return triggerObj;
		else
			print("Trailer empty or not closed");
		end;
	end;
end;

function AlternativeTipTrigger:adjustPlaneToGround(trigger)
	local adjustNode = getChildAt(trigger.triggerId,1);
	local x,y,z = getWorldTranslation(adjustNode);
	local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x,y,z);	
	local dirX, dirY, dirZ =  localDirectionToWorld(adjustNode, 0,-1,0);
	
	raycastClosest(x, y, z, dirX, dirY, dirZ, "adjustCallback", 10, self);
	if self.adjustPosition ~= nil then
		x,y,z = unpack(self.adjustPosition);		
		y = math.max(y, terrainHeight);
	else
		y = terrainHeight;
	end;	
	
	local node = getChildAt(trigger.triggerId,0);
	x,y,z = worldToLocal(getParent(node), x, y, z);
	setTranslation(node, 0, y + 0.3, 0);
end;

function AlternativeTipTrigger:isCallbackValid(transformId)
	local isValid = true;
	for k, vehicle in pairs(g_currentMission.vehicles) do	
		for j, component in pairs(vehicle.components) do
			if component.node == transformId then
				isValid = false;
				break;
			end;
		end;
		if not isValid then
			break;
		end;
	end;
	if isValid then
		for _, trigger in pairs(g_currentMission.tipTriggers) do	
			if trigger.isExtendedTrigger then
				if trigger.triggerId == transformId then
					isValid = false;
					break;
				end;
			end;
		end;
	end;
	return isValid;
end;

function AlternativeTipTrigger:triggerNodeCallback(transformId, x, y, z, distance)
	if transformId == g_currentMission.terrainRootNode or  self:isCallbackValid(transformId) then
		self.triggerNodePosition = {x,y,z};	
		return false;
	else
		return true;
	end;
end;

function AlternativeTipTrigger:fixpointCallback(transformId, x, y, z, distance)	
	if transformId == g_currentMission.terrainRootNode or self:isCallbackValid(transformId) then
		self.fixpointPosition = {x,y,z};
		return false;
	else
		return true;
	end;
end;

function AlternativeTipTrigger:adjustCallback(transformId, x, y, z, distance)	
	if transformId == g_currentMission.terrainRootNode or self:isCallbackValid(transformId) then
		self.adjustPosition = {x,y,z};
		return false;
	else
		return true;
	end;
end;

-- Add event listener
addModEventListener(AlternativeTipTrigger);
	

-- Save-Game-Wrapper
local oldSaveFunction = CareerScreen.saveSelectedGame;
CareerScreen.saveSelectedGame = function(self)
	oldSaveFunction(self);
	local dir = self.savegames[self.selectedIndex].savegameDirectory;
    createFolder(dir);

	local xmlFile = io.open (dir .. "/alternativeTipTrigger.xml", "w");
    if xmlFile ~= nil then
        xmlFile:write('<?xml version="1.0" encoding="utf-8" standalone="no" ?>\n<alternativeTipTrigger>\n');

		xmlFile:write('  <triggers>\n');
		for k,trigger in pairs(g_currentMission.tipTriggers) do
			if trigger:isa(ExtendedTipTrigger) then
				local transform = trigger.transform;
				local x,y,z = getWorldTranslation(transform);
				local a,b,c = getWorldRotation(transform);
				local ax, ay, az = getTranslation(getChildAt(trigger.triggerId, 0));
				local fruit = Fillable.fillTypeIntToName[trigger.currentFillType];
				local attributes = "";
				if trigger.onSave ~= nil then
					attributes = Utils.getNoNil(trigger:onSave(), "");
				end;
				xmlFile:write('    <trigger position="'.. x .. ' ' .. y .. ' ' .. z .. '" rotation="'.. a .. ' ' .. b .. ' ' .. c .. '" offset="'.. ay ..'" fillLevel="'.. string.format("%d", trigger.fillLevel) ..'" fruit="'.. fruit ..'" ' .. attributes .. ' />\n');
			end;
		end;
		xmlFile:write('  </triggers>\n');
        xmlFile:write("</alternativeTipTrigger>");		
        xmlFile:close();
    end;
end;

