--
-- dietFeeder
-- Class for diet feeder wagon
--
-- @author  PeterJ_LS-UK modteam
-- @date  09/10/2011
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
  

dietFeeder = {};

function dietFeeder.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Fillable, specializations) and SpecializationUtil.hasSpecialization(PowerShaft, specializations);
end;
 
function dietFeeder:load(xmlFile)
	
	local youHaveDLC = SpecializationUtil.getSpecialization("pdlc_dlcPack2.newShovel");	
	if youHaveDLC ~= nil then
		self.addSilage = true;
	else
		self.addSilage = false;
	end;

	self.setIsTurnedOn = SpecializationUtil.callSpecializationsFunction("setIsTurnedOn");
	self.isTurnedOn = false;
	self.baleValue = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.bale#value"), 0.0);
	self.baleDestroyer = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.bale#destroyer"));
	if self.baleDestroyer == nil then
		self.baleDestroyer = self.components[1].node;
	end;

	--- rotation parts ---
	self.rotateSpinners = {};
	local i=0;
	while true do
		local baseName = string.format("vehicle.rotateSpinners.node(%d)", i);
		local index = getXMLString(xmlFile, baseName.. "#index");
		local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.. "#rotationSpeed"));
		local rotationSpeed = {x,y,z};
		local runOutTime = Utils.getNoNil(getXMLFloat(xmlFile, baseName.. "#runOutTime"), 2)*1000;
		if index == nil or rotationSpeed == nil or runOutTime == nil then
			break;
		end;
		local node = Utils.indexToObject(self.components, index);
		if node ~= nil then
			local entry = {};
			entry.node = node;
			entry.runOutTime = runOutTime;
			entry.rotationSpeedMax = rotationSpeed;
			entry.rotationSpeedMin = {0,0,0};
			entry.rotationSpeedCurrent = {0,0,0};
			table.insert(self.rotateSpinners, entry);
		end;
		i = i+1;
	end;	
	
	--- sounds ---	
	local rotarySound = getXMLString(xmlFile, "vehicle.rotarySound#file");
	if rotarySound ~= nil and rotarySound ~= "" then
		rotarySound = Utils.getFilename(rotarySound, self.baseDirectory);
		self.rotarySound = createSample("rotarySound");
		loadSample(self.rotarySound, rotarySound, false);
		self.rotarySoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.rotarySound#pitchOffset"), 0);
		self.rotarySoundVolume = 0.0;
		self.rotarySoundVolumeMin = 0.0;
		self.rotarySoundVolumeMax = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.rotarySound#volume"), 1.0);
		self.rotarySoundEnabled = false;
	end;

	--- particle system ---
	self.unloadLeftParticleSystems = {};
	Utils.loadParticleSystem(xmlFile, self.unloadLeftParticleSystems, "vehicle.ParticleSystems.unloadLeft", self.components, false, nil, self.baseDirectory);
	
	self.unloadRightParticleSystems = {};
	Utils.loadParticleSystem(xmlFile, self.unloadRightParticleSystems, "vehicle.ParticleSystems.unloadRight", self.components, false, nil, self.baseDirectory);
	
	--- tipping side ---
	self.tipSideLeft = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.tipReferencePoint#Left"));
	self.tipSideRight = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.tipReferencePoint#Right"));
	self.tipReferencePoint = self.tipSideLeft
	self.unloadSide = 0;
  	self.setUnloadSide = SpecializationUtil.callSpecializationsFunction("setUnloadSide");
	
	---- tipping animation ----
	self.setLeftDoor = SpecializationUtil.callSpecializationsFunction("setLeftDoor");
	self.LeftDoorAnimation = getXMLString(xmlFile, "vehicle.leftDoor#animationName");
	self.LeftDoor = false;
	
	self.setRightDoor = SpecializationUtil.callSpecializationsFunction("setRightDoor");
	self.RightDoorAnimation = getXMLString(xmlFile, "vehicle.rightDoor#animationName");
	self.RightDoor = false;
	
	---- manual fruit type change ----
	self.setFruitType = SpecializationUtil.callSpecializationsFunction("setFruitType");
	self.fruitType = 0;
	
	--- convoyer ---
	self.UVScrollLeft = {};
	local i = 0;
	while true do
		local key = string.format("vehicle.UVScrollParts.UVScrollLeft(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;
		local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key.."#index"));
		local speed = Utils.getVectorNFromString(getXMLString(xmlFile, key.."#speed"), 2);
		if node ~= nil and speed then
			table.insert(self.UVScrollLeft, {node=node, speed=speed});
		end;
		i = i +1;
	end;
	
	self.UVScrollRight = {};
	local i = 0;
	while true do
		local key = string.format("vehicle.UVScrollParts.UVScrollRight(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;
		local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key.."#index"));
		local speed = Utils.getVectorNFromString(getXMLString(xmlFile, key.."#speed"), 2);
		if node ~= nil and speed then
			table.insert(self.UVScrollRight, {node=node, speed=speed});
		end;
		i = i +1;
	end;
	
 	self.setVehicleRpmUp = SpecializationUtil.callSpecializationsFunction("setVehicleRpmUp");
	self.saveMinRpm = 0; 

	--- plane movement ---
	self.movingPlane = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.movingPlane#index"));
	
	self.lastFillLevel = self.fillLevel;
	self.unloadWarningTime = 0;
	self.printWarningTime = 0;
end;
  
function dietFeeder:delete()
	if self.rotarySound ~= nil then
		delete(self.rotarySound);
	end;
	Utils.deleteParticleSystem(self.unloadLeftParticleSystems);
	Utils.deleteParticleSystem(self.unloadRightParticleSystems);
end;
  
function dietFeeder:readStream(streamId, connection)
	local turnedOn = streamReadBool(streamId);
	self:setIsTurnedOn(turnedOn, true);
	self:setLeftDoor(streamReadBool(streamId), true);
	self:setRightDoor(streamReadBool(streamId), true);
end;
  
function dietFeeder:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isTurnedOn);
	streamWriteInt8(streamId, self.unloadSide);
	streamWriteBool(streamId, self.LeftDoor);
	streamWriteBool(streamId, self.RightDoor);
	streamWriteInt8(streamId, self.fruitType);
end;

function dietFeeder:mouseEvent(posX, posY, isDown, isUp, button)
end;
  
function dietFeeder:keyEvent(unicode, sym, modifier, isDown)
end;
 
function dietFeeder:update(dt)
  
	if self.isClient then
		if self:getIsActiveForInput() then
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
				self:setIsTurnedOn(not self.isTurnedOn);
			end;
			if self.tipState == Trailer.TIPSTATE_CLOSED then
				if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA2) then
					self:setUnloadSide(math.abs(self.unloadSide + 1));
				end;
				if self.fillLevel > 0 then
					if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA3) then
						self:setFruitType(math.abs(self.fruitType + 1));
					end;
				end;
			end;
			if InputBinding.isPressed(InputBinding.IMPLEMENT_EXTRA) and self.PTOId then
				self.printWarningTime = self.time + 1500;
			end;
		end;
	end;
end;
  
function dietFeeder:updateTick(dt)
	if self:getIsActive() or not self:getIsActive() then
		if self.isTurnedOn then
				for k, spinner in pairs(self.rotateSpinners) do
				local values = Utils.getMovedLimitedValues(spinner.rotationSpeedCurrent, spinner.rotationSpeedMax, spinner.rotationSpeedMin, 3, spinner.runOutTime, dt, not self.isTurnedOn);
				spinner.rotationSpeedCurrent = values;
				rotate(spinner.node, unpack(spinner.rotationSpeedCurrent));
			end;
		end;
	end;
	if self:getIsActive() then
	
		if not self.attacherVehicle.isMotorStarted then
			self:setIsTurnedOn(false, true);
		end;
		if self.PTOId then
			self:setIsTurnedOn(false, true);
		end;
		local isKeyEvent = false;	
		if self.isTurnedOn then
 			isKeyEvent = true;	
		else
 			isKeyEvent = false;	
		end;
		
			self:setVehicleRpmUp(dt, isKeyEvent);	
			local volume = Utils.getMovedLimitedValues({self.rotarySoundVolume}, {self.rotarySoundVolumeMax}, {self.rotarySoundVolumeMin}, 1, 1200, dt, not self.isTurnedOn);
			self.rotarySoundVolume = volume[1];
			if not (self.PTOId and self.attacherVehicle.isMotorStarted) then
				if not self.rotarySoundEnabled and self:getIsActiveForSound() then
					playSample(self.rotarySound, 0, self.rotarySoundVolume, 0);
					setSamplePitch(self.rotarySound, self.rotarySoundPitchOffset);
					self.rotarySoundEnabled = true;
				end;
				if self.rotarySoundVolume < self.rotarySoundVolumeMax then
					setSampleVolume(self.rotarySound, self.rotarySoundVolume);
				end;
				else
				if self.rotarySoundEnabled then
					if self.rotarySoundVolume <= self.rotarySoundVolumeMin then
						stopSample(self.rotarySound);
						self.rotarySoundEnabled = false;
					else
						setSampleVolume(self.rotarySound, self.rotarySoundVolume);
					end;
				end;

			end;
	end;
	if self.isClient then
		local nearestDistance = 5.0;
		local px, py, pz = getWorldTranslation(self.baleDestroyer);
		for i=1, table.getn(g_currentMission.attachables) do
			local vx, vy, vz = getWorldTranslation(g_currentMission.attachables[i].rootNode);
			local distance = Utils.vector3Length(px-vx, py-vy, pz-vz);
			if distance < nearestDistance then
				local attachable = g_currentMission.attachables[i];		
				if attachable.fruitTypes == self.fruitTypes and attachable.fillLevel ~= nil and attachable ~= self then
					if self.currentFillType ~= attachable.currentFillType and attachable.currentFillType ~= Fillable.FILLTYPE_UNKNOWN then
						if self.fillLevel < self.capacity and self.fillLevel > 1 then
							self.currentFillType = attachable.currentFillType;
						end;
					end;
				end;
			end;
		end;
		if self.isTurnedOn then
			local baleDistance = 1.2;
			local px, py, pz = getWorldTranslation(self.baleDestroyer);
			for index, item in pairs(g_currentMission.itemsToSave) do
				if item.item:isa(Bale) then
					local vx, vy, vz = getWorldTranslation(item.item.nodeId);
					local distance = Utils.vector3Length(px-vx, py-vy, pz-vz);
					if distance < baleDistance then
						local baleObject = item.item;
						if baleObject ~= nil and baleObject:isa(Bale) then
							if (self.fillLevel + self.baleValue) < self.capacity then
								baleObject:delete();
								if self.fillLevel > 0 then 
									self:setFillLevel(self.fillLevel + self.baleValue, self.currentFillType);
								else
									self:setFillLevel(self.fillLevel + self.baleValue, Fillable.FILLTYPE_GRASS);
								end;
							end;
						end;
					end;
				end;
			end;
			setVisibility(self.movingPlane, true);
		else
			self.tipState = Trailer.TIPSTATE_CLOSED;
			setVisibility(self.movingPlane, false);
			for _, part in pairs(self.UVScrollLeft) do
				setShaderParameter(part.node, "uvScrollSpeed", 0, 0, 0, 0, false);
			end;
			for _, part in pairs(self.UVScrollRight) do
				setShaderParameter(part.node, "uvScrollSpeed", 0, 0, 0, 0, false);
			end;
			if self.lastFillLevel > self.fillLevel then
				self.unloadWarningTime = self.time + 1500;
			end;
		end;
		self.lastFillLevel = self.fillLevel;
		if (self.tipState == Trailer.TIPSTATE_OPENING or self.tipState == Trailer.TIPSTATE_OPEN) and self.fillLevel > 0 then
			if self.unloadSide == 0 then
				Utils.setEmittingState(self.unloadLeftParticleSystems, true);
				self:setLeftDoor(true,false);
				for _, part in pairs(self.UVScrollLeft) do
					setShaderParameter(part.node, "uvScrollSpeed", part.speed[1], part.speed[2], part.speed[3], 0, false);
				end;
			else
				Utils.setEmittingState(self.unloadRightParticleSystems, true);
				self:setRightDoor(true,false);
				for _, part in pairs(self.UVScrollRight) do
					setShaderParameter(part.node, "uvScrollSpeed", part.speed[1], part.speed[2], part.speed[3], 0, false);
				end;
			end;
		else
			Utils.setEmittingState(self.unloadLeftParticleSystems, false);
			Utils.setEmittingState(self.unloadRightParticleSystems, false);
			self:setLeftDoor(false,true);
			self:setRightDoor(false,true);
		end;
		if self.tipState == Trailer.TIPSTATE_CLOSED then
			for _, part in pairs(self.UVScrollLeft) do
				setShaderParameter(part.node, "uvScrollSpeed", 0, 0, 0, 0, false);
			end;
			for _, part in pairs(self.UVScrollRight) do
				setShaderParameter(part.node, "uvScrollSpeed", 0, 0, 0, 0, false);
			end;
		end;
		if self.unloadSide == 0  then
			self.tipReferencePoint = self.tipSideLeft;
		else
			self.tipReferencePoint = self.tipSideRight;
		end;
	end;

end;

function dietFeeder:draw()
	if self.isClient then
		if self.printWarningTime > self.time then
			g_currentMission:addWarning(g_i18n:getText("turnON_Error"), 0.018, 0.033);
		end;
		if self.unloadWarningTime > self.time then
			g_currentMission:addWarning(g_i18n:getText("unload_Error"), 0.018, 0.033);
		end;
		if self.isTurnedOn then
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_off_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
		else
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_on_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
		end;
		if self.tipState == Trailer.TIPSTATE_CLOSED then
			if self.unloadSide == 0 then
				g_currentMission:addHelpButtonText(g_i18n:getText("Right"), InputBinding.IMPLEMENT_EXTRA2);
			else
				g_currentMission:addHelpButtonText(g_i18n:getText("Left"), InputBinding.IMPLEMENT_EXTRA2);
			end;
			if self.fillLevel > 0 then
				g_currentMission:addHelpButtonText(g_i18n:getText("OutPutFruit"), InputBinding.IMPLEMENT_EXTRA3);
			end;
		end;
	end;
end;

function dietFeeder:onAttach(attacherVehicle)
	self.attacherVehicle = attacherVehicle;
	if self.attacherVehicleCopy == nil then
		self.attacherVehicleCopy = self.attacherVehicle;
	end;
	self.saveMinRpm = self.attacherVehicle.motor.minRpm;
end;
 
function dietFeeder:onDetach()
	if self.deactivateOnDetach then
		dietFeeder.onDeactivate(self);
	else
		dietFeeder.onDeactivateSounds(self)
	end;
	for k, steerable in pairs(g_currentMission.steerables) do
		if self.attacherVehicleCopy == steerable then
			steerable.motor.minRpm = self.saveMinRpm;
			self.attacherVehicleCopy = nil;
		end;
	end;
end;
  
function dietFeeder:onLeave()
	if self.deactivateOnLeave then
		dietFeeder.onDeactivate(self);
	else
		dietFeeder.onDeactivateSounds(self)
	end;
end;

function dietFeeder:onDeactivate()
	dietFeeder.onDeactivateSounds(self)
  
	self.isTurnedOn = false;
	
	for _, part in pairs(self.UVScrollLeft) do
		setShaderParameter(part.node, "uvScrollSpeed", 0, 0, 0, 0, false);
	end;
	for _, part in pairs(self.UVScrollRight) do
		setShaderParameter(part.node, "uvScrollSpeed", 0, 0, 0, 0, false);
	end;
end;

function dietFeeder:onDeactivateSounds()
	if self.isClient then
	if self.rotarySoundEnabled then
		stopSample(self.rotarySound);
		self.rotarySoundEnabled = false;
	end;
	end;
end;

function dietFeeder:setVehicleRpmUp(dt, isActive)
	if self.attacherVehicle ~= nil and self.saveMinRpm ~= 0 then
		if dt ~= nil then
			if isActive == true then
				self.attacherVehicle.motor.minRpm = math.max(self.attacherVehicle.motor.minRpm-dt, -1000);
			else
				self.attacherVehicle.motor.minRpm = math.min(self.attacherVehicle.motor.minRpm+dt*2, self.saveMinRpm);
			end;
		else
			self.attacherVehicle.motor.minRpm = self.saveMinRpm;
		end;
		if self.attacherVehicle.isMotorStarted then
			local fuelUsed = 0.0000011*math.abs(self.attacherVehicle.motor.minRpm);
			self.attacherVehicle:setFuelFillLevel(self.attacherVehicle.fuelFillLevel-fuelUsed);
			g_currentMission.missionStats.fuelUsageTotal = g_currentMission.missionStats.fuelUsageTotal + fuelUsed;
			g_currentMission.missionStats.fuelUsageSession = g_currentMission.missionStats.fuelUsageSession + fuelUsed;
		end;
	end;
end;

function dietFeeder:setIsTurnedOn(turnedOn, noEventSend)
	SetTurnedOnEvent.sendEvent(self, turnedOn, noEventSend)
	self.isTurnedOn = turnedOn;
end;

function dietFeeder:setUnloadSide(unloadSide, noEventSend)
	SetUnloadSideEvent.sendEvent(self, unloadSide, noEventSend);

	self.unloadSide = unloadSide;
	if self.unloadSide == 0  then
		self.tipReferencePoint = self.tipSideLeft;
	end;
	if self.unloadSide == 1 then
		self.tipReferencePoint = self.tipSideRight;
	end;
	if self.unloadSide > 1 then
		self.unloadSide = 0;
	end;
end;

function dietFeeder:setLeftDoor(isLeftDoor,noEventSend)
	SetLeftDoorEvent.sendEvent(self, isLeftDoor, noEventSend);
	-- Play LeftDoor animation --
	self.isLeftDoorOn = isLeftDoor;
	if self.isLeftDoorOn then
		if self.LeftDoorAnimation ~= nil and self.playAnimation ~= nil then
			self:playAnimation(self.LeftDoorAnimation, -1, nil, true);
			self.LeftDoor = true;
		end;
	else
		if self.LeftDoorAnimation ~= nil and self.playAnimation ~= nil then
			self:playAnimation(self.LeftDoorAnimation, 1, nil, true);
			self.LeftDoor = false;
		end;
	end;	
end;

function dietFeeder:setRightDoor(isRightDoor,noEventSend)
	SetRightDoorEvent.sendEvent(self, isRightDoor, noEventSend);
	-- Play RightDoor animation --
	self.isRightDoorOn = isRightDoor;
	if self.isRightDoorOn then
		if self.RightDoorAnimation ~= nil and self.playAnimation ~= nil then
			self:playAnimation(self.RightDoorAnimation, -1, nil, true);
			self.RightDoor = true;
		end;
	else
		if self.RightDoorAnimation ~= nil and self.playAnimation ~= nil then
			self:playAnimation(self.RightDoorAnimation, 1, nil, true);
			self.RightDoor = false;
		end;
	end;	
end;

function dietFeeder:setFruitType(fruitType, noEventSend)
	SetFruitTypeEvent.sendEvent(self, fruitType, noEventSend);

	self.fruitType = fruitType;
	if self.fruitType == 1 then
		self.currentFillType = Fillable.FILLTYPE_GRASS;
	end;
	if self.fruitType == 2  then
		self.currentFillType = Fillable.FILLTYPE_CHAFF;
	end;
	if self.fruitType == 3 then
		self.currentFillType = Fillable.FILLTYPE_SILAGE;
	end;
	if self.addSilage then
		if self.fruitType > 2 then
			self.fruitType = 0;
		end;
	else
		if self.fruitType > 1 then
			self.fruitType = 0;
		end;
	end;
end;
