--
-- NHPowerCommand
-- Specialization for NHPowerCommand gearbox
--
-- @author  JoXXer
-- @date  21/09/11
--

NHPowerCommand = {};

function NHPowerCommand.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Steerable, specializations);
end;

function NHPowerCommand:load(xmlFile)
	-- Functions
	-- Change forward gear
	self.changeFwdGear = SpecializationUtil.callSpecializationsFunction("changeFwdGear");
	-- Change reverse gear
	self.changeRevGear = SpecializationUtil.callSpecializationsFunction("changeRevGear");
	-- Change shuttle direction
	self.changeShuttleDirection = SpecializationUtil.callSpecializationsFunction("changeShuttleDirection");
	-- Change shuttle to neutral
	self.shuttleToNeutral = SpecializationUtil.callSpecializationsFunction("shuttleToNeutral");
	-- Change handbrake state
	self.changeHandbrakeState = SpecializationUtil.callSpecializationsFunction("changeHandbrakeState");
	-- Set acceleration
	self.setLastAcceleration = SpecializationUtil.callSpecializationsFunction("setLastAcceleration");
	-- Set brakeforce
	self.setMotorBrakeForce = SpecializationUtil.callSpecializationsFunction("setMotorBrakeForce");
	-- Set gearRatio
	self.setGearRatio = SpecializationUtil.callSpecializationsFunction("setGearRatio");
	-- Set wheel variables
	self.setWheelVariables = SpecializationUtil.callSpecializationsFunction("setWheelVariables");
	-- Update gear display
	self.updateGearDisplay = SpecializationUtil.callSpecializationsFunction("updateGearDisplay");
	-- For managing footThrottle movement
	self.footThrottleRotation = SpecializationUtil.callSpecializationsFunction("footThrottleRotation");
	-- For managing clutch and brakes movement
	self.clutchAndBrakesRotation = SpecializationUtil.callSpecializationsFunction("clutchAndBrakesRotation");
	-- END Functions

	-- Forward gearratios
	self.numFwdGears = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.forwardGearRatios#count"),18);
	self.fwdRatios = {};
	for i=1, self.numFwdGears do
		local objname = string.format("vehicle.forwardGearRatios.forwardGearRatio" .. "%d", i);
		self.fwdRatios[i] = {};
		local fwdRatio = getXMLString(xmlFile, objname .. "#ratio");
		self.fwdRatios[i] = tonumber(fwdRatio);
	end;
	-- Reverse gearratios
	self.numRevGears = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.reverseGearRatios#count"),8);
	self.revRatios = {};
	for i=1, self.numRevGears do
		local objname = string.format("vehicle.reverseGearRatios.reverseGearRatio" .. "%d", i);
		self.revRatios[i] = {};
		local revRatio = getXMLString(xmlFile, objname .. "#ratio");
		self.revRatios[i] = tonumber(revRatio);
	end;

	-- Sounds NEW
	local powerCommandMotorSound = getXMLString(xmlFile, "vehicle.motorSound#file");
    if powerCommandMotorSound ~= nil and powerCommandMotorSound ~= "" then
        powerCommandMotorSound = Utils.getFilename(powerCommandMotorSound, self.baseDirectory);
        self.powerCommandMotorSound = createSample("powerCommandMotorSound");
        loadSample(self.powerCommandMotorSound, powerCommandMotorSound, false);
        self.powerCommandMotorSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSound#pitchOffset"), 0);
        self.powerCommandMotorSoundPitchScale = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSound#pitchScale"), 0.05);
        self.powerCommandMotorSoundPitchMax = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSound#pitchMax"), 2.0);
        self.powerCommandMotorSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSound#volume"), 1.0);
    end;

    local powerCommandMotorSoundRun = getXMLString(xmlFile, "vehicle.motorSoundRun#file");
    if powerCommandMotorSoundRun ~= nil and powerCommandMotorSoundRun ~= "" then
        powerCommandMotorSoundRun = Utils.getFilename(powerCommandMotorSoundRun, self.baseDirectory);
        self.powerCommandMotorSoundRun = createSample("powerCommandMotorSoundRun");
        loadSample(self.powerCommandMotorSoundRun, powerCommandMotorSoundRun, false);
        self.powerCommandMotorSoundRunPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSoundRun#pitchOffset"), 0);
        self.powerCommandMotorSoundRunPitchScale = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSoundRun#pitchScale"), 0.05);
        self.powerCommandMotorSoundRunPitchMax = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSoundRun#pitchMax"), 2.0);
        self.powerCommandMotorSoundRunVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motorSoundRun#volume"), 1.0);
    end;

	--ExhaustParticles
	self.extraExhaustParticleSystems = {};
    local exhaustParticleSystemCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.extraExhaustParticleSystems#count"), 0);
    for i=1, exhaustParticleSystemCount do
        local namei = string.format("vehicle.extraExhaustParticleSystems.extraExhaustParticleSystem%d", i);
        Utils.loadParticleSystem(xmlFile, self.extraExhaustParticleSystems, namei, self.components, false, nil, self.baseDirectory)
    end;

	self.gearDigits = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.digitGears#index"));
	self.gearDigitsLight = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.digitGearsLight#index"));
	if self.gearDigits ~= nil and self.gearDigitsLight ~= nil then
		setVisibility(self.gearDigits,true);
		setVisibility(self.gearDigitsLight,false);
	end;

	-- ShuttleControl indicators
	self.shuttleIndicatorsIndex = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.shuttleIndicators#index"));
	self.numShuttleIndicators = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.shuttleIndicators#count"), 0);
	self.shuttleIndicators = {};
    for i=1, self.numShuttleIndicators do
        local objname = string.format("vehicle.shuttleIndicators.shuttleIndicator" .. "%d", i);
		self.shuttleIndicators[i] = {};
        self.shuttleIndicators[i].index = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#index"));
		setVisibility(self.shuttleIndicators[i].index,false);
    end;
	if self.shuttleIndicators[3] ~= nil then
		setVisibility(self.shuttleIndicators[3].index,true);
	end;
	if self.shuttleIndicators[1] == nil then
		self.shuttleIndicators = nil;
	end;

	-- ShuttleControl indicators light
	self.shuttleIndicatorsLightIndex = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.shuttleIndicatorsLight#index"));
	self.numShuttleIndicatorsLight = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.shuttleIndicatorsLight#count"), 0);
	self.shuttleIndicatorsLight = {};
    for i=1, self.numShuttleIndicatorsLight do
        local objname = string.format("vehicle.shuttleIndicatorsLight.shuttleIndicator" .. "%d", i);
		self.shuttleIndicatorsLight[i] = {};
        self.shuttleIndicatorsLight[i].index = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#index"));
		setVisibility(self.shuttleIndicatorsLight[i].index,false);
    end;
	if self.shuttleIndicatorsLight[3] ~= nil then
		setVisibility(self.shuttleIndicatorsLight[3].index,true);
	end;

	-- Digital Display digits Gears
	-- Forward Gears
	self.numDigitGearForwardOne = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.displayDigitGEAROne#count"), 0);
	self.digitGearForwardOne = {};
    for i=1, self.numDigitGearForwardOne do
        local objname = string.format("vehicle.displayDigitGEAROne.digitGEAR" .. "%d", i);
		self.digitGearForwardOne[i] = {};
        self.digitGearForwardOne[i].rotNode = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#rotNode"));
		setVisibility(self.digitGearForwardOne[i].rotNode,false);
    end;
	if self.digitGearForwardOne[self.numDigitGearForwardOne] ~= nil then
		setVisibility(self.digitGearForwardOne[self.numDigitGearForwardOne].rotNode,true);
	end;

	-- Forward Gears tens
	self.numDigitGearForwardTwo = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.displayDigitGEARTwo#count"), 0);
	self.digitGearForwardTwo = {};
    for i=1, self.numDigitGearForwardTwo do
        local objname = string.format("vehicle.displayDigitGEARTwo.digitGEAR" .. "%d", i);
		self.digitGearForwardTwo[i] = {};
        self.digitGearForwardTwo[i].rotNode = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#rotNode"));
		setVisibility(self.digitGearForwardTwo[i].rotNode,false);
    end;

	-- Reverse Gears
	self.numDigitGearReverseThree = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.displayDigitGEARThree#count"), 0);
	self.digitGearReverseThree = {};
    for i=1, self.numDigitGearReverseThree do
        local objname = string.format("vehicle.displayDigitGEARThree.digitGEAR" .. "%d", i);
		self.digitGearReverseThree[i] = {};
        self.digitGearReverseThree[i].rotNode = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#rotNode"));
		setVisibility(self.digitGearReverseThree[i].rotNode,false);
    end;

	-- Light digits
	-- Forward Gears
	self.numDigitGearForwardOneLight = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.displayDigitGEAROneLight#count"), 0);
	self.digitGearForwardOneLight = {};
    for i=1, self.numDigitGearForwardOneLight do
        local objname = string.format("vehicle.displayDigitGEAROneLight.digitGEAR" .. "%d", i);
		self.digitGearForwardOneLight[i] = {};
        self.digitGearForwardOneLight[i].rotNode = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#rotNode"));
		setVisibility(self.digitGearForwardOneLight[i].rotNode,false);
    end;

	-- Forward Gears tens
	self.numDigitGearForwardTwoLight = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.displayDigitGEARTwoLight#count"), 0);
	self.digitGearForwardTwoLight = {};
    for i=1, self.numDigitGearForwardTwoLight do
        local objname = string.format("vehicle.displayDigitGEARTwoLight.digitGEAR" .. "%d", i);
		self.digitGearForwardTwoLight[i] = {};
        self.digitGearForwardTwoLight[i].rotNode = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#rotNode"));
		setVisibility(self.digitGearForwardTwoLight[i].rotNode,false);
    end;

	-- Reverse Gears
	self.numDigitGearReverseThreeLight = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.displayDigitGEARThreeLight#count"), 0);
	self.digitGearReverseThreeLight = {};
    for i=1, self.numDigitGearReverseThreeLight do
        local objname = string.format("vehicle.displayDigitGEARThreeLight.digitGEAR" .. "%d", i);
		self.digitGearReverseThreeLight[i] = {};
        self.digitGearReverseThreeLight[i].rotNode = Utils.indexToObject(self.components, getXMLString(xmlFile, objname .. "#rotNode"));
		setVisibility(self.digitGearReverseThreeLight[i].rotNode,false);
    end;

	-- Backup maxRpm
    local motorMaxRpmStr = getXMLString(xmlFile, "vehicle.motor#maxRpm");
    local motorMaxRpm1, motorMaxRpm2, motorMaxRpm3 = Utils.getVectorFromString(motorMaxRpmStr);
    motorMaxRpm1 = Utils.getNoNil(motorMaxRpm1, 800);
    motorMaxRpm2 = Utils.getNoNil(motorMaxRpm2, 1000);
    motorMaxRpm3 = Utils.getNoNil(motorMaxRpm3, 1800);
    local motorMaxRpm = {motorMaxRpm1, motorMaxRpm2, motorMaxRpm3};
    self.motorMaxRpmLimit = motorMaxRpm;


	-- Animation parts
	self.handThrottle = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.handThrottle#index"));
	self.footThrottlePedal = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.footThrottlePedal#index"));
	self.clutchPedal = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.clutchPedal#index"));
	self.brakePedal = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.brakePedal#index"));
	self.handBrakeLever = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.handBrakeLever#index"));

	-- Turbowhistle
	local turboWhistle = getXMLString(xmlFile, "vehicle.turboWhistle#file");
	if turboWhistle ~= nil and turboWhistle ~= "" then
        turboWhistle = Utils.getFilename(turboWhistle, self.baseDirectory);
        self.turboWhistle = createSample("turboWhistle");
        self.playTurboWhistle = false;
		self.playedTurboWhistle = false;
        loadSample(self.turboWhistle, turboWhistle, false);
        self.turboWhistlePitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.turboWhistle#pitchOffset"), 1);
        self.turboWhistleVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.turboWhistle#volume"), 1);
    end;
	-- Turbowhistle interior
	local turboWhistleInterior = getXMLString(xmlFile, "vehicle.turboWhistleInterior#file");
	if turboWhistleInterior ~= nil and turboWhistleInterior ~= "" then
        turboWhistleInterior = Utils.getFilename(turboWhistleInterior, self.baseDirectory);
        self.turboWhistleInterior = createSample("turboWhistleInterior");
		self.playedTurboWhistleInterior = false;
        loadSample(self.turboWhistleInterior, turboWhistleInterior, false);
        self.turboWhistleInteriorPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.turboWhistleInterior#pitchOffset"), 1);
        self.turboWhistleInteriorVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.turboWhistleInterior#volume"), 1);
    end;

	-- HUD --
	self.HUDActive = true;
	self.HUDpath = Utils.getFilename("Textures/NHTM_HUD.png", self.baseDirectory);
	self.HUDWidth = 0.100;
    self.HUDHeight = 0.22;
	self.HUDPoxX = 0.89;
    self.HUDPoxY = 0.51
    self.HUDOverlay = Overlay:new("HUD", self.HUDpath, self.HUDPoxX, self.HUDPoxY, self.HUDWidth, self.HUDHeight);

	-- Variables

	-- Strings
	-- Shuttle directions, default neutral
	self.shuttleDirection = "neutral";

	-- Booleans
	-- Handbrake
	self.handbrake = true;
	-- Tells the overridden Steerable.updateVehiclePhysics that the vehicle has specialization NHPowerCommand
	self.hasNHPowerCommand = true;
	-- If you have changed geared fwd, should rev gear change?
	self.shouldChangeRevGear = false;
	-- If you have changed geared rev, should fwd gear change?
	self.shouldChangeFwdGear = false;
	-- To tell if you have geared up
	self.gearedUp = false;
	-- To tell if you have geared down
	self.gearedDown = false;
	-- If user wants the vehicle to gear down automatically
	self.automaticGearDown = true;

	-- Integers
	-- Current forward gear
	self.currentFwdGear = 12;
	-- Last forward gear
	self.lastFwdGear = 11;
	-- Current reverse gear
	self.currentRevGear = 6;
	-- Last reverse gear
	self.lastRevGear = 5;
	-- Updated acceleration
	self.updatedAcceleration = 0;
	-- New acceleration
	self.newAcceleration = 0;
	-- Power Command minumim RPM
	self.powerCommandMinRpm = 850;
	-- Power Command throttle percentage
	self.powerCommandThrottlePercentage = self.powerCommandMinRpm/self.motorMaxRpmLimit[3];
	-- Acceleration offset
	self.myAccelerationOffset = 0;
	-- Last acceleration
	self.myLastAcceleration = 0;
	-- Gear ratio
	self.gearRatio = 0;
	-- Brakeforce
	self.NHPowerCommandBrakeForce = 0;
	-- Brakepedal
	self.NHPowerCommandBrakePedal = 0;
	-- Motor torque
	self.NHPowerCommandMotorTorque = 0;
	-- Last RPM
	self.lastRoundPerMinute = 0;
	-- RPM value
	self.rpmMeterValue = 0;
	-- Maximum amount of increased RPM on geardown
	self.maxRevOnGearDown = 0;
	-- Backup of minumum RPM
	self.backupMinRpmManual = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.motor#minRpm"), 1000);
	-- Fake maximum RPM coeff
	self.rpmCoefficient = 2200/motorMaxRpm3;

	self.originalForwardGearRatio = self.motor.forwardGearRatios;
	self.originalBackwardGearRation = self.motor.backwardGearRatio;

	-- END Variables
end;

function NHPowerCommand:delete()
	if self.turboWhistle ~= nil then
		delete(self.turboWhistle);
		self.playTurboWhistle = false;
	end;
	if self.turboWhistleInterior ~= nil then
		delete(self.turboWhistleInterior);
		self.playTurboWhistleInterior = false;
	end;
end;

function NHPowerCommand:readStream(streamId, connection)
	self.shuttleDirection = streamReadString(streamId);
	self.currentFwdGear = streamReadInt8(streamId);
	self.currentRevGear = streamReadInt8(streamId);
	self.handbrake = streamReadBool(streamId);
	self.automaticGearDown = streamReadBool(streamId);
	self.powerCommandThrottlePercentage = streamReadFloat32(streamId);
end;

function NHPowerCommand:writeStream(streamId, connection)
	streamWriteString(streamId, self.shuttleDirection);
	streamWriteInt8(streamId, self.currentFwdGear);
	streamWriteInt8(streamId, self.currentRevGear);
	streamWriteBool(streamId, self.handbrake);
	streamWriteBool(streamId, self.automaticGearDown);
	streamWriteFloat32(streamId, self.powerCommandThrottlePercentage);
end;

function NHPowerCommand:readUpdateStream(streamId, timestamp, connection)

end;

function NHPowerCommand:writeUpdateStream(streamId, connection, dirtyMask)

end;

function NHPowerCommand:mouseEvent(posX, posY, isDown, isUp, button)
end;

function NHPowerCommand:keyEvent(unicode, sym, modifier, isDown)
end;

function NHPowerCommand:update(dt)
	if self:getIsActiveForInput() then
		if not self.isMouseActive or self.isMouseActive == nil then
			if InputBinding.hasEvent(InputBinding.NHPOWERCOMMANDGEARUP) then
				if self.shuttleDirection == "forward" and self.currentFwdGear ~= self.numFwdGears then
					self.shouldChangeRevGear = true;
					self:changeFwdGear(self.currentFwdGear+1);
				elseif self.shuttleDirection == "reverse" and self.currentRevGear ~= self.numRevGears then
					self.shouldChangeFwdGear = true;
					self:changeRevGear(self.currentRevGear+1);
				end;
			elseif InputBinding.hasEvent(InputBinding.NHPOWERCOMMANDGEARDOWN) then
				if self.shuttleDirection == "forward" and self.currentFwdGear ~= 1 then
					self.shouldChangeRevGear = true;
					self:changeFwdGear(self.currentFwdGear-1);
				elseif self.shuttleDirection == "reverse" and self.currentRevGear ~= 1 then
					self.shouldChangeFwdGear = true;
					self:changeRevGear(self.currentRevGear-1);
				end;
			end;
		end;

		-- HUD --
		if InputBinding.hasEvent(InputBinding.NHHUD) then
			self.HUDActive = not self.HUDActive;
		end;

		if InputBinding.hasEvent(InputBinding.NHPOWERSHUTTLEDIRECTION) then
			self:changeShuttleDirection(self.shuttleDirection);
		end;

		if InputBinding.hasEvent(InputBinding.NHPOWERSHUTTLENEUTRAL) then
			self:shuttleToNeutral(true);
		end;

		if InputBinding.hasEvent(InputBinding.NHHANDBRAKE) then
			self:changeHandbrakeState(self.handbrake);
		end;

		g_currentMission:addExtraPrintText(g_i18n:getText("GEARCHANGES"));
		if self.shuttleDirection == "forward" then
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("SHUTTLEFORWARD"), self.typeDesc), InputBinding.NHPOWERSHUTTLEDIRECTION);
		elseif self.shuttleDirection == "reverse" or self.shuttleDirection == "neutral" then
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("SHUTTLEREVERSE"), self.typeDesc), InputBinding.NHPOWERSHUTTLEDIRECTION);
		end;
		if self.shuttleDirection ~= "neutral" then
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("SHUTTLENEUTRAL"), self.typeDesc), InputBinding.NHPOWERSHUTTLENEUTRAL);
		end;

		if self.handbrake then
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("HANDBRAKEON"), self.typeDesc), InputBinding.NHHANDBRAKE);
		else
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("HANDBRAKEOFF"), self.typeDesc), InputBinding.NHHANDBRAKE);
		end;

		-- Raise and lower the throttle percentage
		if InputBinding.isPressed(InputBinding.NHPOWERCOMMANDMORERPM) and self.powerCommandThrottlePercentage < 1 then
			self.powerCommandThrottlePercentage = self.powerCommandThrottlePercentage + 0.0002 * dt;
		elseif InputBinding.isPressed(InputBinding.NHPOWERCOMMANDLESSRPM) and self.powerCommandThrottlePercentage > self.powerCommandMinRpm/self.motorMaxRpmLimit[3] then
			self.powerCommandThrottlePercentage = self.powerCommandThrottlePercentage - 0.0002 * dt;
		end;

		if self.powerCommandThrottlePercentage > 1 then
			self.powerCommandThrottlePercentage = 1;
		end;

		--renderText(0.5, 0.2, 0.020, string.format(self.powerCommandThrottlePercentage));

		-- Disable hand throttle
		if InputBinding.hasEvent(InputBinding.NHPOWERCOMMANDRESETRPM) then
			self.motor.minRpm = self.backupMinRpmManual;
			self.powerCommandThrottlePercentage = self.powerCommandMinRpm/self.motorMaxRpmLimit[3];
		end;

		if InputBinding.hasEvent(InputBinding.CHANGEAUTOMATICGEARDOWNSTATE) then
			self.automaticGearDown = not self.automaticGearDown;
		end;

		g_currentMission:addHelpButtonText(string.format(g_i18n:getText("CHANGEAUTOMATICGEARDOWNSTATE"), self.typeDesc), InputBinding.CHANGEAUTOMATICGEARDOWNSTATE);
	end;
end;

function NHPowerCommand:updateTick(dt)
	if not self.isAITractorActivated then
		if self.isEntered then
			if self.isMotorStarted then
				if self.lightsActive then
					if self.gearDigits ~= nil and self.gearDigitsLight then
						setVisibility(self.gearDigits,false);
						setVisibility(self.gearDigitsLight,true);
					end;
					if self.shuttleIndicatorsIndex ~= nil and self.shuttleIndicatorsLightIndex then
						setVisibility(self.shuttleIndicatorsIndex,false);
						setVisibility(self.shuttleIndicatorsLightIndex,true);
					end;
				else
					if self.gearDigits ~= nil and self.gearDigitsLight then
						setVisibility(self.gearDigits,true);
						setVisibility(self.gearDigitsLight,false);
					end;
					if self.shuttleIndicatorsIndex ~= nil and self.shuttleIndicatorsLightIndex then
						setVisibility(self.shuttleIndicatorsIndex,true);
						setVisibility(self.shuttleIndicatorsLightIndex,false);
					end;
				end;
			end;

			if not self.isAITractorActivated then
				local acceleration = self.newAcceleration;

				local brake = false;

				if self.handbrake or self.isBraking then
					brake = true;
					acceleration = 0;
				end;
				if self.firstTimeRun then
					local accelerationPedal = 0;
					self.NHPowerCommandBrakePedal = 0;

					if brake then
						accelerationPedal = 0;
						self.NHPowerCommandBrakePedal = 1;
					else
						accelerationPedal = acceleration;
					end;

					local numTouching = 4;
					local numNotTouching = 0;

					self.NHPowerCommandMotorTorque = 0;
					if numTouching > 0 and math.abs(accelerationPedal) > 0.01 then
						local axisTorque, brakePedalMotor = WheelsUtil.computeAxisTorque(self, accelerationPedal);
						if axisTorque ~= 0 then
							self.NHPowerCommandMotorTorque = axisTorque / (numTouching+numNotTouching);
						else
							if not brake then
								self.NHPowerCommandBrakePedal = brakePedalMotor;
							end;
						end;
					else
						self.motor:computeMotorRpm(self.wheelRpm, accelerationPedal);
					end;

					for k, implement in pairs(self.attachedImplements) do
						if brake then
							implement.object:onBrake();
						else
							implement.object:onReleaseBrake();
						end;
					end;

					-- Set motor torque and brake pedal on the server, as it's already set on client
					if not self.isServer then
					g_client:getServerConnection():sendEvent(SetWheelVariablesEvent:new(self, self.NHPowerCommandMotorTorque, self.NHPowerCommandBrakePedal));
					end;
				end
			end;

			if self.shuttleDirection == "forward" then
				self.motor.forwardGearRatios = {self.fwdRatios[self.currentFwdGear]/2,self.fwdRatios[self.currentFwdGear]/1.5,self.fwdRatios[self.currentFwdGear]};
				self.motor.backwardGearRatio = -1 * self.fwdRatios[self.currentFwdGear];
				-- Set the gear ratios on both server and client
				if not self.isServer then
					g_client:getServerConnection():sendEvent(SetGearRatiosEvent:new(self, self.fwdRatios[self.currentFwdGear]));
				end;
			elseif self.shuttleDirection == "reverse" then
				self.motor.forwardGearRatios = {self.revRatios[self.currentRevGear]/2,self.revRatios[self.currentRevGear]/1.5,self.revRatios[self.currentRevGear]};
				self.motor.backwardGearRatio = -1 * self.revRatios[self.currentRevGear];
				-- Set the gear ratios on both server and client
				if not self.isServer then
					g_client:getServerConnection():sendEvent(SetGearRatiosEvent:new(self, self.revRatios[self.currentRevGear]));
				end;
			elseif self.shuttleDirection == "neutral" then
				self.motor.forwardGearRatios = {0,0,0};
				self.motor.backwardGearRatio = 0;
				-- Set the gear ratios on both server and client
				if not self.isServer then
					g_client:getServerConnection():sendEvent(SetGearRatiosEvent:new(self, 0));
				end;
			end;

			self.gearRatio = 1;
			local acceleration = 0;
			self.updatedAcceleration = 0;
			local throttleAcceleration = 0;

			acceleration = -InputBinding.getAnalogInputAxis(InputBinding.AXIS_MOVE_FORWARD_VEHICLE);
			if InputBinding.isAxisZero(acceleration) then
				acceleration = -InputBinding.getDigitalInputAxis(InputBinding.AXIS_MOVE_FORWARD_VEHICLE);
			end;

			self.updatedAcceleration = acceleration + self.myAccelerationOffset;

			if acceleration <= 0 and self.powerCommandThrottlePercentage < self.powerCommandMinRpm/self.motorMaxRpmLimit[3] then
				self.updatedAcceleration = 0;
			elseif self.powerCommandThrottlePercentage > self.powerCommandMinRpm/self.motorMaxRpmLimit[3] and acceleration <= 0 then
				self.updatedAcceleration = self.powerCommandThrottlePercentage + self.myAccelerationOffset;
				throttleAcceleration = self.powerCommandThrottlePercentage;
			end;

			-- Handthrottle movement
			if self.handThrottle ~= nil then
				setTranslation(self.handThrottle, 0,0,((self.powerCommandThrottlePercentage-0.33)/5.6));
			end;

			--Pedal animation management for pedals
			if InputBinding.getAnalogInputAxis(InputBinding.AXIS_MOVE_FORWARD_VEHICLE) ~= 0 then
				--user has a speeder pedal
				self:footThrottleRotation(dt, acceleration, self.updatedAcceleration);
			else
				--user is using buttons
				self:footThrottleRotation(dt, self.updatedAcceleration, self.updatedAcceleration);
			end;

			self:clutchAndBrakesRotation(dt, acceleration);

			-- Slow rev up and down
			if math.floor(self.updatedAcceleration*100) >  math.floor(self.myLastAcceleration*100) and (self.shuttleDirection == "neutral" or acceleration <= 0) then
				self.updatedAcceleration = self.myLastAcceleration + 0.0005 * dt;
				self.myLastAcceleration = self.updatedAcceleration;
			elseif math.floor(self.updatedAcceleration*100) <  math.floor(self.myLastAcceleration*100) and (self.shuttleDirection == "neutral" or acceleration <= 0) then
				self.updatedAcceleration = self.myLastAcceleration - 0.0004 * dt;
				self.myLastAcceleration = self.updatedAcceleration;
			elseif math.floor(self.updatedAcceleration*100) >  math.floor(self.myLastAcceleration*100) then
				if self.shuttleDirection == "forward" then
					self.updatedAcceleration = self.myLastAcceleration + (0.0003 * dt - (self.fwdRatios[self.currentFwdGear]/2000));
					self.myLastAcceleration = self.updatedAcceleration;
				elseif self.shuttleDirection == "reverse" then
					self.updatedAcceleration = self.myLastAcceleration + (0.0003 * dt - (self.revRatios[self.currentRevGear]/2000));
					self.myLastAcceleration = self.updatedAcceleration;
				end;
			elseif math.floor(self.updatedAcceleration*100) <  math.floor(self.myLastAcceleration*100) then
				if self.shuttleDirection == "forward" then
					self.updatedAcceleration = self.myLastAcceleration - (0.0002 * dt - (self.fwdRatios[self.currentFwdGear]/2000));
					self.myLastAcceleration = self.updatedAcceleration;
				elseif self.shuttleDirection == "reverse" then
					self.updatedAcceleration = self.myLastAcceleration - (0.0002 * dt - (self.revRatios[self.currentRevGear]/2000));
					self.myLastAcceleration = self.updatedAcceleration;
				end;
			end;

			if ((self.updatedAcceleration < acceleration and acceleration > 0) or (self.updatedAcceleration < throttleAcceleration and throttleAcceleration > 0)) and self.isMotorStarted then
				self.showExtraParticles = true;
			else
				self.showExtraParticles = false;
			end;
			--Extra smoke particles
			Utils.setEmittingState(self.extraExhaustParticleSystems, self.showExtraParticles)

			if self.updatedAcceleration <= self.powerCommandMinRpm/self.motorMaxRpmLimit[3] then
				self.updatedAcceleration = self.powerCommandMinRpm/self.motorMaxRpmLimit[3];
			end;

			local mySpeed = self.lastSpeed*3600;

			if self.updatedAcceleration > 0.6 and acceleration == 0 and self:getIsActiveForSound() then
				if self.powerCommandThrottlePercentage > self.powerCommandMinRpm/self.motorMaxRpmLimit[3] then
					self.playTurboWhistle = false;
				else
					self.playTurboWhistle = true;
				end;
			elseif acceleration > 0 then
				self.playTurboWhistle = false;
			end;

			-- Acceleration adjustment when gearing up or down and gearshift modes
			if self.gearedDown or self.gearedUp then
				if self.gearedUp then
					if mySpeed > 2 then
						if acceleration > 0 and self.updatedAcceleration > 0 and self.myLastAcceleration > 0.6 then
							self.myLastAcceleration = self.myLastAcceleration - 0.1;
							if self.myLastAcceleration < 0.6 then
								self.myLastAcceleration = 0.6;
								self.gearedUp = false;
							end;
						elseif acceleration <= 0 then
							self.gearedUp = false;
						end;
					end;
				elseif self.gearedDown then
					if mySpeed > 2 then
						if acceleration >= 0 and self.updatedAcceleration > 0 then
							self.myLastAcceleration = self.myLastAcceleration + 0.0015 * dt;
							if self.maxRevOnGearDown == 0 then
								self.maxRevOnGearDown = self.updatedAcceleration + 0.1
							end;
							if self.myLastAcceleration > self.maxRevOnGearDown then
								self.myLastAcceleration = self.maxRevOnGearDown;
								self.maxRevOnGearDown = 0;
								self.gearedDown = false;
							end;
						elseif acceleration < 0 then
							self.gearedDown = false;
						end;
					end;
				end;
			end;

			if self.myLastAcceleration < 0.1 then
				self.myLastAcceleration = 0.1;
			end;

			--RPM meter value
			if (self.updatedAcceleration * (self.motorMaxRpmLimit[3] * self.rpmCoefficient)) > self.powerCommandMinRpm then
				if (self.updatedAcceleration * (self.motorMaxRpmLimit[3] * self.rpmCoefficient)) < 2300 then
					self.rpmMeterValue = self.updatedAcceleration * (self.motorMaxRpmLimit[3] * self.rpmCoefficient);
				end;
			else
				self.rpmMeterValue = self.updatedAcceleration * self.motorMaxRpmLimit[3];
			end;

			--Sound adjustments NEW

			--Throttle Sound
			self.lastRoundPerMinute = 0;
			self.motor.lastMotorRpm = 0;

			local roundPerSecond = 0;

			if self.myLastAcceleration > 0 then
				if acceleration >= 0 and self.shuttleDirection == "forward" then
					roundPerSecond = self.myLastAcceleration * 48 - (self.currentFwdGear * 0.3);
				elseif acceleration >= 0 and self.shuttleDirection == "reverse" then
					roundPerSecond = self.myLastAcceleration * 48 - (self.currentRevGear * 0.3);
				else
					roundPerSecond = self.myLastAcceleration * 48;
				end;
				if roundPerSecond < 0 then
					roundPerSecond = 0;
				end;
			end;


			if self.isMotorStarted then
				if self.powerCommandMotorSound ~= nil then
					local pitch = math.min(self.powerCommandMotorSoundPitchOffset - (0.25 - self.updatedAcceleration * 0.25)  + self.powerCommandMotorSoundPitchScale*math.abs(roundPerSecond), self.powerCommandMotorSoundPitchMax);

					--renderText(0.5, 0.1, 0.0250, string.format("myLastAcceleration " .. self.myLastAcceleration));
					--renderText(0.5, 0.2, 0.0250, string.format("roundPerSecond " .. roundPerSecond));
					--renderText(0.5, 0.3, 0.0250, string.format("pitch " .. pitch));
					setSamplePitch(self.powerCommandMotorSound, pitch);

					if self.powerCommandMotorSoundRun ~= nil then
						setSamplePitch(self.powerCommandMotorSoundRun, math.min(self.powerCommandMotorSoundRunPitchOffset + self.powerCommandMotorSoundRunPitchScale*math.abs(roundPerSecond), self.powerCommandMotorSoundRunPitchMax));
					end;
				end;

				if self.powerCommandMotorSoundRun ~= nil then
					local maxRpm  = self.motor.maxRpm[3];

					if self.updatedAcceleration > 0 then
						local rpmVolume = Utils.clamp(math.abs(roundPerSecond*60)/((maxRpm - self.motor.minRpm)), 0.0, 1.0);
						setSampleVolume(self.powerCommandMotorSoundRun, rpmVolume);
					else
						local rpmVolume = Utils.clamp(math.abs(roundPerSecond*60)/((maxRpm - self.motor.minRpm)*2), 0.0, 1.0);
						setSampleVolume(self.powerCommandMotorSoundRun, rpmVolume);
					end;
				end;
				if self.camIndex ~= self.intCameraIndex then
					if self.turboWhistle ~= nil then
						local maxRpm  = self.motor.maxRpm[3];
						local rpmVolume = Utils.clamp(math.abs(roundPerSecond*60)/((maxRpm - self.motor.minRpm)), 0.0, 1.0);
						if self.playTurboWhistle and not self.playedTurboWhistle then
							playSample(self.turboWhistle, 1, self.turboWhistleVolume, 0);
							setSamplePitch(self.turboWhistle, self.turboWhistlePitchOffset);
							setSampleVolume(self.turboWhistle, rpmVolume);
							self.playedTurboWhistle = true;
						elseif not self.playTurboWhistle then
							stopSample(self.turboWhistle);
							self.playedTurboWhistle = false;
						end;
					end;
				else
					if self.turboWhistleInterior ~= nil then
						local maxRpm  = self.motor.maxRpm[3];
						local rpmVolume = Utils.clamp(math.abs(roundPerSecond*60)/((maxRpm - self.motor.minRpm)), 0.0, 1.0);
						if self.playTurboWhistle and not self.playedTurboWhistleInterior then
							playSample(self.turboWhistleInterior, 1, self.turboWhistleVolumeInterior, 0);
							setSamplePitch(self.turboWhistleInterior, self.turboWhistleInteriorPitchOffset);
							setSampleVolume(self.turboWhistleInterior, rpmVolume);
							self.playedTurboWhistleInterior = true;
						elseif not self.playTurboWhistle then
							stopSample(self.turboWhistleInterior);
							self.playedTurboWhistleInterior = false;
						end;
					end;
				end;
			end;

			--END NEW

			if self.shuttleDirection == "forward" then
				self.gearRatio = (0.5 + (0.5 * self.currentFwdGear/self.numFwdGears));
			elseif self.shuttleDirection == "reverse" then
				self.gearRatio = (0.5 + (0.5 * self.currentRevGear/self.numRevGears))
			end;

			--renderText(0.5, 0.6, 0.0250, string.format(self.gearRatio));

			if self.isMotorStarted and (self.shuttleDirection == "forward" or self.shuttleDirection == "reverse") then
				self.updatedAcceleration = (self.gearRatio) * self.updatedAcceleration;
			else
				self.updatedAcceleration = 0;
			end;

			if self.updatedAcceleration > 1 then
				self.updatedAcceleration = 1;
			elseif self.updatedAcceleration < -1 then
				self.updatedAcceleration = -1;
			end;

			self.newAcceleration = self.updatedAcceleration;

			if acceleration < 0 then
				self.isBraking = true;
			else
				self.isBraking = false;
			end;

			if self.isBraking then
				if self.automaticGearDown then
					if self.currentFwdGear > 14 then
						self:changeFwdGear(14);
						if self.gearedUp then
							self.gearedUp = false;
						elseif self.gearedDown then
							self.gearedDown = false;
						end;
					end;
				end;
			end;
			self.NHPowerCommandBrakeForce = 0;
			--Handle brake as brake
			if acceleration < 0 then
				self.NHPowerCommandBrakeForce = 10 * -acceleration;
			elseif self.handbrake then
				self.NHPowerCommandBrakeForce = 5;
			elseif self.isBraking then
				self.NHPowerCommandBrakeForce = 2;
			elseif self.updatedAcceleration > 0 then
				self.NHPowerCommandBrakeForce = 0;
			end;

			if (self.shuttleDirection == "reverse" and self.movingDirection == 1) or (self.shuttleDirection == "forward" and self.movingDirection == -1) then
				self.NHPowerCommandBrakeForce = 6;
			end;

			if self.fuelFillLevel == 0 then
				self.updatedAcceleration = 0;
			end;

			-- Set the brakeforce on both server and client
			self.motor.brakeForce = self.NHPowerCommandBrakeForce;
			if not self.isServer then
				g_client:getServerConnection():sendEvent(SetMotorBrakeForceEvent:new(self, self.NHPowerCommandBrakeForce));
			end;

			-- Set the lastAcceleration on both server and client
			self.lastAcceleration = self.newAcceleration;
			if not self.isServer then
				g_client:getServerConnection():sendEvent(SetLastAccelerationEvent:new(self, self.newAcceleration));
			end;
		end;
	else
		-- Set the original gearratios of tractor is hired
		if self.motor.forwardGearRatios ~= self.originalForwardGearRatio then
			self.motor.forwardGearRatios = self.originalForwardGearRatio;
			if not self.isServer then
				g_client:getServerConnection():sendEvent(SetGearRatiosEvent:new(self, self.originalForwardGearRatio));
			end;
		end;
		if self.motor.backwardGearRatio ~= self.originalBackwardGearRation then
			self.motor.backwardGearRatio = self.originalBackwardGearRation;
			if not self.isServer then
				g_client:getServerConnection():sendEvent(SetGearRatiosEvent:new(self, self.originalBackwardGearRation));
			end;
		end;
		if self.motor.brakeForce ~= 6 then
			self.motor.brakeForce = 6;
			if not self.isServer then
				g_client:getServerConnection():sendEvent(SetMotorBrakeForceEvent:new(self, 6));
			end;
		end;
	end;
end;

function NHPowerCommand:draw()

	if 	self.HUDActive and self.isMotorStarted then
		if self.HUDOverlay ~= nil then
			self.HUDOverlay:render();
		end;

		setTextBold(true);
		setTextColor(1, 1, 1, 1.0);

		if not self.isAITractorActivated and not self.isAutopilotActivated then
			renderText(0.905, 0.698, 0.0180,"RPM:");
			if self.rpmMeterValue > 1000 then
				renderText(0.94, 0.698, 0.0180, string.format("%d ",self.rpmMeterValue));
			else
				renderText(0.949, 0.698, 0.0180, string.format("%d ",self.rpmMeterValue));
			end;
			if self.motor.speedLevel == 0 or self.motor.speedLevel == 3 then
				renderText(0.905, 0.678, 0.0180,"FWD:");
				renderText(0.905, 0.658, 0.0180,"REV:");
				if self.currentFwdGear > 10 then
					renderText(0.955, 0.678, 0.0180,string.format(self.currentFwdGear));
				else
					renderText(0.961, 0.678, 0.0180,string.format(self.currentFwdGear));
				end;
				renderText(0.961, 0.658, 0.0180, string.format(self.currentRevGear));
			end;
			renderText(0.905, 0.623, 0.0180, string.format("Shuttle:"));
			if self.shuttleDirection == "forward" then
				renderText(0.961, 0.623, 0.0180, string.format("F"));
			elseif self.shuttleDirection == "reverse" then
				renderText(0.961, 0.623, 0.0180, string.format("R"));
			else
				renderText(0.961, 0.623, 0.0180, string.format("N"));
			end;
		end;

		--renderText(0.93, 0.215, 0.0180,string.format("%d ",self.lastSpeed * 3600).."km/h");

		if self.handbrake then
			setTextBold(true);
			setTextColor(1, 0, 0, 1.0);
			renderText(0.913, 0.52, 0.018, string.format("P-Brake"));
		end;
	end;
end;

function NHPowerCommand:onLeave()
	self.myLastAcceleration = 0;
	NHPowerCommand.stopSounds(self);

	newAcceleration = 0;
	self.newAcceleration = 0;
	self:shuttleToNeutral(true);
	self.handbrake = true;
end;

function NHPowerCommand:stopMotor()
    self.motor.minRpm = self.backupMinRpmManual;
	self.myLastAcceleration = 0;
	NHPowerCommand.stopSounds(self);
end;

function NHPowerCommand:stopSounds()
	self.playPowerCommandMotorSound = false;
	if self.powerCommandMotorSound ~= nil then
        stopSample(self.powerCommandMotorSound);
    end;

    if self.powerCommandMotorSoundRun ~= nil then
        stopSample(self.powerCommandMotorSoundRun);
    end;
end;

function NHPowerCommand:startMotor()
end;

function NHPowerCommand:onEnter()
	self:updateGearDisplay();
end;

function NHPowerCommand:changeFwdGear(newFwdGear, noEventSend)
	ChangeFwdGearEvent.sendEvent(self,newFwdGear,noEventSend);

	self.lastFwdGear = self.currentFwdGear;
	self.currentFwdGear = newFwdGear;

	if self.shouldChangeRevGear then
		if self.currentFwdGear == 7 and self.currentRevGear == 2 then
			self:changeRevGear(1);
		elseif self.currentFwdGear == 8 and ((self.currentRevGear == 3 and self.lastFwdGear == 9) or (self.currentRevGear == 1 and self.lastFwdGear == 7)) then
			self:changeRevGear(2);
		elseif self.currentFwdGear == 9 and ((self.currentRevGear == 4 and self.lastFwdGear == 10) or (self.currentRevGear == 2 and self.lastFwdGear == 8)) then
			self:changeRevGear(3);
		elseif self.currentFwdGear == 10 and ((self.currentRevGear == 5 and self.lastFwdGear == 11) or (self.currentRevGear == 3 and self.lastFwdGear == 9)) then
			self:changeRevGear(4);
		elseif self.currentFwdGear == 11 and ((self.currentRevGear == 6 and self.lastFwdGear == 12) or (self.currentRevGear == 4 and self.lastFwdGear == 10)) then
			self:changeRevGear(5);
		elseif self.currentFwdGear == 12 and (self.currentRevGear == 5 and self.lastFwdGear == 11) then
			self:changeRevGear(6);
		end;
	end;

	if self.lastFwdGear > self.currentFwdGear then
		self.gearedDown = true;
	elseif self.lastFwdGear < self.currentFwdGear then
		self.gearedUp = true;
	end;
	self.shouldChangeRevGear = false;
	self:updateGearDisplay();
end;

function NHPowerCommand:changeRevGear(newRevGear, noEventSend)
	ChangeRevGearEvent.sendEvent(self,newRevGear,noEventSend);

	self.lastRevGear = self.currentRevGear;
	self.currentRevGear = newRevGear;

	if self.shouldChangeFwdGear then
		if self.currentRevGear == 1 and (self.currentFwdGear == 6 or self.currentFwdGear == 8) then
			self:changeFwdGear(7);
		elseif self.currentRevGear == 2 and (self.currentFwdGear == 7 or self.currentFwdGear == 9) then
			self:changeFwdGear(8);
		elseif self.currentRevGear == 3 and (self.currentFwdGear == 8 or self.currentFwdGear == 10) then
			self:changeFwdGear(9);
		elseif self.currentRevGear == 4 and (self.currentFwdGear == 9 or self.currentFwdGear == 11) then
			self:changeFwdGear(10);
		elseif self.currentRevGear == 5 and (self.currentFwdGear == 10 or self.currentFwdGear == 12) then
			self:changeFwdGear(11);
		elseif self.currentRevGear == 6 and (self.currentFwdGear == 11 or self.currentFwdGear == 13) then
			self:changeFwdGear(12);
		end;
	end;

	if self.lastRevGear > self.currentRevGear then
		self.gearedDown = true;
	elseif self.lastRevGear < self.currentFwdGear then
		self.gearedUp = true;
	end;
	self.shouldChangeFwdGear = false;
	self:updateGearDisplay();
end;

function NHPowerCommand:changeShuttleDirection(currentShuttleDirection, noEventSend)
	ChangeShuttleDirectionEvent.sendEvent(self,currentShuttleDirection,noEventSend);

	if currentShuttleDirection == "neutral" then
		self.shuttleDirection = "forward";
		if self.shuttleIndicators ~= nil then
			setVisibility(self.shuttleIndicators[3].index, false);
			setVisibility(self.shuttleIndicators[1].index, true);
			setVisibility(self.shuttleIndicatorsLight[3].index, false);
			setVisibility(self.shuttleIndicatorsLight[1].index, true);
		end;
	elseif currentShuttleDirection == "forward" then
		self.shuttleDirection = "reverse";
		if self.shuttleIndicators ~= nil then
			setVisibility(self.shuttleIndicators[1].index, false);
			setVisibility(self.shuttleIndicators[2].index, true);
			setVisibility(self.shuttleIndicatorsLight[1].index, false);
			setVisibility(self.shuttleIndicatorsLight[2].index, true);
		end;
		self.gearedUp = true;
	elseif currentShuttleDirection == "reverse" then
		self.shuttleDirection = "forward";
		if self.shuttleIndicators ~= nil then
			setVisibility(self.shuttleIndicators[2].index, false);
			setVisibility(self.shuttleIndicators[1].index, true);
			setVisibility(self.shuttleIndicatorsLight[2].index, false);
			setVisibility(self.shuttleIndicatorsLight[1].index, true);
		end;
		self.gearedUp = true;
	end;
end;

function NHPowerCommand:shuttleToNeutral(shuttleDirectionNeutral, noEventSend)
	ShuttleToNeutralEvent.sendEvent(self,shuttleDirectionNeutral,noEventSend);

	if shuttleDirectionNeutral then
		self.shuttleDirection = "neutral";
		if self.shuttleIndicators ~= nil then
			setVisibility(self.shuttleIndicators[1].index, false);
			setVisibility(self.shuttleIndicators[2].index, false);
			setVisibility(self.shuttleIndicators[3].index, true);
			setVisibility(self.shuttleIndicatorsLight[1].index, false);
			setVisibility(self.shuttleIndicatorsLight[2].index, false);
			setVisibility(self.shuttleIndicatorsLight[3].index, true);
		end;
	end;
end;

function NHPowerCommand:changeHandbrakeState(currentHandbrakeState, noEventSend)
	ChangeHandBrakeStateEvent.sendEvent(self,currentHandbrakeState,noEventSend);

	if currentHandbrakeState then
		self.handbrake = false;
		if self.handBrakeLever ~= nil then
			setRotation(self.handBrakeLever, math.rad(15), 0, 0);
		end;
	else
		self.handbrake = true;
		if self.handBrakeLever ~= nil then
			setRotation(self.handBrakeLever, 0, 0, 0);
		end;
	end;
end;

function NHPowerCommand:setLastAcceleration(newAcceleration)
	self.lastAcceleration = newAcceleration;
end;

function NHPowerCommand:setMotorBrakeForce(brakeForce)
	self.motor.brakeForce = brakeForce;
end;

function NHPowerCommand:setGearRatio(currentRatio)
	self.motor.forwardGearRatios = {currentRatio/2,currentRatio/1.5,currentRatio};
	self.motor.backwardGearRatio = -1 * currentRatio;
end;

function NHPowerCommand:setWheelVariables(motorTorque, brakePedal)
	self.NHPowerCommandMotorTorque = motorTorque;
	self.NHPowerCommandBrakePedal = brakePedal;
end;

function NHPowerCommand:updateGearDisplay()
	-- Handle the digitRPMs in the display
	if self.numDigitGearForwardOne > 0 and self.numDigitGearForwardTwo > 0 and self.numDigitGearReverseThree > 0  and self.digitGearForwardOne ~= nil then
		if self.isMotorStarted then
			local forwardGear = self.currentFwdGear;
			local reverseGear = self.currentRevGear;

			local tensForward = false;

			local gearDifference = 0;

			gearDifference = forwardGear-10;

			for i=1, self.numDigitGearForwardOne do
				setVisibility(self.digitGearForwardOne[i].rotNode,false);
				setVisibility(self.digitGearForwardOneLight[i].rotNode,false);
			end;
			for i=1, self.numDigitGearForwardTwo do
				setVisibility(self.digitGearForwardTwo[i].rotNode,false);
				setVisibility(self.digitGearForwardTwoLight[i].rotNode,false);
			end;
			for i=1, self.numDigitGearReverseThree do
				setVisibility(self.digitGearReverseThree[i].rotNode,false);
				setVisibility(self.digitGearReverseThreeLight[i].rotNode,false);
			end;

			if gearDifference >= 0 then
				tensForward = true;
				setVisibility(self.digitGearForwardOne[gearDifference+1].rotNode,true);
				setVisibility(self.digitGearForwardOneLight[gearDifference+1].rotNode,true);
			else
				setVisibility(self.digitGearForwardOne[forwardGear+1].rotNode,true);
				setVisibility(self.digitGearForwardOneLight[forwardGear+1].rotNode,true);
			end;

			if tensForward then
				setVisibility(self.digitGearForwardTwo[1].rotNode,true);
				setVisibility(self.digitGearForwardTwoLight[1].rotNode,true);
			end;

			setVisibility(self.digitGearReverseThree[reverseGear+1].rotNode,true);
			setVisibility(self.digitGearReverseThreeLight[reverseGear+1].rotNode,true);
		end;
	end;
end;

function NHPowerCommand:footThrottleRotation(dt, acceleration, newAcceleration)
	if self.footThrottlePedal ~= nil then
		if acceleration > 0 then
			setRotation(self.footThrottlePedal, math.rad(-20 * math.abs(acceleration)),0,0);
		elseif newAcceleration > 0 then
			setRotation(self.footThrottlePedal, math.rad(20 * math.abs(newAcceleration)),0,0);
		else
			setRotation(self.footThrottlePedal, math.rad(0),0,0);
		end;
	end;
end;

function NHPowerCommand:clutchAndBrakesRotation(dt, acceleration)
	if self.clutchPedal ~= nil then
		if acceleration < 0 then
			setRotation(self.clutchPedal, math.rad(-25 * math.abs(acceleration)),0,0);
		else
			setRotation(self.clutchPedal, math.rad(0),0,0);
		end;
	end;

	if self.brakePedal ~= nil then
		local brakeActive = false;
		if acceleration < 0 then
			brakeActive = true;
		end;

		local brakeMax = {};
		brakeMax[1] = math.rad(-25);
		brakeMax[2] = math.rad(0);
		brakeMax[3] = math.rad(0);
		local brakeRotationTime = 500;

		local brakeMin = {};
		brakeMin[1] = math.rad(0);
		brakeMin[2] = math.rad(0);
		brakeMin[3] = math.rad(0);

		local x, y, z = getRotation(self.brakePedal);
		local rot = {x,y,z};
		local newRot = Utils.getMovedLimitedValues(rot, brakeMin, brakeMax, 3, brakeRotationTime, dt, brakeActive);
		setRotation(self.brakePedal, unpack(newRot));
	end;
end;
