﻿--
-- Sprayer
-- Class for all sprayers
--
-- @author  Stefan Geiger
-- @date  24/02/08
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.

Sprayer = {};


Sprayer.SPRAYTYPE_UNKNOWN = 0;
Sprayer.NUM_SPRAYTYPES = 0;

Sprayer.sprayTypes = {};
Sprayer.sprayTypeIndexToDesc = {};

Sprayer.sprayTypeToFillType = {};
Sprayer.fillTypeToSprayType = {};

function Sprayer.registerSprayType(name, pricePerLiter, litersPerSqmPerSecond, hudOverlayFilename)
    local key = "SPRAYTYPE_"..string.upper(name);
    if Sprayer[key] == nil then
        Sprayer.NUM_SPRAYTYPES = Sprayer.NUM_SPRAYTYPES+1;
        Sprayer[key] = Sprayer.NUM_SPRAYTYPES;

        local desc = {name = name, index = Sprayer.NUM_SPRAYTYPES};
        desc.pricePerLiter = pricePerLiter;
        desc.litersPerSqmPerSecond = litersPerSqmPerSecond;
        desc.hudOverlayFilename = hudOverlayFilename;


        Sprayer.sprayTypes[name] = desc;
        Sprayer.sprayTypeIndexToDesc[Sprayer.NUM_SPRAYTYPES] = desc;

        local fillType = Fillable.registerFillType(name)
        Sprayer.sprayTypeToFillType[Sprayer.NUM_SPRAYTYPES] = fillType;
        Sprayer.fillTypeToSprayType[fillType] = Sprayer.NUM_SPRAYTYPES;
    end;
end;

Sprayer.registerSprayType("fertilizer", 0.3, 0.5, "");
Sprayer.registerSprayType("manure", 0.01, 0.5, "");
Sprayer.registerSprayType("liquidManure", 0.01, 0.5, "");

function Sprayer.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Fillable, specializations);
end;

function Sprayer:load(xmlFile)
    assert(self.setIsSprayerFilling == nil, "Sprayer needs to be the first specialization which implements setIsSprayerFilling");
    self.setIsSprayerFilling = Sprayer.setIsSprayerFilling;
    self.addSprayerFillTrigger = Sprayer.addSprayerFillTrigger;
    self.removeSprayerFillTrigger = Sprayer.removeSprayerFillTrigger;
    self.fillLitersPerSecond = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fillLitersPerSecond"), 500);
    self.isSprayerFilling = false;
    self.sprayerFillTriggers = {};
    self.sprayerFillActivatable = SprayerFillActivatable:new(self);
end;

function Sprayer:delete()
    g_currentMission:removeActivatableObject(self.sprayerFillActivatable);
end;

function Sprayer:mouseEvent(posX, posY, isDown, isUp, button)
end;

function Sprayer:keyEvent(unicode, sym, modifier, isDown)
end;

function Sprayer:update(dt)
end;

function Sprayer:updateTick(dt)
    if self.isSprayerFilling and self.isServer then
        local disableFilling = false;
        if self:allowFillType(self.sprayerFillingFillType, false) then
            local oldFillLevel = self.fillLevel;
            local delta = self.fillLitersPerSecond*dt*0.001;
            local silo = g_currentMission:getSiloAmount(self.sprayerFillingFillType);
            if self.sprayerFillingIsSiloTrigger then
                if silo <= 0 then
                    disableFilling = true;
                end;
                delta = math.min(delta, silo);
            end;
            self:setFillLevel(self.fillLevel + delta, self.sprayerFillingFillType, true);
            local delta = self.fillLevel - oldFillLevel;
            if delta > 0 then
                if self.sprayerFillingIsSiloTrigger then
                    g_currentMission:setSiloAmount(self.sprayerFillingFillType, silo - delta);
                else
                    local sprayType = Sprayer.fillTypeToSprayType[self.sprayerFillingFillType];
                    if sprayType ~= nil then
                        local sprayTypeDesc = Sprayer.sprayTypeIndexToDesc[sprayType]

                        local price = delta*sprayTypeDesc.pricePerLiter;
                        g_currentMission.missionStats.expensesTotal = g_currentMission.missionStats.expensesTotal + price;
                        g_currentMission.missionStats.expensesSession = g_currentMission.missionStats.expensesSession + price;

                        g_currentMission:addSharedMoney(-price);
                    end;
                end;
            elseif self.fillLevel == self.capacity then
                disableFilling = true;
            end;
        else
            disableFilling = true;
        end;
        if disableFilling then
            self:setIsSprayerFilling(false);
        end;
    end;
end;

function Sprayer:draw()
    if self.isClient then
        if self.fillLevel <= 0 and self.capacity ~= 0 then
            g_currentMission:addExtraPrintText(g_i18n:getText("FirstFillTheTool"));
        end;
    end;
end;

function Sprayer:onDetach()
end;

function Sprayer:onLeave()
end;

function Sprayer:onDeactivate()
end;

function Sprayer:onDeactivateSounds()
end;

function Sprayer:setIsSprayerFilling(isFilling, fillType, isSiloTrigger, noEventSend)
    SprayerSetIsFillingEvent.sendEvent(self, isFilling, fillType, isSiloTrigger, noEventSend)
    if self.isSprayerFilling ~= isFilling then
        self.isSprayerFilling = isFilling;
        self.sprayerFillingFillType = fillType;
        self.sprayerFillingIsSiloTrigger = isSiloTrigger;
    end;
end;

function Sprayer:addSprayerFillTrigger(trigger)
    if table.getn(self.sprayerFillTriggers) == 0 then
        g_currentMission:addActivatableObject(self.sprayerFillActivatable);
    end;
    table.insert(self.sprayerFillTriggers, trigger);
end;

function Sprayer:removeSprayerFillTrigger(trigger)
    for i=1, table.getn(self.sprayerFillTriggers) do
        if self.sprayerFillTriggers[i] == trigger then
            table.remove(self.sprayerFillTriggers, i);
            break;
        end;
    end;
    if table.getn(self.sprayerFillTriggers) == 0 then
        if self.isServer then
            self:setIsSprayerFilling(false);
        end;
        g_currentMission:removeActivatableObject(self.sprayerFillActivatable);
    end;
end;

SprayerFillActivatable = {}
local SprayerFillActivatable_mt = Class(SprayerFillActivatable);

function SprayerFillActivatable:new(sprayer)
    local self = {};
    setmetatable(self, SprayerFillActivatable_mt);
    self.sprayer = sprayer;
    self.activateText = "unknown";

    self.currentTrigger = nil;

    return self;
end;


function SprayerFillActivatable:getIsActivatable()
    self.currentTrigger = nil;
    if not self.sprayer:getIsActiveForInput() or self.sprayer.fillLevel == self.sprayer.capacity then
        return false;
    end;
    -- find the first trigger which is activable
    for i=1, table.getn(self.sprayer.sprayerFillTriggers) do
        local trigger = self.sprayer.sprayerFillTriggers[i];
        if trigger:getIsActivatable(self.sprayer) then
            self.currentTrigger = trigger;
            self:updateActivateText();
            return true;
        end;
    end;
    return false;
end;

function SprayerFillActivatable:onActivateObject()
    self.sprayer:setIsSprayerFilling(not self.sprayer.isSprayerFilling, self.currentTrigger.fillType, self.currentTrigger.isSiloTrigger);
    self:updateActivateText();
    g_currentMission:addActivatableObject(self);
end;

function SprayerFillActivatable:drawActivate()
    -- TODO draw icon
end;

function SprayerFillActivatable:updateActivateText()
    if self.sprayer.isSprayerFilling then
        self.activateText = string.format(g_i18n:getText("stop_refill_OBJECT"), self.sprayer.typeDesc);
    else
        self.activateText = string.format(g_i18n:getText("refill_OBJECT"), self.sprayer.typeDesc);
    end;
end;