--
-- Joskin water tank
-- Class for water tank
--
-- @author  PeterJ_LS-UK modteam
-- @date  08/03/2012
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.


water_tank = {};


water_tank.SPRAYTYPE_UNKNOWN = 0;
water_tank.NUM_SPRAYTYPES = 0;
  
water_tank.tankTypes = {};
water_tank.tankTypeIndexToDesc = {};
  
water_tank.tankTypeToFillType = {};
water_tank.fillTypeToSprayType = {};
  
function water_tank.registerSprayType(name, pricePerLiter, litersPerSqmPerSecond, hudOverlayFilename)
	local key = "SPRAYTYPE_"..string.upper(name);
	if water_tank[key] == nil then
		water_tank.NUM_SPRAYTYPES = water_tank.NUM_SPRAYTYPES+1;
		water_tank[key] = water_tank.NUM_SPRAYTYPES;
  
		local desc = {name = name, index = water_tank.NUM_SPRAYTYPES};
		desc.pricePerLiter = pricePerLiter;
		desc.litersPerSqmPerSecond = litersPerSqmPerSecond;
		desc.hudOverlayFilename = hudOverlayFilename;
  
 
		water_tank.tankTypes[name] = desc;
		water_tank.tankTypeIndexToDesc[water_tank.NUM_SPRAYTYPES] = desc;
  
		local fillType = Fillable.registerFillType(name)
		water_tank.tankTypeToFillType[water_tank.NUM_SPRAYTYPES] = fillType;
		water_tank.fillTypeToSprayType[fillType] = water_tank.NUM_SPRAYTYPES;
	end;
end;
  
water_tank.registerSprayType("water", 0.01, 0.5, "");


function water_tank.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Fillable, specializations);
end;

function water_tank:load(xmlFile)

	local youHaveDLC3 = SpecializationUtil.getSpecialization("pdlc_dlcPack3.waterTrailer");	
	if youHaveDLC3 ~= nil then
		self.addWater = false;
	else
		self.addWater = true;
	end;

	self.setIsTurnedOn = water_tank.setIsTurnedOn;
	self.getIsTurnedOnAllowed = water_tank.getIsTurnedOnAllowed;
	self.setIsTankFilling = water_tank.setIsTankFilling;
	self.setIsTankUnloading = water_tank.setIsTankUnloading;
	self.fillLitersPerSecond = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fillLitersPerSecond"), 500);
	self.isTankFilling = false;
	self.isTankUnloading = false;
	self.isTurnedOn = false;
	self.isFillingAllowed = false;
	
	self.unloadLitersPerSecond = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.unloadLitersPerSecond"), 500);
	self.ableCowsEat = false;
	
	if self.isClient then
		local tankSound = getXMLString(xmlFile, "vehicle.tankSound#file");
		if tankSound ~= nil and tankSound ~= "" then
			tankSound = Utils.getFilename(tankSound, self.baseDirectory);
			self.tankSound = createSample("tankSound");
			self.tankSoundEnabled = false;
			loadSample(self.tankSound, tankSound, false);
			self.tankSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tankSound#pitchOffset"), 1);
			self.tankSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tankSound#volume"), 1);
		end;
	end;

	local tankAISound = getXMLString(xmlFile, "vehicle.tankAISound#file");
		if tankAISound ~= nil and tankAISound ~= "" then
		tankAISound = Utils.getFilename(tankAISound, self.baseDirectory);
		self.tankAISoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tankAISound#pitchOffset"), 0);
		self.tankAISoundRadius = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tankAISound#radius"), 50);
		self.tankAISoundInnerRadius = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tankAISound#innerRadius"), 10);
		self.tankAISoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tankAISound#volume"), 1);
		self.tankAISound = createAudioSource("tankAISound", tankAISound, self.tankAISoundRadius, self.tankAISoundInnerRadius, self.tankAISoundVolume, 0);
		link(self.components[1].node, self.tankAISound);
		setVisibility(self.tankAISound, false);
	end;
	
	self.unloadParticleSystems = {};
	Utils.loadParticleSystem(xmlFile, self.unloadParticleSystems, "vehicle.ParticleSystems.unload", self.components, false, nil, self.baseDirectory);

	self.setRotationMax = SpecializationUtil.callSpecializationsFunction("setRotationMax");
	local rotationPartNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.rotationPart#index"));
	if rotationPartNode ~= nil then
		self.rotationPart = {};
		self.rotationPart.node = rotationPartNode;
		local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.rotationPart#minRot"));
		self.rotationPart.minRot = {};
		self.rotationPart.minRot[1] = Utils.degToRad(Utils.getNoNil(x, 0));
		self.rotationPart.minRot[2] = Utils.degToRad(Utils.getNoNil(y, 0));
		self.rotationPart.minRot[3] = Utils.degToRad(Utils.getNoNil(z, 0));
  
		x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.rotationPart#maxRot"));
		self.rotationPart.maxRot = {};
		self.rotationPart.maxRot[1] = Utils.degToRad(Utils.getNoNil(x, 0));
		self.rotationPart.maxRot[2] = Utils.degToRad(Utils.getNoNil(y, 0));
		self.rotationPart.maxRot[3] = Utils.degToRad(Utils.getNoNil(z, 0));
  
		self.rotationPart.rotTime = Utils.getNoNil(getXMLString(xmlFile, "vehicle.rotationPart#rotTime"), 2)*1000;
		self.rotationPart.touchRotLimit = Utils.degToRad(Utils.getNoNil(getXMLString(xmlFile, "vehicle.rotationPart#touchRotLimit"), 10));
	end;
	self.rotationMax = false;

	self.tapIndex = Utils.indexToObject(self.components,getXMLString(xmlFile,"vehicle.rotationPart#index"));
	if self.tapIndex == nil then
		self.tapIndex = self.components[1].node;
	end;

	self.lastFillLevel = self.fillLevel;
	self.drinkSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.animals#drinkSpeed"), 0.0);
	
	self.overlay = Utils.getFilename("textures/water_hud.png", self.baseDirectory);
    self.WaterOverlay = Overlay:new("hudPDAControl", self.overlay, g_currentMission.fruitSymbolX, g_currentMission.fruitSymbolY, g_currentMission.fruitSymbolSize, g_currentMission.fruitSymbolSize * (4 / 3)); 
	
	self.waterPlane = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.drinkingTrough#plane"));
	self.planeLevel = 0.495;
	
	self.fillingTrigger = false;
	self.speedViolationMaxTime = 1000;
	self.speedViolationTimer = self.speedViolationMaxTime;
	
	self.isFillableInRange = nil;
end;

function water_tank:delete()
	if self.tankSound ~= nil then
		delete(self.tankSound);
	end;
	Utils.deleteParticleSystem(self.unloadParticleSystems);
end;

function water_tank:readStream(streamId, connection)
	local turnedOn = streamReadBool(streamId);
	local isTankFilling = streamReadBool(streamId);
	self:setIsTurnedOn(turnedOn, true);
	self:setIsTankFilling(isTankFilling, true);
	self:setIsTankUnloading(isTankUnloading, true);
	self.rotateMax = streamReadBool(streamId);
	if self.rotationPart ~= nil then
		local x = streamReadFloat32(streamId);
		local y = streamReadFloat32(streamId);
		local z = streamReadFloat32(streamId);
		setRotation(self.rotationPart.node, x, y, z);
	end;
end;
  
function water_tank:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isTurnedOn);
	streamWriteBool(streamId, self.isTankFilling);
	streamWriteBool(streamId, self.isTankUnloading);
	streamWriteBool(streamId, self.rotateMax);
	if self.rotationPart ~= nil then
		local x, y, z = getRotation(self.rotationPart.node);
		streamWriteFloat32(streamId, x);
		streamWriteFloat32(streamId, y);
		streamWriteFloat32(streamId, z);
	end;
end;

function water_tank:mouseEvent(posX, posY, isDown, isUp, button)
end;

function water_tank:keyEvent(unicode, sym, modifier, isDown)
end;

function water_tank:update(dt)

	if self.isClient then
		if self:getIsActiveForInput() then
			if self.isFillingAllowed then
				if InputBinding.hasEvent(InputBinding.ACTIVATE_OBJECT) then
					self:setIsTankFilling(not self.isTankFilling);
				end;
			end;
		end;
	end;
	if not self:getIsActive() then
		if self.playerInRange and self.fillLevel > 0 then	
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
				self:setIsTankUnloading(not self.isTankUnloading);
			end;
			if self.rotationMax then
				g_currentMission:addHelpButtonText(g_i18n:getText("stop_unloading"), InputBinding.IMPLEMENT_EXTRA);
			else
				g_currentMission:addHelpButtonText(g_i18n:getText("start_unloading"), InputBinding.IMPLEMENT_EXTRA);
				if InputBinding.hasEvent(InputBinding.ACTIVATE_OBJECT) then
					self:setIsTurnedOn(not self.isTurnedOn);
				end;
				if self.isTurnedOn then
					g_currentMission:addHelpButtonText(g_i18n:getText("drinking_off"), InputBinding.ACTIVATE_OBJECT);
				else
					g_currentMission:addHelpButtonText(g_i18n:getText("drinking_on"), InputBinding.ACTIVATE_OBJECT);
				end;
			end;
		end;
	end;
end;

function water_tank:updateTick(dt)

	if self.rotationPart ~= nil then
		local x, y, z = getRotation(self.rotationPart.node);
		local maxRot = self.rotationPart.maxRot;
		local minRot = self.rotationPart.minRot;
		local eps = self.rotationPart.touchRotLimit;
  		if (math.abs(x-maxRot[1]) > eps and math.abs(x-minRot[1]) > eps) or
			(math.abs(y-maxRot[2]) > eps and math.abs(y-minRot[2]) > eps) or
			(math.abs(z-maxRot[3]) > eps and math.abs(z-minRot[3]) > eps) then
			updateDensity = false;
		end;
  		local x, y, z = getRotation(self.rotationPart.node);
		local rot = {x,y,z};
		local newRot = Utils.getMovedLimitedValues(rot, self.rotationPart.maxRot, self.rotationPart.minRot, 3, self.rotationPart.rotTime, dt, not self.rotationMax);
		setRotation(self.rotationPart.node, unpack(newRot));
	end;
	if self.ableCowsEat then
		local cowsStored = AnimalHusbandry.getNumberOfAnimals();
		if cowsStored > 0 then
			self:setFillLevel(self.fillLevel - (cowsStored / self.eatSpeed), self.currentFillType);
		end;
	end;
	local xt,yt,zt = getTranslation(self.components[1].node);
	local deltaWater = yt-g_currentMission.waterY+2.0;
	if deltaWater < 2 then
		self.isFillingAllowed = true;
	else
		self.isFillingAllowed = false;
	end;
	if self.isFillingAllowed then
		if self.isTankFilling and self.isServer then
			local disableFilling = false;
			local delta = self.fillLitersPerSecond*dt*0.001;
			self:setFillLevel(self.fillLevel + delta, Fillable.FILLTYPE_WATER);
			if self.fillLevel == self.capacity then
				disableFilling = true;
			end;
			if disableFilling then
				self:setIsTankFilling(false);
			end;
		end;
	else
		self:setIsTankFilling(false);
	end;
	if self.isTurnedOn then
		local cowsStored = AnimalHusbandry.getNumberOfAnimals();
		if cowsStored > 0 then
			self:setFillLevel(self.fillLevel - (cowsStored / self.drinkSpeed), self.currentFillType);
		end;
		if self.planeLevel < 0.674 then
			self.planeLevel = self.planeLevel + 0.001;
		end;
		setTranslation(self.waterPlane, 0, self.planeLevel, -2.07);
		if self.fillLevel <= 0 and self.capacity ~= 0 then
			self:setIsTurnedOn(false, true);
		end;
	else
		if self.planeLevel > 0.495 then
			self.planeLevel = self.planeLevel - 0.001;
		end;
		setTranslation(self.waterPlane, 0, self.planeLevel, -2.07);
		if self.lastFillLevel > self.fillLevel then
			self:setRotationMax(true, false);
			self.fillingTrigger = true;
		else
			if self.fillingTrigger then
				self:setRotationMax(false, true);
			end;
		end;
	end;
	if self.rotationMax then
		if self.fillLevel <= 0 and self.capacity ~= 0 then
			self:setRotationMax(false, true);
			self:setIsTankUnloading(false, true);
		end;
		if not self.tankSoundEnabled and self:getIsActiveForSound() then
			playSample(self.tankSound, 0, self.tankSoundVolume, 0);
			setSamplePitch(self.tankSound, self.tankSoundPitchOffset);
			self.tankSoundEnabled = true;
		end;
		Utils.setEmittingState(self.unloadParticleSystems, true);
		if self:doCheckSpeedLimit() and self.lastSpeed*3600 > 31 then
			self.speedViolationTimer = self.speedViolationTimer - dt;
		else
			self.speedViolationTimer = self.speedViolationMaxTime;
		end;
		if self.tankAISound ~= nil then
			setVisibility(self.tankAISound, true);
		end;
	else
		Utils.setEmittingState(self.unloadParticleSystems, false);
		stopSample(self.tankSound);
		self.tankSoundEnabled = false;
		self.speedViolationTimer = self.speedViolationMaxTime;
		if self.tankAISound ~= nil then
			setVisibility(self.tankAISound, false);
		end;
	end;
	self.lastFillLevel = self.fillLevel;
	if self.isTankUnloading then
		self:setIsTurnedOn(false, true);
		if self.fillLevel > 0 then
			local delta = self.unloadLitersPerSecond*dt*0.001;
			self:setFillLevel(self.fillLevel - delta, self.currentFillType);
			--- fill an attachable in range ---
			local nearestDistance = 5.0;
			local px, py, pz = getWorldTranslation(self.tipReferencePoint);
			for i=1, table.getn(g_currentMission.attachables) do
				local vx, vy, vz = getWorldTranslation(g_currentMission.attachables[i].rootNode);
				local distance = Utils.vector3Length(px-vx, py-vy, pz-vz);
				if distance < nearestDistance then
					local attachable = g_currentMission.attachables[i];		
					if attachable.fillTypes ~= nil and attachable.setFillLevel ~= nil and attachable.fillLevel ~= nil and attachable ~= self then
						if self.currentFillType == attachable.currentFillType or attachable.currentFillType == FruitUtil.FRUITTYPE_UNKNOWN then
							if attachable.fillLevel < attachable.capacity and self.fillLevel > 0 then							
								attachable:setFillLevel(attachable.fillLevel + delta, self.currentFillType);
							end; 
						end;
					end;
				end;
			end;
		end;
	end;

	if self.tipState == Trailer.TIPSTATE_OPENING or self.tipState == Trailer.TIPSTATE_CLOSING or self.tipState == Trailer.TIPSTATE_OPEN then
		self:setIsTurnedOn(false, true);
	end;
	if g_currentMission.player ~= nil then
		local nearestDistance = 1.7;
		local x1,y1,z1 = getWorldTranslation(self.tapIndex);
		local x2,y2,z2 = getWorldTranslation(g_currentMission.player.rootNode);
		local distance = Utils.vector3Length(x1-x2,y1-y2,z1-z2);
		if distance < nearestDistance then
			self.playerInRange = true;
		else
			self.playerInRange = false;
		end;
	end;
end;

function water_tank:draw()

	if self.isClient then
		if self.fillLevel <= 0 and self.capacity ~= 0 then
			g_currentMission:addExtraPrintText(g_i18n:getText("FirstFillTheTool"));
		end;
		if self.addWater and self.fillLevel > 0 and self.currentFillType == Fillable.FILLTYPE_WATER then
			self.WaterOverlay:render();
		end;
		if self.isTankFilling then
			g_currentMission:addHelpButtonText(string.format(g_i18n:getText("stop_refill_OBJECT"), self.typeDesc), InputBinding.ACTIVATE_OBJECT);
		else
			if self.isFillingAllowed and self.fillLevel < self.capacity then
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("refill_OBJECT"), self.typeDesc), InputBinding.ACTIVATE_OBJECT);
			end;
		end;
		if math.abs(self.speedViolationTimer - self.speedViolationMaxTime) > 2 then
			g_currentMission:addWarning(g_i18n:getText("Dont_drive_to_fast") .. "\n" .. string.format(g_i18n:getText("Cruise_control_levelN"), "2", InputBinding.getKeyNamesOfDigitalAction(InputBinding.SPEED_LEVEL2)), 0.07+0.022, 0.019+0.029);
		end;
	end;
end;

function water_tank:onAttach(attacherVehicle)
end;

function water_tank:onDetach()
	if self.deactivateOnDetach then
		water_tank.onDeactivate(self);
	else
		water_tank.onDeactivateSounds(self);
	end;
end;
  
function water_tank:onLeave()
	if self.deactivateOnLeave then
		water_tank.onDeactivate(self);
	else
		water_tank.onDeactivateSounds(self);
	end;
end;
  
function water_tank:onDeactivate()
	self.speedViolationTimer = self.speedViolationMaxTime;
	water_tank.onDeactivateSounds(self);
end;
  
function water_tank:onDeactivateSounds()
	if self.tankSoundEnabled then
		stopSample(self.tankSound);
		self.tankSoundEnabled = false;
	end;
end;

function water_tank:getIsTurnedOnAllowed(isTurnedOn)
	if not isTurnedOn or self.fillLevel > 0 then
		return true;
	end;
end;

function water_tank:setIsTurnedOn(isTurnedOn, noEventSend)
	if isTurnedOn ~= self.isTurnedOn then
		if self:getIsTurnedOnAllowed(isTurnedOn) then
			SetTurnedOnEvent.sendEvent(self, isTurnedOn, noEventSend)
  
			self.isTurnedOn = isTurnedOn;
		end;
	end;
end;
  
function water_tank:setIsTankFilling(isTankFilling, noEventSend)
	SetIsFillingEvent.sendEvent(self, isTankFilling, noEventSend)
	if self.isTankFilling ~= isTankFilling then
		self.isTankFilling = isTankFilling;
	end;
end;

function water_tank:setIsTankUnloading(isTankUnloading, noEventSend)
	SetIsUnloadingEvent.sendEvent(self, isTankUnloading, noEventSend)
	if self.isTankUnloading ~= isTankUnloading then
		self.isTankUnloading = isTankUnloading;
	end;
end;

function water_tank:setRotationMax(rotationMax, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(PloughRotationEvent:new(self, rotationMax), nil, nil, self);
		else
			g_client:getServerConnection():sendEvent(PloughRotationEvent:new(self, rotationMax));
		end;
	end;
	self.rotationMax = rotationMax;
	self.speedViolationTimer = self.speedViolationMaxTime;
end;
