--
-- Name: KuhnGMD4010
-- Description: Specialization for the Kuhn GMD 4010 mower
-- Version: 1.0
-- Author: Face & Saracaen
-- Date: 21/10/2011
--

KuhnGMD4010 = {};

function KuhnGMD4010.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(EnhancedMower, specializations);
end;

function KuhnGMD4010:load(xmlFile)
	
	-- Register specializations --
	self.setIsArmDown = SpecializationUtil.callSpecializationsFunction("setIsArmDown");
	self.setIsArmExpanded = SpecializationUtil.callSpecializationsFunction("setIsArmExpanded");
	
	-- Arm --
	self.arm = {};
	self.arm.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.arm#index"));
	
	local y = Utils.degToRad(Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.arm#minRot"), 0));
	
	self.arm.minRot = {0,y,0};
	
	y = Utils.degToRad(Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.arm#maxRot"), 0));
	self.arm.maxRot = {0,y,0};
	self.arm.rotTime = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.arm#rotTime"), 3) * 1000;
	self.arm.liftTime = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.arm#liftTime"), 3) * 1000;
	self.arm.currentRotLimit = {0};
	
	self.arm.isDown = false;
	self.arm.isExpanded = false;
	
	-- Hydraulics --
	self.hydraulics = {};	
	local hydraulicsCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.hydraulics#count"), 0);	
	local i = 0;
	for i = 1, hydraulicsCount do
		local baseName = string.format("vehicle.hydraulics.hydraulic%d", i);
		
		self.hydraulics[i] = {};		
		self.hydraulics[i].node = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#index"));
		self.hydraulics[i].punch = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#punch"));
		self.hydraulics[i].translationPunch = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#punchFixpoint"));
		self.hydraulics[i].fixPoint = Utils.indexToObject(self.components, getXMLString(xmlFile, baseName .. "#fixpoint"));
		
		local ax, ay, az = getWorldTranslation(self.hydraulics[i].punch);
		local bx, by, bz = getWorldTranslation(self.hydraulics[i].translationPunch);		
		self.hydraulics[i].punchDistance = Utils.vector3Length(ax-bx, ay-by, az-bz);	
	end;
		
	-- Joint orientation --
	self.backup = nil;
	self.doJointOrientation = false;
	self.doJointRepositioning = false;
		
	-- Other variables --
	self.isExpanded = false;
	self.isDown = false;
	
end;

function KuhnGMD4010:delete()
end;

function KuhnGMD4010:readStream(streamId, connection)
	local isArmDown = streamReadBool(streamId);
	self:setIsArmDown(isArmDown, true);
	
	local isArmExpanded = streamReadBool(streamId);
	self:setIsArmExpanded(isArmExpanded, true);
end;

function KuhnGMD4010:writeStream(streamId, connection)
	streamWriteBool(streamId, self.arm.isDown);
	streamWriteBool(streamId, self.arm.isExpanded);
end;

function KuhnGMD4010:mouseEvent(posX, posY, isDown, isUp, button)
end;

function KuhnGMD4010:keyEvent(unicode, sym, modifier, isDown)
end;

function KuhnGMD4010:update(dt)
	if self:getIsActiveForInput() then
		if self.isExpanded then		
			if InputBinding.hasEvent(InputBinding.LOWER_IMPLEMENT) then
				self:setIsArmDown(not self.arm.isDown);
			end;
		end;
		
		if not self.isTurnedOn and not self.isDown and not self:isLowered(false) then
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA2) then
				self:setIsArmExpanded(not self.arm.isExpanded);
			end;
		end;
	end;
end;

function KuhnGMD4010:updateTick(dt)
	if self:getIsActive() then
		
		-- Joint orientation --
		if self.doJointOrientation then
			for k,v in pairs(self.attacherVehicle.attachedImplements) do
				if v.object == self then
					local joint = self.attacherVehicle.attacherJoints[v.jointDescIndex];
					
					self.backup = {};
					self.backup.joint = joint;
					self.backup.minRot2_1 = joint.minRot2[1];
					
					if self.isExpanded then
						joint.minRot2[1] = -joint.minRot[1] + math.rad(2);
					end;
				end;
			end;
			self.doJointOrientation = false;
		end;
		
		-- Joint repositioning --
		if self.doJointRepositioning then
			local joint = self.backup.joint;
			if self.arm.isExpanded then
				joint.minRot2[1] = -joint.minRot[1] + math.rad(2);
			else
				joint.minRot2[1] = self.backup.minRot2_1;					
			end;
			self.doJointRepositioning = false;
		end;
		
		local x, y, z = getRotation(self.arm.node);
		self.isExpanded = y > Utils.degToRad(85);
		self.isDown = self.arm.currentRotLimit[1] > 1;
		
		-- Prevent lowering the linkage when expanded
		self.allowsLowering = not self.arm.isExpanded;
		
		-- Arm rotation --
		local x, y, z = getRotation(self.arm.node);	
		local rot = {x,y,z};
		self.arm.maxRot[1] = x;
		self.arm.minRot[1] = x;
		local newRot = Utils.getMovedLimitedValues(rot, self.arm.maxRot, self.arm.minRot, 3, self.arm.rotTime, dt, not self.arm.isExpanded);
		setRotation(self.arm.node, unpack(newRot));	
		
		setJointFrame(self.componentJoints[1].jointIndex, 0, self.componentJoints[1].jointNode);
			
		if not self.isExpanded then
			setJointFrame(self.componentJoints[2].jointIndex, 0, self.componentJoints[2].jointNode);
			setJointRotationLimit(self.componentJoints[2].jointIndex, 2, true, Utils.degToRad(0), Utils.degToRad(0));
		else
			setJointRotationLimit(self.componentJoints[2].jointIndex, 2, true, Utils.degToRad(-19), Utils.degToRad(19));
		end;
		
		x, y, z = getRotation(self.arm.node);
		rot = {x,y,z};			
		
		local newRotLimit = {0,0,0};
		newRotLimit = Utils.getMovedLimitedValues(self.arm.currentRotLimit, {30}, {0}, 1, self.arm.liftTime * 2, dt, not self.arm.isDown);
		if math.abs(newRotLimit[1] - self.arm.currentRotLimit[1]) > 0.001 then
			setJointRotationLimit(self.componentJoints[1].jointIndex, 0, true, Utils.degToRad(-newRotLimit[1]), Utils.degToRad(newRotLimit[1]));
		end;	
		
		self.arm.currentRotLimit = newRotLimit;		
		
		-- Hydraulics --
		for i=1, table.getn(self.hydraulics) do
			local ax, ay, az = getWorldTranslation(self.hydraulics[i].node);
			local bx, by, bz = getWorldTranslation(self.hydraulics[i].fixPoint);
			local x, y, z = worldDirectionToLocal(getParent(self.hydraulics[i].node), bx-ax, by-ay, bz-az);
			
			setDirection(self.hydraulics[i].node, x, y, z, 0, 1, 0);
			if self.hydraulics[i].punch ~= nil then
				local distance = Utils.vector3Length(ax-bx, ay-by, az-bz);
				setTranslation(self.hydraulics[i].punch, 0, 0, distance-self.hydraulics[i].punchDistance);
			end;	
		end;
		
	elseif not self.doJointOrientation then
		self.doJointOrientation = true;
	end;
end;

function KuhnGMD4010:draw()
	if self.isClient then
		
		-- Expand arm --
		if not self.isTurnedOn and not self.isDown and not self:isLowered(false) then
			if self.arm.isExpanded then
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("arm_fold"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA2);
			else
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("arm_expand"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA2);
			end;
		end;
		
	end;
end;

function KuhnGMD4010:onDetach()
	if self.backup ~= nil then
		local joint = self.backup.joint;
		
		joint.minRot2[1] = self.backup.minRot2_1;
	end;
end;

function KuhnGMD4010:setIsArmDown(isArmDown, noEventSend)
	SetArmDownEvent.sendEvent(self, isArmDown, noEventSend);
	self.arm.isDown = isArmDown;
end;

function KuhnGMD4010:setIsArmExpanded(isArmExpanded, noEventSend)
	SetArmExpandedEvent.sendEvent(self, isArmExpanded, noEventSend);
	self.arm.isExpanded = isArmExpanded;
	self.doJointRepositioning = true;
end;
