--
-- Name: EnhancedMower
-- Description: Enhanced specialization for mowers
-- Version: 1.0
-- Author: Saracaen
-- Date: 22/10/2011
--

source("dataS/scripts/vehicles/specializations/SetTurnedOnEvent.lua");
source("dataS/scripts/vehicles/specializations/MowerAreaEvent.lua");

EnhancedMower = {};

function EnhancedMower.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Attachable, specializations);
end;

function EnhancedMower:load(xmlFile)

	-- Register specializations --
	self.setIsTurnedOn = SpecializationUtil.callSpecializationsFunction("setIsTurnedOn");

	-- Cutting areas --
	local i = 0;
    local numCuttingAreas = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.cuttingAreas#count"), 0);
    for i = 1, numCuttingAreas do
        local baseName = string.format("vehicle.cuttingAreas.cuttingArea%d", i);
        self.cuttingAreas[i].foldMinLimit = Utils.getNoNil(getXMLFloat(xmlFile, baseName .. "#foldMinLimit"), 0);
        self.cuttingAreas[i].foldMaxLimit = Utils.getNoNil(getXMLFloat(xmlFile, baseName .. "#foldMaxLimit"), 1);
    end;
	
	-- Ground reference --
	self.groundReferenceThreshold = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.groundReferenceNode#threshold"), 0.2);
    self.groundReferenceNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.groundReferenceNode#index"));
    if self.groundReferenceNode == nil then
        self.groundReferenceNode = self.components[1].node;
    end;
	
    self.mowerGroundFlag = self.nextDirtyFlag;
    self.nextDirtyFlag = self.mowerGroundFlag * 2;
	
	-- Particle systems --
	self.groundParticleSystems = {};
	local psName = "vehicle.groundParticleSystem";
	Utils.loadParticleSystem(xmlFile, self.groundParticleSystems, psName, self.components, false, nil, self.baseDirectory)

	self.newGroundParticleSystems = {};
	local i = 0;
	while true do
		local baseName = string.format("vehicle.groundParticleSystems.groundParticleSystem(%d)", i);
		if not hasXMLProperty(xmlFile, baseName) then
			break;
		end;
		
		local entry = {};
		entry.ps = {};
		
		Utils.loadParticleSystem(xmlFile, entry.ps, baseName, self.components, false, nil, self.baseDirectory);
		if table.getn(entry.ps) > 0 then
			entry.isActive = false;
			table.insert(self.newGroundParticleSystems, entry);
		end;
		
		i = i + 1;
	end;
	
	-- Sounds --
    local mowerSound = getXMLString(xmlFile, "vehicle.mowerSound#file");
    if mowerSound ~= nil and mowerSound ~= "" then
        mowerSound = Utils.getFilename(mowerSound, self.baseDirectory); 
        self.mowerSound = createSample("mowerSound");
        self.mowerSoundEnabled = false;
        loadSample(self.mowerSound, mowerSound, false);
        self.mowerSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mowerSound#pitchOffset"), 1);
        self.mowerSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mowerSound#volume"), 1);
    end;
	
	-- Speed violation --
	self.maxSpeedLevel = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.maxSpeedLevel#value"), 2);
    self.speedViolationMaxTime = 2500;
    self.speedViolationTimer = self.speedViolationMaxTime;
	
	-- Other variables --
	self.isTurnedOn = false;
	
end;

function EnhancedMower:delete()
	
	-- Particle systems
    Utils.deleteParticleSystem(self.groundParticleSystems);
       
    for _, entry in ipairs(self.newGroundParticleSystems) do
        Utils.deleteParticleSystem(entry.ps);
    end;
	
	-- Sounds --
	if self.mowerSound ~= nil then
		delete(self.mowerSound);
	end;
end;

function EnhancedMower:readStream(streamId, connection)
	local isTurnedOn = streamReadBool(streamId);
	self:setIsTurnedOn(isTurnedOn, true);
end;

function EnhancedMower:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isTurnedOn);
end;

function EnhancedMower:mouseEvent(posX, posY, isDown, isUp, button)
end;

function EnhancedMower:keyEvent(unicode, sym, modifier, isDown)
end;

function EnhancedMower:update(dt)
	if self:getIsActiveForInput() then
		if self.isExpanded == nil or self.isExpanded then
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
				self:setIsTurnedOn(not self.isTurnedOn);
			end;
		end;
	end;
end;

function EnhancedMower:updateTick(dt)
	if self:getIsActive() then
		if self.isServer then
		
			-- Cutting areas --
			if self.isTurnedOn and (self.isDown or self:isLowered(false)) then
				local x,y,z = getWorldTranslation(self.groundReferenceNode);
				local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 0, z);
				
				if terrainHeight + self.groundReferenceThreshold >= y then
					local cuttingAreasSend = {};
					
					for k, cuttingArea in pairs(self.cuttingAreas) do
						if self:getIsAreaActive(cuttingArea) then
							local x,y,z = getWorldTranslation(cuttingArea.start);
							local x1,y1,z1 = getWorldTranslation(cuttingArea.width);
							local x2,y2,z2 = getWorldTranslation(cuttingArea.height);
							
							table.insert(cuttingAreasSend, {x,z,x1,z1,x2,z2});
						end;
					end;
					
					if table.getn(cuttingAreasSend) > 0 then
						MowerAreaEvent.runLocally(cuttingAreasSend);
						g_server:broadcastEvent(MowerAreaEvent:new(cuttingAreasSend));
					end;
				end;
			end;
		
		end;
	
		if self.isClient then
		
			-- Particle systems --
			local enableGroundParticleSystems = (self.lastSpeed * 3600 > 5) and self.isTurnedOn and (self.isDown or self:isLowered(false));
			Utils.setEmittingState(self.groundParticleSystems, enableGroundParticleSystems);

			if enableGroundParticleSystems then
				for k, cuttingArea in pairs(self.cuttingAreas) do
					local ps = self.newGroundParticleSystems[k];
					if self:getIsAreaActive(cuttingArea) then
						if ps ~= nil then
							if not ps.isActive then
								ps.isActive = true;
								Utils.setEmittingState(ps.ps, true);
							end;
						end;
					else
						if ps ~= nil and ps.isActive then
							ps.isActive = false;
							Utils.setEmittingState(ps.ps, false);
						end;
					end;
				end;		
			else
				for k, ps in pairs(self.newGroundParticleSystems) do
					if ps.isActive then
						ps.isActive = false;
						Utils.setEmittingState(ps.ps, false);
					end;
				end;
			end;
		
			-- Sounds --
			if self:getIsActiveForSound() then
				if self.mowerSound ~= nil then
					if self.isTurnedOn then
						if not self.mowerSoundEnabled then
							playSample(self.mowerSound, 0, self.mowerSoundVolume, 0);
							setSamplePitch(self.mowerSound, self.mowerSoundPitchOffset);
							self.mowerSoundEnabled = true;
						end;
					else 
						if self.mowerSoundEnabled then
							stopSample(self.mowerSound);
							self.mowerSoundEnabled = false;
						end;
					end;
				end;	
			end;
		
		end;
	
		-- Speed violation --
		local speedLimit = 20;
		if self.maxSpeedLevel == 2 then
			speedLimit = 30;
		elseif self.maxSpeedLevel == 3 then
			speedLimit = 100;
		end;
		
		if self.isDown and self.isTurnedOn then	
			if self:doCheckSpeedLimit() and self.lastSpeed * 3600 > speedLimit then
				self.speedViolationTimer = self.speedViolationTimer - dt;
				
				if self.isServer then
					if self.speedViolationTimer < 0 then
						if self.attacherVehicle then
							self.attacherVehicle:detachImplementByObject(self);
						end;
					end;
				end;
			else
				self.speedViolationTimer = self.speedViolationMaxTime;
			end;
		else
			self.speedViolationTimer = self.speedViolationMaxTime;
		end;
	end;
end;

function EnhancedMower:draw()
	if self.isClient then
	
		-- Turned on --
		if self.isExpanded == nil or self.isExpanded then
			if self.isTurnedOn then
		        g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_off_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
		    else
		        g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_on_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
		    end;
		end;
		
		-- Speed violation --
		if math.abs(self.speedViolationTimer - self.speedViolationMaxTime) > 2 then
			local buttonName = InputBinding.SPEED_LEVEL1;
			if self.maxSpeedLevel == 2 then
				buttonName = InputBinding.SPEED_LEVEL2;
			elseif self.maxSpeedLevel == 3 then
				buttonName = InputBinding.SPEED_LEVEL3;
			end;
			
			g_currentMission:addWarning(g_i18n:getText("Dont_drive_to_fast") .. "\n" .. string.format(g_i18n:getText("Cruise_control_levelN"), tostring(self.maxSpeedLevel), InputBinding.getKeyNamesOfDigitalAction(buttonName)), 0.07+0.022, 0.019+0.029);
        end;
		
	end;
end;

function EnhancedMower:onDetach()
	if self.deactivateOnDetach then
        EnhancedMower.onDeactivate(self);
    else
		EnhancedMower.onDeactivateSounds(self)
    end;
end;

function EnhancedMower:onLeave()
    if self.deactivateOnLeave then
		EnhancedMower.onDeactivate(self);
    else
        EnhancedMower.onDeactivateSounds(self)
    end;
end;

function EnhancedMower:onDeactivate()
	EnhancedMower.onDeactivateSounds(self)
	self.isTurnedOn = false;
end;

function EnhancedMower:onDeactivateSounds()
	if self.mowerSoundEnabled then
		stopSample(self.mowerSound);
		self.mowerSoundEnabled = false;
	end;
end;

function EnhancedMower:setIsTurnedOn(isTurnedOn, noEventSend)
	SetTurnedOnEvent.sendEvent(self, isTurnedOn, noEventSend);
	self.isTurnedOn = isTurnedOn;
end;