-- NOTE: The copyright statement below affect this source file only. Whatever mod it is included in may have another license.

--[[
Copyright 2010, Christer Folkesson ('Aqualize'). All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are
permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright notice, this list
      of conditions and the following disclaimer in the documentation and/or other materials
      provided with the distribution.

THIS SOFTWARE IS PROVIDED BY CHRISTER FOLKESSON ``AS IS'' AND ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--]]





-- BE AWARE. DOCUMENTATION (WHERE AVAILABLE) TO FUNCTIONS MAY BE OUTDATED; DON NOT TRUST IT BLINDLY!

-- This script started out as a very ambitious idea and is in it current stage and usage a lot over designed and too generally written, more than it
-- needs to for current functionality.

--[[
TODO list. This is a list of things that needs to be fixed and/or was envisioned for inclusion into this script:

* 	Automatic raise and lowering of the parking stand when trailer is attached and deattached. Consider the problem when using multiple trailers and
	also unhitching them all once (shift+Q).
*	Fix animations in general so they are more stable and work correctly (that they activate when they should).
*	Get the inter mod communication routines to work. This is a big concept where mods would be able to communicate with each other, exchanging
	info first on their different functionalities. A simple thing would be to have multiple trailers agree on which parking stand should be lowered and
	synchronize loading and unloading operations. On a broader perspective a baler could implement this and make cool things having a bale thrower
	that aims for the wagon and calculates angle and force depending on forward motion speed and elevation.
*	Redo the grab point and drop points so they work as the stack (attach) points. Then one could make a grid of load and drop points in the same
	positions and have the script work for a bale grab implement for a FEL.
*	Add some animation on the movement of bales from grabbing to attaching. This important if the script should be used for a bale stacker with
	different "lanes" that it places the bales in before releasing them.

--]]

-- You can perhaps reach me via ls-uk.info forum or at farmsim@aqualize.se, I can at least try to explain my dumb ideas in this script ;-)


function __LINE__() return debug.getinfo(2, 'l').currentline end
----print("At line: " .. tostring(__LINE__()))
scriptName="Bale Vehicle: " -- Prefixes some error messages to console

GENERIC_TYPE = "Bale vehicle"
MOD_TYPE = "Bale trailer" -- Needs to be changed if script is developed for other than trailer usage.

BaleVehicle = {}


function BaleVehicle.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Trailer, specializations);
end;


--[[
Called when mod is loaded. Registers methods, verifyies XML structure, reads necessary XML spatial data and creates
a bale stack (bale handler).

param (number): xmlFile:
	The handle to the xmlFile used by the mod.

side-effects:
	Creates fields to member-methods in "self".
	Creates (by method call) "self.sh" (the stack handler), "self.grabPoints" (where to snatch the bales from) and
	"self.dropBales" (where to let the bales drop from when unloading).
--]]
function BaleVehicle:load(xmlFile)
	MOD_NAME = getXMLString(xmlFile, "vehicle.typeDesc.en")


	if hasXMLProperty == nil then
		self.originalHasXMLProperty = hasXMLProperty

		-- OK this is a makeshift solution for Farming Simulator 2009 - NON gold edition. It lacks the hasXMLProperty function,
		-- so I implemented my own that exploits the fact that all MY properties have the "#name" attribute. So testing a
		-- property that isn't a attribute (something with #) and doesn't have the name attribute WILL NOT WORK!
		hasXMLProperty = function(objectId, propertyPathPLUSname)
			local firstOccurency = string.find(propertyPathPLUSname, "#")
			if firstOccurency == nil then
				propertyPathPLUSname = propertyPathPLUSname .. "#name"
			end
			local strRetVal = getXMLString(objectId, propertyPathPLUSname)
			if strRetVal == nil then
				return false
			else
				return true
			end
		end
	end

	-- Register member methods
	self.registerUpdateCallback = BaleVehicle.registerUpdateCallback
	self.unregisterUpdateCallback = BaleVehicle.unregisterUpdateCallback
	self.registerDrawCallback = BaleVehicle.registerDrawCallback
	self.unregisterDrawCallback = BaleVehicle.unregisterDrawCallback
	self.resetHandlers = BaleVehicle.resetHandlers
	self.setupSimpleOperationMode = BaleVehicle.setupSimpleOperationMode
	self.addDefaultHandlers = BaleVehicle.addDefaultHandlers
	self.simpleLoadingTimerCallback = BaleVehicle.simpleLoadingTimerCallback
	self.simpleUnloadingTimerCallback = BaleVehicle.simpleUnloadingTimerCallback
	self.setupGrabbers = BaleVehicle.setupGrabbers
	self.setupDroppers = BaleVehicle.setupDroppers
	self.registerGrabber = BaleVehicle.registerGrabber
	self.registerStacker = BaleVehicle.registerStacker
	self.registerDropper = BaleVehicle.registerDropper
	self.registerFastener = BaleVehicle.registerFastener
	self.registerRelinquisher = BaleVehicle.registerRelinquisher
	self.registerHandlers = BaleVehicle.registerHandlers
	self.unregisterHandlers = BaleVehicle.unregisterHandlers
	self.buildHandlerTables = BaleVehicle.buildHandlerTables
	self.verifyXMLstructure = BaleVehicle.verifyXMLstructure
	self.getSubProperties = BaleVehicle.getSubProperties
	self.loadPropertiesFromXML = BaleVehicle.loadPropertiesFromXML
	self.getInterModCommunicationHandler = BaleVehicle.getInterModCommunicationHandler
	self.playDropPointAnimationInReverse = BaleVehicle.playDropPointAnimationInReverse

	-- end of registration

	if not self:verifyXMLstructure(xmlFile) then
		print(scriptName .. ": XML failed verification, I won't load anything")
		return
	end

	--self.myImch = ModComm.instantiateObject(self, GENERIC_TYPE, MOD_NAME)
    --self.imchs = {}
	self.updateCallbacks = {}
	self.drawCallbacks = {}
	self.grabPoints = {}
	self.dropPoints = {}
	self.baleSpecs = {}
	self.loadingOrders = {}
	self.unloadingOrders = {}

	self.stackHandlers = {}
	self.grabBaleHandlers = {}
	self.dropBaleHandlers = {}
	self.fastenBaleHandlers = {}
	self.relinquishBaleHandlers = {}

	self.standClip = {}

	self:loadPropertiesFromXML(xmlFile)
    -- Use the vanilla configuration as standard
    self:setupSimpleOperationMode()

	if self.originalHasXMLProperty ~= nil then
		HASXMLProperty = self.originalHasXMLProperty
	end
end


function BaleVehicle:delete()
end


function BaleVehicle:mouseEvent(posX, posY, isDown, isUp, button)
end

-- The keyEvent is using a very complicated way of knowing which keys were pressed but it was the only way I found to distinguish if the SHIFT modifier button was
-- pressed or not. So sorry - no easy way of changing keys.
function BaleVehicle:keyEvent(unicode, sym, modifier, isDown)
	if isDown and modifier ~= 4097 and unicode == 47 then -- The key is the slash character and the modifier mask is NOT shift
		if not self.isInSimpleBaleLoadingMode or (self.currentBaleWorker ~= "simpleLoadingTimerCallback" and self.currentBaleWorker ~= nil) then
			return
		end

		if self.isLoading then
			self.loadingTimerID = 0
			self.currentBaleWorker = nil
		else
			local occupied, total = self.stackHandlerInUse:getNrOfLoadedBalesOutOfTotalNr()
			if occupied ~= total then
				self.currentBaleWorker = "simpleLoadingTimerCallback"
				self.loadingTimerID = addTimer(self.grabPoints.interval, "simpleLoadingTimerCallback", self)
			end
		end
		self.isLoading = not self.isLoading

	elseif isDown and modifier ~= 4097 and unicode == 42 then -- The key is the asterisk character and the modifier mask is NOT shift
		if not self.isInSimpleBaleUnLoadingMode or (self.currentBaleWorker ~= "simpleUnloadingTimerCallback" and self.currentBaleWorker ~= nil) then
			return
		end
		if self.isUnloading then
			self.unloadingTimerID = 0
			self.currentBaleWorker = nil
			self.firstRunForUnloading = false
		else
			local occupied, total = self.stackHandlerInUse:getNrOfLoadedBalesOutOfTotalNr()
			if occupied ~= 0 then
				self.currentBaleWorker = "simpleUnloadingTimerCallback"
				self.firstRunForUnloading = true
				self.unloadingTimerID = addTimer(self.dropPoints.interval, "simpleUnloadingTimerCallback", self)
			end
		end
		self.isUnloading = not self.isUnloading

	elseif isDown and modifier == 4097 and unicode == 47 then -- The key is the slash character and the modifier mask IS shift
		if #self.grabPoints == 1 then
			return -- Nothing to do
		end
		self.grabBaleHandlers[1]:toogleGrabberPoint(self.grabPoints[self.grabPoints.currentGrabPoint].name, false) -- Disables current
		if self.grabPoints.currentGrabPoint == #self.grabPoints then
			self.grabPoints.currentGrabPoint = 1
		else
			self.grabPoints.currentGrabPoint = self.grabPoints.currentGrabPoint + 1
		end
		self.grabBaleHandlers[1]:toogleGrabberPoint(self.grabPoints[self.grabPoints.currentGrabPoint].name, true) -- Enables the new one

	elseif isDown and modifier == 4097 and unicode == 42 then -- The key is the asterisk character and the modifier mask IS shift
		if #self.dropPoints == 1 then
			return -- Nothing to do
		end
		if self.dropPoints.currentDropPoint == #self.dropPoints then
			self.dropPoints.currentDropPoint = 1
		else
			self.dropPoints.currentDropPoint = self.dropPoints.currentDropPoint + 1
		end
	end
end


function BaleVehicle:update(dt)
--~ 	for i = 1, #self.updateCallbacks do
--~ 		self.updateCallbacks[i]()
--~ 	end
end


function BaleVehicle:draw()
	local bales, capacity = self.stackHandlerInUse:getNrOfLoadedBalesOutOfTotalNr()

	if self.currentBaleWorker == "simpleUnloadingTimerCallback" then
		g_currentMission:addExtraPrintText("BT (" .. tostring(bales) .. "/" .. tostring(capacity) .."): Is unloading with dropper " .. tostring(self.dropPoints[self.dropPoints.currentDropPoint].name))
	elseif self.currentBaleWorker == "simpleLoadingTimerCallback" then
		g_currentMission:addExtraPrintText("BT (" .. tostring(bales) .. "/" .. tostring(capacity) .."): Is loading with grabber " .. tostring(self.grabPoints[self.grabPoints.currentGrabPoint].name))
	else
		g_currentMission:addExtraPrintText("BT (" .. tostring(bales) .. "/" .. tostring(capacity) .."): 'Folding' raises/lowers the parking stand") -- Very mod specific, should be handled better
	end

	for i = 1, #self.drawCallbacks do
		self.drawCallbacks[i]()
	end
end


function BaleVehicle:onAttach(attacherVehicle)
	-- This would have been used to initated contact between mods to have them interact. Cutting it out for now...
    --self.myImch:testNewVehicle(attacherVehicle)

--~ 	for i = 1, #self.standClip do
--~ 		print("At line: " .. tostring(__LINE__()))
--~ 		local clip = self.standClip[i]
--~ 		setAnimTrackSpeedScale(clip.animCharSet, clip.trackId, 1.0)
--~ 		setAnimTrackTime(clip.animCharSet, clip.trackId, 0.0, false)
--~ 		enableAnimTrack(clip.animCharSet, clip.trackId)
--~ 		print("At line: " .. tostring(__LINE__()))
--~ 	end
end


function BaleVehicle:onDetach(detacherVehicle)
--~ 	if attacherVehicle.getInterModCommunicationHandler ~= nil then
--~ 		local imch = detacherVehicle:getInterModCommunicationHandler()
--~         for i = 1, #self.imchs do
--~                 if imch == self.imchs[i] then
--~                     self.imchs[i] = nil

--~ 		if imch:isCompatibleWith("BaleVehicle", 1.0) then
--~ 			self.imchs[#self.imchs + 1] = imch
--~ 			imch:recieveSuccessfullCoupling (self.myImch)
--~ 		end
--~ 	end

--~ 	for i = 1, #self.standClip do
--~ 		print("At line: " .. tostring(__LINE__()))
--~ 		local clip = self.standClip[i]
--~ 		setAnimTrackSpeedScale(clip.animCharSet, clip.trackId, -1.0)
--~ 		enableAnimTrack(clip.animCharSet, clip.trackId)
--~ 		print("At line: " .. tostring(__LINE__()))
--~ 	end
end

function BaleVehicle:getInterModCommunicationHandler()
	return self.myImch
end

function BaleVehicle:registerUpdateCallback(handler)
	self.updateCallbacks[#self.updateCallbacks + 1] = handler
end

function BaleVehicle:unregisterUpdateCallback(handler)
	for i = 1, #self.updateCallbacks do
		if self.updateCallbacks[i] == handler then
			for j = i, #self.updateCallbacks do
				self.updateCallbacks[j] = self.updateCallbacks[j + 1]
			end
		end
	end
end


function BaleVehicle:registerDrawCallback(handler)
	self.drawCallbacks[#self.drawCallbacks + 1] = handler
end

function BaleVehicle:unregisterDrawCallback(handler)
	for i = 1, #self.drawCallbacks do
		if self.drawCallbacks[i] == handler then
			for j = i, #self.drawCallbacks do
				self.drawCallbacks[j] = self.drawCallbacks[j + 1]
			end
		end
	end
end


function BaleVehicle:resetHandlers()
	for i = 2, #self.stackHandlers do
		self.stackHandlers[i] = nil
	end

    for i = 2, #self.grabBaleHandlers do
		self.grabBaleHandlers[i] = nil
	end

	for i = 2, #self.dropBaleHandlers do
		self.dropBaleHandlers[i] = nil
	end

	for i = 2, #self.fastenBaleHandlers do
		self.fastenBaleHandlers[i] = nil
	end

	for i = 2, #self.relinquishBaleHandlers do
		self.relinquishBaleHandlers[i] = nil
	end
end


-- The standard function mode of the vehicle.
function BaleVehicle:setupSimpleOperationMode()
	self:resetHandlers()
	self:addDefaultHandlers(self.baleSpecs[1], self.loadingOrders, self.unloadingOrders)

	self.stackHandlerInUse = self.stackHandlers[1]
	self.grabBaleHandlerInUse = self.grabBaleHandlers[1]
	self.dropBaleHandlerInUse = self.dropBaleHandlers[1]
	self.fastenBaleHandlerInUse = self.fastenBaleHandlers[1]
	self.relinquishBaleHandlerInUse = self.relinquishBaleHandlers[1]

	self.isInSimpleBaleLoadingMode = true
	self.isInSimpleBaleUnLoadingMode = true
end


-- In the case were there are no foreign handlers (could have been provided via IMCH) we should use the simple handlers for solo-usage
function BaleVehicle:addDefaultHandlers(baleSpec, loadingOrders, unloadingOrders)
	self.stackHandlers[1] = StackHandler.instantiateObject(baleSpec, loadingOrders, unloadingOrders)
    self.grabBaleHandlers[1] = self:setupGrabbers(baleSpec.baleTypes, loadingOrders)
    self.dropBaleHandlers[1] = self:setupDroppers(unloadingOrders)
	self.fastenBaleHandlers[1] = FastenBaleHandler.instantiateObject()
	self.relinquishBaleHandlers[1] = RelinquishBaleHandler.instantiateObject()
end


function BaleVehicle:simpleLoadingTimerCallback()
	if self.loadingTimerID == 0 then
		return false -- We have been asked to stop unloading
	end

    if self.currentBaleWorker ~= "simpleLoadingTimerCallback" then
		self.loadingTimerID = 0
        return false -- Some one else is doing something with bales now, this operation is not concurrent
    end

    if self.grabBaleHandlerInUse ~= self.grabBaleHandlers[1] then
		self.loadingTimerID = 0
		self.currentBaleWorker = nil
        return false -- We are not the grabber that should do work now
    end

    if self.fastenBaleHandlerInUse ~= self.fastenBaleHandlers[1] then
		self.loadingTimerID = 0
		self.currentBaleWorker = nil
        return false -- We don't have the default fastener
    end

    if self.stackHandlerInUse ~= self.stackHandlers[1] then
		self.loadingTimerID = 0
		self.currentBaleWorker = nil
        return false -- We don't have the default stack handler
    end

    local baleScanResult

    if self.savedBaleScanResult ~= nil then
        baleScanResult = self.savedBaleScanResult -- Load all previously (earlier calls) found bales before scanning for new ones
    else
        baleScanResult = self.grabBaleHandlerInUse:getBalesWithinGrabbers()
    end

    if baleScanResult == nil then
        return true -- We didn't have any bale to load but were not interrupted, so wait a while and try again.
    end

    local fastenBaleRetVal = self.fastenBaleHandlerInUse:fastenBale(self.stackHandlerInUse, baleScanResult[1].loadingView, baleScanResult[1].bale)

	if not fastenBaleRetVal then
		-- Probably there werent any free bale attachers left (i.e. the trailer is full) so stop loading
		self.loadingTimerID = 0
		self.currentBaleWorker = nil
		self.savedBaleScanResult = nil
	else
		if #baleScanResult > 1 then
			for i = 1, #baleScanResult do
				baleScanResult[i] = baleScanResult[i + 1]
			end
			self.savedBaleScanResult = baleScanResult
		else
			self.savedBaleScanResult = nil
		end
    end

	return fastenBaleRetVal
end


function BaleVehicle:playDropPointAnimationInReverse()
	local unloadingPoint, unloadingView, animationCS, trackId, animationClipIndex = self.dropBaleHandlerInUse:getDropPointWithAttributes()
	if animationCS ~= nil then
		--setAnimTrackTime(animcationCS, trackId, getAnimClipDuration(animationCS, animationClipIndex), false)
		setAnimTrackSpeedScale(animationCS, trackId, -1.0)
		enableAnimTrack(animationCS, trackId)
	end
end


function BaleVehicle:simpleUnloadingTimerCallback()
	if self.unloadingTimerID == 0 then
		self.firstRunForUnloading = false
		self:playDropPointAnimationInReverse()
		return false -- We have been asked to stop unloading
	end

    if self.currentBaleWorker ~= "simpleUnloadingTimerCallback" then
		self.unloadingTimerID = 0
		self.firstRunForUnloading = false
        return false -- Some one else is doing something with bales now, this operation is not concurrent
    end

    if self.dropBaleHandlerInUse ~= self.dropBaleHandlers[1] then
		self.unloadingTimerID = 0
		self.currentBaleWorker = nil
		self.firstRunForUnloading = false
        return false -- We don't have the default drop handler
    end

    if self.relinquishBaleHandlerInUse ~= self.relinquishBaleHandlers[1] then
		self.unloadingTimerID = 0
		self.currentBaleWorker = nil
		self.firstRunForUnloading = false
        return false -- We don't have the default relinquisher
    end

    if self.stackHandlerInUse ~= self.stackHandlers[1] then
		self.unloadingTimerID = 0
		self.currentBaleWorker = nil
		self.firstRunForUnloading = false
        return false -- We don't have the default stack handler
    end

	-- Always refresh with drop point to use (perhaps not only key input changes this...)
	self.dropBaleHandlerInUse:switchToDropPoint(self.dropPoints[self.dropPoints.currentDropPoint].name)

	local unloadingPoint, unloadingView, animationCS, trackId, animationClipIndex = self.dropBaleHandlerInUse:getDropPointWithAttributes()

	if self.firstRunForUnloading and animationCS ~= nil then
		setAnimTrackSpeedScale(animationCS, trackId, 1.0)
		setAnimTrackTime(animcationCS, trackId, 0.0, false)
		enableAnimTrack(animationCS, trackId)
		setTimerTime(self.unloadingTimerID, getAnimClipDuration(animationCS, animationClipIndex))
		self.firstRunForUnloading = false
		return true
	else
		setTimerTime(self.unloadingTimerID, self.dropPoints.interval)
	end

	-- Return value from function tells if it could relinquish a bale or not. If not it is probably the case that the trailer is empty and we should stop unloading callbacks
	local relinquishRetVal = self.relinquishBaleHandlerInUse:relinquishBale(self.stackHandlerInUse, unloadingView, unloadingPoint)

	if not relinquishRetVal then
		self.currentBaleWorker = nil
		self.unloadingTimerID = 0
		self:playDropPointAnimationInReverse()
	end

	return relinquishRetVal
end


function BaleVehicle:setupGrabbers(baleTypes, loadingOrders)
	local gbh = GrabBaleHandler.instantiateObject(baleTypes)
	for i = 1, #self.grabPoints do
		gbh:provideGrabPoint(self.grabPoints[i].name, self.grabPoints[i].point, self.grabPoints[i].radius, loadingOrders[i].name)
	end
	gbh:toogleGrabberPoint(self.grabPoints[1].name, true) -- Enables the first one
	self.grabPoints.currentGrabPoint = 1
	return gbh
end

function BaleVehicle:setupDroppers(unloadingOrders)
	local dbh = DropBaleHandler.instantiateObject()
	for i = 1, #self.dropPoints do
		dbh:provideDropPoint(self.dropPoints[i].name, self.dropPoints[i].point, unloadingOrders[i].name, self.dropPoints[i].animationCS, self.dropPoints[i].trackId, self.dropPoints[i].clipIndex)
	end
	dbh:switchToDropPoint(self.dropPoints[1].name)
	self.dropPoints.currentDropPoint = 1
	return dbh
end

--[[
Looks through some of the XML properties etc. that needs to be present for this script to work.
Note that not all fields are checked, the script may fail anyway.

param (number): xmlFile:
	The handle to the xmlFile used by the mod.

return:
	'false' if any property that is searched for is not found. 'true' if all seems fine.

side-effects:
	If the method returns 'false' there will be a debug message written to console/log.txt
--]]
function BaleVehicle:verifyXMLstructure(xmlFile)
	if not hasXMLProperty(xmlFile, "vehicle.baleHandling#stackHandlingRootNode") then
		print(scriptName .. "Cant find XML property: vehicle.baleHandling#stackHandlingRootNode")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling#grabAndDropRootNode") then
		print(scriptName .. "Cant find XML property: vehicle.baleHandling#grabAndDropRootNode")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling#grabInterval") then
		print(scriptName .. "Cant find XML property: vehicle.baleHandling#grabInterval")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling#dropInterval") then
		print(scriptName .. "Cant find XML property: vehicle.baleHandling#dropInterval")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.baleSpec_1") then
		print(scriptName .. "Missing required bale specification")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.unloadingOrder_1") then
		print(scriptName .. "At least one unloading point must be specified")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.loadingOrder_1") then
		print(scriptName .. "At least one loading point must be specified")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.baleSpec_1.baleBox_1") then
		print(scriptName .. "At least one bale box must be specified")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.baleSpec_1.allowedType_1") then
		print(scriptName .. "At least one allowed baletype must be specified")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.grabBalePoint_1") then
		print(scriptName .. "At least point to grab bales from must be specified")
		return false
	end

	if not hasXMLProperty(xmlFile, "vehicle.baleHandling.dropBalePoint_1") then
		print(scriptName .. "At least point to drop bales to must be specified")
		return false
	end

	return true
end


--[[
Reads from the XML file some properties that are used for describing the bale stack, grab and drop points.

param (number): xmlFile:
	The handle to the xmlFile used by the mod.

param (string): pointType:
	What this kind of point is. It should be any of the valid property names without underscore and unique number.

param (string): indexRoot:
	The prefix string of where this object is to be found in the hierarchy, e.g. "17" which will then perhaps become "17|4".
	Note that the indexRoot CAN NOT be a top level component (like "2>" is).

return:
	A table list with fields "name" (XML property) and "point" which is a index to a FS object for each index in the table.
	Also the returned table itself has a field "length" that tells how many indices there are in the table.

side-effects: none
--]]
function BaleVehicle:getSubProperties(xmlFile, xmlPathPrefix, pointType, indexRoot)
	local result = {}
	local i = 1

	while hasXMLProperty(xmlFile,  xmlPathPrefix .. pointType .. "_" .. tostring(i)) do
		result[i] = {}
		local xmlPath = xmlPathPrefix .. pointType .. "_" .. tostring(i)

		result[i].name = getXMLString(xmlFile, xmlPath .. "#name")

		result[i].point = Utils.indexToObject(self.rootNode, indexRoot  .. "|" .. getXMLString(xmlFile, xmlPath .. "#subIndex"))

		if pointType == "baleBox" then
			result[i].width = getUserAttribute(result[i].point, "width")
			result[i].height = getUserAttribute(result[i].point, "height")
			result[i].depth = getUserAttribute(result[i].point, "depth")
		end

		if pointType == "grabBalePoint" then
			result[i].radius = getUserAttribute(result[i].point, "radius")
		end

		if pointType == "dropBalePoint" then
			local animationObject = getXMLString(xmlFile, xmlPath .. "#animationObject")
			local animationClipName = getXMLString(xmlFile, xmlPath .. "#animationClipName")
			if animationObject ~= nil and animationClipName ~= nil then
				result[i].animationCS = getAnimCharacterSet(Utils.indexToObject(self.rootNode, animationObject))
				result[i].clipIndex = getAnimClipIndex(result[i].animationCS, animationClipName)
			end
		end

		i = i + 1
	end

	return result
end



--[[
Gets and stores data related to the spatial objects and properties related to them given described in the XML file for the script.

param (number): xmlFile:
	The handle to the xmlFile used by the mod.

return (table, table, table, table):
	1st return value is the bale specification. 2nd is the list of bale boxes. 3rd is the loading orders. 4th is unloading orders.

side-effects:
	Tables self.grabPoints and self.dropPoints will be set.
--]]
function BaleVehicle:loadPropertiesFromXML(xmlFile)
	local shRoot = getXMLString(xmlFile, "vehicle.baleHandling#stackHandlingRootNode")
	local gpRoot = getXMLString(xmlFile, "vehicle.baleHandling#grabAndDropRootNode")


	self.loadingOrders = self:getSubProperties(xmlFile, "vehicle.baleHandling.", "loadingOrder", shRoot)
	self.unloadingOrders = self:getSubProperties(xmlFile, "vehicle.baleHandling.", "unloadingOrder", shRoot)

	self.grabPoints = self:getSubProperties(xmlFile, "vehicle.baleHandling.", "grabBalePoint", gpRoot)

	self.dropPoints = self:getSubProperties(xmlFile, "vehicle.baleHandling.", "dropBalePoint", gpRoot)

	for i = 1, #self.dropPoints do
		if self.dropPoints[i].animationCS ~= nil then
			self.dropPoints[i].trackId = i
			assignAnimTrackClip(self.dropPoints[i].animationCS, self.dropPoints[i].trackId, self.dropPoints[i].clipIndex)
			setAnimTrackLoopState(self.dropPoints[i].animationCS, self.dropPoints[i].trackId, false)
		end
	end

    local i = 1
	while hasXMLProperty(xmlFile, "vehicle.baleHandling.baleSpec_" .. tostring(i)) do
		local xmlPathPrefix = "vehicle.baleHandling.baleSpec_" .. tostring(i)
        self.baleSpecs[i] = {}
		self.baleSpecs[i].name = getXMLString(xmlFile, xmlPathPrefix .. "#name")
	    self.baleSpecs[i].width = getXMLFloat(xmlFile, xmlPathPrefix .. "#width")
	    self.baleSpecs[i].height = getXMLFloat(xmlFile, xmlPathPrefix .. "#height")
	    self.baleSpecs[i].depth = getXMLFloat(xmlFile, xmlPathPrefix .. "#depth")

		self.baleSpecs[i].baleBoxes = self:getSubProperties(xmlFile, xmlPathPrefix .. ".", "baleBox", shRoot)

		self.baleSpecs[i].baleTypes = {}
        local j = 1
        while hasXMLProperty(xmlFile, xmlPathPrefix .. ".allowedType_" .. tostring(j)) do
            self.baleSpecs[i].baleTypes[j] = getXMLString(xmlFile, xmlPathPrefix .. ".allowedType_" .. tostring(j) .. "#name")
            j = j + 1
        end

		i = i + 1
    end

	self.grabPoints.interval = getXMLInt(xmlFile, "vehicle.baleHandling#grabInterval")
	self.dropPoints.interval = getXMLInt(xmlFile, "vehicle.baleHandling#dropInterval")


-- Seems like that if I use this code to load the animation the normal folding doesn't work. Perhaps it can't be used twice.
-- If so, the question is how do I get hold of the animation FS has created when it found the foldingParts section in the XML path.

--~ 	if hasXMLProperty(xmlFile, "vehicle.foldingParts") then
--~ 		local xmlFoldPath = "vehicle.foldingParts.foldingPart"
--~ 		-- Exploiting the short circuit evaluation in lua
--~ 		if hasXMLProperty(xmlFile, xmlFoldPath) then
--~ 			and hasXMLProperty(xmlFile,xmlFoldPath .. "#isAStand")
--~ 			and getXMLBool(xmlFile, xmlFoldPath .. "#isAStand") then
--~ 			local index = #self.standClip + 1
--~ 			self.standClip[index] = {}
--~ 			self.standClip[index].animationClipName = getXMLString(xmlFile, xmlFoldPath .. "#animationClip")
--~ 			local animObjectPath = getXMLString(xmlFile, xmlFoldPath .. "#rootNode")
--~ 			self.standClip[index].animCharSet = getAnimCharacterSet(Utils.indexToObject(self.rootNode, animObjectPath))
--~ 			self.standClip[index].clipIndex = getAnimClipIndex(self.standClip[index].animCharSet, self.standClip[index].animationClipName)
--~ 			self.standClip[index].trackId = #self.dropPoints + 1
--~ 			assignAnimTrackClip(self.standClip[index].animCharSet, self.standClip[index].trackId, self.standClip[index].clipIndex)
--~ 			setAnimTrackLoopState(self.standClip[index].animCharSet, self.standClip[index].trackId, false)
--~ 		end
--~ 	end

end





function BaleVehicle:registerGrabber(grabberObject)
    self.grabBaleHandlers[#self.grabBaleHandlers + 1] = grabberObject
end


function BaleVehicle:registerStacker(stackerObject)
    self.stackHandlers[#self.stackHandlers + 1] = stackerObject
end


function BaleVehicle:registerDropper(dropperObject)
    self.dropBaleHandlers[#self.dropBaleHandlers + 1] = dropperObject
end


function BaleVehicle:registerFastener(fastenerObject)
    self.fastenBaleHandlers[#self.fastenBaleHandlers + 1] = fastenerObject
end


function BaleVehicle:registerRelinquisher(relinquisherObject)
    self.relinquishBaleHandlers[#self.relinquishBaleHandlers + 1] = relinquisherObject
end


function BaleVehicle:registerHandlers(foreignHandlers)
	self.handlerSets[#self.handlerSets + 1] = foreignHandlers
	self:buildHandlerTables()
end

function BaleVehicle:unregisterHandlers(foreignHandlerSet)
	local foundForeignSet
	for i = 1, #self.handlerSets do
		if self.handlerSets[i].id == foreignHandlerSet.id then
			for j = i + 1, #self.handlerSets do
				self.handlerSets[j - 1] = self.handlerSets[j] -- Shifting indices to the left
			end
			foundForeignSet = true
			break
		end
	end
	if foundForeignSet then
		self.buildHandlerTables()
	end
end


function BaleVehicle:buildHandlerTables()
	for i = 1, #self.handlerSets do
		for j = 1, #self.handlerSets[i].stack do
			self.stackHandlers[#self.stackHandlers + 1] = self.handlerSets[i].stack[j]
		end
		for j = 1, #self.handlerSets[i].grab do
			self.grabBaleHandlers[#self.grabBaleHandlers + 1] = self.handlerSets[i].grab[j]
		end
		for j = 1, #self.handlerSets[i].drop do
			self.dropBaleHandlers[#self.dropBaleHandlers + 1] = self.handlerSets[i].drop[j]
		end
		for j = 1, #self.handlerSets[i].fasten do
			self.fastenBaleHandlers[#self.fastenBaleHandlers + 1] = self.handlerSets[i].fasten[j]
		end
		for j = 1, #self.handlerSets[i].relinquish do
			self.relinquishBaleHandlers[#self.relinquishBaleHandlers + 1] = self.handlerSets[i].relinquish[j]
		end

		self.stackHandlerInUse = self.stackHandlers[1]
		self.grabBaleHandlerInUse = self.grabBaleHandlers[1]
		self.dropBaleHandlerInUse = self.dropBaleHandlers[1]
		self.fastenBaleHandlerInUse = self.fastenBaleHandlers[1]
		self.relinquishBaleHandlerInUse = self.relinquishBaleHandlers[1]
	end
end




--[[ 	AND THIS IS THE CURRENT NON OPERATIONAL CODE OF INTER MOD COMMUNICATION FACILITY.
		AS IT IS NOT USED IT IS ALL COMMENTED OUT.
--]]



--~ ModComm = {}
--~ ModComm.__index = ModComm


--~ function ModComm.instantiateObject(modObject, GENERIC_TYPE, MOD_TYPE, MOD_NAME)
--~ 	local mc = {}
--~ 	setmetatable(mc, ModComm)

--~     mc.MOD_COMM_VERSION = 1

--~ 	mc.modObject = modObject

--~     mc.knownModComms = {}

--~ 	mc.thisModGenericType = GENERIC_TYPE
--~ 	mc.thisModVehicleType = MOD_TYPE
--~     mc.thisModID = MOD_NAME

--~     mc.couplingActions = {}

--~     mc.compatibleMods = {}
--~ 	mc.compatibleMods[1] = {}
--~ 	mc.compatibleMods[1].ID = MOD_NAME
--~ 	mc.decouplingActions = {}
--~     return mc
--~ end

--~ function ModComm:testNewVehicle(newVehicle)
--~     if newVehicle.getInterModCommunicationHandler ~= nil then
--~ 		print("Found a new vehicle that has an IMCH")
--~ 		local imch = newVehicle.getInterModCommunicationHandler()
--~ 		if imch:isCompatibleWith(self.thisModGenericType, self.thisModVehicleType, self.thisModID) then
--~ 			self.imchs[#self.imchs + 1] = imch
--~ 			imch:recieveSuccessfullCoupling (self.myImch)
--~ 			print("And it was compatible with this mod")
--~ 		end
--~ 	end
--~ end

--~ function ModComm:receiveModCommInstance(mc)
--~     self.knownModComms[#self.knownModComms + 1] = mc
--~ 	self.decoupleFunctions[#self.decoupleFunctions + 1] = mc.modObject.decoupleMod
--~ end

--~ function ModComm:receive(remoteFunction, modID
--~ 	if modID == self.thisModID then
--~ 		remoteFunction(self.modObject)
--~ 	end
--~ end

--~ function ModComm:removeModCommInstance(mc)
--~     local total = #self.knownModComms

--~     for i = 1, total do
--~         if mc == self.knownModComms[i] then
--~             self.knownModComms[i] = nil
--~             for j = i, total do
--~                 self.knownModComms[j] = self.knownModComms[j + 1]
--~             end
--~ 			mc.modObject.decoupleMod(self.modObject, mc.modObject)
--~         end
--~     end
--~ end


--~ function ModComm:addModCouplingAction(modID, action)
--~     if not self:isCompatibleWith(modID, modVersion) then
--~         return
--~ 	end

--~     for i = 1, #self.couplingActions do
--~         if self.couplingActions[i].ID == modID then
--~             self.couplingActions[i].actions[#self.couplingActions[i].actions + 1] = action
--~         else
--~             local newIndex = #self.couplingActions + 1
--~             self.couplingActions[newIndex] = {}
--~             self.couplingActions[newIndex].ID = modID
--~             self.couplingActions[newIndex].actions = {}
--~             self.couplingActions[newIndex].actions[1] = action
--~         end
--~     end
--~ end

-- Only accepts the same mod
--~ function ModComm:isCompatibleWith(genericType, vehicleType, modID)
--~     for i = 1, #self.compatibleMods do
--~         if self.compatibleMods[i].ID == modID then
--~             return true
--~         else
--~             return false
--~         end
--~     end
--~ end


--~ function ModComm:recieveSuccessfullCoupling(mc)
--~     local performedActions = {}
--~     for i = 1, #self.couplingActions do
--~         if self.couplingActions[i].ID == mc.thisModID then
--~             local alreadyDone = false
--~             for j = 1, #performedActions do
--~                 if self.couplingActions[i].action == performeActions[j] then
--~                     alreadyDone = true
--~                     break
--~                 end
--~             end
--~             if not alreadyDone then
--~                 self.couplingActions[i].action(self.modObect)
--~                 performeActions[#performedActions + 1] = self.couplingActions[i].action
--~             end
--~         end
--~     end
--~ end



BVUtils = {}
BVUtils.__index = BVUtils


--[[	BVUtils.constructBoxGrid (box, spacingX, spacingY, spacingZ, balePoints)
Takes a bale box and add it to the balePoints total set of bale attachers.

param (double): spacingX/Y/Z:
	How far away (spacing between) the attachment points should be in this 3D box grid.

return: nothing

side-effects:
	self.baleAttacher.length will increased with the number of bales in the bale box, as will their be indices for them
	created in the self.baleAttacher[x] table, starting from "offset".
--]]
function BVUtils.constructBoxGrid (box, spacingX, spacingY, spacingZ, balePoints)
	local width = spacingX
	local height = spacingY
	local depth = spacingZ

	local onEntryNrOfAttachers = #balePoints

	for i = 0, box.depth*box.height*box.width - 1, box.height*box.width do
		for j = 0, box.height*box.width - 1, box.width do
			for k = 0, box.width - 1 do
				local index = 1 + onEntryNrOfAttachers + i+j+k
				balePoints[index] = {}
				local transformPoint = createTransformGroup("Bale transformPoint: depth: " .. (i/(box.height*box.width)) .. ", height: " .. (j/box.width) .. ", width: " .. k)

				link(box.point, transformPoint)
				--setRotation(transformPoint, rotX, rotY, rotZ)
				setTranslation(transformPoint, k * width, (j/box.width) * height, (i/(box.height*box.width)) * depth)

				-- Thanks to Stefan Geiger (GIANTS Software) for this piece of code that enables the box to be rotated in any direction.
				local x,y,z = worldToLocal(getParent(box.point), getWorldTranslation(transformPoint))
				local dx,dy,dz = worldDirectionToLocal(getParent(box.point), localDirectionToWorld(transformPoint, 0,0,1))
				local upx, upy, upz = worldDirectionToLocal(getParent(box.point), localDirectionToWorld(transformPoint, 0,1,0))
				link(getParent(box.point), transformPoint)
				setTranslation(transformPoint, x,y,z)
				setDirection(transformPoint, dx,dy,dz, upx,upy,upz)

				balePoints[index].transformPoint = transformPoint
			end
		end
	end
end











--[[
Class:
	StackHandler

Descripction:
	The stack handler is a helper class for making it easy to manage the complex world of putting bales in a stack!

	Bale boxes are a non-trivial concept. A bale box is at least 1x1x1 (width x height x depth) where the units are number of
	bales. The bale width, height and depth (set in the constructor) determines the spatial distance between them.
	A box can have its bales placed perpendicular (90 degrees in the 3 dimensions) to the normal way a bale is placed on the
	ground.
	Several bale boxes can (and probably should) be used to make up the stack that will be built up in the trailer's loading
	area (in the "cage"). Example of this is that usually one places a few more bales in the middle at the top most layer(s).

	Also there are loading/unloading orders, where order means (wiktionary): "(uncountable) Arrangement, disposition, sequence".
	So it will put bales to or grab bales from the stack, first to last. These are a "view" to the bale in a specific order,
	instead of going to the order they are added as bale boxes.
	When loading (the loading orders) the first bale attachment point is the one furthest away from the point.
	Naturally as one starts stacking bales at the back/at the bottom. The unloading is the reverse, closest is the first bale
	attachment point.

	All the transform groups of bale boxes, loading orders and unloading orders should share the same parent object.

	When everything is setup the user of this class only need to worry about putting bales (they will be automatically
	transported to the stack) and remove from the stack (providing a drop point).
--]]
StackHandler = {}
StackHandler.__index = StackHandler

--[[	StackHandler.instantiateObject(baleSpec)
Constructor of the StackHandler class

Param (table): baleSpec:
	Contains three self-explaining values, "width", "height" and "depth" of a bale, measured in world spatial units
--]]
function StackHandler.instantiateObject(baleSpec, loadingOrders, unloadingOrders)
	local sh = {}
	setmetatable(sh, StackHandler)
	-- Register member methods
	sh.appendBox = StackHandler.appendBox
	sh.createStack = StackHandler.createStack
	sh.calculateSpatialOrder = StackHandler.calculateSpatialOrder
	sh.addOrderSourcePoint = StackHandler.addOrderSourcePoint
	sh.getNextAttachPointToUnload = StackHandler.getNextAttachPointToUnload
	sh.getNextFreeAttachPointToLoad = StackHandler.getNextFreeAttachPointToLoad
	sh.getNrOfLoadedBalesOutOfTotalNr = StackHandler.getNrOfLoadedBalesOutOfTotalNr

	sh.baleSpec = baleSpec
	sh.loadingDirection = {}
	sh.unloadingDirection = {}
    sh.putBaleFunctions = {}
    sh.dropBaleFunctions = {}

	sh:createStack()

	for i = 1, #loadingOrders do
		sh:addOrderSourcePoint(loadingOrders[i].name, loadingOrders[i].point, true)
	end

	for i = 1, #unloadingOrders do
		sh:addOrderSourcePoint(unloadingOrders[i].name, unloadingOrders[i].point, false)
	end

	return sh
end


--[[	StackHandler:createStack(boxList)
Constructs the stack by using the balespecifications given in the constructor.

return: nothing

side-effects:
	A new table self.baleAttacher is created. It will be populated if any bale box is non-empty (size is at least (1,1,1)).
--]]
function StackHandler:createStack()
	self.baleAttacher = {}

	for i = 1, #self.baleSpec.baleBoxes do
		local box = self.baleSpec.baleBoxes[i]
		if box.width > 0 and box.height > 0 and box.depth > 0  and box.point then
			--self:appendBox(box, self.baleSpec.width, self.baleSpec.height, self.baleSpec.depth)
            BVUtils.constructBoxGrid(box, self.baleSpec.width, self.baleSpec.height, self.baleSpec.depth, self.baleAttacher)
		end
	end
	self.occupiedAttachPoints = 0
end


--[[	StackHandler:calculateSpatialOrder(sourcePoint, beginClosest)
Constructs a spatial order of the bale attachers seen from a specific point. The stack should be built and ready when this
method is called.

param (number): sourcePoint:
	Index of a point (preferable a transform group) in FS. This is the source of the ordering.

param (boolean): beginClosest:
	If 'true' the first index in the result table will be the spatial point that is the closest to source point.
	If 'false' it will be the other way around.

return (table):
	Table indexed by numbers (1 to [number of bale attachers]) which field "attacher" points to the baleAttacher table and
	"distance" is the distance between this attacher and the source point (used for sorting the table).

side-effects: none
--]]
function StackHandler:calculateSpatialOrder(sourcePoint, beginClosest)
	local sX, sY, sZ = getTranslation(sourcePoint)
	local result = {};

	for i = 1, #self.baleAttacher do
		local hX, hY, hZ = getTranslation(self.baleAttacher[i].transformPoint)
		result[i] = {}
		result[i].attacher = self.baleAttacher[i]
		result[i].distance = Utils.vector3Length(sX - hX, sY - hY, sZ - hZ);
	end

	if beginClosest then
		table.sort(result, function (v1, v2) return v1.distance < v2.distance; end)
	else
		table.sort(result, function (v1, v2) return v1.distance > v2.distance; end)
	end

	return result;
end


--[[	StackHandler:addOrderSourcePoint(name, point, isLoader)
Adds a source point and calculates the ordering of this "view" of the bales.

param (string): name:
	Will become a table field in the table for the loading directions (orders) or unloading directions,
	which contains number-indexed order of the bales.

param (number): point:
	Index to a FS object (preferable a transform group) from which the ordering should be computed.

param (boolean): isLoader:
	If this direction is meant as where the bales should be loaded (back-to-front) this should be 'true'.
	If it is the unloading direction (front-to-back) this should be 'false'.

return: nonthing

side-effects:
	Depending on "isLoader" either self.loadingDirection or self.unloadingDirection will get a new mapping between
	"name" and an order of bales.
--]]
function StackHandler:addOrderSourcePoint(name, point, isLoader)
	if isLoader then
		self.loadingDirection[name] = self:calculateSpatialOrder(point, false)
	else
		self.unloadingDirection[name] = self:calculateSpatialOrder(point, true)
	end
end


--[[    StackHandler:getNextAttachPointToUnload(unloaderName)
When someone want to unload a bale they should call this. It will look through the stack based and pick a bale, if any exist at all in the stack.

param (string): unloderName:
    Name of the "view" from which the unloading should take place.

return (table):
    An attachpoint. If there are no available bales (the stack is empty) this will be nill.

side-effects:
    If any bale is released the occupiedAttachPoints will be updated to tell the number of bales except the one that is removed.
    Note that the bale in the attachpoint is not changed, that's the responsibility of the relinquisher to do.
--]]
function StackHandler:getNextAttachPointToUnload(unloaderName)
	local attachPoint = nil
	local foundOccupied = false

	if self.unloadingDirection[unloaderName] ~= nil then
		for i = 1, #self.baleAttacher do
			attachPoint = self.unloadingDirection[unloaderName][i].attacher
			if attachPoint.bale ~= nil then
				foundOccupied = true
				break
			end
		end
	end

	if foundOccupied then
		self.occupiedAttachPoints = 0
		for i = 1, #self.baleAttacher do
			if self.baleAttacher[i].bale then
				self.occupiedAttachPoints = self.occupiedAttachPoints + 1
			end
		end
		-- We assume that the one requesting to unload will do so instantely. Thus we should count one less bale.
		self.occupiedAttachPoints = self.occupiedAttachPoints - 1

		return attachPoint
	else
		return nil
	end
end


--[[    StackHandler:getNextAttachPointToLoad(loaderName)
When someone want to load a bale they should call this. It will look through the stack based and pick an empty bale attachpoint, if any exist at all in the stack.

param (string): loderName:
    Name of the "view" from which the loading should take place.

return nothing

side-effects:
    If free bale attachpoint is found the occupiedAttachPoints will be updated to tell the number of bales including the one that is about to be loaded.
    Note that the bale in the attachpoint is not changed, that's the responsibility of the fastener to do.
--]]
function StackHandler:getNextFreeAttachPointToLoad(loaderName)
	local attachPoint = nil
	local foundFree = false

	if self.loadingDirection[loaderName] ~= nil then
		for i = 1, #self.baleAttacher do
			attachPoint = self.loadingDirection[loaderName][i].attacher
			if attachPoint.bale == nil then
				foundFree = true
                break
			end
		end
	end

	if foundFree then
		self.occupiedAttachPoints = 0
		for i = 1, #self.baleAttacher do
			if self.baleAttacher[i].bale then
				self.occupiedAttachPoints = self.occupiedAttachPoints + 1
			end
		end
		-- We assume that the one requesting to load will do so instantely. Thus we should count one extra bale.
		self.occupiedAttachPoints = self.occupiedAttachPoints + 1
		return attachPoint
	else
		return nil
	end
end

--[[ StackHandler:getNrOfLoadedBalesOutOfTotalNr()
return (int, int):
    returns the number of bales that it has counted earlier as should be in the stack. No new count is done for each run of this function. Second return value is the capacity of the stack.

side-effects:
    none
--]]
function StackHandler:getNrOfLoadedBalesOutOfTotalNr()
	return self.occupiedAttachPoints, #self.baleAttacher
end






FastenBaleHandler = {}
FastenBaleHandler.__index = FastenBaleHandler

function FastenBaleHandler.instantiateObject()
	local fbh = {}
	setmetatable(fbh, FastenBaleHandler)

	-- Register member methods
	fbh.fastenBale = FastenBaleHandler.fastenBale

	return fbh
end


--[[ FastenBaleHandler:fastenBale(sh, loaderName, bale)
Called when we have a bale that we want to load onto a stack.

param (table): sh:
    A valid stackhandler. A full stack is handled.

param (string): loaderName:
    The name of the "view" from which the bale should be loaded onto the stack.

param (int): bale:
    Reference to the giants object which is a valid bale.

return:
    true if the bale was successfully fastened, false if not (e.g. the stack was full)

side-effects:
    If the stackhandler returns an attachpoint it will be changed so the bale is attached their.
    The bale will be moved to that spot in the world.
--]]
function FastenBaleHandler:fastenBale(sh, loaderName, bale)
    local attachPoint = sh:getNextFreeAttachPointToLoad(loaderName)

    if attachPoint == nil then
        return false
    end

    setRigidBodyType(bale,"none")
	link(attachPoint.transformPoint, bale) -- Links the bale spatially to this attachment point
    setTranslation(bale, 0, 0, 0)
	setRotation(bale, 0, 0, 0)
	attachPoint.bale = bale -- Occupy this abstract attachment point with the bale

	return true
end






RelinquishBaleHandler = {}
RelinquishBaleHandler.__index = RelinquishBaleHandler

function RelinquishBaleHandler.instantiateObject()
	local rbh = {}
	setmetatable(rbh, RelinquishBaleHandler)

	-- Register member methods
	rbh.relinquishBale = RelinquishBaleHandler.relinquishBale

	return rbh
end


--[[ RelinqushBaleHandler:relinquishBale(sh, unloaderName, dropPoint)
Called when we want to try to relinquish (release) a bale from the stack.

param (table): sh:
    Stackhandler that we want to use.

param (string): unloaderName:
    The "view" from which bales should be unloaded (the order).

param (int): dropPoint:
    Reference to the point in the world where the bale should be released.

return:
    false if it couldn't release any bale (e.g. the stack "was empty"). True if it was successfull.

side-effects:
    The bale attachment point returned from the stackhandler will be released of its bale.
--]]
function RelinquishBaleHandler:relinquishBale(sh, unloaderName, dropPoint)
    local attachPoint = sh:getNextAttachPointToUnload(unloaderName)

    if attachPoint == nil then
        return false
    end

	local bale = attachPoint.bale
	attachPoint.bale = nil -- Remove the association to the bale that is about to be dropped

	link(getRootNode(), bale)
	setTranslation(bale, getWorldTranslation(dropPoint))
	setRotation(bale, getWorldRotation(dropPoint))
	setRigidBodyType(bale,"Dynamic")

	return true
end




GrabBaleHandler = {}
GrabBaleHandler.__index = GrabBaleHandler

--[[	StackHandler.instantiateObject(baleTypes)
Constructor of the GrabBaleHandler class

Param (table): baleTypes:
	Array of strings with names of different baletypes, used for matching against bale objects.
--]]
function GrabBaleHandler.instantiateObject(baleTypes)
	local gbh = {}
	setmetatable(gbh, GrabBaleHandler)
	-- Register member methods
	gbh.provideGrabPoint = GrabBaleHandler.provideGrabPoint
	gbh.toogleGrabberPoint = GrabBaleHandler.toogleGrabberPoint
	gbh.getBalesWithinGrabbers = GrabBaleHandler.getBalesWithinGrabbers
	gbh.scanForBalesWithinGrabbers = GrabBaleHandler.scanForBalesWithinGrabbers


	gbh.baleTypes = baleTypes
	gbh.grabPoints = {}
	gbh.activeGrabbers = {}

	return gbh
end


--[[ GrabBaleHandler:provideGrabPoint(name, grabPoint, grabRadius, loadingView)
Provides a new grab point, where the scanner can look for bales. By default it is deactivated.

param (string): name:
    The name of this point.

param (int): grabPoint:
    reference to the giants object for the point in the model where it should grab bales.

param (float): grabRadius:
    In which radius the grabber should scan for bales.

param (string): loadingView:
    If a bale is found it need to be loaded with the proper view to the stack. This remebers it.

return:
    nothing

side-effects:
    A newe entry in the grabPoints table will be create where all this data is stored.
--]]
function GrabBaleHandler:provideGrabPoint(name, grabPoint, grabRadius, loadingView)
	if grabRadius > 0.01 then -- Require some sane functionality out of this grabPoint.
		local index = #self.grabPoints + 1
		self.grabPoints[index] = {}
		self.grabPoints[index].name = name
		self.grabPoints[index].point = grabPoint
		self.grabPoints[index].radius = grabRadius
        self.grabPoints[index].loadingView = loadingView
		self.grabPoints[index].enabled = false
	end
end


--[[ GrabBaleHandler:toogleGrabberPoint(name, state)
Sets a grabbers state (activated or deactivated) regardless of current state.

param (string): name:
    Name of the grabber.

param (boolean): state:
    To which state to set the grabber to. true means enabled and false disabled.

return:
    nothing.

side-effects:
    the enabled-membervariable of the grabber will be set (but possible not changed if already at that value).
--]]
function GrabBaleHandler:toogleGrabberPoint(name, state)
	self.activeGrabbers = {} -- resets it
	local j = 1

	for i = 1, #self.grabPoints do
		if self.grabPoints[i].name == name then
			self.grabPoints[i].enabled = state
		end

		if self.grabPoints[i].enabled then
			self.activeGrabbers[j] = self.grabPoints[i]
			j = j + 1
		end
	end
end


function GrabBaleHandler:getBalesWithinGrabbers()
	if #self.activeGrabbers == 0 then
		return nil -- If there aren't any grabbers activated, then it's pointless to have it scan for bales.
	end

	local searchResults = self:scanForBalesWithinGrabbers(self.activeGrabbers)
	local result = {}

	if searchResults == nil then
		return nil
	else
	    local j = #searchResults
		for i = 1, #self.activeGrabbers do
			if searchResults[i].bale ~= nil then
				local duplicateBale = false
				for k = #searchResults, 1, -1 do
					if result[k] == nil then
						break
					end
					if result[k].bale == searchResults[i].bale then
						duplicateBale = true
						break
					end
				end
				if not duplicateBale then
					result[j] = {}
					result[j].bale = searchResults[i].bale
					result[j].grabberName = self.grabPoints[i].name
					result[j].loadingView = self.grabPoints[i].loadingView
					j = j - 1
				end
			end
		end
	end

	return result
end



function GrabBaleHandler:scanForBalesWithinGrabbers(grabbers)
	local closestBaleForEachGrabber = {}
	local candidateItem
	local distance
	local gpX = {}
	local gpY = {}
	local gpZ = {}
	local foundAnyBale = false

	for i = 1, #grabbers do
		local x, y, z = getWorldTranslation(grabbers[i].point)
		gpX[i] = x
		gpY[i] = y
		gpZ[i] = z
		closestBaleForEachGrabber[i] = {}
	end

	for i = 1, table.getn(g_currentMission.itemsToSave) do

		if getParent(g_currentMission.itemsToSave[i].node) == getRootNode() then
			candidateItem = g_currentMission.itemsToSave[i].node
			for j = 1, #self.baleTypes do

				if getUserAttribute(candidateItem, self.baleTypes[j]) then
					-- We have found a valid bale
					local bX, bY, bZ = getWorldTranslation(candidateItem)

					for k = 1, #grabbers do
						distance = Utils.vector3Length(gpX[k] - bX, gpY[k] - bY, gpZ[k] - bZ)
						if distance < grabbers[k].radius then
							-- It is within this grabber
							if closestBaleForEachGrabber[k].distance == nil then
								closestBaleForEachGrabber[k].distance = math.huge -- start out with positive infinity
							end

							if distance < closestBaleForEachGrabber[k].distance then
								-- It is the bales closest to the the grabber's center, thus far
								closestBaleForEachGrabber[k].distance = distance
								closestBaleForEachGrabber[k].bale = candidateItem
							end

							foundAnyBale = true
						end
					end
				end
			end
		end
	end

	if foundAnyBale then
		return closestBaleForEachGrabber
	else
		return nil
	end
end



DropBaleHandler = {}
DropBaleHandler.__index = DropBaleHandler

function DropBaleHandler.instantiateObject()
	local dbh = {}
	setmetatable(dbh, DropBaleHandler)
	-- Register member methods
	dbh.provideDropPoint = DropBaleHandler.provideDropPoint
	dbh.switchToDropPoint = DropBaleHandler.switchToDropPoint
	dbh.getDropPointWithAttributes = DropBaleHandler.getDropPointWithAttributes


	dbh.dropPoints = {}

	return dbh
end


--[[ DropBaleHandler:provideDropPoint(name, dropPoint, loadingView, animationCS)
Provied a drop point to this handler, telling where a bale can be dropped.

param (string): name:
    Name of this new drop point.

param (int): dropPoint:
    Reference to the giants world object (preferable an transformation group in empty space)

param (string): loadingView:
    Name of the loading view (for the stack handler) that should be associated with this drop point.

param (?): animationCS:
    Reference to which animation character set that should be associated with this drop point.
    Note that this should not be an animation for the bale but for something else, like a hatch or fence door that needs to be opened
    before the unloading starts.

return:
    nothing.

side-effects:
    A new entry into the dropPoints table with the information from the parameters.
--]]
function DropBaleHandler:provideDropPoint(name, dropPoint, loadingView, animationCS, trackId, animationClipIndex)
	self.dropPoints[name] = {}
	self.dropPoints[name].point = dropPoint
    self.dropPoints[name].loadingView = loadingView
	self.dropPoints[name].animationCS = animationCS
	self.dropPoints[name].trackId = trackId
	self.dropPoints[name].animationClipIndex = animationClipIndex
end


--[[ DropBaleHandler:switchToDropPoint(name)
Only one drop point is activated at a time. This switches it.

param (string): name:
    Name of the drop point to switch to.

return:
    true if a switch was made, false if not (e.g. no drop point with that name).

side-effects:
    The variable remembering which drop point is active may be updated.
--]]
function DropBaleHandler:switchToDropPoint(name)
	if self.dropPoints[name] then -- If not found it will be nil which evaluates to false
		self.currentDropPointName = name
	end
end


function DropBaleHandler:getDropPointWithAttributes()
	local dp = self.dropPoints[self.currentDropPointName]
	return dp.point, dp.loadingView, dp.animationCS, dp.trackId, dp.animationClipIndex
end

