--
-- BuyableTwinWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--

BuyableFordsonWeights = {};

function BuyableFordsonWeights.prerequisitesPresent(specializations)
    return true;
end;

function BuyableFordsonWeights:load(xmlFile)

	self.twinWheelTriggerCallback = BuyableFordsonWeights.twinWheelTriggerCallback;
	self.wheelDelete = BuyableFordsonWeights.wheelDelete;
	self.assembleFordsonWheights = SpecializationUtil.callSpecializationsFunction("assembleFordsonWheights");
	self.disassembleFordsonWheights = SpecializationUtil.callSpecializationsFunction("disassembleFordsonWheights");

	self.checkFordsonString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.twinWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#activationTrigger"));
	
	
	addTrigger(self.usingTrigger, "twinWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#belts"));
	
    self.buyableFordsonWeights = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.twinWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		setRigidBodyType(wheel.collision, "Kinematic");
		setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableFordsonWeights, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.fordsonWeightsUser = nil;
end;

function BuyableFordsonWeights:delete()
	if self.fordsonWeightsUser ~= nil then
		self.fordsonWeightsUser:onDisassembling(true);
		for _,twinWheel in pairs(self.buyableFordsonWeights) do
			delete(twinWheel.node);
			delete(twinWheel.collision);
		end;
		self.buyableFordsonWeights = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableFordsonWeights:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableFordsonWeights:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local fordsonWeightsUser = networkGetObject(id);
		if fordsonWeightsUser ~= nil then
			self:assembleFordsonWheights(fordsonWeightsUser, true);
		end;
	end;
end;

function BuyableFordsonWeights:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.fordsonWeightsUser ~= nil then
		idToWrite = networkGetObjectId(self.fordsonWeightsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableFordsonWeights:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableFordsonWeights:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableFordsonWeights:update(dt)
end;

function BuyableFordsonWeights:updateTick(dt)

	-- try to load saved twinwheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then				
				self:assembleFordsonWheights(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;
end;

function BuyableFordsonWeights:draw()	
end;

function BuyableFordsonWeights:twinWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then		
			if vehicle.rootNode == otherId then
				if vehicle.buyableFordsonWeights ~= nil then
					if vehicle.buyableFordsonWeights.checkFordsonString == self.checkFordsonString then
						if self.fordsonWeightsUser ~= nil then
							if self.fordsonWeightsUser == vehicle then
								vehicle.buyableFordsonWeights.fordsonWeightsInRange = self;
							end;
						else
							vehicle.buyableFordsonWeights.fordsonWeightsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.buyableFordsonWeights ~= nil then
					vehicle.buyableFordsonWeights.fordsonWeightsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableFordsonWeights:assembleFordsonWheights(vehicle, noEventSend)
	BuyableFordsonWeightsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;
	if self.fordsonWeightsUser == nil then
		if vehicle.buyableFordsonWeights ~= nil then
			self.fordsonWeightsUser = vehicle;	
			for k,wheel in pairs(vehicle.wheels) do
				for l,twinWheel in pairs(self.buyableFordsonWeights) do 
					if k == l then	
						unlink(twinWheel.node);
						unlink(twinWheel.collision);
						link(wheel.driveNode, twinWheel.node);	
						link(getParent(wheel.repr), twinWheel.collision);	
						local x,y,z = getWorldTranslation(wheel.driveNode);
						x,y,z = worldToLocal(getParent(twinWheel.collision), x,y,z);
						setTranslation(twinWheel.collision,x,y,z);
						--setRigidBodyType(twinWheel.collision, "Kinematic");
						break;
					end;
				end;
			end;	
			self.fordsonWeightsUser:onAssembling(self);
		end;
	end;
end;

function BuyableFordsonWeights:disassembleFordsonWheights(noEventSend)
	BuyableFordsonWeightsDetachEvent.sendEvent(self, noEventSend);

	self.fordsonWeightsUser:onDisassembling();

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;
	
	for k,wheel in pairs(self.buyableFordsonWeights) do
		setRigidBodyType(wheel.collision, "NoRigidBody");
		unlink(wheel.collision);
		unlink(wheel.node);
		setRotation(wheel.collision, 0,0,0);
		setTranslation(wheel.collision, 0,0,0);
		link(wheel.savePosition, wheel.collision);
		link(wheel.collision, wheel.node);
	end;	
	self.fordsonWeightsUser = nil;	
end;


function BuyableFordsonWeights:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");		
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};			
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableFordsonWeights:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	
	if self.fordsonWeightsUser ~= nil then
		local x,y,z = getWorldTranslation(self.fordsonWeightsUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;
	
    return attributes, nil;
end;


BuyableFordsonWeightsAttachEvent = {};
BuyableFordsonWeightsAttachEvent_mt = Class(BuyableFordsonWeightsAttachEvent, Event);

InitEventClass(BuyableFordsonWeightsAttachEvent, "BuyableFordsonWeightsAttachEvent");

function BuyableFordsonWeightsAttachEvent:emptyNew()
    local self = Event:new(BuyableFordsonWeightsAttachEvent_mt);
    self.className="BuyableFordsonWeightsAttachEvent";
    return self;
end;

function BuyableFordsonWeightsAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableFordsonWeightsAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableFordsonWeightsAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableFordsonWeightsAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableFordsonWeightsAttachEvent:run(connection)
	self.vehicle:assembleFordsonWheights(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableFordsonWeightsAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableFordsonWeightsAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableFordsonWeightsAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableFordsonWeightsAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableFordsonWeightsDetachEvent = {};
BuyableFordsonWeightsDetachEvent_mt = Class(BuyableFordsonWeightsDetachEvent, Event);

InitEventClass(BuyableFordsonWeightsDetachEvent, "BuyableFordsonWeightsDetachEvent");

function BuyableFordsonWeightsDetachEvent:emptyNew()
    local self = Event:new(BuyableFordsonWeightsDetachEvent_mt);
    self.className="BuyableFordsonWeightsDetachEvent";
    return self;
end;

function BuyableFordsonWeightsDetachEvent:new(vehicle)
    local self = BuyableFordsonWeightsDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableFordsonWeightsDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableFordsonWeightsDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableFordsonWeightsDetachEvent:run(connection)
	self.vehicle:disassembleFordsonWheights(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableFordsonWeightsDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;


function BuyableFordsonWeightsDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableFordsonWeightsDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableFordsonWeightsDetachEvent:new(vehicle));
		end;
	end;
end;