--
-- BuyableTwinWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--

BuyableDoeWeights = {};

function BuyableDoeWeights.prerequisitesPresent(specializations)
    return true;
end;

function BuyableDoeWeights:load(xmlFile)

	self.twinWheelTriggerCallback = BuyableDoeWeights.twinWheelTriggerCallback;
	self.wheelDelete = BuyableDoeWeights.wheelDelete;
	self.assembleDoeWheights = SpecializationUtil.callSpecializationsFunction("assembleDoeWheights");
	self.disassembleDoeWheights = SpecializationUtil.callSpecializationsFunction("disassembleDoeWheights");

	self.checkDoeString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.twinWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#activationTrigger"));
	
	
	addTrigger(self.usingTrigger, "twinWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#belts"));
	
    self.buyableDoeWeights = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.twinWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		setRigidBodyType(wheel.collision, "Kinematic");
		setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableDoeWeights, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.doeWeightsUser = nil;
end;

function BuyableDoeWeights:delete()
	if self.doeWeightsUser ~= nil then
		self.doeWeightsUser:onDisassembling(true);
		for _,twinWheel in pairs(self.buyableDoeWeights) do
			delete(twinWheel.node);
			delete(twinWheel.collision);
		end;
		self.buyableDoeWeights = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableDoeWeights:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableDoeWeights:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local doeWeightsUser = networkGetObject(id);
		if doeWeightsUser ~= nil then
			self:assembleDoeWheights(doeWeightsUser, true);
		end;
	end;
end;

function BuyableDoeWeights:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.doeWeightsUser ~= nil then
		idToWrite = networkGetObjectId(self.doeWeightsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableDoeWeights:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableDoeWeights:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableDoeWeights:update(dt)
end;

function BuyableDoeWeights:updateTick(dt)

	-- try to load saved twinwheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then				
				self:assembleDoeWheights(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;
end;

function BuyableDoeWeights:draw()	
end;

function BuyableDoeWeights:twinWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then		
			if vehicle.rootNode == otherId then
				if vehicle.buyableDoeWeights ~= nil then
					if vehicle.buyableDoeWeights.checkDoeString == self.checkDoeString then
						if self.doeWeightsUser ~= nil then
							if self.doeWeightsUser == vehicle then
								vehicle.buyableDoeWeights.doeWeightsInRange = self;
							end;
						else
							vehicle.buyableDoeWeights.doeWeightsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.buyableDoeWeights ~= nil then
					vehicle.buyableDoeWeights.doeWeightsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableDoeWeights:assembleDoeWheights(vehicle, noEventSend)
	BuyableDoeWeightsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;
	if self.doeWeightsUser == nil then
		if vehicle.buyableDoeWeights ~= nil then
			self.doeWeightsUser = vehicle;	
			for k,wheel in pairs(vehicle.wheels) do
				for l,twinWheel in pairs(self.buyableDoeWeights) do 
					if k == l then	
						unlink(twinWheel.node);
						unlink(twinWheel.collision);
						link(wheel.driveNode, twinWheel.node);	
						link(getParent(wheel.repr), twinWheel.collision);	
						local x,y,z = getWorldTranslation(wheel.driveNode);
						x,y,z = worldToLocal(getParent(twinWheel.collision), x,y,z);
						setTranslation(twinWheel.collision,x,y,z);
						--setRigidBodyType(twinWheel.collision, "Kinematic");
						break;
					end;
				end;
			end;	
			self.doeWeightsUser:onAssembling(self);
		end;
	end;
end;

function BuyableDoeWeights:disassembleDoeWheights(noEventSend)
	BuyableDoeWeightsDetachEvent.sendEvent(self, noEventSend);

	self.doeWeightsUser:onDisassembling();

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;
	
	for k,wheel in pairs(self.buyableDoeWeights) do
		setRigidBodyType(wheel.collision, "NoRigidBody");
		unlink(wheel.collision);
		unlink(wheel.node);
		setRotation(wheel.collision, 0,0,0);
		setTranslation(wheel.collision, 0,0,0);
		link(wheel.savePosition, wheel.collision);
		link(wheel.collision, wheel.node);
	end;	
	self.doeWeightsUser = nil;	
end;


function BuyableDoeWeights:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");		
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};			
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableDoeWeights:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	
	if self.doeWeightsUser ~= nil then
		local x,y,z = getWorldTranslation(self.doeWeightsUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;
	
    return attributes, nil;
end;


BuyableDoeWeightsAttachEvent = {};
BuyableDoeWeightsAttachEvent_mt = Class(BuyableDoeWeightsAttachEvent, Event);

InitEventClass(BuyableDoeWeightsAttachEvent, "BuyableDoeWeightsAttachEvent");

function BuyableDoeWeightsAttachEvent:emptyNew()
    local self = Event:new(BuyableDoeWeightsAttachEvent_mt);
    self.className="BuyableDoeWeightsAttachEvent";
    return self;
end;

function BuyableDoeWeightsAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableDoeWeightsAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableDoeWeightsAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableDoeWeightsAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableDoeWeightsAttachEvent:run(connection)
	self.vehicle:assembleDoeWheights(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableDoeWeightsAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableDoeWeightsAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableDoeWeightsAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableDoeWeightsAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableDoeWeightsDetachEvent = {};
BuyableDoeWeightsDetachEvent_mt = Class(BuyableDoeWeightsDetachEvent, Event);

InitEventClass(BuyableDoeWeightsDetachEvent, "BuyableDoeWeightsDetachEvent");

function BuyableDoeWeightsDetachEvent:emptyNew()
    local self = Event:new(BuyableDoeWeightsDetachEvent_mt);
    self.className="BuyableDoeWeightsDetachEvent";
    return self;
end;

function BuyableDoeWeightsDetachEvent:new(vehicle)
    local self = BuyableDoeWeightsDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableDoeWeightsDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableDoeWeightsDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableDoeWeightsDetachEvent:run(connection)
	self.vehicle:disassembleDoeWheights(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableDoeWeightsDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;


function BuyableDoeWeightsDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableDoeWeightsDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableDoeWeightsDetachEvent:new(vehicle));
		end;
	end;
end;