--
-- BuyableTwinWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--

BuyableCountyWeights = {};

function BuyableCountyWeights.prerequisitesPresent(specializations)
    return true;
end;

function BuyableCountyWeights:load(xmlFile)

	self.twinWheelTriggerCallback = BuyableCountyWeights.twinWheelTriggerCallback;
	self.wheelDelete = BuyableCountyWeights.wheelDelete;
	self.assembleCountyWheights = SpecializationUtil.callSpecializationsFunction("assembleCountyWheights");
	self.disassembleCountyWheights = SpecializationUtil.callSpecializationsFunction("disassembleCountyWheights");

	self.checkCountyString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.twinWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#activationTrigger"));
	
	
	addTrigger(self.usingTrigger, "twinWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#belts"));
	
    self.buyableCountyWeights = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.twinWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		setRigidBodyType(wheel.collision, "Kinematic");
		setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableCountyWeights, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.countyWeightsUser = nil;
end;

function BuyableCountyWeights:delete()
	if self.countyWeightsUser ~= nil then
		self.countyWeightsUser:onDisassembling(true);
		for _,twinWheel in pairs(self.buyableCountyWeights) do
			delete(twinWheel.node);
			delete(twinWheel.collision);
		end;
		self.buyableCountyWeights = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableCountyWeights:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableCountyWeights:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local countyWeightsUser = networkGetObject(id);
		if countyWeightsUser ~= nil then
			self:assembleCountyWheights(countyWeightsUser, true);
		end;
	end;
end;

function BuyableCountyWeights:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.countyWeightsUser ~= nil then
		idToWrite = networkGetObjectId(self.countyWeightsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableCountyWeights:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableCountyWeights:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableCountyWeights:update(dt)
end;

function BuyableCountyWeights:updateTick(dt)

	-- try to load saved twinwheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then				
				self:assembleCountyWheights(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;
end;

function BuyableCountyWeights:draw()	
end;

function BuyableCountyWeights:twinWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then		
			if vehicle.rootNode == otherId then
				if vehicle.buyableCountyWeights ~= nil then
					if vehicle.buyableCountyWeights.checkCountyString == self.checkCountyString then
						if self.countyWeightsUser ~= nil then
							if self.countyWeightsUser == vehicle then
								vehicle.buyableCountyWeights.countyWeightsInRange = self;
							end;
						else
							vehicle.buyableCountyWeights.countyWeightsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.buyableCountyWeights ~= nil then
					vehicle.buyableCountyWeights.countyWeightsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableCountyWeights:assembleCountyWheights(vehicle, noEventSend)
	BuyableCountyWeightsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;
	if self.countyWeightsUser == nil then
		if vehicle.buyableCountyWeights ~= nil then
			self.countyWeightsUser = vehicle;	
			for k,wheel in pairs(vehicle.wheels) do
				for l,twinWheel in pairs(self.buyableCountyWeights) do 
					if k == l then	
						unlink(twinWheel.node);
						unlink(twinWheel.collision);
						link(wheel.driveNode, twinWheel.node);	
						link(getParent(wheel.repr), twinWheel.collision);	
						local x,y,z = getWorldTranslation(wheel.driveNode);
						x,y,z = worldToLocal(getParent(twinWheel.collision), x,y,z);
						setTranslation(twinWheel.collision,x,y,z);
						--setRigidBodyType(twinWheel.collision, "Kinematic");
						break;
					end;
				end;
			end;	
			self.countyWeightsUser:onAssembling(self);
		end;
	end;
end;

function BuyableCountyWeights:disassembleCountyWheights(noEventSend)
	BuyableCountyWeightsDetachEvent.sendEvent(self, noEventSend);

	self.countyWeightsUser:onDisassembling();

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;
	
	for k,wheel in pairs(self.buyableCountyWeights) do
		setRigidBodyType(wheel.collision, "NoRigidBody");
		unlink(wheel.collision);
		unlink(wheel.node);
		setRotation(wheel.collision, 0,0,0);
		setTranslation(wheel.collision, 0,0,0);
		link(wheel.savePosition, wheel.collision);
		link(wheel.collision, wheel.node);
	end;	
	self.countyWeightsUser = nil;	
end;


function BuyableCountyWeights:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");		
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};			
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableCountyWeights:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	
	if self.countyWeightsUser ~= nil then
		local x,y,z = getWorldTranslation(self.countyWeightsUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;
	
    return attributes, nil;
end;


BuyableCountyWeightsAttachEvent = {};
BuyableCountyWeightsAttachEvent_mt = Class(BuyableCountyWeightsAttachEvent, Event);

InitEventClass(BuyableCountyWeightsAttachEvent, "BuyableCountyWeightsAttachEvent");

function BuyableCountyWeightsAttachEvent:emptyNew()
    local self = Event:new(BuyableCountyWeightsAttachEvent_mt);
    self.className="BuyableCountyWeightsAttachEvent";
    return self;
end;

function BuyableCountyWeightsAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableCountyWeightsAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableCountyWeightsAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableCountyWeightsAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableCountyWeightsAttachEvent:run(connection)
	self.vehicle:assembleCountyWheights(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableCountyWeightsAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableCountyWeightsAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableCountyWeightsAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableCountyWeightsAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableCountyWeightsDetachEvent = {};
BuyableCountyWeightsDetachEvent_mt = Class(BuyableCountyWeightsDetachEvent, Event);

InitEventClass(BuyableCountyWeightsDetachEvent, "BuyableCountyWeightsDetachEvent");

function BuyableCountyWeightsDetachEvent:emptyNew()
    local self = Event:new(BuyableCountyWeightsDetachEvent_mt);
    self.className="BuyableCountyWeightsDetachEvent";
    return self;
end;

function BuyableCountyWeightsDetachEvent:new(vehicle)
    local self = BuyableCountyWeightsDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableCountyWeightsDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableCountyWeightsDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableCountyWeightsDetachEvent:run(connection)
	self.vehicle:disassembleCountyWheights(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableCountyWeightsDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;


function BuyableCountyWeightsDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableCountyWeightsDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableCountyWeightsDetachEvent:new(vehicle));
		end;
	end;
end;