--
-- InteractiveCamera
-- Specialization for InteractiveCamera
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v1.0
-- @date  		06/09/11
-- @history:	v1.0 - Initial version
--
InteractiveCamera = {};

function InteractiveCamera.prerequisitesPresent(specializations)
    return true;
end;

function InteractiveCamera:load(xmlFile)
	
	self.interactiveCamera = {};
	self.interactiveCamera.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.interactiveCamera#node"));
	self.interactiveCamera.seat = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.interactiveCamera#seat"));
	self.interactiveCamera.trans = {getTranslation(self.interactiveCamera.node)};
	self.interactiveCamera.seatTrans = {getTranslation(self.interactiveCamera.seat)};
	self.interactiveCamera.wheel1 = self.wheels[getXMLInt(xmlFile, "vehicle.interactiveCamera#wheel1")].repr;
	self.interactiveCamera.wheel2 = self.wheels[getXMLInt(xmlFile, "vehicle.interactiveCamera#wheel2")].repr;
	self.interactiveCamera.suspTravel = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.interactiveCamera#suspTravel"), 0.3);
	self.interactiveCamera.camIndex = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.interactiveCamera#camIndex"), 2);
	self.interactiveCamera.damper = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.interactiveCamera#damper"), 0.4) * 0.001;
	
	self.interactiveCamera.firstRun = true;
	local _,y1,_ = getTranslation(self.interactiveCamera.wheel1);
	self.interactiveCamera.wheel1Y = y1;	
	local _,y2,_ = getTranslation(self.interactiveCamera.wheel2);
	self.interactiveCamera.wheel2Y = y2;
	self.interactiveCamera.average = (y1 + y2) / 2;
	self.interactiveCamera.lastAverage = self.interactiveCamera.average;
	self.interactiveCamera.lastSpeed = 0;
	self.interactiveCamera.transSpeed = 0;
	
	local soundFile = getXMLString(xmlFile, "vehicle.interactiveCamera#sound");
	if soundFile ~= nil then
		local sound = {};
		seatSoundFile = Utils.getFilename(soundFile, self.baseDirectory);
		sound.id = createSample("seatSound");
		loadSample(sound.seatSoundId, seatSoundFile, false);
		sound.volume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.interactiveCamera#volume"), 1.0);
		self.interactiveCamera.sound = sound;
	end;
	
	self.interactiveCamera.newCamTrans = 0;
	self.interactiveCamera.hasChanged = false;
end;

function InteractiveCamera:delete()
	if self.interactiveCamera.sound ~= nil then
		delete(self.interactiveCamera.sound.id);
        self.interactiveCamera.sound = nil;
	end;
end;

function InteractiveCamera:mouseEvent(posX, posY, isDown, isUp, button)
end;

function InteractiveCamera:keyEvent(unicode, sym, modifier, isDown)
end;

function InteractiveCamera:update(dt)
end;

function InteractiveCamera:updateTick(dt)

	if self.isClient then
        if self:getIsActiveForSound() then
			if self.camIndex == self.interactiveCamera.camIndex then
				local cam = self.interactiveCamera;
				local _,y1,_ = getTranslation(cam.wheel1);		
				local _,y2,_ = getTranslation(cam.wheel2);	
				
				local deltaY1 = self.interactiveCamera.wheel1Y - y1;
				local deltaY2 = self.interactiveCamera.wheel2Y - y2;
				local average = (deltaY1 + deltaY2) / 2;
				
				if cam.firstRun then
					cam.lastAverage = average;
					cam.firstRun = false;
				end;
				local speed =  math.min(math.max((average - cam.lastAverage) / 0.045, -1), 1);			

				local movingAcceleration = (speed - cam.lastSpeed) / dt;
				cam.lastSpeed = speed;
				cam.lastAverage = average;
				
				local transForce = math.cos(cam.newCamTrans) * movingAcceleration * 0.0006;
				local downForce = 0.000003;
				local sideForce = -math.sin(cam.newCamTrans) * downForce;
				local dampingForce = cam.damper * cam.transSpeed;
				
				cam.transSpeed = cam.transSpeed + (transForce + sideForce - dampingForce) * dt;
				cam.newCamTrans = cam.newCamTrans + cam.transSpeed * dt;
				cam.newCamTrans = math.min(math.max(cam.newCamTrans, -cam.suspTravel), cam.suspTravel);	
				
				setTranslation(cam.node, cam.trans[1], cam.trans[2] + cam.newCamTrans, cam.trans[3]);
				setTranslation(cam.seat, cam.seatTrans[1], cam.seatTrans[2] + cam.newCamTrans, cam.seatTrans[3]);
			
				if cam.sound ~= nil then
					if speed > 0.6 or speed < -0.6 then
						playSample(cam.sound.id, 1, cam.sound.volume, 0);
					end;
				end;
				
				cam.hasChanged = true;
			else
				if self.interactiveCamera.hasChanged then
					setTranslation(self.interactiveCamera.node, unpack(self.interactiveCamera.trans));
					setTranslation(self.interactiveCamera.seat, unpack(self.interactiveCamera.seatTrans));
					self.interactiveCamera.hasChanged = false;
				end;
			end;
		end;
	end;
end;

function InteractiveCamera:draw()
end;

function InteractiveCamera:onLeave()
	setTranslation(self.interactiveCamera.node, unpack(self.interactiveCamera.trans));
end;