--
-- LightFunctions.lua
-- Specialization for Lights, Breaklights, Turnlights.
--
-- @author Ellinor Strm
-- @date 28/11/2010
--
-- This work is licensed under a
-- Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
-- http://creativecommons.org/licenses/by-nc-sa/3.0/
-- 

LightFunctions = {};

function LightFunctions.prerequisitesPresent(specializations)
    return true;
end;

LightFunctions.TURNLIGHTS_NONE	= -1;
LightFunctions.TURNLIGHTS_LEFT	= 1;
LightFunctions.TURNLIGHTS_RIGHT	= 2;
LightFunctions.TURNLIGHTS_WARN	= 3;

function LightFunctions:load(xmlFile)
	self.turnTimerId = nil;
	self.turnSel = LightFunctions.TURNLIGHTS_NONE;
	self.turnMod = false;
	self.breaking = false;
	self.reversing = false;
	self.isActor = false;

	self.turnLightsTimerCallback = LightFunctions.turnLightsTimerCallback;

	self.Lights = {};
	self.Lights.revlights = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.reverseLights#index"));
	self.Lights.breaklights = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.breakLights#index"));
	self.Lights.turnR = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.blinkers#rightindex"));
	self.Lights.turnL = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.blinkers#leftindex"));
	self.Lights.turnR2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.blinkers#rightindex2"));
	self.Lights.turnL2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.blinkers#leftindex2"));
	
	LightFunctions.TURNLIGHTS_INTERVAL = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.blinkers#interval"), 500);

	local pitch = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.blinkers#pitch"), 0);
	self.Lights.volume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.blinkers#volume"), 1);

	local tSound = nil;
	tSound = getXMLString(xmlFile, "vehicle.blinkers#sound1");
	if tSound ~= nil then
		tSound = Utils.getFilename(tSound, self.baseDirectory);
		self.Lights.turnS1 = createSample("turn1");
    		loadSample(self.Lights.turnS1, tSound, false);
    		setSamplePitch(self.Lights.turnS1, pitch);
    	end;

	tSound = getXMLString(xmlFile, "vehicle.blinkers#sound2");
	if tSound ~= nil then
		tSound = Utils.getFilename(tSound, self.baseDirectory);
		self.Lights.turnS2 = createSample("turn2");
    		loadSample(self.Lights.turnS2, tSound, false);
    		setSamplePitch(self.Lights.turnS2, pitch);
    	end;
end;

function LightFunctions:mouseEvent(posX, posY, isDown, isUp, button)
end;

function LightFunctions:keyEvent(unicode, sym, modifier, isDown)
	if InputBinding.isPressed(InputBinding.TURNLEFT) then
		if self.turnSel ~= LightFunctions.TURNLIGHTS_LEFT then
			self.turnSel = LightFunctions.TURNLIGHTS_LEFT;
			LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.turnsignal, self.turnSel);
		else
			self.turnSel = LightFunctions.TURNLIGHTS_NONE;
			LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.turnsignal, self.turnSel);
		end;
	elseif InputBinding.isPressed(InputBinding.TURNRIGHT) then
		if self.turnSel ~= LightFunctions.TURNLIGHTS_RIGHT then
			self.turnSel = LightFunctions.TURNLIGHTS_RIGHT;
			LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.turnsignal, self.turnSel);
		else
			self.turnSel = LightFunctions.TURNLIGHTS_NONE;
			LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.turnsignal, self.turnSel);
		end;
	elseif InputBinding.isPressed(InputBinding.WARNINGLIGHT) then
		if self.turnSel ~= LightFunctions.TURNLIGHTS_WARN then
			self.turnSel = LightFunctions.TURNLIGHTS_WARN;
			LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.turnsignal, self.turnSel);
		else
			self.turnSel = LightFunctions.TURNLIGHTS_NONE;
			LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.turnsignal, self.turnSel);
		end;
	end;
end;

function LightFunctions:update(dt)
	if self:getIsActive() then
		-- break lights
		if self.Lights.breaklights ~= nil then
   			acceleration = -InputBinding.getAnalogInputAxis(InputBinding.AXIS_MOVE_FORWARD_VEHICLE);
   			if InputBinding.isAxisZero(acceleration) then
        			acceleration = -InputBinding.getDigitalInputAxis(InputBinding.AXIS_MOVE_FORWARD_VEHICLE);
   			end;
	
			if self.isActor then
				if (self.movingDirection*self.lastSpeed*(acceleration)) < -0.001 then
					self.breaking = true;
					LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.breaking, self.breaking);

				else
					self.breaking = false;
					LightFunctionsEvent.sendEvent(self, LightFunctionsEventType.breaking, self.breaking);
				end;
			end;
			
			if self.breaking then
				setVisibility(self.Lights.breaklights, true);
			else
				setVisibility(self.Lights.breaklights, false);
			end;
		end;

		-- reverse lights
		if self.Lights.revlights ~= nil then
			if self.movingDirection == -1 then
				self.reversing = true;
			elseif self.movingDirection ~= -1  then 
				self.reversing = false;
			end;
			
			if self.reversing then
				setVisibility(self.Lights.revlights, true);
			else
				setVisibility(self.Lights.revlights, false);
			end;
		end;

		-- turn lights
		if self.isActor then
			if self.TurnTimerId ~= nil and self.turnSel ~= LightFunctions.TURNLIGHTS_WARN then
   				axis = -InputBinding.getAnalogInputAxis(InputBinding.AXIS_MOVE_SIDE_VEHICLE);
   				if InputBinding.isAxisZero(axis) then
        				axis = -InputBinding.getDigitalInputAxis(InputBinding.AXIS_MOVE_SIDE_VEHICLE);
				end;

				if axis > -0.001 and axis < 0.001 then
					self.turnSel = LightFunctions.TURNLIGHTS_NONE;
				end;
			end;
		end;

		if self.turnSel ~= LightFunctions.TURNLIGHTS_NONE and self.turnTimerId == nil then
			self.turnTimerId = addTimer(LightFunctions.TURNLIGHTS_INTERVAL, "turnLightsTimerCallback", self);
		end;
	end;
end;

function LightFunctions:readStream(streamId, connection)
	self.breaking = streamReadBool(streamId);
	self.reversing = streamReadBool(streamId);
	self.turnSel = streamReadInt8(streamId);
end;

function LightFunctions:writeStream(streamId, connection)
	streamWriteBool(streamId, self.breaking);
	streamWriteBool(streamId, self.reversing);
	streamWriteInt8(streamId, self.turnSel);
end

function LightFunctions:draw()
end;

function LightFunctions:onEnter()
	-- store if the player is inside the vehicle - this is an important information in multiplayer games,
	-- because onEnter and onLeave are called on ALL clients, not just the one actually using the vehicle.
	self.isActor = self.isEntered;
end;

function LightFunctions:onLeave()
	self.isActor = false;
	self.breaking = false;
	self.reversing = false;
	self.turnMod = false;
	self.turnSel = LightFunctions.TURNLIGHTS_NONE;
	self.turnTimerId = nil;

	if self.Lights.revlights ~= nil then
		setVisibility(self.Lights.revlights, false);
	end;

	if self.Lights.breaklights ~= nil then
		setVisibility(self.Lights.breaklights, false);
	end;

	if self.Lights.turnR ~= nil and self.Lights.turnL ~= nil and self.Lights.turnR2 ~= nil and self.Lights.turnL2 ~= nil then
		setVisibility(self.Lights.turnL, false);
		setVisibility(self.Lights.turnR, false);
		setVisibility(self.Lights.turnL2, false);
		setVisibility(self.Lights.turnR2, false);
	end;

	if self.Lights.turnS1 ~= nil then
		stopSample(self.Lights.turnS1);
	end;

	if self.Lights.turnS2 ~= nil then
		stopSample(self.Lights.turnS2);
	end;
end;

function LightFunctions:delete()
	if self.Lights.turnS1 ~= nil then
        	delete(self.Lights.turnS1);
        	self.Lights.turnS1 = nil;
   	end;

	if self.Lights.turnS2 ~= nil then
        	delete(self.Lights.turnS2);
        	self.Lights.turnS2 = nil;
   	end;
end;

function LightFunctions:turnLightsTimerCallback()
	self.turnMod = not self.turnMod;

	if self.Lights.turnR ~= nil and self.Lights.turnL ~= nil and self.Lights.turnR2 ~= nil and self.Lights.turnL2 ~= nil then
		if self.turnSel == LightFunctions.TURNLIGHTS_LEFT then
			setVisibility(self.Lights.turnL, self.turnMod);
			setVisibility(self.Lights.turnL2, self.turnMod);			
			setVisibility(self.Lights.turnR, false);
			setVisibility(self.Lights.turnR2, false);
		elseif self.turnSel == LightFunctions.TURNLIGHTS_RIGHT then
			setVisibility(self.Lights.turnR, self.turnMod);
			setVisibility(self.Lights.turnR2, self.turnMod);
			setVisibility(self.Lights.turnL, false);
			setVisibility(self.Lights.turnL2, false);
		elseif self.turnSel == LightFunctions.TURNLIGHTS_WARN then
			setVisibility(self.Lights.turnL, self.turnMod);
			setVisibility(self.Lights.turnL2, self.turnMod);
			setVisibility(self.Lights.turnR, self.turnMod);
			setVisibility(self.Lights.turnR2, self.turnMod);		
		elseif self.turnSel == LightFunctions.TURNLIGHTS_NONE then
			self.turnMod = false;
			self.turnTimerId = nil;
			setVisibility(self.Lights.turnL, self.turnMod);
			setVisibility(self.Lights.turnL2, self.turnMod);
			setVisibility(self.Lights.turnR, self.turnMod);
			setVisibility(self.Lights.turnR2, self.turnMod);
			return false;
		end;
	else
		return false;
	end;

	if self.turnSel ~= LightFunctions.TURNLIGHTS_NONE then
		if self.turnMod then
			if self.Lights.turnS1 ~= nil then
				playSample(self.Lights.turnS1, 1, self.Lights.volume, 0);
			end;
		else
			if self.Lights.turnS2 ~= nil then
				playSample(self.Lights.turnS2, 1, self.Lights.volume, 0);
			end;
		end;
	end;

	return true;
end;
