--
-- manureFiller
-- Class for fertilizer filler
--
-- @author  PeterJ_LS-UK modteam
-- @date  20/07/2011
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
  

manureFiller = {};

function manureFiller.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Fillable, specializations);
end;
 
function manureFiller:load(xmlFile)
  
	--Filling
	assert(self.setIsSprayerFilling == nil, "manureFiller needs to be the first specialization which implements setIsSprayerFilling");
	self.setIsSprayerFilling = Sprayer.setIsSprayerFilling;
	self.addSprayerFillTrigger = Sprayer.addSprayerFillTrigger;
	self.removeSprayerFillTrigger = Sprayer.removeSprayerFillTrigger;
  
	self.fillLitersPerSecond = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fillLitersPerSecond"), 500);
	self.isSprayerFilling = false;
  
	self.sprayerFillTriggers = {};
	self.sprayerFillActivatable = SprayerFillActivatable:new(self);
 
	self.fillTypes[Fillable.FILLTYPE_LIQUIDMANURE] = true;
	
	self.overlay = Utils.getFilename("textures/manure_hud.png", self.baseDirectory);
    self.FruitTypeOverlay = Overlay:new("hudPDAControl", self.overlay, g_currentMission.fruitSymbolX, g_currentMission.fruitSymbolY, g_currentMission.fruitSymbolSize, g_currentMission.fruitSymbolSize * (4 / 3)); 

	--Unloading
	assert(self.setIsTurnedOn == nil, "manureFiller needs to be the first specialization which implements setIsTurnedOn");
	self.setIsTurnedOn = manureFiller.setIsTurnedOn;
	self.getIsTurnedOnAllowed = manureFiller.getIsTurnedOnAllowed;
  
	self.unloadLitersPerSecond = {};
	local i=0;
	while true do
		local key = string.format("vehicle.unloadPipe(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;
		local fillType = getXMLString(xmlFile, key.. "#unloadType");
		local litersPerSecond = getXMLFloat(xmlFile, key.. "#litersPerSecond");
		if fillType ~= nil and litersPerSecond ~= nil then
			local fillTypeInt = Fillable.fillTypeNameToInt[fillType];
			if fillTypeInt ~= nil then
				self.unloadLitersPerSecond[fillTypeInt] = litersPerSecond;
				if self.defaultUnloadLitersPerSecond == nil then
					self.defaultUnloadLitersPerSecond = litersPerSecond;
				end;
			else
				print("Warning: Invalid unload type '"..fillType.."' in '" .. self.configFileName.. "'");
			end;
		end;
		i = i+1;
	end;
	if self.defaultUnloadLitersPerSecond == nil then
		print("Warning: No unload type specified for '" .. self.configFileName.. "'. This bag will not use any unload type.");
		self.defaultUnloadLitersPerSecond = 0;
	end;
  
	self.unloadValves = {};
  
	if self.isClient then
		local psFile = getXMLString(xmlFile, "vehicle.unloadParticleSystem#file");
		if psFile ~= nil then
			local i=0;
			while true do
				local baseName = string.format("vehicle.unloadParticleSystem(%d)", i);
				local node = getXMLString(xmlFile, baseName.. "#index");
				if node == nil then
					break;
				end;
				node = Utils.indexToObject(self.components, node);
				if node ~= nil then
					local unloadValve = {};
					unloadValve.particleSystems = {};
					Utils.loadParticleSystem(xmlFile, unloadValve.particleSystems, "vehicle.unloadParticleSystem", node, false, nil, self.baseDirectory);
					table.insert(self.unloadValves, unloadValve);
				end;
				i = i+1;
			end;
		end;
		local unloadSound = getXMLString(xmlFile, "vehicle.unloadSound#file");
		if unloadSound ~= nil and unloadSound ~= "" then
			unloadSound = Utils.getFilename(unloadSound, self.baseDirectory);
			self.unloadSound = createSample("unloadSound");
			self.unloadSoundEnabled = false;
			loadSample(self.unloadSound, unloadSound, false);
			self.unloadSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.unloadSound#pitchOffset"), 1);
			self.unloadSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.unloadSound#volume"), 1);
		end;
	end;
	
	self.minThreshold = 0.05;
	self.isTurnedOn = false;

	--Trigger
	self.FillableTrigger = manureFiller.FillableTrigger;
	self.FillableTriggers = {};
  
	local FillableTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fillableTrigger#index"));
	if FillableTrigger ~= nil then
		self.FillableTriggers[FillableTrigger] = {node=FillableTrigger, pipeState=2};
	end;
	for _, FillableTrigger in pairs(self.FillableTriggers) do
		addTrigger(FillableTrigger.node, "FillableTrigger", self);
	end;
  
	self.isSpreaderInRange = nil;
	
	--Show fill level of machine in range
	self.hudFillLevelBasePosX = 0.01;
	self.hudFillLevelBasePosY = 0.0149;
	self.hudFillLevelBaseWidth = 0.2;
	self.hudFillLevelBaseHeight = 0.043;
	self.hudFillLevelBarWidth = 0.168;
	self.hudFillLevelOverlay = Overlay:new("hudFillLevelOverlay", "dataS2/menu/hud/vehicleHUD_barGold.png", self.hudFillLevelBasePosX + 0.0228, self.hudFillLevelBasePosY + 0.003, self.hudFillLevelBaseWidth - 0.03, self.hudFillLevelBaseHeight - 0.012);
	self.FillLevelBasePath = Utils.getFilename("textures/FillLevelBase.png", self.baseDirectory);
	self.hudFillLevelBase = Overlay:new("hudFillLevelBase", self.FillLevelBasePath, self.hudFillLevelBasePosX, self.hudFillLevelBasePosY, self.hudFillLevelBaseWidth, self.hudFillLevelBaseHeight);
	self.FillLevelFramePath = Utils.getFilename("textures/FillLevelFrame.png", self.baseDirectory);
	self.hudFillLevelFrame = Overlay:new("hudFillLevelFrame", self.FillLevelFramePath, self.hudFillLevelBasePosX, self.hudFillLevelBasePosY, self.hudFillLevelBaseWidth, self.hudFillLevelBaseHeight);

 	self.setVehicleRpmUp = SpecializationUtil.callSpecializationsFunction("setVehicleRpmUp");
	self.saveMinRpm = 0; 
    self.printWarningTime = 0;
    self.printWarningTime1 = 0;
	self.beaconLight = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.beaconLights.beaconLight#index1"));
	
    if self.isClient then
        local fillingSound = getXMLString(xmlFile, "vehicle.fillingSound#file");
        if fillingSound ~= nil and fillingSound ~= "" then
            fillingSound = Utils.getFilename(fillingSound, self.baseDirectory);
            self.fillingSound = createSample("fillingSound");
            self.fillingSoundEnabled = false;
            loadSample(self.fillingSound, fillingSound, false);
            self.fillingSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fillingSound#pitchOffset"), 1);
            self.fillingSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fillingSound#volume"), 1);
        end;
    end; 
	
end;
  
function manureFiller:delete()
  
	g_currentMission:removeActivatableObject(self.sprayerFillActivatable);
  
	for k,unloadValve in pairs(self.unloadValves) do
		Utils.deleteParticleSystem(unloadValve.particleSystems);
	end;
  
	if self.unloadSound ~= nil then
		delete(self.unloadSound);
	end;
    if self.fillingSound ~= nil then
        delete(self.fillingSound);
    end;
	for _, FillableTrigger in pairs(self.FillableTriggers) do
		removeTrigger(FillableTrigger.node);
	end;
end;
  
function manureFiller:readStream(streamId, connection)
	local turnedOn = streamReadBool(streamId);
	local isSprayerFilling = streamReadBool(streamId);
	self:setIsTurnedOn(turnedOn, true);
	self:setIsSprayerFilling(isSprayerFilling, true);
end;
  
function manureFiller:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isTurnedOn);
	streamWriteBool(streamId, self.isSprayerFilling);
end;
  
function manureFiller:readUpdateStream(streamId, timestamp, connection)
end;
  
function manureFiller:writeUpdateStream(streamId, connection, dirtyMask)
end;
  
function manureFiller:mouseEvent(posX, posY, isDown, isUp, button)
end;
  
function manureFiller:keyEvent(unicode, sym, modifier, isDown)
end;
 
function manureFiller:update(dt)
  
	if self.isClient then
		if self:getIsActiveForInput() then
			if InputBinding.isPressed(InputBinding.ACTIVATE_OBJECT) and self.PTOId then
				self.printWarningTime1 = self.time + 1000;
			end;
			if InputBinding.hasEvent(InputBinding.TOGGLE_BEACON_LIGHTS) then
				setVisibility(self.beaconLight, true);
			end;
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
				if self.PTOId then
					self.printWarningTime = self.time + 1000;
				else
					self:setIsTurnedOn(not self.isTurnedOn);
				end;
			end;
		end;
	end;
end;
  
function manureFiller:updateTick(dt)

	if self:getIsActive() then
		if not self.attacherVehicle.isMotorStarted then
			self:setIsTurnedOn(false);
            self:setIsSprayerFilling(false);
		end;
		if self.PTOId then
			self:setIsTurnedOn(false);
            self:setIsSprayerFilling(false);
		end;
		local isKeyEvent = false;	
		if self.isTurnedOn or self.isSprayerFilling then
 			isKeyEvent = true;	
		else
 			isKeyEvent = false;	
		end;
		self:setVehicleRpmUp(dt, isKeyEvent);
		if self.isTurnedOn then
			if self.isServer then
				local litersPerSecond = self.unloadLitersPerSecond[self.currentFillType];
				if litersPerSecond == nil then
					litersPerSecond = self.defaultUnloadLitersPerSecond;
				end;
				local usage = litersPerSecond * dt*0.001;
				if self.fillLevel > 0 then
					self:setFillLevel(self.fillLevel - usage, self.currentFillType);
				end;  
			end;
			if self.isClient then
				if not self.unloadSoundEnabled and self:getIsActiveForSound() then
					playSample(self.unloadSound, 0, self.unloadSoundVolume, 0);
					setSamplePitch(self.unloadSound, self.unloadSoundPitchOffset);
					self.unloadSoundEnabled = true;
				end;
			end;
			if self.fillLevel <= 0 and self.capacity ~= 0 then
				self:setIsTurnedOn(false, true);
			end;
		end;
	end;

	if self.isSprayerFilling and self.isServer then

		local disableFilling = false;
		if self:allowFillType(self.sprayerFillingFillType, false) then
			local oldFillLevel = self.fillLevel;
  
			local delta = self.fillLitersPerSecond*dt*0.001;
  
			local silo = g_currentMission:getSiloAmount(self.sprayerFillingFillType);
			if self.sprayerFillingIsSiloTrigger then
				if silo <= 0 then
					disableFilling = true;
				end;
				delta = math.min(delta, silo);
			end;
  
			self:setFillLevel(self.fillLevel + delta, self.sprayerFillingFillType, true);
			local delta = self.fillLevel - oldFillLevel;
			if delta > 0 then
				if self.sprayerFillingIsSiloTrigger then
					g_currentMission:setSiloAmount(self.sprayerFillingFillType, silo - delta);
				else
					local sprayType = Sprayer.fillTypeToSprayType[self.sprayerFillingFillType];
					if sprayType ~= nil then
						local sprayTypeDesc = Sprayer.sprayTypeIndexToDesc[sprayType]
  
						local price = delta*sprayTypeDesc.pricePerLiter;
						g_currentMission.missionStats.expensesTotal = g_currentMission.missionStats.expensesTotal + price;
						g_currentMission.missionStats.expensesSession = g_currentMission.missionStats.expensesSession + price;
  
						g_currentMission:addSharedMoney(-price);
					end;
				end;
			elseif self.fillLevel == self.capacity then
				disableFilling = true;
			end;
		else
			disableFilling = true;
		end;
		if disableFilling then
			self:setIsSprayerFilling(false);
		end;
	end;

	if self.isServer and self:getIsActive() then
		if self.isTurnedOn and self.isSpreaderInRange ~= nil then
			local fillable = self.isSpreaderInRange;
			if fillable.setFillLevel ~= nil then
				for fillType,v in pairs(fillable.fillTypes) do
					if fillType == self.currentFillType then
						if  fillable.fillLevel ~= fillable.capacity  then
							local litersPerSecond = self.unloadLitersPerSecond[self.currentFillType];
							if litersPerSecond == nil then
								litersPerSecond = self.defaultUnloadLitersPerSecond;
							end;
							local usage = litersPerSecond * dt*0.001;
							fillable:resetFillLevelIfNeeded(Fillable.FILLTYPE_LIQUIDMANURE);
							fillable:setFillLevel(fillable.fillLevel + usage, Fillable.FILLTYPE_LIQUIDMANURE);
						end; 	
					end;
				end;
			end;
		end;
	end;
    if self.isClient then		
		if self.isSprayerFilling then
			if not self.fillingSoundEnabled and self:getIsActiveForSound() then
                playSample(self.fillingSound, 0, self.fillingSoundVolume, 0);
                setSamplePitch(self.fillingSound, self.fillingSoundPitchOffset);
                self.fillingSoundEnabled = true;
            end;
        end;
		if not self.isSprayerFilling then
			stopSample(self.fillingSound);
			self.fillingSoundEnabled = false;
		end;			
	end;
end;
  
function manureFiller:draw()

	if self.isClient then
		if self.fillLevel <= 0 and self.capacity ~= 0 then
				g_currentMission:addExtraPrintText(g_i18n:getText("FirstFillTheTool"));
		end;
		if self.fillLevel > 0 then
			self.FruitTypeOverlay:render();
			if self.isTurnedOn then
				g_currentMission:addHelpButtonText(g_i18n:getText("StopUnlod"), InputBinding.IMPLEMENT_EXTRA);
			else
				g_currentMission:addHelpButtonText(g_i18n:getText("StartUnload"), InputBinding.IMPLEMENT_EXTRA);
			end;
		end;
		if self.printWarningTime > self.time then
			g_currentMission:addWarning(g_i18n:getText("turnON_Error"), 0.018, 0.033);
		end;
		if self.printWarningTime1 > self.time then
			g_currentMission:addWarning(g_i18n:getText("fill_Error"), 0.018, 0.033);
		end;
	end;
	if self:getIsActive() then
		if self.isSpreaderInRange ~= nil then
			local fillable = self.isSpreaderInRange;
			if self.isSpreaderInRange.fillLevel ~= nil then
				for fillType,v in pairs(fillable.fillTypes) do
					if fillType == self.currentFillType and fillable.rootNode ~= self.rootNode then
						self.hudFillLevelBase:render();
						local percent = (self.isSpreaderInRange.fillLevel / self.isSpreaderInRange.capacity) * 100;
						setTextBold(true);
						setTextAlignment(RenderText.ALIGN_CENTER);
						setTextColor(0, 0, 0, 1);
						renderText(self.hudFillLevelBasePosX + self.hudFillLevelBaseWidth / 2 + 0.003, self.hudFillLevelBasePosY + 0.0083, 0.024, string.format("%d (%d%%)", self.isSpreaderInRange.fillLevel, percent));
						setTextColor(1, 1, 1, 1);
						renderText(self.hudFillLevelBasePosX + self.hudFillLevelBaseWidth / 2 + 0.003, self.hudFillLevelBasePosY + 0.0113, 0.024, string.format("%d (%d%%)", self.isSpreaderInRange.fillLevel, percent));
						setTextBold(false);
						self.hudFillLevelOverlay.width = self.hudFillLevelBarWidth * (self.isSpreaderInRange.fillLevel / self.isSpreaderInRange.capacity);
						setOverlayUVs(self.hudFillLevelOverlay.overlayId, 0, 0.05, 0, 1, self.isSpreaderInRange.fillLevel / self.isSpreaderInRange.capacity, 0.05, self.isSpreaderInRange.fillLevel / self.isSpreaderInRange.capacity, 1);
						self.hudFillLevelOverlay:render();
						self.hudFillLevelFrame:render();
					end;
				end;
			end;
		end;
	end;
end;

function manureFiller:onAttach(attacherVehicle)
	self.attacherVehicle = attacherVehicle;
	if self.attacherVehicleCopy == nil then
		self.attacherVehicleCopy = self.attacherVehicle;
	end;
	self.saveMinRpm = self.attacherVehicle.motor.minRpm;
end;

function manureFiller:onDetach()
	if self.deactivateOnDetach then
		manureFiller.onDeactivate(self);
	else
		manureFiller.onDeactivateSounds(self);
	end;
	for k, steerable in pairs(g_currentMission.steerables) do
		if self.attacherVehicleCopy == steerable then
			steerable.motor.minRpm = self.saveMinRpm;
			self.attacherVehicleCopy = nil;
		end;
	end;
	setVisibility(self.beaconLight, false);
end;
  
function manureFiller:onLeave()
	if self.deactivateOnLeave then
		manureFiller.onDeactivate(self);
	else
		manureFiller.onDeactivateSounds(self);
	end;
	setVisibility(self.beaconLight, false);
	self.beaconLightsActive = false;
end;
  
function manureFiller:onDeactivate()
	self:setIsTurnedOn(false, true)
	manureFiller.onDeactivateSounds(self);
end;
  
function manureFiller:onDeactivateSounds()
	if self.unloadSoundEnabled then
		stopSample(self.unloadSound);
		self.unloadSoundEnabled = false;
	end;
      if self.fillingSoundEnabled then
          stopSample(self.fillingSound);
          self.fillingSoundEnabled = false;
      end;
end;
  
function manureFiller:getIsTurnedOnAllowed(isTurnedOn)
	if not isTurnedOn or self.fillLevel > 0 or self.capacity == 0 then
		return true;
	end;
end;
  
function manureFiller:setIsTurnedOn(isTurnedOn, noEventSend)
	if isTurnedOn ~= self.isTurnedOn then
		if self:getIsTurnedOnAllowed(isTurnedOn) then
			SetTurnedOnEvent.sendEvent(self, isTurnedOn, noEventSend)
  
			self.isTurnedOn = isTurnedOn;
			if self.isClient then
				for k,unloadValve in pairs(self.unloadValves) do
					Utils.setEmittingState(unloadValve.particleSystems, self.isTurnedOn);
				end;
				if not self.isTurnedOn and self.unloadSoundEnabled then
					stopSample(self.unloadSound);
					self.unloadSoundEnabled = false;
				end;
			end;
		end;
	end;
end;

function manureFiller:setIsSprayerFilling(isFilling, fillType, isSiloTrigger, noEventSend)
	SprayerSetIsFillingEvent.sendEvent(self, isFilling, fillType, isSiloTrigger, noEventSend)
	if self.isSprayerFilling ~= isFilling then
		self.isSprayerFilling = isFilling;
		self.sprayerFillingFillType = fillType;
		self.sprayerFillingIsSiloTrigger = isSiloTrigger;
	end;
end;
  
function manureFiller:addSprayerFillTrigger(trigger)
	if table.getn(self.sprayerFillTriggers) == 0 then
		g_currentMission:addActivatableObject(self.sprayerFillActivatable);
	end;
	table.insert(self.sprayerFillTriggers, trigger);
end;
  
function manureFiller:removeSprayerFillTrigger(trigger)
	for i=1, table.getn(self.sprayerFillTriggers) do
		if self.sprayerFillTriggers[i] == trigger then
			table.remove(self.sprayerFillTriggers, i);
			break;
		end;
	end;
	if table.getn(self.sprayerFillTriggers) == 0 then
		if self.isServer then
			self:setIsSprayerFilling(false);
		end;
		g_currentMission:removeActivatableObject(self.sprayerFillActivatable);
	end;
end;
 
function manureFiller:FillableTrigger(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	if onEnter or onLeave then
		local spreader = g_currentMission.nodeToVehicle[otherShapeId];
		if spreader ~= nil then
			if spreader ~= self and spreader.setFillLevel	then
				if onEnter then
					self.isSpreaderInRange = (spreader);
				else
					self.isSpreaderInRange = nil;
				end;
			end;
		end;
	end;
end;


function manureFiller:setVehicleRpmUp(dt, isActive)
	if self.attacherVehicle ~= nil and self.saveMinRpm ~= 0 then
		if dt ~= nil then
			if isActive == true then
				self.attacherVehicle.motor.minRpm = math.max(self.attacherVehicle.motor.minRpm-dt, -1000);
			else
				self.attacherVehicle.motor.minRpm = math.min(self.attacherVehicle.motor.minRpm+dt*2, self.saveMinRpm);
			end;
		else
			self.attacherVehicle.motor.minRpm = self.saveMinRpm;
		end;
		if self.attacherVehicle.isMotorStarted then
			local fuelUsed = 0.0000011*math.abs(self.attacherVehicle.motor.minRpm);
			self.attacherVehicle:setFuelFillLevel(self.attacherVehicle.fuelFillLevel-fuelUsed);
			g_currentMission.missionStats.fuelUsageTotal = g_currentMission.missionStats.fuelUsageTotal + fuelUsed;
			g_currentMission.missionStats.fuelUsageSession = g_currentMission.missionStats.fuelUsageSession + fuelUsed;
		end;
	end;
end;

 
SprayerFillActivatable = {}
local SprayerFillActivatable_mt = Class(SprayerFillActivatable);
  
function SprayerFillActivatable:new(sprayer)
	local self = {};
	setmetatable(self, SprayerFillActivatable_mt);
  
	self.sprayer = sprayer;
	self.activateText = "unknown";
  
	self.currentTrigger = nil;
  
	return self;
end;

function SprayerFillActivatable:getIsActivatable()
	self.currentTrigger = nil;
	if not self.sprayer:getIsActiveForInput() or self.sprayer.fillLevel == self.sprayer.capacity then
		return false;
	end;
	-- find the first trigger which is activable
   for i=1, table.getn(self.sprayer.sprayerFillTriggers) do
		local trigger = self.sprayer.sprayerFillTriggers[i];
		if trigger:getIsActivatable(self.sprayer) then
			self.currentTrigger = trigger;
			self:updateActivateText();
			return true;
		end;
	end;
	return false;
end;

function SprayerFillActivatable:onActivateObject()
	self.sprayer:setIsSprayerFilling(not self.sprayer.isSprayerFilling, self.currentTrigger.fillType, self.currentTrigger.isSiloTrigger);
	self:updateActivateText();
	g_currentMission:addActivatableObject(self);
end;
  
function SprayerFillActivatable:drawActivate()
	-- TODO draw icon
end;
  
function SprayerFillActivatable:updateActivateText()
	if self.sprayer.isSprayerFilling then
		self.activateText = string.format(g_i18n:getText("stop_refill_OBJECT"), self.sprayer.typeDesc);
	else
		self.activateText = string.format(g_i18n:getText("refill_OBJECT"), self.sprayer.typeDesc);
	end;
end;