--
-- BuyableFrontDualWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--
-- @edit		JoXXer
-- @date		08/11/11

BuyableFrontDualWheels = {};

function BuyableFrontDualWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableFrontDualWheels:load(xmlFile)

	self.frontDualsWheelTriggerCallback = BuyableFrontDualWheels.frontDualsWheelTriggerCallback;
	self.wheelDelete = BuyableFrontDualWheels.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.frontDualWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.frontDualWheels#activationTrigger"));


	addTrigger(self.usingTrigger, "frontDualsWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);

    self.buyableFrontDualWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.frontDualWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;

		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.transFix = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#transFix"));
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		wheel.connectors = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#connectors"));
		table.insert(self.buyableFrontDualWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.frontDualWheelsUser = nil;

	self.savedVehicleID = -1;
	self.shouldUpdateLater = false;
	
	local i = 0;
	self.dirtComponents = {};
	while true do
		local path = string.format("vehicle.dirt.dirtComponent(%d)", i);
		local component = Utils.indexToObject(self.components, getXMLString(xmlFile, path .. "#index"));	
		if component == nil then
			break;
		end;
		table.insert(self.dirtComponents, component);
		i = i + 1;
	end;	
end;

function BuyableFrontDualWheels:delete()
	if self.frontDualWheelsUser ~= nil then
		self.frontDualWheelsUser:onDisassembling(self);
		for _,frontDualWheel in pairs(self.buyableFrontDualWheels) do
			delete(frontDualWheel.node);
			delete(frontDualWheel.transFix);
		end;
		self.buyableFrontDualWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableFrontDualWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableFrontDualWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local frontDualWheelsUser = networkGetObject(id);
		if frontDualWheelsUser ~= nil then
			self:assembleWheels(frontDualWheelsUser, true);
		else
			self.savedVehicleID = id;
			self.shouldUpdateLater = true;
		end;
	end;
end;

function BuyableFrontDualWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.frontDualWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.frontDualWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableFrontDualWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableFrontDualWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableFrontDualWheels:update(dt)
end;

function BuyableFrontDualWheels:updateTick(dt)

	-- try to load saved frontDualWheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;

	-- Attach wheels on client if not attached on load
	if self.shouldUpdateLater then
		local frontDualWheelsUser = networkGetObject(self.savedVehicleID);
		if frontDualWheelsUser ~= nil then
			self:assembleWheels(frontDualWheelsUser, true);
			self.shouldUpdateLater = false;
			self.savedVehicleID = -1;
		end;
	end;
end;

function BuyableFrontDualWheels:draw()
end;

function BuyableFrontDualWheels:frontDualsWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then
			if vehicle.rootNode == otherId then
				if vehicle.buyableWheels ~= nil then
					if vehicle.buyableWheels.checkString == self.checkString then
						if self.frontDualWheelsUser ~= nil then
							if self.frontDualWheelsUser == vehicle then
								vehicle.buyableWheels.frontDualwheelsInRange = self;
							end;
						else
							vehicle.buyableWheels.frontDualwheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then
			if otherId == vehicle.rootNode then
				if vehicle.buyableWheels ~= nil then
					vehicle.buyableWheels.frontDualwheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableFrontDualWheels:assembleWheels(vehicle, noEventSend)
	BuyableWheelsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.frontDualWheelsUser == nil then
		if vehicle.buyableWheels ~= nil then
			self.frontDualWheelsUser = vehicle;
			--for k,wheel in pairs(vehicle.wheels) do
				for l,frontDualWheel in pairs(self.buyableFrontDualWheels) do
				
					unlink(frontDualWheel.node);
					unlink(frontDualWheel.transFix);
						
					if l == 1 then
						link(self.frontDualWheelsUser.animatedWheel[1].wheel, frontDualWheel.node);
						--link(getParent(wheel.repr), frontDualWheel.transFix);
						local x,y,z = getWorldTranslation(self.frontDualWheelsUser.animatedWheel[1].wheel);
						x,y,z = worldToLocal(getParent(frontDualWheel.transFix), x,y,z);
						--setTranslation(frontDualWheel.transFix,x,y,z);
						--setRigidBodyType(frontDualWheel.transFix, "Kinematic");
					elseif l == 2 then
						link(self.frontDualWheelsUser.animatedWheel[2].wheel, frontDualWheel.node);
						--link(getParent(wheel.repr), frontDualWheel.transFix);
						local x,y,z = getWorldTranslation(self.frontDualWheelsUser.animatedWheel[2].wheel);
						x,y,z = worldToLocal(getParent(frontDualWheel.transFix), x,y,z);
						--setTranslation(frontDualWheel.transFix,x,y,z);
						--setRigidBodyType(frontDualWheel.transFix, "Kinematic");
					end;
					
					setVisibility(frontDualWheel.connectors, true);
				end;
			--end;
			self.frontDualWheelsUser:onAssembling(self);
		end;
	end;
end;

function BuyableFrontDualWheels:disassembleWheels(vehicle, noEventSend)
	BuyableWheelsDetachEvent.sendEvent(self, vehicle, noEventSend);

	self.frontDualWheelsUser:onDisassembling(self);

	for k,wheel in pairs(self.buyableFrontDualWheels) do
		unlink(wheel.transFix);
		unlink(wheel.node);
		setRotation(wheel.transFix, 0,0,0);
		setTranslation(wheel.transFix, 0,0,0);
		link(wheel.savePosition, wheel.transFix);
		link(wheel.transFix, wheel.node);
					
		setVisibility(wheel.connectors, false);
	end;
	self.frontDualWheelsUser = nil;
end;


function BuyableFrontDualWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoordsFrontDuals");
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};
		end;
	end;

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableFrontDualWheels:getSaveAttributesAndNodes(nodeIdent)
    local attributes = nil;

	if self.frontDualWheelsUser ~= nil then
		local x,y,z = getWorldTranslation(self.frontDualWheelsUser.rootNode);
		attributes = 'attacherCoordsFrontDuals="'.. x .. " " .. y .. " " .. z ..'"';
	end;

    return attributes, nil;
end;

function BuyableFrontDualWheels:onAttach(attacherVehicle)
end;

function BuyableFrontDualWheels:onDetach()
end;


