--[[
@title:       FollowingCamera
@author:      Jakob Tischler
@date:        25 Mar 2014
@version:     0.2
]]

FollowingCamera = {};
FollowingCamera.modName = 'Following camera';
FollowingCamera.author = 'Jakob Tischler';
FollowingCamera.version = 0.2;
addModEventListener(FollowingCamera);

local pow, floor, min, max, pi, abs, deg, rad = math.pow, math.floor, math.min, math.max, math.pi, math.abs, math.deg, math.rad;

function FollowingCamera:loadMap(name)
	self.active = false;
	self.curVehicle, self.curVehicleIdx, self.curVehicleName = self:getVehicle(1);

	self.vehX, self.vehY, self.vehZ = 0, 0, 0;
	self.plX, self.plY, self.plZ = 0, 0, 0;

	-- self.defaultFov = 60;
	-- self.fov = 60;

	self.zoomActive = false;

	print(('## %s v%.1f by %s loaded'):format(FollowingCamera.modName, FollowingCamera.version, FollowingCamera.author));
end;

local deletePlayer = function(self)
	if self.cameraId then
		setFovy(self.cameraId, 60);
	end;
end;
Player.delete = Utils.prependedFunction(Player.delete, deletePlayer);

function FollowingCamera:deleteMap()
	self:setIsActive(false, false);
	self.curVehicle, self.curVehicleIdx, self.curVehicleName = nil, nil, nil;
end;

function FollowingCamera:keyEvent(unicode, sym, modifier, isDown)
end;

function FollowingCamera:update(dt)
	if g_currentMission.isPlayerFrozen or g_gui.currentGui ~= nil or g_currentMission.controlledVehicle ~= nil then
		return;
	end;

	if self.active and (self.curVehicle == nil or self.curVehicleIdx == nil) then
		self.curVehicle, self.curVehicleIdx, self.curVehicleName = self:getVehicle(1);
	end;
	if self.defaultFov == nil then
		self.defaultFov = getFovy(g_currentMission.player.cameraId);
		self.fov = self.defaultFov;
	end;

	if g_currentMission.showHelpText then
		if self.active then
			g_currentMission:addHelpButtonText(g_i18n:getText('FOLLOWINGCAMERA_TURNOFF'), InputBinding.FOLLOWINGCAMERA_TOGGLE);
			g_currentMission:addHelpButtonText(g_i18n:getText('FOLLOWINGCAMERA_PREV_VEHICLE'), InputBinding.FOLLOWINGCAMERA_PREV_VEHICLE);
			g_currentMission:addHelpButtonText(g_i18n:getText('FOLLOWINGCAMERA_NEXT_VEHICLE'), InputBinding.FOLLOWINGCAMERA_NEXT_VEHICLE);
			g_currentMission:addExtraPrintText(g_i18n:getText('FOLLOWINGCAMERA_CURRENTVEHICLE'));
			g_currentMission:addExtraPrintText(self.curVehicleName);
			g_currentMission:addHelpButtonText(g_i18n:getText(self.zoomActive and 'FOLLOWINGCAMERA_TURNOFF_ZOOM' or 'FOLLOWINGCAMERA_TURNON_ZOOM'), InputBinding.FOLLOWINGCAMERA_ZOOM_TOGGLE);
		else
			g_currentMission:addHelpButtonText(g_i18n:getText('FOLLOWINGCAMERA_TURNON'), InputBinding.FOLLOWINGCAMERA_TOGGLE);
		end;
	end;

	if InputBinding.hasEvent(InputBinding.FOLLOWINGCAMERA_TOGGLE) then
		self:setIsActive(not self.active);
	elseif self.active and InputBinding.hasEvent(InputBinding.FOLLOWINGCAMERA_PREV_VEHICLE) then
		self.curVehicle, self.curVehicleIdx, self.curVehicleName = self:getVehicle(-1);
	elseif self.active and InputBinding.hasEvent(InputBinding.FOLLOWINGCAMERA_NEXT_VEHICLE) then
		self.curVehicle, self.curVehicleIdx, self.curVehicleName = self:getVehicle(1);
	elseif self.active and InputBinding.hasEvent(InputBinding.FOLLOWINGCAMERA_ZOOM_TOGGLE) then
		self:setZoomActive(not self.zoomActive);
	end;
end;

local FollowingCameraUpdateTick = function(self, dt)
	if g_currentMission.player == self then
		FollowingCamera:updateRotationAndZoom();
	end;
end;
-- Player.updateTick = Utils.appendedFunction(Player.updateTick, FollowingCameraUpdateTick);

function FollowingCamera:draw()
	self:updateRotationAndZoom();
end;

local maxFov = 3.5;
function FollowingCamera:updateRotationAndZoom()
	if g_currentMission.isPlayerFrozen or g_gui.currentGui ~= nil or g_currentMission.controlledVehicle ~= nil then
		if self.active then
			self:setIsActive(false);
		end;
		return;
	end;
	if not self.active or self.curVehicle == nil then return; end;

	-- CALCULATION
	local pl = g_currentMission.player;
	local changed = false;

	local vx,vy,vz = getWorldTranslation(self.curVehicle.rootNode);
	if vx ~= self.vehX or vy ~= self.vehY or vz ~= self.vehZ then
		self.vehX, self.vehY, self.vehZ = vx, vy, vz;
		changed = true;
	end;

	local px,py,pz = pl.lastTranslation[1], pl.lastTranslation[2], pl.lastTranslation[3];
	if px ~= self.plX or py ~= self.plY or pz ~= self.plZ then
		self.plX, self.plY, self.plZ = px, py, pz;
		changed = true;
	end;

	local dx,dy,dz = px - vx, py - vy, pz - vz;

	-- if changed then
		-- Y ROTATION
		local rotY = Utils.getYRotationFromDirection(dx, dz);
		-- pl.rotY = rotY;
		-- setRotation(pl.graphicsRootNode, 0, rotY, 0);
		if pl.rotY ~= rotY then
			self:rotSmoothly('Y', rotY);
		end;

		-- X ROTATION
		local dist2D = Utils.vector2Length(dx, dz);
		local rotX = -Utils.getYRotationFromDirection(dy, dist2D);
		-- pl.rotX = rotX;
		-- setRotation(pl.cameraId, rotX, 0, 0);
		if pl.rotX ~= rotX then
			self:rotSmoothly('X', rotX);
		end;
	-- end;

	-- ZOOM
	-- if not self.zoomActive then return; end;
	local dist3D = Utils.vector3Length(dx, dy, dz);
	if dist3D <= 10 or not self.zoomActive then
		self:zoomSmoothly(self.defaultFov, dist3D);
	elseif dist3D > 80 then
		self:zoomSmoothly(maxFov, dist3D);
	else
		local targetFov = 1190.1 * pow(dist3D, -1.318);
		self:zoomSmoothly(targetFov, dist3D);
	end;
end;

function FollowingCamera:mouseEvent(posX, posY, isDown, isUp, button)
end;

function FollowingCamera:setIsActive(active, setFov)
	if setFov == nil then setFov = true; end;
	self.active = active;
	if not active and setFov then
		self:setFov(self.defaultFov);
	end;
	-- print(string.format('FollowingCamera:setIsActive(%s)', tostring(state)));
end;

function FollowingCamera:getVehicle(dir)
	local veh, idx, name;
	if self.curVehicleIdx == nil then
		idx,veh = next(g_currentMission.steerables);
	else
		if dir == 1 then
			idx,veh = next(g_currentMission.steerables, self.curVehicleIdx);
			if idx == nil or veh == nil then
				idx,veh = next(g_currentMission.steerables);
			end;
		else
			idx = self.curVehicleIdx - 1;
			if idx == 0 then
				idx = #g_currentMission.steerables;
			end;
			veh = g_currentMission.steerables[idx];
		end;
	end;

	if veh and veh.configFileName and StoreItemsUtil.storeItemsByXMLFilename[veh.configFileName] then
		name = StoreItemsUtil.storeItemsByXMLFilename[veh.configFileName].name;
	elseif veh then
		name = veh.name or '(no name)';
	end;
	name = ('\t%q'):format(tostring(name));

	-- print(string.format('FollowingCamera:getVehicle(%d): veh=%q, idx=%s', dir, name, tostring(idx)));
	return veh, idx, name;
end;

function FollowingCamera:setFov(angle)
	self.fov = angle;
	setFovy(g_currentMission.player.cameraId, angle);
end;

function FollowingCamera:setZoomActive(active)
	self.zoomActive = active;
end;

local zoomPerMs = 0.001 * 0.666666666666666666666667;
function FollowingCamera:zoomSmoothly(targetFov, dist)
	if self.fov ~= targetFov then
		local deltaFov = targetFov - self.fov;

		local add = zoomPerMs * g_physicsDt * deltaFov;
		local fov;
		if deltaFov > 0 then
			add = max(0.05, add);
			fov = min(targetFov, self.fov + add);
		else
			add = min(-0.05, add);
			fov = max(targetFov, self.fov + add);
		end;

		-- print(('dist=%.2f, targetFov=%.3f -> set fov to %.3f'):format(dist, targetFov, fov));
		self:setFov(fov);
	end;
end;

-- local rotPerMs = 0.005;
local rotPerMs = rad(0.28);
function FollowingCamera:rotSmoothly(axis, targetRot)
	local pl = g_currentMission.player;
	local curRot = axis == 'X' and pl.rotX or pl.rotY;
	if curRot == targetRot then return; end;

	local deltaRot = targetRot - curRot;

	-- prevent switching from -pi to +pi (full 360° turn) when at 180° (looking directly south)
	if axis == 'Y' and abs(deltaRot) > pi then
		if deltaRot > 0 then
			curRot = curRot + pi * 2;
		else
			curRot = curRot - pi * 2;
		end;
		pl.rotY = curRot;
		deltaRot = targetRot - curRot;
	end;

	local add = rotPerMs * g_physicsDt * deltaRot;
	local rot;
	if deltaRot > 0 then
		add = max(0.0025, add);
		rot = min(targetRot, curRot + add);
	else
		add = min(-0.0025, add);
		rot = max(targetRot, curRot + add);
	end;

	if axis == 'X' then
		pl.rotX = rot;
		setRotation(pl.cameraId, rot, 0, 0);
	else
		pl.rotY = rot;
		setRotation(pl.graphicsRootNode, 0, rot, 0);
	end;
end;
