--
-- BuyableDualRear
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--				v2.1 - edit by fruktor for Fendt 615 ()
--

BuyableDualRear = {};

function BuyableDualRear.prerequisitesPresent(specializations)
    return true;
end;

function BuyableDualRear:load(xmlFile)

	self.twinWheelTriggerCallback = BuyableDualRear.twinWheelTriggerCallback;
	self.wheelDelete = BuyableDualRear.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.twinWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#activationTrigger"));
	
	
	addTrigger(self.usingTrigger, "twinWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.twinWheels#belts"));
	-- self.side  = getXMLString(xmlFile, "vehicle.twinWheels#side");
	
    self.buyableWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.twinWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		--wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		--setRigidBodyType(wheel.collision, "Kinematic");
		--setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.twinWheelsUser = nil;
end;

function BuyableDualRear:delete()
	if self.twinWheelsUser ~= nil then
		self.twinWheelsUser:onDisassembling(true);
		for _,twinWheel in pairs(self.buyableWheels) do
			--delete(twinWheel.collision);
		end;
		self.buyableWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableDualRear:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableDualRear:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local twinWheelsUser = networkGetObject(id);
		if twinWheelsUser ~= nil then
			self:assembleWheels(twinWheelsUser, true);
		end;
	end;
end;

function BuyableDualRear:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.twinWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.twinWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableDualRear:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableDualRear:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableDualRear:update(dt)
end;

function BuyableDualRear:updateTick(dt)

	-- try to load saved twinwheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then				
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;
end;

function BuyableDualRear:draw()	
end;

function BuyableDualRear:twinWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter then		
			if vehicle.rootNode == otherId then
				if vehicle.BuyableDualRear ~= nil then
					if vehicle.BuyableDualRear.checkString == self.checkString then
						if self.twinWheelsUser ~= nil then
							if self.twinWheelsUser == vehicle then
								vehicle.BuyableDualRear.wheelsInRange = self;
							end;
						else
							vehicle.BuyableDualRear.wheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.BuyableDualRear ~= nil then
					vehicle.BuyableDualRear.wheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableDualRear:assembleWheels(vehicle, noEventSend)
	BuyableDualRearAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;

	if self.twinWheelsUser == nil then
		if vehicle.BuyableDualRear ~= nil then
			self.twinWheelsUser = vehicle;	
			for k,wheel in pairs(vehicle.wheels) do
				for l,twinWheel in pairs(self.buyableWheels) do 
					--if k == l then	
					if k == twinWheel.wheelIndex then
						unlink(twinWheel.node);
						link(wheel.driveNode, twinWheel.node);	
						--link(getParent(wheel.repr), twinWheel.node);	
						local x,y,z = getWorldTranslation(wheel.driveNode);
						x,y,z = worldToLocal(getParent(twinWheel.node), x,y,z);
						setTranslation(twinWheel.node,x,y,z);
						--setRigidBodyType(twinWheel.collision, "Kinematic");
						break;
					end;
				end;
			end;	
			self.twinWheelsUser:onAssembling(self);
		end;
	end;
end;

function BuyableDualRear:disassembleWheels(noEventSend)
	BuyableDualRearDetachEvent.sendEvent(self, noEventSend);

	self.twinWheelsUser:onDisassembling();

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;
	
	for k,wheel in pairs(self.buyableWheels) do
		--setRigidBodyType(wheel.collision, "NoRigidBody");
		--unlink(wheel.collision);
		unlink(wheel.node);
		link(wheel.savePosition, wheel.node);
		setRotation(wheel.node, 0,0,0);
		setTranslation(wheel.node, 0,0,0);
		--link(wheel.collision, wheel.node);
	end;	
	self.twinWheelsUser = nil;	
end;


function BuyableDualRear:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");		
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};			
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableDualRear:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	
	if self.twinWheelsUser ~= nil then
		local x,y,z = getWorldTranslation(self.twinWheelsUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;
	
    return attributes, nil;
end;

--------------event------------------

BuyableDualRearAttachEvent = {};
BuyableDualRearAttachEvent_mt = Class(BuyableDualRearAttachEvent, Event);

InitEventClass(BuyableDualRearAttachEvent, "BuyableDualRearAttachEvent");

function BuyableDualRearAttachEvent:emptyNew()
    local self = Event:new(BuyableDualRearAttachEvent_mt);
    self.className="BuyableDualRearAttachEvent";
    return self;
end;

function BuyableDualRearAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableDualRearAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableDualRearAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableDualRearAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableDualRearAttachEvent:run(connection)
	self.vehicle:assembleWheels(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableDualRearAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableDualRearAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableDualRearAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableDualRearAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableDualRearDetachEvent = {};
BuyableDualRearDetachEvent_mt = Class(BuyableDualRearDetachEvent, Event);

InitEventClass(BuyableDualRearDetachEvent, "BuyableDualRearDetachEvent");

function BuyableDualRearDetachEvent:emptyNew()
    local self = Event:new(BuyableDualRearDetachEvent_mt);
    self.className="BuyableDualRearDetachEvent";
    return self;
end;

function BuyableDualRearDetachEvent:new(vehicle)
    local self = BuyableDualRearDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableDualRearDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableDualRearDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableDualRearDetachEvent:run(connection)
	self.vehicle:disassembleWheels(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableDualRearDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;


function BuyableDualRearDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableDualRearDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableDualRearDetachEvent:new(vehicle));
		end;
	end;
end;
