--
-- PZ Zweegers CM185H and CM212F
-- Class for all foldable drum mowers
--
-- @author  PeterJ - FS-UK modteam
-- @date  11/05/12
--
-- Copyright (C) LS-UK modteam, Confidential, All Rights Reserved.
  

PZ_mower = {};
  
function PZ_mower.prerequisitesPresent(specializations)
	return true;
end;
  
function PZ_mower:load(xmlFile)
  
	self.setIsTurnedOn = SpecializationUtil.callSpecializationsFunction("setIsTurnedOn");
  
	self.groundReferenceThreshold = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.groundReferenceNode#threshold"), 0.2);
	self.groundReferenceNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.groundReferenceNode#index"));
	if self.groundReferenceNode == nil then
		self.groundReferenceNode = self.components[1].node;
	end;
      
      
	if self.isClient then
		local mowerSound = getXMLString(xmlFile, "vehicle.mowerSound#file");
		if mowerSound ~= nil and mowerSound ~= "" then
			mowerSound = Utils.getFilename(mowerSound, self.baseDirectory); 
			self.mowerSound = createSample("mowerSound");
			self.mowerSoundEnabled = false;
			loadSample(self.mowerSound, mowerSound, false);
			self.mowerSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mowerSound#pitchOffset"), 1);
			self.mowerSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mowerSound#volume"), 1);
		end;
	end;
  
	local numCuttingAreas = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.cuttingAreas#count"), 0);
	for i=1, numCuttingAreas do
		local areanamei = string.format("vehicle.cuttingAreas.cuttingArea%d", i);
		self.cuttingAreas[i].foldMinLimit = Utils.getNoNil(getXMLFloat(xmlFile, areanamei .. "#foldMinLimit"), 0);
		self.cuttingAreas[i].foldMaxLimit = Utils.getNoNil(getXMLFloat(xmlFile, areanamei .. "#foldMaxLimit"), 1);
		self.cuttingAreas[i].grassParticleSystemIndex = getXMLInt(xmlFile, areanamei .. "#particleSystemIndex");
	end;

	self.rotateSpinners = {};
	local i=0;
	while true do
		local baseName = string.format("vehicle.rotateSpinners.node(%d)", i);
		local index = getXMLString(xmlFile, baseName.. "#index");
		local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, baseName.. "#rotationSpeed"));
		local rotationSpeed = {x,y,z};
		local runOutTime = Utils.getNoNil(getXMLFloat(xmlFile, baseName.. "#runOutTime"), 2)*1000;
		if index == nil or rotationSpeed == nil or runOutTime == nil then
			break;
		end;
		local node = Utils.indexToObject(self.components, index);
		if node ~= nil then
			local entry = {};
			entry.node = node;
			entry.runOutTime = runOutTime;
			entry.rotationSpeedMax = rotationSpeed;
			entry.rotationSpeedMin = {0,0,0};
			entry.rotationSpeedCurrent = {0,0,0};
			table.insert(self.rotateSpinners, entry);
		end;
		i = i+1;
	end;
  
	self.isTurnedOn = false;
	self.wasToFast = false;

	self.mowerGroundFlag = self.nextDirtyFlag;
	self.nextDirtyFlag = self.mowerGroundFlag*2;
	
	self.printPTOWarningTime = 0;

	self.mowerParticleSystems = {};
	local i = 0;
	while true do
        local namei = string.format("vehicle.grassParticleSystems.grassParticleSystem(%d)", i);
		local nodei = Utils.indexToObject(self.components, getXMLString(xmlFile, namei .. "#index"));
		if nodei == nil then
			break;
		end; 
        Utils.loadParticleSystem(xmlFile, self.mowerParticleSystems, namei, nodei, false, nil, self.baseDirectory)		
		i = i +1;		
	end;

	self.lastArea = 0;
	self.lastAreaBiggerZero = self.lastArea > 0;
	
	self.setHydraulicDirection = SpecializationUtil.callSpecializationsFunction("setHydraulicDirection");
	
	local hydraulicsCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.hydraulics#count"), 0);	
	self.hydraulics = {};	
	for i=1, hydraulicsCount do
		local hydraulicName = string.format("vehicle.hydraulics.hydraulic%d", i);		
		self.hydraulics[i] = {};		
		self.hydraulics[i].node = Utils.indexToObject(self.components, getXMLString(xmlFile, hydraulicName .. "#index"));
		self.hydraulics[i].punch = Utils.indexToObject(self.components, getXMLString(xmlFile, hydraulicName .. "#punch"));
		self.hydraulics[i].translationPunch = Utils.indexToObject(self.components, getXMLString(xmlFile, hydraulicName .. "#punchFixpoint"));
		self.hydraulics[i].fixPoint = Utils.indexToObject(self.components, getXMLString(xmlFile, hydraulicName .. "#fixpoint"));	
		local xUp, yUp, zUp = Utils.getVectorFromString(Utils.getNoNil(getXMLString(xmlFile, hydraulicName .. "#upVectors"),"0 1 0"));
		self.hydraulics[i].upVectors = {xUp, yUp, zUp};
		
		local ax, ay, az;		
		if self.hydraulics[i].punch ~= nil then
			ax, ay, az = getWorldTranslation(self.hydraulics[i].punch);
		else
			ax, ay, az = getWorldTranslation(self.hydraulics[i].node);
		end;
		if self.hydraulics[i].translationPunch ~= nil then
			local bx, by, bz = getWorldTranslation(self.hydraulics[i].translationPunch);		
			self.hydraulics[i].punchDistance = Utils.vector3Length(ax-bx, ay-by, az-bz);
		end;
	end;
	self.setHydraulicTime = 30;
	
	self.mowerParticleSystemFlag = self:getNextDirtyFlag();
	
    self.setVehicleRpmUp = SpecializationUtil.callSpecializationsFunction("setVehicleRpmUp");
    self.saveMinRpm = 0; 
	
	self.isPZmower = true;
	self.attachedHose = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.implementLinks.hose#index"));
	self.detachedHose = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.implementLinks.hose#deattached"));
end;

function PZ_mower:delete()
	Utils.deleteParticleSystem(self.mowerParticleSystems);
	if self.mowerSound ~= nil then
		delete(self.mowerSound);
	end;
end;
  
function PZ_mower:readStream(streamId, connection)
	local turnedOn = streamReadBool(streamId);
	self:setIsTurnedOn(turnedOn, true);
	self.lastAreaBiggerZero = streamReadBool(streamId);
end;
  
function PZ_mower:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isTurnedOn);
	streamWriteBool(streamId, self.lastAreaBiggerZero);
end;

function PZ_mower:readUpdateStream(streamId, timestamp, connection)
	if connection:getIsServer() then
		self.lastAreaBiggerZero = streamReadBool(streamId);
	end;
end;
  
function PZ_mower:writeUpdateStream(streamId, connection, dirtyMask)
	if not connection:getIsServer() then
		streamWriteBool(streamId, self.lastAreaBiggerZero);
	end;
end;
 
function PZ_mower:mouseEvent(posX, posY, isDown, isUp, button)
end;
  
function PZ_mower:keyEvent(unicode, sym, modifier, isDown)
end;
  
function PZ_mower:update(dt)

	if self:getIsActiveForInput() then
		if self.foldAnimTime < 0.45 then
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
				if self.PTOId then
					self.printPTOWarningTime = self.time + 1500;
				else
					self:setIsTurnedOn(not self.isTurnedOn);
				end;
			end;
		end;
	end;
	self.setHydraulicTime = 30;
	if self.setHydraulicTime > 0 then
		for k,v in pairs(self.hydraulics) do 
			self:setHydraulicDirection(k);
		end;
		self.setHydraulicTime = self.setHydraulicTime - 1;
	end;
end;
  
function PZ_mower:updateTick(dt)
	self.wasToFast = false;
	self.lastArea = 0;
	if self:getIsActive() then
		if not self.attacherVehicle.isMotorStarted then
			self:setIsTurnedOn(false);
		end;
		if self.PTOId then
			self:setIsTurnedOn(false);
		end;
		if self.attacherVehicle.motor and not self.attacherVehicle.isTriangle then
			self:setVehicleRpmUp(dt, self.isTurnedOn);
		end;
		if self.isTurnedOn then
			local toFast = self:doCheckSpeedLimit() and self.lastSpeed*3600 > 31;
			if not toFast and (self:isLowered(false) or self.attacherVehicle.isTriangle) then
				if self.isServer then
					local x,y,z = getWorldTranslation(self.groundReferenceNode);
					local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 0, z);
					if terrainHeight+self.groundReferenceThreshold >= y then
						local foldAnimTime = self.foldAnimTime;
						local totalArea = 0;
						local cuttingAreasSend = {};
						for k, cuttingArea in pairs(self.cuttingAreas) do
							if self:getIsAreaActive(cuttingArea) then
								local x,y,z = getWorldTranslation(cuttingArea.start);
								local x1,y1,z1 = getWorldTranslation(cuttingArea.width);
								local x2,y2,z2 = getWorldTranslation(cuttingArea.height);
								table.insert(cuttingAreasSend, {x,z,x1,z1,x2,z2});
							end;
						end;
						if (table.getn(cuttingAreasSend) > 0) then
							totalArea = PZMowerAreaEvent.runLocally(cuttingAreasSend);
			
							self.lastArea = totalArea;
							self.lastAreaBiggerZero = (self.lastArea > 0);
							
							if totalArea > 0 then
								if (table.getn(cuttingAreasSend) > 0) then
									g_server:broadcastEvent(PZMowerAreaEvent:new(cuttingAreasSend));
								end;
							end;
						end;
					end;
				end;
			else
				self.lastAreaBiggerZero = false;
			end;
			self.wasToFast = toFast;
			if self.movingDirection ~= 0 and self.lastAreaBiggerZero then
				Utils.setEmittingState(self.mowerParticleSystems, true);
			else
				Utils.setEmittingState(self.mowerParticleSystems, false);
			end;
		end;
		if self.foldAnimTime > 0.45 then
			self:setIsTurnedOn(false, true);
			if self.isClient then
				if self.mowerSoundEnabled then
					stopSample(self.mowerSound);
					self.mowerSoundEnabled = false;
				end;
			end;
		end;
	else
		Utils.setEmittingState(self.mowerParticleSystems, false);
	end;
	if self.isTurnedOn then
		if self.isClient then
			if not self.mowerSoundEnabled and self:getIsActiveForSound() then
				setSamplePitch(self.mowerSound, self.mowerSoundPitchOffset);
				playSample(self.mowerSound, 0, self.mowerSoundVolume, 0);
				self.mowerSoundEnabled = true;
			end;
		end;
	else
		if self.isClient then
			if self.mowerSoundEnabled then
				stopSample(self.mowerSound);
				self.mowerSoundEnabled = false;
			end;
		end;
		Utils.setEmittingState(self.mowerParticleSystems, false);
	end;
	for k, spinner in pairs(self.rotateSpinners) do
		local values = Utils.getMovedLimitedValues(spinner.rotationSpeedCurrent, spinner.rotationSpeedMax, spinner.rotationSpeedMin, 3, spinner.runOutTime, dt, not self.isTurnedOn);
		spinner.rotationSpeedCurrent = values;
		rotate(spinner.node, unpack(spinner.rotationSpeedCurrent));
	end;
end;
  
function PZ_mower:draw()
	if self.isClient then
		if self.foldAnimTime < 0.45 then
			if self.isTurnedOn then
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_off_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
			else
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_on_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
			end;
		end;
		if self.wasToFast then
			g_currentMission:addWarning(g_i18n:getText("Dont_drive_to_fast") .. "\n" .. string.format(g_i18n:getText("Cruise_control_levelN"), "2", InputBinding.getKeyNamesOfDigitalAction(InputBinding.SPEED_LEVEL2)), 0.07+0.022, 0.019+0.029);
		end;
		if self.printPTOWarningTime > self.time then
			g_currentMission:addWarning(g_i18n:getText("turnON_Error"), 0.018, 0.033);
		end;
	end;
end;

function PZ_mower:validateAttacherJoint(implement, jointDesc, dt)
    return true;
end;

function PZ_mower:onAttach(attacherVehicle)
	setVisibility(self.attachedHose,true);
	setVisibility(self.detachedHose,false);
    if self.attacherVehicleCopy == nil then
        self.attacherVehicleCopy = self.attacherVehicle;
    end;
	if self.attacherVehicle.motor ~= nil then
		self.saveMinRpm = self.attacherVehicle.motor.minRpm;
	end;
end;

function PZ_mower:onDetach()
	if self.deactivateOnDetach then
		PZ_mower.onDeactivate(self);
	else
		PZ_mower.onDeactivateSounds(self)
	end;
	if self.mowerParticleSystems ~= nil then
		Utils.setEmittingState(self.mowerParticleSystems, false);
	end;
	setVisibility(self.attachedHose,false);
	setVisibility(self.detachedHose,true);
    for k, steerable in pairs(g_currentMission.steerables) do
        if self.attacherVehicleCopy == steerable then
            steerable.motor.minRpm = self.saveMinRpm;
            self.attacherVehicleCopy = nil;
        end;
    end;
end;

function PZ_mower:onLeave()
	if self.deactivateOnLeave then
		PZ_mower.onDeactivate(self);
	else
		PZ_mower.onDeactivateSounds(self)
	end;
	if self.mowerParticleSystems ~= nil then
		Utils.setEmittingState(self.mowerParticleSystems, false);
	end; 
end;
  
function PZ_mower:onDeactivate()
	PZ_mower.onDeactivateSounds(self)
	self.isTurnedOn = false;
	if self.mowerParticleSystems ~= nil then
		Utils.setEmittingState(self.mowerParticleSystems, false);
	end;
end;
  
function PZ_mower:onDeactivateSounds()
	if self.isClient then
		if self.mowerSoundEnabled then
			stopSample(self.mowerSound);
			self.mowerSoundEnabled = false;
		end;
	end;
end;
  
function PZ_mower:setIsTurnedOn(turnedOn, noEventSend)
	SetTurnedOnEvent.sendEvent(self, turnedOn, noEventSend)
	self.isTurnedOn = turnedOn;
end;

function PZ_mower:setHydraulicDirection(index)
	local hydraulic = self.hydraulics[index];

	if hydraulic.fixPoint ~= nil then
		local ax, ay, az = getWorldTranslation(hydraulic.node);
		local bx, by, bz = getWorldTranslation(hydraulic.fixPoint);
		local x, y, z = worldDirectionToLocal(getParent(hydraulic.node), bx-ax, by-ay, bz-az);
		local xUp, yUp, zUp = unpack(hydraulic.upVectors);
		setDirection(hydraulic.node, x, y, z, xUp, yUp, zUp);
		if hydraulic.punch ~= nil then
			local distance = Utils.vector3Length(ax-bx, ay-by, az-bz);
			setTranslation(hydraulic.punch, 0, 0, distance-hydraulic.punchDistance);
		end;
	end;
end;

function PZ_mower:setVehicleRpmUp(dt, isActive)
    if self.attacherVehicle ~= nil and self.saveMinRpm ~= 0 then
        if dt ~= nil then
            if isActive == true then
                self.attacherVehicle.motor.minRpm = math.max(self.attacherVehicle.motor.minRpm-dt, -1000);
            else
                self.attacherVehicle.motor.minRpm = math.min(self.attacherVehicle.motor.minRpm+dt*2, self.saveMinRpm);
            end;
        else
            self.attacherVehicle.motor.minRpm = self.saveMinRpm;
        end;
        if self.attacherVehicle.isMotorStarted then
            local fuelUsed = 0.000001*math.abs(self.attacherVehicle.motor.minRpm);
            self.attacherVehicle:setFuelFillLevel(self.attacherVehicle.fuelFillLevel-fuelUsed);
            g_currentMission.missionStats.fuelUsageTotal = g_currentMission.missionStats.fuelUsageTotal + fuelUsed;
            g_currentMission.missionStats.fuelUsageSession = g_currentMission.missionStats.fuelUsageSession + fuelUsed;
        end;
    end;
end;