--
-- InteractiveButtons
-- Specialization for an interactive control button
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.1
-- @date  		29/08/11
-- @history:	v1.0 - Initial version
--				v2.0 - converted to ls2011
--				v2.1 - improvements
--

InteractiveButtons = {};

function InteractiveButtons.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(InteractiveControl, specializations);
end;

function InteractiveButtons:load(xmlFile)

	local i=0;
	while true do
		local buttonName = string.format("vehicle.interactiveComponents.buttons.button(%d)", i);	
		if not hasXMLProperty(xmlFile, buttonName) then
			break;
		end;
		local name = Utils.getNoNil(g_i18n:getText(getXMLString(xmlFile, buttonName .. "#name")), "ERROR");
		local mark = Utils.indexToObject(self.components, getXMLString(xmlFile, buttonName .. "#mark"));
		local highlight = getChildAt(mark, 0);
		local size = Utils.getNoNil(getXMLFloat(xmlFile, buttonName .. "#size"), 0.1);
		local event = getXMLString(xmlFile, buttonName .. "#event");
		local onMessage = g_i18n:getText(Utils.getNoNil(getXMLString(xmlFile, buttonName .. "#onMessage"), "ic_button_on"));
		local offMessage =  g_i18n:getText(Utils.getNoNil(getXMLString(xmlFile, buttonName .. "#offMessage") , "ic_button_off"));
	
		local button = Button:new(nil, highlight, name, mark, size, event, self, onMessage, offMessage, self.infoBar);
		
		button.synch = Utils.getNoNil(getXMLBool(xmlFile, buttonName .. "#synch"), true);

		table.insert(self.interactiveObjects, button);
		i = i + 1;
	end;
end;

function InteractiveButtons:delete()
end;

function InteractiveButtons:mouseEvent(posX, posY, isDown, isUp, button)
end;

function InteractiveButtons:keyEvent(unicode, sym, modifier, isDown)
end;

function InteractiveButtons:update(dt)	
end;

function InteractiveButtons:draw()
end;



--
-- Button Class
-- Specifies an interactive Button
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date  29/08/11
--

Button = {};

function Button:new(node, highlight, name, mark, size, event, vehicle, onMessage, offMessage, infobar)

	local Button_mt = Class(Button, InteractiveComponentInterface);	
    local instance = InteractiveComponentInterface:new(node, highlight, name, mark, size, onMessage, offMessage, infobar, Button_mt);

	instance.vehicle = vehicle;
	instance.event = event;
	
	return instance;	
end;

function Button:delete()
	InteractiveComponentInterface.delete(self);
end;

function Button:mouseEvent(posX, posY, isDown, isUp, button)
	InteractiveComponentInterface.mouseEvent(self, posX, posY, isDown, isUp, button);
end;

function Button:keyEvent(unicode, sym, modifier, isDown)
	InteractiveComponentInterface.keyEvent(self, unicode, sym, modifier, isDown);
end;

function Button:update(dt)
	if self.vehicle ~= nil then
		if self.event == "toggleDoor" then
			self.isOpen = self.vehicle.animationParts[10].clipEndTime;
		elseif self.event == "toggleWindow" then
			self.isOpen = self.vehicle.animationParts[9].clipEndTime;
		elseif self.event == "foldCombine" then
			self.isOpen = self.vehicle:getToggledFoldDirection() > 0;
		elseif self.event == "foldCutter" then
			self.isOpen = true;
			if self.vehicle.numAttachedCutters > 0 then
				for cutter, _ in pairs(self.vehicle.attachedCutters) do
					self.isOpen = cutter:getToggledFoldDirection() > 0;
				end
			end
		elseif self.event == "toggleConveyor" then
			self.isOpen = self.vehicle.targetPipeState ~= 1;
		elseif self.event == "startThreshing" then
			self.isOpen = self.vehicle.isThreshing;
		elseif self.event == "toggleAutosteer" then
			self.isOpen = not self.vehicle.steeringLocked;
		end;
	end;
	InteractiveComponentInterface.update(self, dt);
end;

function Button:draw()
	InteractiveComponentInterface.draw(self);
end;

function Button:doAction(noEventSend, forceAction)
	if self.vehicle ~= nil then
		if self.event == "toggleDoor" then
			self.vehicle:toggleDoorState();
		elseif self.event == "toggleWindow" then
			self.vehicle:toggleWindowState();
		elseif self.event == "foldCombine" then
			if self.vehicle:getIsFoldAllowed() then
				self.vehicle:setFoldState(self.vehicle:getToggledFoldDirection(), false);
			end
		elseif self.event == "foldCutter" then
			if self.vehicle.numAttachedCutters > 0 then
				for cutter, _ in pairs(self.vehicle.attachedCutters) do
					if cutter:getIsFoldAllowed() then
						cutter:setFoldState(cutter:getToggledFoldDirection(), false);
					end
				end
			end
		elseif self.event == "offsetSteeringRight" then
			if self.vehicle.isMotorStarted then
				self.vehicle:setSteeringMode(3);
			end;
		elseif self.event == "offsetSteeringLeft" then
			if self.vehicle.isMotorStarted then
				self.vehicle:setSteeringMode(2);
			end;
		elseif self.event == "toggleConveyor" then
			if self.vehicle:getIsPipeStateChangeAllowed() then
				local nextState = self.vehicle.targetPipeState+1;
				if nextState > self.vehicle.numPipeStates then
					nextState = 1;
				end;
				self.vehicle:setPipeState(nextState, true);
			end;
		elseif self.event == "startThreshing" then
			if self.vehicle:getIsStartThreshingAllowed() then
				self.vehicle:setIsThreshing(not self.vehicle.isThreshing);
			end;
		elseif self.event == "toggleAutosteer" then
			self.vehicle:setAutoSteer(not self.vehicle.steeringLocked);
		elseif self.event == "synchronizedSteering" then
			if self.vehicle.isMotorStarted then
				self.vehicle:setSteeringMode(1);
			end;
		elseif self.event == "totalSteer" then
			if self.vehicle.isMotorStarted then
				self.vehicle:setSteeringMode(4);
			end;
		elseif self.event == "allWheelSteer" then
			if self.vehicle.isMotorStarted then
				self.vehicle:setSteeringMode(5);
			end;
		end;
	end;
end;

function Button:onEnter(dt)
	InteractiveComponentInterface.onEnter(self, dt);
end;

function Button:onExit(dt)
	InteractiveComponentInterface.onExit(self, dt);
end;

function Button:setActive()
	InteractiveComponentInterface.setActive(self, isActive);
end;

function Button:setVisible(isVisible)
	InteractiveComponentInterface.setVisible(self, isVisible);
end;