﻿RealisticMotorized = {};

function RealisticMotorized.prerequisitesPresent(specializations) 
	return SpecializationUtil.hasSpecialization(RealisticVehicle, specializations) and SpecializationUtil.hasSpecialization(Motorized, specializations);
end;


function RealisticMotorized:load(xmlFile) 

	self.isRealistic = true;

	self.realIsMotorized = true;	
	self.motor.isRealistic = true;
	self.motor.accelerations = {1, 1, 1, 1};
	
	--self.realRollingResistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realRollingResistance"), 0.045); -- this value overrides the value in realisticVehicle
	--self.realRollingResistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realRollingResistance"), 0.035); -- 20131217 - new default value
	self.realRollingResistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realRollingResistance"), 0.02); -- 20140219 - new default value - tractor tyre rr on gravel/asphalt => 200N/T
	
	--20140219 - override default vehicle.realVehicleOnFieldRollingResistanceFx too
	self.realVehicleOnFieldRollingResistanceFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realVehicleOnFieldRollingResistanceFx"), 2.75);	-- tractor default rr in field = 2.75 x 0.02 = 0.07 = 550N/T
	
	local maxPtoPower = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realPtoPowerKW"), 0); -- max measured PTO power
	self.realBaseMaxFuelUsage = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMaxFuelUsage"), 0); -- fuel usage in liter per hour when the motor is at full power (without boost activated)
	self.realTransmissionEfficiency = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realTransmissionEfficiency"), 0.85); -- efficiency between transmission input and output power (input power = DIN power)
	self.realPtoDriveEfficiency = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realPtoDriveEfficiency"), 0.94); -- efficiency between pto transmission input and output power (input power = DIN power)
	
	self.realMotorizedWheelsDriveLossFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMotorizedWheelsDriveLossFx"), 0.0035); -- loss factor when moving with motorized wheels engaged
		
	self.realPower = maxPtoPower / self.realPtoDriveEfficiency; -- this is the max "net engine power" or DIN power
	
	self.realMaxReverseSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMaxReverseSpeed"), self.realMaxVehicleSpeed); --kph
	self.realAiManeuverSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realAiManeuverSpeed"), 10);	--kph
	self.realAiMaxWorkingSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realAiMaxWorkingSpeed"), 999);	--kph
	
	
	self.realMinReverseIdleTime = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMinReverseIdleTime"), 0.05); -- time (s) needed to stay idle for reversing gearBox when wheels are stopped
	self.realDirectInverser = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realDirectInverser"), false); -- no need waiting for ReverseIdleTime, true or false ?
	self.realHydrostaticTransmission = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realHydrostaticTransmission"), false); -- more "motor brake", no stop for reversing gearBox
	self.realReverseAccelerationKept = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realReverseAccelerationKept"), 0);
	
	self.realMinSpeedForMaxPower = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMinSpeedForMaxPower"), 2)/3.6; -- max power at 2 kph  -> drive Train Protection (max torque at the wheel == torque when the motor is at full power and the wheel at 2 kph)
	
	self.realSpeedBoostAddPwr = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realSpeedBoost"), 0);
	self.realSpeedBoostMinSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realSpeedBoost#minSpeed"), 0);
	self.realImplementNeedsBoostAddPwr = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realImplementNeedsBoost"), 0);
	self.realImplementNeedsBoostMinPowerCons = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realImplementNeedsBoost#minPowerCons"), 0);
	self.realMaxBoostAddPwr = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMaxBoost"), self.realSpeedBoostAddPwr + self.realImplementNeedsBoostAddPwr);
	self.realMaxPowerToTransmission = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMaxPowerToTransmission"), self.realPower+self.realMaxBoostAddPwr); -- useful for harvesters and others heavy machine with hydraulic transmission and big motor total power (usually, the hydraulic motors are les powerful than the motor)
		
	self.realForceApplicationOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realForceApplicationOffset"), 0); -- wheelforce application offset. useful to make a vehicle lift the front wheel when accelerating (ATV for example)
	self.realTyreGripFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realTyreGripFx"), 1); -- useful to give more grip to wheel for traction (example : ATV because less weight but much power)
	self.realMotorBrakeFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMotorBrakeFx"), 1); -- useful to give less motorBrake (example : ATV because less weight but much power and so, the "computed" motorbrake is too strong)
	
	
	RealisticMotorized.loadRealManualGearXML(self, xmlFile);
	
	
	
	
	
	--**********************************************************************************************************************
	--20140218 - realEngine
	self.realEngine = {};
	self.realEngine.maxTorque = 0;
	self.realEngine.maxTorqueRpm = 0;
	self.realEngine.maxRpm = 0;
	self.realEngine.torqueCurve = AnimCurve:new(linearInterpolator1);
	self.realEngine.fuelUsageRatioCurve = AnimCurve:new(linearInterpolator1);
	local torqueI = 0;
	while true do
		local key = string.format("vehicle.realEngine.torque(%d)", torqueI);
		local rpm = getXMLFloat(xmlFile, key.."#rpm");
		local torque = getXMLFloat(xmlFile, key.."#ptoTorque");
		local fuelUsageRatio = getXMLFloat(xmlFile, key.."#fuelUsageRatio");
		if torque == nil or rpm == nil or fuelUsageRatio==nil then
			break;
		end;
		torque = torque / self.realPtoDriveEfficiency; -- take into account "realPtoDriveEfficiency" !!! => we are talking of the pto torque curve here.
		self.realEngine.torqueCurve:addKeyframe({v=torque, time = rpm});
		self.realEngine.fuelUsageRatioCurve:addKeyframe({v=fuelUsageRatio, time = rpm});
		
		if torque>self.realEngine.maxTorque then
			self.realEngine.maxTorqueRpm = rpm;
			self.realEngine.maxTorque = torque;
		end;
		
		self.realEngine.maxRpm = math.max(self.realEngine.maxRpm, rpm);
		torqueI = torqueI + 1;		
	end;
	
	self.realEngine.motorSoundPitchCurve = AnimCurve:new(linearInterpolator1);
	self.realEngine.motorSoundRunPitchCurve = AnimCurve:new(linearInterpolator1);
	local pitchI = 0;
	while true do
		local key = string.format("vehicle.realEngine.pitch(%d)", pitchI);
		local rpm = getXMLFloat(xmlFile, key.."#rpm");
		local pitch = getXMLFloat(xmlFile, key.."#motorSoundPitch");
		local pitchRun = getXMLFloat(xmlFile, key.."#motorSoundRunPitch");
		if pitch == nil or rpm == nil or pitchRun==nil then
			break;
		end;
		self.realEngine.motorSoundPitchCurve:addKeyframe({v=pitch, time = rpm});	
		self.realEngine.motorSoundRunPitchCurve:addKeyframe({v=pitchRun, time = rpm});	
		pitchI = pitchI + 1;
	end;
	
	self.realEngine.accelerationSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#accelerationSpeed"), self.maxAccelerationSpeed*1000)*0.001;
	self.realEngine.decelerationSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#decelerationSpeed"), self.decelerationSpeed*1000)*0.001;
	
	self.realEngine.speedLimiter = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#speedLimiter"),9999)/3.6; -- kph to mph
	
	self.realEngine.revUpTimeMs = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#revUpTimeMs"), 1400);
	self.realEngine.revDownTimeMs = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#revDownTimeMs"), 2000);
	
	self.realEngine.idleRpm = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#idleRpm"), 800);
	self.realEngine.ratedRpm = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realEngine#ratedRpm"), 2100);
	
	self.realEngine.idleRpmRatio = self.realEngine.idleRpm/self.realEngine.ratedRpm;
	self.realEngine.ratedToMaxRpmRatio = self.realEngine.maxRpm/self.realEngine.ratedRpm;
	self.realEngine.ratedToMaxTorqueRpmRatio = self.realEngine.maxRpm/self.realEngine.maxTorque;
	
	self.realEngine.currentGearRatedSpeed = 0;
	self.realEngine.ratedEngineRpmWheelSpeedRatio = 0;				
	self.realEngine.engineMaxTorqueToWheelPossible = 0;
		
	--** end realEngine
	--**********************************************************************************************************************
	
	
	local motorSpeedLevelStr = getXMLString(xmlFile, "vehicle.realSpeedLevel");
	local motorSpeedLevel1, motorSpeedLevel2, motorSpeedLevel3 = Utils.getVectorFromString(motorSpeedLevelStr);
	motorSpeedLevel1 = Utils.getNoNil(motorSpeedLevel1, 6);
	motorSpeedLevel2 = Utils.getNoNil(motorSpeedLevel2, 12);
	motorSpeedLevel3 = Utils.getNoNil(motorSpeedLevel3, 18);	
	self.motor.realSpeedLevels = {motorSpeedLevel1, motorSpeedLevel2, motorSpeedLevel3, self.realMaxVehicleSpeed};	
	self.motor.realSpeedLevelsAI = {motorSpeedLevel1, self.realAiManeuverSpeed, motorSpeedLevel3, self.realMaxVehicleSpeed};	
	
	--print("RealisticMotorized:load => " .. tostring(xmlFile));
		
	self.realAWDModePossible = false;
		
	local i = 0;	
	while true do
		local wheelnamei = string.format("vehicle.wheels.wheel(%d)", i);
		if self.wheels[i+1] == nil then
			break;
		end;		
		self.wheels[i+1].realGroundForce = 0;		
		self.wheels[i+1].realHasNoGroundContactWithPower = false;
		self.wheels[i+1].realLastMotorBrakeForce = 0;
		self.wheels[i+1].realStartingSlippingFx = 0;
		if self.wheels[i+1].driveMode==1 then
			self.realAWDModePossible = true;		
		end;
		i = i + 1;
	end;
	
	self.requiredDriveMode = 2;
	
	self.realReverserPossible = false;
	self.realReverseTimer = 0;
	
		
	self.realNbMotorizedWheels = 0;
	self.realTotalMassOnMotorizedWheels = 0;
	self.realLastAxisPower = 0;
	
	
	
	self.realCurrentTotalPowerConsumption = 0;
	self.realCurrentTotalForceNeeded = 0;
	--self.realResistanceForceApplicationOffset = 0;
	
	--RealisticMotorized.updateMotorizedWheels(self);
	

	
	self.realTractionMode = false;
	self.realTractionModeHasChanged = false;
	self.realAutoParkBrakeValue = 1;
	self.realReverseGearEngaged = false;
	
	
	--self.realGeneralWheelCoeff = 0;	
	
	
	
	self.realMaxFuelUsage = self.realBaseMaxFuelUsage;
	
	self.realMotorLoad = 0;
	self.realMotorLoadS = 0;	
	self.realMotorThrottleForTools = 0;
	self.realMotorThrottleForWheels = 0;
	--self.realRegulatorFx = 0;
	self.realRegulatorFxS = 0;
	
	self.realLastEngineAccForTransmission = 0;
	
	self.realCurrentPowerBoost = 0;
	self.realCurrentPowerBoost2 = 0;
	self.realAiWorkingSpeedLevel = 1; -- updated in OverrideAITractor.lua / OverrideAICombine.lua
	
	
	self.realFuelTankFillMass = 0;
	
	self.realStiffFactor = 1;
	--self.realStiffRandTimer = 0;
	self.realStiffFactorRnd = 1;
	self.realStiffFactorRndTarget = 1;
	
	--self.realCurrentRefuelGasTank = nil;
	--self.realSetHasRefuelTankInRange = RealisticGasTank.realSetHasRefuelTankInRange;
	
	
	
	self.realDeliveredToolsPowerRatio = 1; -- ratio of delivered power to tools => if tools get all the power they need, =1, if tools get no power, =0
	
	
	self.realTransmissionMode = {};
	self.realTransmissionMode.switchPossible = true;
	
	--self.realTransmissionMode.currentMode = 1; -- we want to set the mode 2 by default
	--if self.realHydrostaticTransmission or self.realDirectInverser then
	--	self.realTransmissionMode.currentMode = 2; --we want to set the mode 3 by default
	--end;
	
	--EDIT 20131204 -> T3 mode by default for all vehicles (easier to understand for newbies)
	self.realTransmissionMode.currentMode = 2;
		
	self.realTransmissionMode.maxMode = 3;
	
	self.realTransmissionMode.modes = {};
	self.realTransmissionMode.modes[1] = {};
	self.realTransmissionMode.modes[1].enabled = true; -- manual shuttle 
	self.realTransmissionMode.modes[1].text = "T1";
	
	self.realTransmissionMode.modes[2] = {};
	self.realTransmissionMode.modes[2].enabled = true; -- double "tap" mode
	self.realTransmissionMode.modes[2].text = "T2";
	
	self.realTransmissionMode.modes[3] = {};
	self.realTransmissionMode.modes[3].enabled = true; -- direct inverser mode
	self.realTransmissionMode.modes[3].text = "T3";	
	
	
	
	
	
	self.realShuttleDirection = 1; -- 1 = forward / -1 = reverse / 0 = neutral
	self.realShuttleIsManual = false;
	
	self.realParkbrakeIsManual = false;
	self.realParkbrakeIsActive = false;
	
	self.realManualGearSet = false;
	self.realManualGearMaxSpeed = 0;
	self.realManualGearMinSpeed = 0;
	self.realManualGearMinSpeedForMaxPower = 0;
	self.realManualGearClutchIsSlipping = false;
	--self.realManualGearMotorRpmRatio = 0;
	self.realManualGearIdleRpmRatio = 0;
	self.realManualGearIdleSpeedFx = 0;
	
	self.realManualGearUseOldModel = false;
	
	self.realThrottleIsManual = false;
	self.realThrottleCurrentValue = 0; -- between 0 and 1 => 0 = motor stopped, 1 = motor at full rpm
	self.realThrottleMinValue = 0; -- % of throttle when motor is at idle speed
	self.realThrottleOnlyManual = false; -- if true, pedals or keyboard standard acceleration key is not used
	
	self.realClutchEngaged = true;
	self.realClutchPercent = 1;
	
	self.realIsMotorStarted = false; -- replace "self.isMotorStarted and self.motorStartTime <= self.time"
	
	self.realCurrentTargetSlip = 0;
		
	--if Vehicle.debugRendering then
	--	self.motor.realDebugCurrentFadingOut = 1;
	--end;
	
	if self.isServer then
		RealisticSteerable.realSetNextTransmissionMode(self);
	end;
	
	self.realSetShuttleDirection = SpecializationUtil.callSpecializationsFunction("realSetShuttleDirection");
	self.realSetReverseGearActive = SpecializationUtil.callSpecializationsFunction("realSetReverseGearActive"); 
	
	--self.realMotorBrakeActive = false;
	self.realCurrentMotorBrakeForceFx = 0;
	self.realLastMotorFx = 0;
	
	
	self.realSoundSpdFx = 0;
	self.realSoundMotorFx = 0;
	self.realSoundEngineRevFx = 0;
	self.realBrakePedal = 0;
	
	self.realOverSpeed = 0;
	self.realMaxSpeedWanted = 0;
	
	self.realEngineSoundVolumeFx = 1; -- allow modders to implement a "incab" sound for example, by setting this parameter to 0.3 or whatever they think is good.
	
	self.realEngineSound3dEnabled = false;
	
	self.realMotorizedDirtyFlag = self:getNextDirtyFlag(); --see "overrideMotorized" and "overrideWheelUtils" files for the raise events
	
end;

function RealisticMotorized:delete()    
end;

function RealisticMotorized:mouseEvent(posX, posY, isDown, isUp, button)
end;

function RealisticMotorized:keyEvent(unicode, sym, modifier, isDown)
end;

function RealisticMotorized:readStream(streamId, connection)
	self.realManualGearSet = streamReadBool(streamId);		
end;
 
function RealisticMotorized:writeStream(streamId, connection)
	streamWriteBool(streamId, self.realManualGearSet);	
end;

function RealisticMotorized:readUpdateStream(streamId, timestamp, connection)
	if connection:getIsServer() then
		if streamReadBool(streamId) then
			self.realSoundSpdFx = 0.01*streamReadUIntN(streamId, 8);
			self.realSoundMotorFx = 0.01*streamReadUIntN(streamId, 8);
			self.realBrakePedal = 0.01*streamReadInt8(streamId);
			if self.realManualGearSet then
				self.realSoundEngineRevFx = 0.01*streamReadUIntN(streamId, 8);
			end;
		end;
	end;
end;
 
function RealisticMotorized:writeUpdateStream(streamId, connection, dirtyMask)
	if not connection:getIsServer() then
		if streamWriteBool(streamId, bitAND(dirtyMask, self.realMotorizedDirtyFlag) ~= 0) then
			streamWriteUIntN(streamId, math.min(255, tonumber(self.realSoundSpdFx*100)), 8); -- uint8 => 0 / +255
			streamWriteUIntN(streamId, math.min(255, tonumber(self.realSoundMotorFx*100)), 8); -- uint8 => 0 / +255			
			streamWriteInt8(streamId, tonumber(self.realBrakePedal*100)); --int8 => -127 / +127
			if self.realManualGearSet then
				streamWriteUIntN(streamId, math.min(255, tonumber(self.realSoundEngineRevFx*100)), 8); -- uint8 => 0 / +255	
			end;
		end;
	end;
end;




function RealisticMotorized:update(dt)
		
	if self.isServer and self.isActive and self.isMotorStarted then		
		RealisticMotorized.updateWheelsForces(self, dt/1000);	
		
		--update the realManualGearMotorRpmRatio if needed
		--[[20140214 - deprecated
		if self.realManualGearSet then	
			RealisticMotorized.updateMotorRpmRatio(self);
		else			
			self.realClutchEngaged = self.lastRealAcceleration~=0 or self.realAutoParkBrakeValue<1;			
		end;]]
		
		if not self.realManualGearSet then
			self.realClutchEngaged = self.lastRealAcceleration~=0 or self.realAutoParkBrakeValue<1;
		end;
		
	end;

end;


function RealisticMotorized:updateTick(dt)

	if self.isActive then

		self.realIsMotorStarted = false;
		--DURAL 20131004 - fix the bug that set the realIsMotorStarted to false when the engine is running
		
		--if self.isMotorStarted and (not isSamplePlaying(self.motorStartSound) or getSamplePlayOffset(self.motorStartSound) >= (getSampleDuration(self.motorStartSound)-1.5*dt)) then
		if self.isMotorStarted and self.time>=self.motorStartTime then  
			self.realIsMotorStarted = true;
		end;
		
		if self.isServer and self.realIsMotorStarted then			
			--print(self.time .. " RealisticMotorized:updateTick - realIsMotorStarted=" .. tostring(self.realIsMotorStarted));				
			RealisticMotorized.updateStiffFactor(self, false, dt);
			RealisticMotorized.updateFuelMass(self);						
		end;	
		
	end;
	
	--DURAL 20131020 - better support for manual ignition mod
	if self.isServer and self.realIsMotorStarted then
		--print(self.time .. " RealisticMotorized:updateTick - realIsMotorStarted=" .. tostring(self.realIsMotorStarted));	
		RealisticMotorized.updateImplementsNeeds(self);
		RealisticMotorized.updatePowerBoost(self);
	end;
	
end;

function RealisticMotorized:draw() 
end;









function RealisticMotorized:realSetShuttleDirection(currentDirection, newDirection)
	self.realShuttleDirection = newDirection;
end;




--[[
function RealisticMotorized.DEPRECATEDupdateMotorizedWheels(self)

	local nbWheels = 0;	
	local sumMass = 0;
	local avgSpd = 0;
	
	
	for k, wheel in pairs(self.wheels) do
		if wheel.driveMode>=self.requiredDriveMode then	
			nbWheels = nbWheels + 1;			
			sumMass = sumMass + wheel.realSupportedMass;	
			avgSpd = avgSpd + wheel.axleSpeed * wheel.radius;
		end;		
	end;
	
	self.realNbMotorizedWheels = nbWheels;
	self.realTotalMassOnMotorizedWheels = sumMass;
	self.realAvgMotorizedWheelSpd = avgSpd;

end;--]]

function RealisticMotorized.updateImplementsNeeds(self)
	
	local needTraction, powerConsumption, forceNeeded, needEngineRev = self:getImplementsNeeds();
	
	self.realTractionModeHasChanged = false;
	
	if self.realTractionMode ~= needTraction then
		--RealisticMotorized.updateStiffFactor(self, true);
		self.realTractionModeHasChanged = true;
		self.realPreviousTractionMode=self.realTractionMode;
	end;
	self.realTractionMode = needTraction;
	self.realCurrentTotalPowerConsumption = powerConsumption;
	--self.realCurrentTotalForceNeeded = forceNeeded;
		
	--fictive resistance force to simulate PTO tools inertia
	self.realCurrentTotalForceNeeded = forceNeeded + 0.02*self.realCurrentTotalPowerConsumption;
	
	self.realNeedEngineRev = needEngineRev;
	
	--if self.realCurrentTotalForceNeeded>0 then
		--resistance force application offset
	--	self.realResistanceForceApplicationOffset = sumAttacherHeight / self.realCurrentTotalForceNeeded;
	--else
		--self.realResistanceForceApplicationOffset = 0;
	--end;
	
end;



function RealisticMotorized.updateStiffFactor(self, force, dt)


	
	if self.realStiffFactorRnd==self.realStiffFactorRndTarget or force then
		--self.realStiffRandTimer = 0;
		
		if self.realVehicleOnField then
			-- on field variation
			self.realStiffFactorRndTarget = 0.9 + 0.2*math.random() - 0.2*RealisticGlobalListener.realRainFx*math.random();	-- between 0.9 and 1.1   -- should be on field -> 20% variation			
		else
			-- on road variation
			self.realStiffFactorRndTarget = 0.95 + 0.1*math.random(); -- between 0.95 and 1.05   -- on concrete ground -> 10% variation
		end;	
		
	end;
	
	--print(self.time .. " RealisticMotorized.updateStiffFactor => target / value : " .. self.realStiffFactorRndTarget .. " / " .. self.realStiffFactorRnd);
	
	if self.realStiffFactorRndTarget>self.realStiffFactorRnd then
		self.realStiffFactorRnd = math.min(self.realStiffFactorRndTarget, self.realStiffFactorRnd + 0.000005*dt);
	else
		self.realStiffFactorRnd = math.max(self.realStiffFactorRndTarget, self.realStiffFactorRnd - 0.000005*dt);
	end;
	
	--self.realStiffFactorRnd = 0.98 * self.realStiffFactorRnd + 0.02 * self.realStiffFactorRndTarget;
	
	
	local tractionFx = RealisticGlobalListener.realRoadTractionFx*(1-0.3*RealisticGlobalListener.realRainFx);
	if self.realVehicleOnField then
		tractionFx = RealisticGlobalListener.realFieldTractionFx*(1-0.4*RealisticGlobalListener.realRainFx);
	end;
	
	self.realStiffFactor = RealisticGlobalListener.realFrictionTuning * tractionFx * self.realStiffFactorRnd;
	
	

end;

function RealisticMotorized.updateFuelMass(self)

	if self.firstTimeRun and self.isServer then
		self.realFuelTankFillMass = 0.001*self.fuelFillLevel*RealisticGlobalListener.fuelDensity;
	end;

end;










-- dt in second !!!!!!!!
function RealisticMotorized.updateWheelsForces(self, dt)
	
	--self.realMotorBrakeActive = false;
	self.realCurrentMotorBrakeForceFx = 0;
	
	if Vehicle.debugRendering then
		for k, wheel in pairs(self.wheels) do
			wheel.realDebugMotorBrakeForce = 0;
		end;
	end;
	
	if self.realClutchEngaged then
	
		--print(self.time .. " RealisticMotorized.updateWheelsForces - realClutchEngaged = true");
	
		--[[
		local forceApplicationOffset = 0.2;
		if self.realTractionMode then		
			forceApplicationOffset = 0;
		end;]]
		--local forceApplicationOffset = self.realForceApplicationOffset;
		
		--print(self.time .. " test RealisticMotorized.updateWheelsForces  forcefx = " .. tostring(forceFx) .. " spdWanted = " .. tostring(spdWanted));
		--[[ no more "steering help force" - we have to set the proper wheel steering angles min and max (different => see "Ackerman" on google) 
		local forceFx2 = 1;
		local speedFx2 = 1;
		local avgWheelAngle = 0;
		
		
		if self.realGroundSpeed>1 then
			avgWheelAngle = Utils.clamp(0.9*self.realTouchingWheelsAvgSteeringAngle, -1.3, 1.3); -- 1.3 rad = 75°
			if avgWheelAngle~= 0 and self.realNumTouchingWheels>1 then
				forceFx2 = 1 / math.cos(avgWheelAngle);			
				--if forceFx2>1.1 then
				--	print("avgWheelAngle / cos : " .. avgWheelAngle .. " / " .. math.cos(avgWheelAngle));
				--end;		
				speedFx2 = math.min(1, 5/self.realGroundSpeed) * self.realSteeringAngleFx;
			end;
		end;]]
		
		local totalMbfFx = 0;
		local nbMbfWheels = 0;
		
		for k, wheel in pairs(self.wheels) do
		
			local newWheelMotorBrakeForce = 0;
							
			if wheel.driveMode>=self.requiredDriveMode then
			
				--if not self.isEntered then
				--	print(self.time .. " RealisticMotorized.updateWheelsForces - wheel.realGroundForce ="..tostring(wheel.realGroundForce));
				--end;	
				
				if wheel.realGroundForce~=0 then						
					
					local gForce = wheel.realGroundForce*dt;
					local xPoint = wheel.netInfo.x;
					
					
					--20140116 - manage skid steer
					--print(self.time .. " realIsSkidSteerVehicle=" .. tostring(self.realIsSkidSteerVehicle) .. " maxRotTime=" .. tostring(self.maxRotTime) .. " self.rotatedTime=" .. tostring(self.rotatedTime));
					if self.realIsSkidSteerVehicle then
						if self.rotatedTime>0 then							
							--turn left
							--limit or reverse left wheels force
							if self.maxRotTime>0 then
								
								local skidFx = self.rotatedTime/self.maxRotTime;
								if skidFx<0.5 then
									--only limit lzft wheel
									if wheel.realSkidIsLeftWheel then 
										gForce = gForce * (1-2*skidFx);								
									end;									
								else
									--"reverse" left wheel force
									if wheel.realSkidIsLeftWheel then 
										gForce = gForce * self.realIsSkidSteerVehicleSteeringForceFx * (1-2*skidFx) * 1.25;
									end;
									--increase right wheel force to keep up with the negative left wheel force
									if wheel.realSkidIsRightWheel then
										gForce = gForce * (1 + (skidFx-0.5)*2*(self.realIsSkidSteerVehicleSteeringForceFx-0.5));
									end; 
								end;
								--print(" -- left factor = " .. tostring(skidFx));
								
							end;
						elseif self.rotatedTime<0 then
							--turn right
							--limit or reverse right wheels force
							if self.minRotTime<0 then
							
								local skidFx = self.rotatedTime/self.minRotTime;
								if skidFx<0.5 then
									--only limit right wheel
									if wheel.realSkidIsRightWheel then 
										gForce = gForce * (1-2*skidFx);								
									end;									
								else
									--"reverse" right wheel force
									if wheel.realSkidIsRightWheel then 
										gForce = gForce * self.realIsSkidSteerVehicleSteeringForceFx * (1-2*skidFx) * 1.25;
									end;
									--increase left wheel force to keep up with the negative right wheel force
									if wheel.realSkidIsLeftWheel then
										gForce = gForce * (1 + (skidFx-0.5)*2*(self.realIsSkidSteerVehicleSteeringForceFx-0.5));
									end; 
								end;
								--print(" -- right factor = " .. tostring(skidFx));
								
							end;
						end;
					end;	
					
					
					
					local xForce = 0;
					local zForce = gForce;
					
					
					if wheel.steeringAngle~=0 and not self.realIsSkidSteerVehicle then
						xForce = math.sin(wheel.steeringAngle)*gForce;
						zForce = math.cos(wheel.steeringAngle)*gForce;
						--print(string.format("%g motorized wheel force - steering angle=%1.2f (degree=%1.2f) / xForce=%1.2f / zForce=%1.2f", self.time, wheel.steeringAngle, math.deg(wheel.steeringAngle), xForce, zForce));
					end;	
											
					local xT, yT, zT = localDirectionToWorld(self.realMainComponent.node, xForce, 0, zForce);						
					
					--print(self.time .. " RealisticMotorized.updateWheelsForces - adding force = " .. tostring(wheel.realGroundForce));
					
					addForce(wheel.realRootComponent.node, xT, yT, zT, xPoint, wheel.realY-(wheel.radius-self.realForceApplicationOffset), wheel.netInfo.z, true);	
					--addForce(wheel.realRootComponent.node, xT, yT, zT, 0, wheel.realY-(wheel.radius-self.realForceApplicationOffset), wheel.netInfo.z, true);	
					--addForce(self.realMainComponent.node, xT, yT, zT, xPoint-2, wheel.realY-(wheel.radius-forceApplicationOffset), wheel.netInfo.z, true);
					
				else
					--motor brake force
					if not (self.realShuttleIsManual and self.realShuttleDirection==0 and not self.realHydrostaticTransmission) then
						local spd = self.realGroundSpeed;
						if wheel.realSupportedMass>0 and spd>0.3 and self.realOverSpeed>0 then
							--local accFx = 1;
							
							--local wheelMotorBrakeForce = accFx * 0.25 * self.realPower * wheel.realSupportedMass / self.realTotalMassOnMotorizedWheels / math.max(1, spd) * dt;	
							--print(self.time .. " RealisticMotorized.updateWheelsForces motorbrakeforce : " .. wheel.realSupportedMass .. " | " .. self.realTotalMassOnMotorizedWheels);
							--local wheelMotorBrakeForce = 0.2 * self.realPower * wheel.realSupportedMass / self.realTotalMassOnMotorizedWheels / math.max(1, spd^0.25) * dt;	
							
							local maxWheelMotorBrakeForce = RealisticGlobalListener.motorBrakeForceTuning * self.realMotorBrakeFx * self.realMaxPowerToTransmission * wheel.realSupportedMass / self.realTotalMassOnMotorizedWheels / math.max(2, spd);
							--more motorbrake for hydrostatic transmission
							if self.realHydrostaticTransmission then
								maxWheelMotorBrakeForce = Utils.clamp(spd, 1, 1.25) * maxWheelMotorBrakeForce; -- 25% more brakemotor than "normal" transmission
							end;
							
							--max brake power starts at 2m.s-1 (7.2 kph)
							--local wheelMotorBrakeForce = RealisticGlobalListener.motorBrakeForceTuning * self.realMotorBrakeFx * self.realPower * wheel.realSupportedMass / self.realTotalMassOnMotorizedWheels / math.max(2, spd)^0.5;	

							local mbfFx = 1;
							local maxTransmissionSpeed = self.realMaxVehicleSpeed/3.6;						
							if self.realManualGearSet and not self.realHydrostaticTransmission then								
								if self.realOldManualGearBoxPresent then
									mbfFx = mbfFx * math.min(5, (spd / self.realManualGearMaxSpeed)^0.5);
								else
									mbfFx = mbfFx * math.min(5, spd / (self.realEngine.ratedToMaxTorqueRpmRatio*self.realEngine.currentGearRatedSpeed));
								end;
							else
								if self.realReverseGearEngaged then
									maxTransmissionSpeed = self.realMaxReverseSpeed/3.6;
								end;
								if spd>maxTransmissionSpeed then
									mbfFx = mbfFx * math.min(6, (spd / maxTransmissionSpeed)^0.5);
								end;
							end;
							
							if not self.realIsBraking and self.motor.speedLevel==0 and not self.realManualGearSet then
								if not self.realHydrostaticTransmission then
									mbfFx = mbfFx * 0.1 * Utils.clamp(3*self.realGroundSpeed-2, 1,10); -- reduce motor brake force when not braking - function of speed. 4mph = 1.  1mph = 0.1
									mbfFx = mbfFx * 0.25 * Utils.clamp(self.realTotalMovingMassOnTheWheels/self.realComponentsCurrentMass, 1,4);
								else
									mbfFx = mbfFx * 0.5 * Utils.clamp(self.realTotalMovingMassOnTheWheels/self.realComponentsCurrentMass, 1,2);
								end;
							end;
							
							--take into account total mass : to be less "harsh" when empty
							--wheelMotorBrakeForce = wheelMotorBrakeForce * 0.5;-- * Utils.clamp(self.realTotalMovingMassOnTheWheels/self.realComponentsCurrentMass, 1,2);
							mbfFx = mbfFx * 4 * math.min(0.25, self.realOverSpeed^2); -- realOverSpeed = 0.5mph (1.8kph) => max brake force
							totalMbfFx = totalMbfFx + mbfFx;
							nbMbfWheels = nbMbfWheels + 1;
							
							newWheelMotorBrakeForce = mbfFx * maxWheelMotorBrakeForce * self.realMovingDirection;
							
							if math.abs(newWheelMotorBrakeForce)>math.abs(wheel.realLastMotorBrakeForce) then
								--newWheelMotorBrakeForce = wheel.realLastMotorBrakeForce * 0.95 + newWheelMotorBrakeForce * 0.05;
								--20140220 - take into account dt
								--decrease motor brake force responsiveness at low speed
								local incFx = 0.5 * spd * dt;
								--print(self.time .. " incFx = " .. tostring(incFx) .. " - dt="..tostring(dt).. " - spd="..tostring(spd));
								newWheelMotorBrakeForce = wheel.realLastMotorBrakeForce * (1-incFx) + newWheelMotorBrakeForce * incFx;
							--else
								--newWheelMotorBrakeForce = wheel.realLastMotorBrakeForce * 0.9 + wheelMotorBrakeForce * 0.1;
								--newWheelMotorBrakeForce = wheelMotorBrakeForce;
							end;
							
							local xT, yT, zT = localDirectionToWorld(self.components[1].node, 0, 0, -newWheelMotorBrakeForce * dt);
							addForce(wheel.realRootComponent.node, xT, yT, zT, wheel.netInfo.x, wheel.realY-wheel.radius, wheel.netInfo.z, true);							
							
							
							
							if Vehicle.debugRendering then
								--print(self.time .. "RealisticMotorized.updateWheelsForces mbfFx : " .. tostring(mbfFx) .. " / maxMbf = " .. tostring(maxWheelMotorBrakeForce));
								wheel.realDebugMotorBrakeForce = newWheelMotorBrakeForce;
							end;
							
						end;
					end;
				end; -- end no power		
				
			end;--end drive wheel
			
			wheel.realLastMotorBrakeForce = newWheelMotorBrakeForce;
			if nbMbfWheels>0 then
				self.realCurrentMotorBrakeForceFx = totalMbfFx / nbMbfWheels;			
			end;
			
		end; -- end "for k,wheel in pairs"	
	end; -- end self.realClutchEngaged
	
	
	

end;


function RealisticMotorized.updatePowerBoost(self)

	-- boost management	
	local totalBoost = 0;
	
	--boost from speed 
	if self.realSpeedBoostAddPwr>0 then
		if self.realGroundSpeed*3.6>self.realSpeedBoostMinSpeed then
			totalBoost = self.realSpeedBoostAddPwr;
		end;
	end;
	
	if self.realImplementNeedsBoostAddPwr>0 then
		if self.realCurrentTotalPowerConsumption>self.realImplementNeedsBoostMinPowerCons then
			totalBoost = totalBoost + self.realImplementNeedsBoostAddPwr;
		end;
	end;
	
	totalBoost = math.min(self.realMaxBoostAddPwr, totalBoost);
	
	--if totalBoost>self.realCurrentPowerBoost then
		self.realCurrentPowerBoost = 0.98 * self.realCurrentPowerBoost + 0.02 * totalBoost;
	--else
	--	self.realCurrentPowerBoost = totalBoost;
	--end;
	
	--print("RealisticMotorized.updatePowerBoost : speedBoost=" .. self.realSpeedBoostAddPwr .. " minSpeed=" .. self.realSpeedBoostMinSpeed .. " totalboost=" .. totalBoost .. " groundspeed=" .. self.realGroundSpeed .. " realCurrentPowerBoost="..tostring(self.realCurrentPowerBoost));

end;






--gearRatedSpeed in m.s-1
function RealisticMotorized:setManualGear3(activateManualGear, gearRatedSpeed, noEventSend)

	if self.isServer then

		self.realManualGearSet = activateManualGear;	
		
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				--send the new value to all clients
				g_server:broadcastEvent(RealisticMotorizedSetManualGearEvent:new(self, activateManualGear), nil, nil, self);
			end;
		end;
			
		if activateManualGear then
		
			if gearRatedSpeed<=0 then
				RealisticUtils.printWarning("RealisticMotorized:setManualGear3", self.realVehicleName .. " : incorrect parameters for 'setManualGear3'.", false);
				self.realManualGearSet = false;
			else
				self.realEngine.currentGearRatedSpeed = gearRatedSpeed;
				self.realEngine.ratedEngineRpmWheelSpeedRatio = self.realEngine.ratedRpm / self.realEngine.currentGearRatedSpeed;
				
				--wheel force (KN) / engine torque (Nm) = engine rpm / (wheel speed (m.s-1) * 9548.8)
				--wheel force (KN) =  engine torque (Nm) * engine rpm / (wheel speed (m.s-1) * 9548.8)
				self.realEngine.engineMaxTorqueToWheelPossible = self.realEngine.maxTorque * self.realEngine.ratedEngineRpmWheelSpeedRatio / 9548.8;
		
			end;
							
		end;
		
	else	
		RealisticUtils.printWarning("RealisticMotorized:setManualGear2", self.realVehicleName .. " : setManualGear must be called server side only.", false);		
	end;

end;







--old function - please use "setManualGear3" now
function RealisticMotorized:setManualGear2(activateManualGear, gearMinSpeed, gearIdleSpeed, gearMinSpeedForMaxPower, gearMaxSpeed, noEventSend)

	if self.isServer then

		self.realManualGearSet = activateManualGear;	
		
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				--send the new value to all clients
				g_server:broadcastEvent(RealisticMotorizedSetManualGearEvent:new(self, activateManualGear), nil, nil, self);
			end;
		end;
			
		if activateManualGear then
			self.realManualGearIdleRpmRatio = gearIdleSpeed/gearMaxSpeed;
			
			--check that gearMaxSpeed>gearMinSpeedForMaxPower>gearIdleSpeed>gearMinSpeed
			if gearMaxSpeed<=0 or gearMinSpeedForMaxPower>=gearMaxSpeed or gearIdleSpeed>=gearMinSpeedForMaxPower or gearMinSpeed>=gearIdleSpeed then
				RealisticUtils.printWarning("RealisticMotorized:setManualGear2", self.realVehicleName .. " : incorrect parameters for 'setManualGear2'.", false);
				self.realManualGearSet = false;
			else
				self.realManualGearMaxSpeed = gearMaxSpeed; --m/s
				self.realManualGearMinSpeedForMaxPower = gearMinSpeedForMaxPower;--m/s
				self.realManualGearIdleSpeed = gearIdleSpeed;
				self.realManualGearMinSpeed = gearMinSpeed; --m/s
				
				self.realManualGearIdleSpeedFx = (self.realManualGearIdleSpeed-self.realManualGearMinSpeed)/(self.realManualGearMaxSpeed-self.realManualGearMinSpeed);
				
				self.realManualGearUseOldModel = false;
			end;			
							
		end;
		
	else	
		RealisticUtils.printWarning("RealisticMotorized:setManualGear2", self.realVehicleName .. " : setManualGear must be called server side only.", false);		
	end;

end;


--old function - please use "setManualGear3" now
function RealisticMotorized:setManualGear(activateManualGear, motorMinMaxRpmRatio, gearMinSpeedForMaxPower, gearMaxSpeed, noEventSend)

	if self.isServer then

		self.realManualGearSet = activateManualGear;	
		
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				--send the new value to all clients
				g_server:broadcastEvent(RealisticMotorizedSetManualGearEvent:new(self, activateManualGear), nil, nil, self);
			end;
		end;
			
		if activateManualGear then
			self.realManualGearIdleRpmRatio = motorMinMaxRpmRatio;
			if gearMinSpeedForMaxPower~=nil and gearMaxSpeed~=nil then
				self.realManualGearMaxSpeed = gearMaxSpeed; --m/s
				self.realManualGearMinSpeedForMaxPower = gearMinSpeedForMaxPower;--m/s
				self.realManualGearIdleSpeed = motorMinMaxRpmRatio * gearMaxSpeed;
				self.realManualGearMinSpeed = 0.99*self.realManualGearIdleSpeed; --m/s
				
				self.realManualGearIdleSpeedFx = (self.realManualGearIdleSpeed-self.realManualGearMinSpeed)/(self.realManualGearMaxSpeed-self.realManualGearMinSpeed);
				
				self.realManualGearUseOldModel = true;
			end;
		end;
		
	else	
		RealisticUtils.printWarning("RealisticMotorized:setManualGear", self.realVehicleName .. " : setManualGear must be called server side only.", false);		
	end;

end;









function RealisticMotorized:realSetReverseGearActive(reverseGearActive, noEventSend)

	if reverseGearActive~=self.realReverseGearEngaged then
		--[[ we are now using RealisticSteerable:realSetShuttleDisplayDirection
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				g_server:broadcastEvent(RealisticSteerableToggleReverseGearEngagedEvent:new(self, reverseGearActive), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(RealisticSteerableToggleReverseGearEngagedEvent:new(self, reverseGearActive));
			end;
		end;]]
		
		self.realReverseGearEngaged = reverseGearActive;

		--print(self.time .. " - RealisticMotorized:realSetReverseGearActive - realReverseGearEngaged is now ="..tostring(reverseGearActive) .. " - self.lastAcceleration="..tostring(self.lastAcceleration) .. " - self.realShuttleDirection="..tostring(self.realShuttleDirection));
					
	end;

end;


function RealisticMotorized:onEnter(isControlling)

	if self.isServer then
		if not self.isControlled then
			--reset all the wheels motorized parameters
			for _, wheel in pairs(self.wheels) do
				WheelsUtil.realResetMotorizedWheel(wheel);
			end;
		end;
	end;
	
end;

--20140802 - we want to be sure the full brake power is applyied when leaving a vehicle
function RealisticMotorized:onLeave()

	--print(self.time .. " - " .. self.realVehicleName .. " - onLeave");

	if self.isServer then
		if not self.isControlled then
			local fixedDt = 1/60;
			WheelsUtil.updateWheelsPhysicsMR(self, fixedDt, self.lastSpeed, 0, true, self.requiredDriveMode);
		end;
	end;
	
end;


function RealisticMotorized.regulatorGetFx(self)


	--local realSpd = self.realAvgTouchingMotorizedWheelSpd; --self.realGroundSpeed*self.movingDirection;
	local spd = self.realAvgTouchingMotorizedWheelSpd; --self.realGroundSpeed; --20140220-use the motorized wheel speed instead of GPS speed
	local spdWanted = 0;
	local forceFx = 1;
	local isFixedGearRatio = false;
	
	local isAiDriven = self:realGetIsAiDriven();
	
	--20140628 - fix the problem of the AI at very low speed = the vehicle doesn't move due to heavy slippage, but the AI does not try to increase the power to the wheels since the "realAvgTouchingMotorizedWheelSpd" is sufficient.
	if isAiDriven then
		spd = self.realGroundSpeed * self.realMovingDirection;
	end;
	
	--local wrongDirection = 0;
	if self.realReverseGearEngaged then		
		if spd>0 then
			--wrongDirection = 1;
			spd = 0;
		end;
		spdWanted = self.realMaxReverseSpeed/3.6;
	else		
		if spd<0 then
			--wrongDirection = -1;
			spd = 0;		
		end;
		spdWanted = self.realMaxVehicleSpeed/3.6;
	end;
	spd = math.abs(spd);
			
	if self.motor.speedLevel>0 then	
		--local test = isAiDriven;
		--print(self.time .. " RealisticMotorized.updateWheelsForces isAiDriven = " .. tostring(test));
		if isAiDriven then
			spdWanted = math.min(spdWanted, self.motor.realSpeedLevelsAI[self.motor.speedLevel]/3.6);
			--print(self.time .. " aidriven - spdLevel / spdWanted = " .. tostring(self.motor.speedLevel) .. " / " .. tostring(self.motor.realSpeedLevelsAI[self.motor.speedLevel]));
		else
			spdWanted = math.min(spdWanted, self.motor.realSpeedLevels[self.motor.speedLevel]/3.6);
		end;
		--print(self.time .. " RealisticMotorized.updateWheelsForces speedLevel / spdWanted = " .. self.motor.speedLevel .. " / " .. spdWanted*3.6);		
	elseif self.realManualGearSet then
		if self.realOldManualGearBoxPresent then
			--20140213 - take into account the speedLimiter feature		
			spdWanted = math.min(self.realManualGearMaxSpeed, self.realManualGearSpeedLimiter);		
		else
			spdWanted = math.min(self.realEngine.currentGearRatedSpeed*self.realEngine.ratedToMaxRpmRatio, self.realEngine.speedLimiter);
		end;
		--spd = realSpd;
		isFixedGearRatio = true;
	end;

	--print(self.time .. " test RealisticMotorized.updateWheelsForces  spdWanted = " .. tostring(spdWanted));

	--if self.realThrottleIsManual and self.realManualGearSet then	
	if self.realManualGearSet then
		if self.realOldManualGearBoxPresent then
			spdWanted = RealisticUtils.linearFx(math.abs(self.lastAcceleration), self.realManualGearIdleRpmRatio, 1) * spdWanted;
		else
			spdWanted = math.abs(self.lastAcceleration) * spdWanted;
			--print(self.time .. " spdWanted = "..tostring(spdWanted));
		end;
	else
		if self.motor.speedLevel==0 then
			if self.axisForwardIsAnalog then
				--smooth the driving experience with pedal to allow us to reach reduced speeds
				spdWanted = math.abs(self.lastAcceleration)^2 * spdWanted;
			else
				spdWanted = math.abs(self.lastAcceleration) * spdWanted;
			end;
		end;
	end;

	--print(self.time .. " test RealisticMotorized.updateWheelsForces  spdWanted = " .. tostring(spdWanted));

	--local spdFadeRange = 1/3.6;-- 1 km/h

	--[[
	local spdFadeRange = 0.1*spdWanted;-- 10%
	local minFadeRange = 1/3.6; -- 1kmh
	--10% of speed is not enough when running at low speed
	if spdFadeRange<minFadeRange then -- fade < 1 km/h
		local diff = minFadeRange - spdFadeRange;
		spdWanted = spdWanted - 0.4*diff;
		spdFadeRange = minFadeRange;
	end;

	local spdMax = spdWanted + spdFadeRange;
			
	if spd>spdWanted then
		forceFx = Utils.clamp((spdMax-spd)/spdFadeRange, 0, 1)^2;
	end;]]
	
	
	
	
	--local spdFadeRange = 0.5;-- 0.5 m.s-1 = 1.8km/h
	local spdFadeRange = 0;
	
	if isFixedGearRatio then
		if self.realOldManualGearBoxPresent then
			--20140213 - take into account the speedLimiter feature		
			spdFadeRange = self.realManualGearRpmFadingPercent*self.realManualGearMaxSpeed*math.abs(self.lastAcceleration);	
			spdFadeRange = math.min(spdFadeRange, math.max(2/3.6, self.realManualGearSpeedLimiter - self.realManualGearMaxSpeed));
			--spdFadeRange = self.realManualGearRpmFadingPercent*math.min(self.realManualGearMaxSpeed, self.realManualGearSpeedLimiter);	
		else
			--if self.realEngine.speedLimiter==spdWanted then
			--spdFadeRange = (self.realEngine.ratedToMaxRpmRatio-1)*spdWanted;
			--end;
			
			--spdFadeRange = 2;-- + 0.5 * (1 - self.realMotorLoadS);
			--spdWanted = spdWanted - spdFadeRange * (1 - math.abs(self.lastAcceleration));			
			
			spdFadeRange = 0.05;
			spdWanted = spdWanted - spdFadeRange;-- * (1 - math.abs(self.lastAcceleration));
		end;
	else
		if isAiDriven then
			--local test = spdWanted;
			spdFadeRange = 1;
			spdWanted = spdWanted - 0.75*(1-self.realRegulatorFxS^0.5)*spdFadeRange;
			--spdWanted = spdWanted - 0.75*spdFadeRange;
			--print(self.time .. " spdWanted before="..tostring(test*3.6).." / after="..tostring(spdWanted*3.6));
		else
			spdFadeRange = 0.3 + 0.5 * (1 - self.realMotorLoadS);		
			spdWanted = spdWanted - 0.2 * (1 - self.realMotorLoadS);
		end;
	end;
	
	self.realMaxSpeedWanted = spdWanted+spdFadeRange;
	 
	if spd>spdWanted then
		--forceFx = Utils.clamp(((spdWanted + spdFadeRange)-spd)/spdFadeRange, 0, 1)^(3-2.5*self.realMotorLoadS);
		self.realOverSpeed = spd-(spdWanted + spdFadeRange);
		forceFx = Utils.clamp(-self.realOverSpeed/spdFadeRange, 0, 1)^2;
		
		--print(self.time .. " spd="..tostring(spd*3.6) .. " - spdWanted=" .. tostring(spdWanted*3.6) .. " - realOverSpeed = " .. tostring(self.realOverSpeed*3.6) .. " - forceFx="..tostring(forceFx) .. " - spdFadeRange="..tostring(spdFadeRange));
		
	else
		self.realOverSpeed = 0;
	end;
	
	--if not self.isEntered then
		--print(self.time .. " RealisticMotorized.regulatorGetFx - spdWanted="..tostring(spdWanted*3.6) .. " /fadeRange="..tostring(spdFadeRange*3.6).." / overSpeed="..tostring(self.realOverSpeed).." / forceFx="..tostring(forceFx));
	--end;
		
	--print(self.time .. " speedlevel = " .. tostring(self.motor.speedLevel) .. "/".. tostring(spdWanted*3.6) .. "kph / acc=" .. tostring(self.lastAcceleration) .. " / regulatorFx = " .. tostring(forceFx) .. " / overSpeed = " .. tostring(self.realOverSpeed));
	
	if Vehicle.debugRendering and self.isServer then
		self.realDebugTargetSpeed = spdWanted;
	end;
	
	
	return forceFx;

end;



--only to support older fixed ratio gear box (before the "realEngine" section exist)
function RealisticMotorized:loadRealManualGearXML(xmlFile)

	if hasXMLProperty(xmlFile, "vehicle.realManualGearBox") then
	
		self.realOldManualGearBoxPresent = true;

		self.realManualGearIdlePowerPercent = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#idlePowerPercent"), 0.33);
		if self.realManualGearIdlePowerPercent<=0 then
			RealisticUtils.printWarning("RealisticMotorized:load", self.realVehicleName .. " : vehicle.realManualGearBox#idlePowerPercent must be greater than 0.", false);
			self.realManualGearIdlePowerPercent = 0.01;
		end;
		
		self.realManualGearMaxTorquePowerPercent = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#maxTorquePowerPercent"), 0.66);
		if self.realManualGearMaxTorquePowerPercent<=self.realManualGearIdlePowerPercent then
			RealisticUtils.printWarning("RealisticMotorized:load", self.realVehicleName .. " : vehicle.realManualGearBox#maxTorquePowerPercent must be greater than idlePowerPercent.", false);
			self.realManualGearMaxTorquePowerPercent = math.min(1, self.realManualGearIdlePowerPercent * 2);
		end;
		
		
		self.realManualGearAccelerationSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#accelerationSpeed"), self.maxAccelerationSpeed*1000)*0.001;
		self.realManualGearDecelerationSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#decelerationSpeed"), self.decelerationSpeed*1000)*0.001;
			
		self.realManualGearRpmFadingPercent = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#rpmFadingPercent"), 0.1); -- fade range = 10% of max rpm
		
		--20140213 - adding a speedLimiter feature for trucks or some tractors with "eco" speed => speed electronically limited
		self.realManualGearSpeedLimiter = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#speedLimiter"),9999)/3.6; -- kph to mph
		
		self.realManualGearNeutralRpmRatio = 1 + self.realManualGearRpmFadingPercent; -- to be specified by the transmission gearbox script => 1 + (fade rpm range / (nominal rpm - idle rpm))
		
		self.realManualGearEngineRevUpTimeMs = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#engineRevUpTimeMs"), 1400);
		self.realManualGearEngineRevDownTimeMs = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#engineRevDownTimeMs"), 2000);
		
		self.realManualGearEngineSoundPitchOffsetFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realManualGearBox#engineSoundPitchOffsetFx"), 0.75);
		
	else
	
		self.realOldManualGearBoxPresent = false;
	
	end;

end;


function RealisticMotorized:developmentReloadFromXML(xmlFile)
	RealisticMotorized.load(self, xmlFile);
end;




--[[
DEPRECATED

function RealisticMotorized:updateMotorRpmRatio()

	local rpmRatio = 0;

	--neutral
	if (self.realShuttleIsManual and self.realShuttleDirection==0) or not self.realClutchEngaged then
		rpmRatio = self.realMotorLoadS;
	 		
		if self.realThrottleIsManual then
			rpmRatio = math.max(rpmRatio, self.realThrottleCurrentValue);	
		end;
		rpmRatio = math.max(rpmRatio, self.lastAcceleration);
		
		if rpmRatio>0 then
			rpmRatio = 1.1 * rpmRatio; --at neutral, if we give some fuel to the motor, it reaches higher rpm than "asked"
		end;
		rpmRatio = RealisticUtils.linearFx(rpmRatio, self.realManualGearIdleRpmRatio, 1);
		
		--print(self.time .. " RealisticMotorized:updateMotorRpmRatio  realThrottleCurrentValue = " .. tostring(self.realThrottleCurrentValue) .. " / self.lastAcceleration = " .. tostring(self.lastAcceleration) .. " / realManualGearIdleRpmRatio = " .. tostring(self.realManualGearIdleRpmRatio));
		
		
		rpmRatio = 0.8 * self.realManualGearMotorRpmRatio + 0.2 * rpmRatio; --avoid going from 1 to 0 when braking
		--self.realManualGearClutchIsSlipping = false;
	else				
		rpmRatio = math.abs(self.realAvgMotorizedWheelSpd/self.realManualGearMaxSpeed);				
	end;

	self.realManualGearMotorRpmRatio = rpmRatio;

end;


]]











