--	WoodCrusher	
--	This script implements the destruction of trunks
--		- incorporates with WoodClaw3 ?
--		- incorporates with "foldable" -> opening/closing of input gate
--
--	author:		fruktor, rafftnix
--	date:		18.12.2013
--	version:	0.1		- 	initial implementation
--
--	rights:		Copyright @ fruktor, rafftnix and BM-Modding
--				free for non-commercial usage

WoodCrusher = {};

function WoodCrusher.prerequisitesPresent(specializations)
    return true;
end;

function WoodCrusher:load(xmlFile)
	self.getIsFoldAllowed = Utils.overwrittenFunction(self.getIsFoldAllowed, WoodCrusher.getIsFoldAllowed);
	self.findTrailerToUnload = WoodCrusher.findTrailerToUnload;
	self.findTrailerRaycastCallback = WoodCrusher.findTrailerRaycastCallback;	
	self.trunkTriggerCallback = WoodCrusher.trunkTriggerCallback;
	self.trunkDeleteTriggerCallback = WoodCrusher.trunkDeleteTriggerCallback;
	self.setIsTurnedOn = SpecializationUtil.callSpecializationsFunction("setIsTurnedOn");
	self.createHeapTipTrigger = SpecializationUtil.callSpecializationsFunction("createHeapTipTrigger");
	
	self.isTurnedOn = false;
	
	self.pipeRaycastFoundTerrain = false;
	self.pipeRaycastFoundTerrainPosition = {0,0,0};
	
	self.wCrusher = {};

	self.wCrusher.trunkTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher#trunkTrigger"));
	self.wCrusher.trunkDeleteTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher#trunkDeleteTrigger"));
	self.wCrusher.contactNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher#contactNode"));
	self.wCrusher.contactDrumNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher#contactDrumNode"));
	self.wCrusher.branchRemovePoint = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher#branchRemovePointIndex"));
	self.wCrusher.branchRemoveDist = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.woodcrusher#branchRemoveDist"), 1.5);
	
	if self.isServer then
		addTrigger(self.wCrusher.trunkTrigger, "trunkTriggerCallback", self);
		addTrigger(self.wCrusher.trunkDeleteTrigger, "trunkDeleteTriggerCallback", self);
	end;
	
	self.wCrusher.trunksInTrigger = {};
	self.wCrusher.trunksInDeleteTrigger = {};
	
	self.wCrusher.woodChipInputRef = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher#woodChipInputRef"));

	self.wCrusher.shader = {};
	self.wCrusher.shader.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher.shader#node"));
	self.wCrusher.shader.para = getXMLString( xmlFile, "vehicle.woodcrusher.shader#parameter" );
	self.wCrusher.shader.speed = getXMLFloat( xmlFile, "vehicle.woodcrusher.shader#speed" );
	
	self.wCrusher.drums = {};
	local i=0;
	while true do
		local str = getXMLString(xmlFile, string.format("vehicle.woodcrusher.drum(%d)#node",i));
		if str == nil then break; end;
		local e = {};
		e.node = Utils.indexToObject(self.components, str);
		e.speed = getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.drum(%d)#speed",i));
		e.curSpeed = 0;
		table.insert(self.wCrusher.drums, e);
		i=i+1;
	end;
	
	self.wCrusher.persistentDrums = {};
	local i=0;
	while true do
		local str = getXMLString(xmlFile, string.format("vehicle.woodcrusher.persistentDrum(%d)#node",i));
		if str == nil then break; end;
		local e = {};
		e.node = Utils.indexToObject(self.components, str);
		e.speed = getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.persistentDrum(%d)#speed",i));
		e.curSpeed = 0;
		table.insert(self.wCrusher.persistentDrums, e);
		i=i+1;
	end;
	
	self.pipeRaycastNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher.pipe#raycastNode"));
	self.pipeRaycastDistance = getXMLFloat(xmlFile, "vehicle.woodcrusher.pipe#maxDistance");
	self.pipeUnloadingSpeed = getXMLFloat(xmlFile, "vehicle.woodcrusher.pipe#unloadingSpeed");

	-- pipe particle system 
	local key = "vehicle.woodcrusher.pipeParticleSystem";
	local currentPS = {};
	local particleNode = Utils.loadParticleSystem(xmlFile, currentPS, key, self.components, false, "$data/vehicles/particleSystems/wheatParticleSystem.i3d", self.baseDirectory, self.components[1].node);
	for _, v in ipairs(currentPS) do
		local normalSpeed,tangentSpeed = getParticleSystemAverageSpeed(v.geometry);
		v.speed = math.sqrt(normalSpeed*normalSpeed + tangentSpeed*tangentSpeed);
		v.originalLifespan = getParticleSystemLifespan(v.geometry);
	end
	self.pipeParticleSystem = currentPS;
	self.pipeParticleSystemActive = false;
	self.pipeParticleSystemExtraDistance = 1.5;
	self.pipeUnloadingDistance = 0;
	self.pipeUnloadingDistanceSet = 0;
	self.trailerFoundDistance = 0; -- required for shader tipping
	
	self.wCrusher.fullStain = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.woodcrusher.stain#fullStain"), 200);
	self.wCrusher.overStain = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.woodcrusher.stain#overstain"), 390);
	self.wCrusher.maxConveyorSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.woodcrusher#maxConveyorSpeed"), 0.02);

	local key = "vehicle.woodcrusher.drumLowParticleSystem";
	local currentPS = {};
	local particleNode = Utils.loadParticleSystem(xmlFile, currentPS, key, self.components, false, "$data/vehicles/particleSystems/wheatParticleSystem.i3d", self.baseDirectory, self.components[1].node);
	self.drumLowParticleSystem = currentPS;
	local key = "vehicle.woodcrusher.drumHighParticleSystem";
	local currentPS = {};
	local particleNode = Utils.loadParticleSystem(xmlFile, currentPS, key, self.components, false, "$data/vehicles/particleSystems/wheatParticleSystem.i3d", self.baseDirectory, self.components[1].node);
	self.drumHighParticleSystem = currentPS;
	
	self.drumParticleSystemsActive = false;
	self.drumParticleSystemsActiveSet = false;

	self.wCrusher.notOverstrained = true;
	self.wasAbove = false;
	self.create_HeapTipTrigger = false;
	
	self.wCrusher.sounds = {};
	for i,name in pairs( {"turnOn", "turnOff", "idle", "run"} ) do
		
		local file = getXMLString(xmlFile, string.format("vehicle.woodcrusher.%sSound#file", name));
		local filename = Utils.getFilename(file, self.baseDirectory);
		
		local radius = Utils.getNoNil( getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#radius", name)), 50 );
		local innerRadius = Utils.getNoNil( getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#innerRadius", name)), 10);
		local volume = Utils.getNoNil( getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#volume", name)), 1 );
		local loops = Utils.getNoNil( getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#loops", name)), 1);
		
		local sound = createAudioSource(name, filename, radius, innerRadius, volume, loops);
		link(self.components[1].node, sound);
		setVisibility(sound, false);	
		
		local volumeMin = Utils.getNoNil(getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#volumeMin", name)), 1);
		local volumeMax = Utils.getNoNil(getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#volumeMax", name)), 1);
		local pitchMin = Utils.getNoNil(getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#pitchMin", name)), 1);
		local pitchMax = Utils.getNoNil(getXMLFloat(xmlFile, string.format("vehicle.woodcrusher.%sSound#pitchMax", name)), 1);
		
		local e={};
		e.sound = sound;
		e.a = false;
		e.sample = getAudioSourceSample(sound);
		e.sampleDuration = getSampleDuration(e.sample);
		e.volume = volume;
		e.volumeMin = volumeMin;
		e.volumeMax = volumeMax;
		e.volumeCur = 0;				
		e.pitchMin = pitchMin;
		e.pitchMax = pitchMax;
		self.wCrusher.sounds[name] = e;
	end;
	self.wCrusher.sounds["turnOff"].a = true;	-- disables playback during start
	self.wCrusher.sounds["turnOff"].stopTime = 0;
	self.wCrusher.sounds["turnOn"].startTime = -1;
	self.wCrusher.sounds["idle"].valPitch = 1.0;
	self.wCrusher.sounds["idle"].valVolume = 1.0;
	self.wCrusher.sounds["run"].valPitch = 1.0;
	self.wCrusher.sounds["run"].valVolume = 1.0;

	self.woodCrusherDirtyFlag = self:getNextDirtyFlag();
	self.fillLevel = 0;
	self.fillLevelPS = 0;
	
	if hasXMLProperty(xmlFile, "vehicle.woodcrusher.drumCollisionPairs") then
		local drumNodeId = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.woodcrusher.drumCollisionPairs#drumIndex"));
		local otherIndicesStr = getXMLString(xmlFile, "vehicle.woodcrusher.drumCollisionPairs#otherIndices");
		local indices = Utils.splitString(" ", otherIndicesStr);
		for k, v in pairs(indices) do
			setPairCollision(Utils.indexToObject(self.components, v), drumNodeId, false);	
		end;
	end;
	
	if hasXMLProperty(xmlFile, "vehicle.woodcrusher.syndeticCollisionPairs") then
		local indicesStr = getXMLString(xmlFile, "vehicle.woodcrusher.syndeticCollisionPairs#indices");
		local indices = Utils.splitString(" ", indicesStr);

		for i,n1 in pairs(indices) do
			for j,n2 in pairs(indices) do
				if i ~= j then
					setPairCollision(Utils.indexToObject(self.components, n1), Utils.indexToObject(self.components, n2), false);	
				end;
			end;
		end;
	end;
	
	if g_currentMission.treeManager ~= nil then
		table.insert(g_currentMission.treeManager.tutorialGuisToOpen, "mobileChipper");
	end;
	
	--# GUI/HUD
	self.wCrusher.gui = {};
	self.wCrusher.gui.x = 0.9;
	self.wCrusher.gui.y = 0.2;
	self.wCrusher.gui.w = 128/g_screenWidth;
	self.wCrusher.gui.h = 128/g_screenHeight;
	self.wCrusher.guiCrusherFL = Overlay:new("crusherFL", Utils.getFilename(getXMLString(xmlFile,"vehicle.woodcrusher.gui.fillLevelImg#fileName"), self.baseDirectory), self.wCrusher.gui.x, self.wCrusher.gui.y+2*self.wCrusher.gui.h, self.wCrusher.gui.w, self.wCrusher.gui.h);
	self.wCrusher.guiCrusherRed = Overlay:new("crusherRed", Utils.getFilename(getXMLString(xmlFile,"vehicle.woodcrusher.gui.progressRed#fileName"), self.baseDirectory), self.wCrusher.gui.x, self.wCrusher.gui.y+2*self.wCrusher.gui.h, self.wCrusher.gui.w, self.wCrusher.gui.h);	
	self.wCrusher.guiCrusherYellow = Overlay:new("crusherYellow", Utils.getFilename(getXMLString(xmlFile,"vehicle.woodcrusher.gui.progressYellow#fileName"), self.baseDirectory), self.wCrusher.gui.x, self.wCrusher.gui.y+2*self.wCrusher.gui.h, self.wCrusher.gui.w, self.wCrusher.gui.h);
	self.wCrusher.guiCrusherYellowTime = 0;
	self.wCrusher.guiCrusherYellowSet = false;
	self.wCrusher.guiCrusherGreen = Overlay:new("crusherGreen", Utils.getFilename(getXMLString(xmlFile,"vehicle.woodcrusher.gui.progressGreen#fileName"), self.baseDirectory), self.wCrusher.gui.x, self.wCrusher.gui.y+2*self.wCrusher.gui.h, self.wCrusher.gui.w, self.wCrusher.gui.h);
end;

function WoodCrusher:delete()
	if self.wCrusher.trunkTrigger ~= nil then
		removeTrigger(self.wCrusher.trunkTrigger);
	end;
	if self.wCrusher.trunkDeleteTrigger ~= nil then
		removeTrigger(self.wCrusher.trunkDeleteTrigger);
	end;
	if self.pipeParticleSystem ~= nil then
		Utils.deleteParticleSystem(self.pipeParticleSystem);
	end;
	if self.drumLowParticleSystem ~= nil then
		Utils.deleteParticleSystem(self.drumLowParticleSystem);
	end;
	if self.drumHighParticleSystem ~= nil then
		Utils.deleteParticleSystem(self.drumHighParticleSystem);
	end;
	for i,j in pairs(self.wCrusher.sounds) do
		if j.sound ~= nil then
			delete(j.sound);
		end;
	end;
	if self.wCrusher.guiCrusherFL ~= nil then
		self.wCrusher.guiCrusherFL:delete();
	end;
	if self.wCrusher.guiClawClosed ~= nil then
		self.wCrusher.guiClawClosed:delete();
	end;
	if self.wCrusher.guiCrusherYellow ~= nil then
		self.wCrusher.guiCrusherYellow:delete();
	end;
	if self.wCrusher.guiCrusherGreen ~= nil then
		self.wCrusher.guiCrusherGreen:delete();
	end;	
end;

function WoodCrusher:readStream(streamId, connection)
	self:setIsTurnedOn(streamReadBool(streamId), true);
end;

function WoodCrusher:writeStream(streamId, connection)	
	streamWriteBool(streamId, self.isTurnedOn);
end;

function WoodCrusher:readUpdateStream(streamId, timestamp, connection)
	if connection.isServer then
		local hasUpdate = streamReadBool(streamId);
		if hasUpdate then
			self.fillLevel = streamReadFloat32(streamId);
			self.pipeUnloadingDistance = streamReadFloat32(streamId);
			self.pipeParticleSystemsActive = streamReadBool(streamId);
			self.drumParticleSystemsActive = streamReadBool(streamId);
		end;
	end;
end;

function WoodCrusher:writeUpdateStream(streamId, connection, dirtyMask)
	if not connection.isServer then
		if streamWriteBool(streamId, bitAND(dirtyMask, self.woodCrusherDirtyFlag) ~= 0) then
			streamWriteFloat32(streamId, self.fillLevel);
			streamWriteFloat32(streamId, self.pipeUnloadingDistance);
			streamWriteBool(streamId, self.pipeParticleSystemsActive);
			streamWriteBool(streamId, self.drumParticleSystemsActive);
		end;
	end;
end;

function WoodCrusher:mouseEvent(posX, posY, isDown, isUp, button)
end;

function WoodCrusher:keyEvent(unicode, sym, modifier, isDown)
end;

function WoodCrusher:update(dt)
	if self.attacherVehicle ~= nil then
		if self.attacherVehicle.isMotorStarted ~= nil then 	
			if not self.attacherVehicle.isMotorStarted and self.isTurnedOn then 	
				self:setIsTurnedOn(false);
			end;			
		end;
	end;
	
	if self:getIsActiveForInput() then
		if self.foldAnimTime == 0 then
			local allowed = false;
			if self.attacherVehicle ~= nil then
				if self.attacherVehicle.isMotorStarted ~= nil then 
					allowed = self.attacherVehicle.isMotorStarted;
				end;
			end;
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
				if allowed then
					self:setIsTurnedOn( not self.isTurnedOn );
				end;
			end;
			if self.isTurnedOn then
				if self.pipeRaycastFoundTerrain and g_currentMission.modDirectoryOfHeapTipTrigger ~= nil and self.trailerFound == nil and self.triggerFound == nil then
					if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA3) then
						if self.isServer then
							self.create_HeapTipTrigger = true;
						else
							g_client:getServerConnection():sendEvent(HeapTipTriggerMPEvent:new(self));
						end;
					end;				
				end;
			end;
		end;			
	end;
	
	if self.fillLevel <= self.wCrusher.fullStain and self.wasAbove then
		self.wCrusher.notOverstrained = true;
		self.wasAbove = false;
	end;
	if self.fillLevel > self.wCrusher.overStain then
		self.wCrusher.notOverstrained = false;
		self.wasAbove = true;
	end;	
	
	if self.isClient and self.isTurnedOn then
		for i,d in pairs(self.wCrusher.persistentDrums) do
			if d.speed > 0 then
				d.curSpeed = math.min( d.speed, d.curSpeed + d.speed*(dt / self.wCrusher.sounds["turnOn"].sampleDuration) );
			else
				d.curSpeed = math.max( d.speed, d.curSpeed + d.speed*(dt / self.wCrusher.sounds["turnOn"].sampleDuration) );
			end;
			rotate(d.node, 0,0,d.curSpeed*(dt/1000) );
		end;	
		if self.wCrusher.notOverstrained then 
			for i,d in pairs(self.wCrusher.drums) do
				if d.speed > 0 then
					d.curSpeed = math.min( d.speed, d.curSpeed + d.speed*(dt / self.wCrusher.sounds["turnOn"].sampleDuration) );
				else
					d.curSpeed = math.max( d.speed, d.curSpeed + d.speed*(dt / self.wCrusher.sounds["turnOn"].sampleDuration) );
				end;
				rotate(d.node, 0,0,d.curSpeed*(dt/1000) );
			end
			if self.wCrusher.shader.a ~= true then
				setShaderParameter( self.wCrusher.shader.node, self.wCrusher.shader.para, self.wCrusher.shader.speed, 0, 0, 0, false );
				self.wCrusher.shader.a = true;
			end;
			self.drumsStopped = false;
		else
			if self.wCrusher.shader.a == true then
				setShaderParameter( self.wCrusher.shader.node, self.wCrusher.shader.para, 0, 0, 0, 0, false );
				self.wCrusher.shader.a = false;
			end;
			self.drumsStopped = true;
		end;
	else
		for i,d in pairs(self.wCrusher.persistentDrums) do
			if d.speed > 0 then
				d.curSpeed = math.max( 0, d.curSpeed - d.speed*(dt / self.wCrusher.sounds["turnOff"].sampleDuration) );
			else
				d.curSpeed = math.min( 0, d.curSpeed - d.speed*(dt / self.wCrusher.sounds["turnOff"].sampleDuration) );
			end;		
			if d.curSpeed ~= 0 then
				rotate(d.node, 0,0,d.curSpeed*(dt/1000) );
			end;
		end;	
		if self.drumsStopped == true then
			for i,d in pairs(self.wCrusher.drums) do	
				d.curSpeed = 0;
			end;
		else
			for i,d in pairs(self.wCrusher.drums) do
				if d.speed > 0 then
					d.curSpeed = math.max( 0, d.curSpeed - d.speed*(dt / self.wCrusher.sounds["turnOff"].sampleDuration) );
				else
					d.curSpeed = math.min( 0, d.curSpeed - d.speed*(dt / self.wCrusher.sounds["turnOff"].sampleDuration) );
				end;
				if d.curSpeed ~= 0 then
					rotate(d.node, 0,0,d.curSpeed*(dt/1000) );
				end;
			end		
		end;
		if self.wCrusher.shader.a == true then
			setShaderParameter( self.wCrusher.shader.node, self.wCrusher.shader.para, 0, 0, 0, 0, false );
			self.wCrusher.shader.a = false;
		end;
	end;
	
	--# move trunks by force which are in the trigger and have contact or are also in deleteTrigger
	local trunksPulled = false;
	if self.isServer and self.isTurnedOn then
		if self.wCrusher.notOverstrained then
			for i,t in pairs(self.wCrusher.trunksInTrigger) do	
				-- remove branches in front of drum
				if t.branches ~= nil then
					local px, py, pz = getWorldTranslation(self.wCrusher.branchRemovePoint);
					for a=1, table.getn(t.branches) do
						local bx, by, bz = getWorldTranslation(t.branches[a]);
						if Utils.vector3Length(px-bx, py-by, pz-bz) > self.wCrusher.branchRemoveDist then
							t:setBranchInvisible(a);
						end;
					end;				
				end;
				
				local x, y, z = getCenterOfMass(t.nodeId);
				local wx, wy, wz = localToWorld(t.nodeId, x, y, z);
				drawDebugPoint(wx, wy, wz, 1, 0, 0, 1);
			
				if t.visNode ~= nil and t.visNodeRefDir ~= nil then 
					
					local m = t:getMassOfAttachedTrees();
					local fac = math.max(0, m/2);
					local force = fac * dt/1000;

					local foundT = false;
					
					if t.woodClaw == nil or (t.trunkJointIndices ~= nil and table.getn(t.trunkJointIndices) == 0 ) then --t.trunkJointIndex == nil then
						force = force * 2; 
						local x,y,z = getWorldTranslation( t.nodeId );
						local lx,ly,lz = worldToLocal( self.components[1].node, x,y,z );
						if t.oldLocalPos == nil then
							t.oldLocalPos = {lx,ly,lz};
							t.oldForce = force;
						else
							local dist = Utils.vector3Length( lx-t.oldLocalPos[1],ly-t.oldLocalPos[2],lz-t.oldLocalPos[3] );
							if dist > 0.05 then 
								t.oldLocalPos = {lx,ly,lz};
								t.oldForce = force;
							else
								force = math.min( 0.4, force + t.oldForce );
								t.oldForce = force;
							end;
						end;

						local pn = { getWorldTranslation(t.visNode) };
						local prn = { getWorldTranslation(t.visRefNode) };
						local lpn = { worldToLocal(self.wCrusher.woodChipInputRef, unpack(pn) ) }; --self.components[1].node, unpack(pn) ) };
						local lprn = { worldToLocal(self.wCrusher.woodChipInputRef, unpack(prn) ) }; --self.components[1].node, unpack(prn) ) };

						local dirT = {0,-1,0};
						local node = t.visNode;
						if lpn[1] < lprn[1] then
							node = t.visRefNode;
							dirT = {0,1,0};
						end;
						local wx,wy,wz = getWorldTranslation(node);
						local wdx,wdy,wdz = localDirectionToWorld(t.nodeId, dirT[1],dirT[2],dirT[3]);
						
						if lpn[1] > 0 and lprn[1] > 0 then
							t.readyToBeDeleted = true;
						else
							t.readyToBeDeleted = false;
						end;
						
						local dirBackUp = { localDirectionToWorld( self.components[1].node, 1,0.1,0 ) };
						
						local p1 = { localToWorld( self.components[1].node, 150,3,-0.5 ) };			-- 
						local com = { getCenterOfMass( t.nodeId ) };
						local wcom = { localToWorld( t.nodeId, unpack(t.centerOfMass) ) }; 
						local dir = { p1[1]-wcom[1], p1[2]-wcom[2], p1[3]-wcom[3] };
						local dirN = {};
						dirN[1] = dir[1] / Utils.vector3Length(dir[1], dir[2], dir[3]);
						dirN[2] = dir[2] / Utils.vector3Length(dir[1], dir[2], dir[3]);
						dirN[3] = dir[3] / Utils.vector3Length(dir[1], dir[2], dir[3]);

						local lx,ly,lz = worldToLocal( t.nodeId, pn[1],pn[2],pn[3]);
						if lpn[1] < lprn[1] then
							lx, ly, lz = worldToLocal( t.nodeId, prn[1],prn[2],prn[3]);
						end;
						addForce( t.nodeId, (dirN[1])*force,(dirN[2])*force,(dirN[3])*force, lx,ly,lz, true );
					end;
				end;							
			end;		
			
			--# rotate drum-comp
			rotate( self.componentJoints[17].jointNode, 0,0,1.8*dt/1000);
			setJointFrame(self.componentJoints[17].jointIndex, 0, self.componentJoints[17].jointNode);
			
			--# scale trunks which are 'inside'
			local woodChips = 0;
			for i,t in pairs(self.wCrusher.trunksInTrigger) do	
				if t.visNode ~= nil and t.visNodeRefDir ~= nil then
					
					local pn = { getWorldTranslation(t.visNode) };
					local prn = { getWorldTranslation(t.visRefNode) };
					local lpn = { worldToLocal(self.wCrusher.woodChipInputRef, unpack(pn) ) };
					local lprn = { worldToLocal(self.wCrusher.woodChipInputRef, unpack(prn) ) };
					
					-- is refNode closer to death? 
					local s = 1;
					if t.sold == nil then
						local sx,sy,sz = getScale(t.visNode);
						s = sy;
						t.sold = sy;
					else 
						s = t.sold;
					end;
					local m = false;
					if lpn[1] < lprn[1] and lprn[1] > 0.0 then		
						s = math.max( 0.0, math.min(1.0, Utils.vector3Length(unpack(lpn))/t.visNodeRefLength));
					elseif lpn[1] > lprn[1] and lpn[1] > 0.0 then
						s = math.max( 0.0, math.min(1.0, Utils.vector3Length(unpack(lprn))/t.visNodeRefLength));
						m = true;
					end;
					
					if math.abs(t.sold - s) > 0.01 then
						t:setVisScale( s, m );
						trunksPulled = true;
						if t.sold > s then
							if t.woodChipAmountFL == nil then	
								t.woodChipAmountFL = t.woodChipAmount;
							end;
							local df = math.min(t.woodChipAmountFL, (t.sold-s)*t.woodChipAmount);
							woodChips = woodChips + df;
							t.sold = s;
							t.woodChipAmountFL = t.woodChipAmountFL - df;
						else
							t:setVisScale( s, m );
						end;
					end;
				end
			end;
			
			if woodChips > 0 then
				self.fillLevel = self.fillLevel + woodChips;
				self:raiseDirtyFlags(self.woodCrusherDirtyFlag);	
			end;
		end;
	end;
	if self.isServer and trunksPulled ~= self.drumParticleSystemsActive then
		self:raiseDirtyFlags(self.woodCrusherDirtyFlag);
		self.drumParticleSystemsActive = trunksPulled;
	end;		
end;

function WoodCrusher:updateTick(dt)
	if self.isServer then
		if self.isTurnedOn and self.create_HeapTipTrigger then
			self.create_HeapTipTrigger = false;
			self:createHeapTipTrigger();
		end;
		
		local activePS = false;
		if self.isTurnedOn and self.fillLevel > 0 then
			local fillType = Fillable.fillTypeNameToInt["woodChip"];
			
			self:findTrailerToUnload(fillType);

			if self.trailerFound ~= nil then
				self.pipeUnloadingDistance = self.pipeUnloadingDistance + self.pipeParticleSystemExtraDistance; -- a little more length for trailers
				self.trailerFound:resetFillLevelIfNeeded(fillType);
				local deltaLevel = math.min(self.fillLevel, self.pipeUnloadingSpeed*dt/1000.0);
				deltaLevel = math.min(deltaLevel, self.trailerFound.capacity - self.trailerFound.fillLevel);
				if deltaLevel > 0 then
					self.fillLevel = self.fillLevel - deltaLevel;
					self:raiseDirtyFlags(self.woodCrusherDirtyFlag);
					activePS = true;
					self.trailerFound:setFillLevel(self.trailerFound.fillLevel+deltaLevel, fillType, nil, 1, 1, self);
				end
			elseif self.triggerFound ~= nil then
				if self.triggerFound.acceptedFillType == fillType and self.triggerFound.isStorageTipTrigger ~= nil then
					local deltaLevel = math.min(self.fillLevel, self.pipeUnloadingSpeed*dt/1000.0);
					deltaLevel = math.min(deltaLevel, self.triggerFound.capacity - self.triggerFound.fillLevel);
					if deltaLevel > 0 then
						self.fillLevel = self.fillLevel - deltaLevel;
						self:raiseDirtyFlags(self.woodCrusherDirtyFlag);
						activePS = true;
						self.triggerFound.fillLevel = self.triggerFound.fillLevel + deltaLevel;
					end;
				elseif self.triggerFound.acceptedFillTypes ~= nil and self.triggerFound.acceptedFillTypes[fillType] ~= nil and self.triggerFound.setFillLevel ~= nil then
					local deltaLevel = math.min(self.fillLevel, self.pipeUnloadingSpeed*dt/1000.0);
					deltaLevel = math.min(deltaLevel, self.triggerFound.capacity - self.triggerFound.fillLevel);
					if deltaLevel > 0 then
						self.fillLevel = self.fillLevel - deltaLevel;
						self:raiseDirtyFlags(self.woodCrusherDirtyFlag);
						activePS = true;
						self.triggerFound:setFillLevel(self.triggerFound.fillLevel+deltaLevel, fillType, nil, 1, 1, self);
					end;					
				end;
			end;	
		end;	
		if self.pipeParticleSystemsActive ~= activePS then
			self.pipeParticleSystemsActive = activePS;			
			self:raiseDirtyFlags(self.woodCrusherDirtyFlag);
		end;
	end;
	
	if self.isClient then
		if self.pipeParticleSystemsActiveSet ~= self.pipeParticleSystemsActive then
			self.pipeParticleSystemsActiveSet = self.pipeParticleSystemsActive;
			Utils.setEmittingState( self.pipeParticleSystem, self.pipeParticleSystemsActive);
		end;
		
		if self.pipeUnloadingDistance ~= self.pipeUnloadingDistanceSet then
			self.pipeUnloadingDistanceSet = self.pipeUnloadingDistance;
			for _, v in ipairs(self.pipeParticleSystem) do
				local lifespan = math.min(v.originalLifespan, self.pipeUnloadingDistance/v.speed);
				setParticleSystemLifespan(v.geometry, lifespan, true);
			end			
		end;
	end;	
	
	if self.isClient then
		if self.drumParticleSystemsActiveSet ~= self.drumParticleSystemsActive then
			self.drumParticleSystemsActiveSet = self.drumParticleSystemsActive;
			Utils.setEmittingState( self.drumHighParticleSystem, self.drumParticleSystemsActive);
		end;
		if (self.fillLevel > 0 or self.drumParticleSystemsActive) and self.isTurnedOn then
			Utils.setEmittingState( self.drumLowParticleSystem, true);		
		else
			Utils.setEmittingState( self.drumLowParticleSystem, false);		
		end;
	end;
		
	-- regulate workRPM
	if self.poweringVehicle ~= nil then
		local t = 0.75; 
		
		if self.isTurnedOn then
			t = t + 0.075;
			if self.drumParticleSystemsActive then
				t = t + 0.15;
			end;
			if self.pipeParticleSystemsActive then
				t = t + 0.075;
			end;		
		end;
		
		if t > self.motorSoundMaxPitchWorkFactor then
			self.motorSoundMaxPitchWorkFactor = self.motorSoundMaxPitchWorkFactor + (0.3 * dt/(900+math.random(0,200)));
		else
			self.motorSoundMaxPitchWorkFactor = self.motorSoundMaxPitchWorkFactor - (0.3 * dt/(900+math.random(0,200)));
		end;
	end;	
	
	if self.isClient then
		if self.wCrusher.sounds["turnOn"] ~= nil and 
			self.wCrusher.sounds["turnOff"] ~= nil and
			self.wCrusher.sounds["idle"] ~= nil and
			self.wCrusher.sounds["run"] ~= nil 
			then

			if self.isTurnedOn then
			
				if self.wCrusher.sounds["turnOn"].a == false and self.wCrusher.sounds["idle"].a == false then
					setVisibility( self.wCrusher.sounds["turnOn"].sound, true );
					self.wCrusher.sounds["turnOn"].a = true;
					self.wCrusher.sounds["turnOn"].startTime = g_currentMission.time;
					
					setVisibility( self.wCrusher.sounds["turnOff"].sound, false );
					self.wCrusher.sounds["turnOff"].a = false;
					
				elseif self.wCrusher.sounds["idle"].a == false and (self.wCrusher.sounds["turnOn"].startTime + self.wCrusher.sounds["turnOn"].sampleDuration) < g_currentMission.time  then
					setVisibility( self.wCrusher.sounds["turnOn"].sound, false );
					self.wCrusher.sounds["turnOn"].a = false;

					setVisibility( self.wCrusher.sounds["idle"].sound, true );
					self.wCrusher.sounds["idle"].a = true;
					setSamplePitch(getAudioSourceSample(self.wCrusher.sounds["idle"].sound), pitch);

					setVisibility( self.wCrusher.sounds["run"].sound, true );
					self.wCrusher.sounds["run"].a = true;
					
				elseif self.wCrusher.sounds["idle"].a == true then 
					if self.motorSoundMaxPitchWorkFactor ~= nil then
					
						local p = (self.motorSoundMaxPitchWorkFactor - 1)/ 0.4;
					
						local vol = self.wCrusher.sounds["idle"].volumeMin + p*(self.wCrusher.sounds["idle"].volumeMax - self.wCrusher.sounds["idle"].volumeMin);
						setSampleVolume(getAudioSourceSample(self.wCrusher.sounds["idle"].sound), vol);
						
						local pitch = self.wCrusher.sounds["idle"].pitchMin + p*(self.wCrusher.sounds["idle"].pitchMax - self.wCrusher.sounds["idle"].pitchMin);
						setSamplePitch(getAudioSourceSample(self.wCrusher.sounds["idle"].sound), pitch);
												
						local vol = self.wCrusher.sounds["run"].volumeMin + p*(self.wCrusher.sounds["run"].volumeMax - self.wCrusher.sounds["run"].volumeMin);
						if self.wCrusher.notOverstrained == false or not self.pipeParticleSystemsActive then
							vol = 0;
						end;
						if self.wCrusher.sounds["run"].volumeCur < vol then
							self.wCrusher.sounds["run"].volumeCur = self.wCrusher.sounds["run"].volumeCur + 1.0*(dt/1000);
						else
							self.wCrusher.sounds["run"].volumeCur = self.wCrusher.sounds["run"].volumeCur - 1.0*(dt/1000);
						end;
						setSampleVolume(getAudioSourceSample(self.wCrusher.sounds["run"].sound), self.wCrusher.sounds["run"].volumeCur);
						
						local pitch = self.wCrusher.sounds["run"].pitchMin + p*(self.wCrusher.sounds["run"].pitchMax - self.wCrusher.sounds["run"].pitchMin);
						setSamplePitch(getAudioSourceSample(self.wCrusher.sounds["run"].sound), pitch);
					end;
				end;
			else
				if self.wCrusher.sounds["turnOff"].a == false and self.wCrusher.sounds["turnOff"].stopTime < self.wCrusher.sounds["turnOn"].startTime then
					setVisibility( self.wCrusher.sounds["turnOff"].sound, true );
					self.wCrusher.sounds["turnOff"].a = true;
					self.wCrusher.sounds["turnOff"].stopTime = g_currentMission.time;
					
					setVisibility( self.wCrusher.sounds["turnOn"].sound, false );
					self.wCrusher.sounds["turnOn"].a = false;
					setVisibility( self.wCrusher.sounds["idle"].sound, false );
					self.wCrusher.sounds["idle"].a = false;
					setVisibility( self.wCrusher.sounds["run"].sound, false );
					self.wCrusher.sounds["run"].a = false;
				end;
				
				if (self.wCrusher.sounds["turnOff"].stopTime + self.wCrusher.sounds["turnOff"].sampleDuration) < g_currentMission.time  then
					setVisibility( self.wCrusher.sounds["turnOff"].sound, false );
					self.wCrusher.sounds["turnOff"].a = false;				
				end;
			end;
		end;
	end;
end;

function WoodCrusher:draw()
	if self:getIsActiveForInput() then
		if self.foldAnimTime == 0 then
			if self.isClient then
				if self.isTurnedOn then
					g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_off_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
				else
					g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_on_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
				end;
			end; 
			if self.isTurnedOn then
				if self.pipeRaycastFoundTerrain and g_currentMission.modDirectoryOfHeapTipTrigger ~= nil and self.trailerFound == nil and self.triggerFound == nil then
					g_currentMission:addHelpButtonText(g_i18n:getText("placeHeap"), InputBinding.IMPLEMENT_EXTRA3);
				end;
			end;
		end;
	end;
	
	self.wCrusher.guiCrusherFL:render();
	if not self.wCrusher.notOverstrained then
		self.wCrusher.guiCrusherRed:render();
	else
		if self.fillLevel > 250 then
			if self.wCrusher.guiCrusherYellowTime < self.time then
				self.wCrusher.guiCrusherYellowTime = self.time + 250;
				if self.wCrusher.guiCrusherYellowSet == true then
					self.wCrusher.guiCrusherYellowSet = false;
				else
					self.wCrusher.guiCrusherYellowSet = true;
				end;
			end;
			if self.wCrusher.guiCrusherYellowSet == true then
				self.wCrusher.guiCrusherYellow:render();
			else
				self.wCrusher.guiCrusherGreen:render();
			end;
		else
			if self.isTurnedOn then
				self.wCrusher.guiCrusherGreen:render();
			end;	
		end;
	end;
end;

function WoodCrusher:onAttach(attacherVehicle) 
end;

function WoodCrusher:onDetach()
	if self.isServer then
		self:setIsTurnedOn(false);
	end;
end;

function WoodCrusher:getIsFoldAllowed(_func)
	return not self.isTurnedOn;
end;

function WoodCrusher:setIsTurnedOn(state, noEventSend)
	SetTurnedOnEvent.sendEvent(self, state, noEventSend);
	self.isTurnedOn = state;
end;

function WoodCrusher:trunkTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay)
	if self.isServer and g_currentMission.treeManager ~= nil then
		local trunk = g_currentMission.treeManager.nodeIdToTrunk[otherId];
		if trunk ~= nil and trunk.isTrunk ~= nil and trunk.isTrunk then
			if onEnter then
				local insert = true;
				for i,t in pairs(self.wCrusher.trunksInTrigger) do
					if t == trunk then
						insert = false;
						break;
					end;
				end;
				if insert then
					table.insert(self.wCrusher.trunksInTrigger, trunk);
				end;
			elseif onLeave then
				for i,t in pairs(self.wCrusher.trunksInTrigger) do
					if t == trunk then 
						local del = false;
						for j,t2 in pairs(self.wCrusher.trunksInDeleteTrigger) do
							if t2 == trunk then
								if trunk.readyToBeDeleted == true then								
									if trunk.woodChipAmountFL ~= nil and trunk.woodChipAmountFL > 0 then
										self.fillLevel = self.fillLevel + trunk.woodChipAmountFL;
										self:raiseDirtyFlags(self.woodCrusherDirtyFlag);			
									end;
									
									del = true;
									if table.getn(t.jointsToOtherTrunks) > 0 then								
										local treePattern = g_currentMission.harvestableTrees.harvestableTreesPattern[t.typeIndex][t.treeIndex];
										local trunkPattern = treePattern.trunks[t.trunkIndex];
										for k,joint in pairs(trunkPattern.joints) do
											trunk:removeTrunkJoint(joint.jointTableIndex, 0,0,0);	
										end;								
									end;
									break;
								end;
							end;
						end;
					
						table.remove(self.wCrusher.trunksInTrigger, i);
						
						if del then
							for j,t2 in pairs(self.wCrusher.trunksInDeleteTrigger) do
								if t2 == trunk then
									table.remove(self.wCrusher.trunksInDeleteTrigger, j);
								end;
							end;	
							trunk:delete();
						end;
						break;
					end;
				end;			
			end;
		end;
	end;
end;

function WoodCrusher:trunkDeleteTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay)
	if self.isServer and g_currentMission.treeManager ~= nil then
		local trunk = g_currentMission.treeManager.nodeIdToTrunk[otherId];
		if trunk ~= nil and trunk.isTrunk ~= nil and trunk.isTrunk then
			if onEnter then
				local insert = true;
				for i,t in pairs(self.wCrusher.trunksInDeleteTrigger) do
					if t == trunk then
						insert = false;
						break;
					end;
				end;
				if insert then
					table.insert(self.wCrusher.trunksInDeleteTrigger, trunk);
					trunk:setCollisionMaskForChopping();
				end;
			elseif onLeave then
				for i,t in pairs(self.wCrusher.trunksInDeleteTrigger) do
					if t == trunk then
						table.remove(self.wCrusher.trunksInDeleteTrigger, i);
						trunk:resetCollisionMask();
						break;
					end;
				end;			
			end;
		end;
	end;
end;

function WoodCrusher:findTrailerToUnload(fruitType)
	local x,y,z = getWorldTranslation(self.pipeRaycastNode);
	local dx,dy,dz = localDirectionToWorld(self.pipeRaycastNode, 0,-1,0);

	self.trailerFound = nil;
	self.pipeRaycastFoundTerrain = false;
	self.triggerFound = nil;	
	raycastAll(x, y, z, dx,dy,dz, "findTrailerRaycastCallback", self.pipeRaycastDistance, self);

	local trailer = self.trailerFound;
	
	if trailer == nil or not trailer:allowFillType(Fillable.fillTypeNameToInt["woodChip"]) or trailer.getAllowFillFromAir == nil or not trailer:getAllowFillFromAir() or trailer.fillLevel >= trailer.capacity then
		self.trailerFound = nil;
	end;
end;

function WoodCrusher:findTrailerRaycastCallback(transformId, x, y, z, distance)
	local vehicle = g_currentMission.nodeToVehicle[transformId];
	if vehicle ~= nil then
		if vehicle.exactFillRootNode == transformId then
			self.trailerFound = vehicle;
			self.pipeUnloadingDistance = distance;
			self.trailerFoundDistance = distance; -- required for shader tipping
		end;
	else
		local obj = g_currentMission:getNodeObject(transformId);
		if obj ~= nil then
			if obj.acceptedFillType ~= nil or obj.acceptedFillTypes ~= nil then
				if (obj.moveNode ~= nil and obj.moveNode == transformId) or (obj.planeId ~= nil and obj.planeId == transformId) or (obj.moveNode == nil and obj.planeId == nil) then -- heapTipTrigger & storageTipTrigger
					self.triggerFound = obj;
					self.pipeUnloadingDistance = distance;
				end;
			end;
		else
			if transformId == g_currentMission.terrainRootNode then
				self.pipeRaycastFoundTerrain = true;
				self.pipeRaycastFoundTerrainPosition = {x,y,z};	
				self.pipeUnloadingDistance = distance;
			end;
		end;
	end;
end;

function WoodCrusher:createHeapTipTrigger()
	local x = self.pipeRaycastFoundTerrainPosition[1];
	local z = self.pipeRaycastFoundTerrainPosition[3];
	local yr = math.random()*math.pi;
	g_currentMission:loadVehicle(g_currentMission.modDirectoryOfHeapTipTrigger, x, 0.5, z, yr);
end;