-- author: rafftnix, fruktor
-- date: 05.12.2013

-- nderungen am Skript nur mit meiner Zustimmung!
-- Modification only with my permission!

-- base class for trunks

Trunk = {}
Trunk_mt = Class(Trunk, PhysicsObject);
InitObjectClass(Trunk, "Trunk");

function Trunk:new(isServer, isClient, customMt)
    local mt = customMt;
    if mt == nil then
        mt = Trunk_mt;
    end;

	local self = PhysicsObject:new(isServer, isClient, mt);
	
	registerObjectClassName(self, "Trunk");
	
    self.nodeId = 0;
    if not self.isServer then
        self.currentPosition = {};
        self.targetPosition = {};
        self.interpolationTime = 0;
    end
    self.forcedClipDistance = 200;	

    self.lastMoveTime = -100000;
	
	self.interpolationAlpha = 0;
	
	self.jointsToOtherTrunks = {};	
	
	return self;
end;

function Trunk:delete()
	unregisterObjectClassName(self);
	g_currentMission:removeItemToSave(self);
	
	-- remove all joints
	if self.isServer then
		self:removeAllJointsToOtherTrunks();
	end;
	
	-- remove entry
	for a=1, table.getn(g_currentMission.harvestableTrees.trunksForJointLoadHelp) do
		if g_currentMission.harvestableTrees.trunksForJointLoadHelp[a] == self then
			table.remove(g_currentMission.harvestableTrees.trunksForJointLoadHelp, a);
			break;
		end;
	end;
	
	-- removeTree
	Trunk:superClass().delete(self);
end;

function Trunk:load(nodeId, x, y, z, rx, ry, rz, typeIndex, treeIndex, trunkIndex, visibleBranches, saveHelpId)
	unlink(nodeId);
	link(getRootNode(), nodeId);
	
	local treePattern = g_currentMission.harvestableTrees.harvestableTreesPattern[typeIndex][treeIndex];
	local trunkPattern = treePattern.trunks[trunkIndex];
	g_currentMission.treeManager.nodeIdToTrunk[nodeId] = self;
	
	self.pos = {x,y,z};
	self.rot = {rx,ry,rz};
	
	self.removeMarks = {};
	
	self:setNodeId(nodeId);
		
	setTranslation(self.nodeId, unpack(self.pos));
	setRotation(self.nodeId, unpack(self.rot));
	local qx,qy,qz,qw = mathEulerToQuaternion(rx, ry, rz);
	self:setPose(x,y,z, qx,qy,qz,qw);

	self.maxBranchCutDist = 0.5;
	self.typeIndex = typeIndex;
	self.treeIndex = treeIndex;
	self.trunkIndex = trunkIndex;
	if trunkPattern.branchGroupIndex ~= nil then
		self.branchGroupIndex = trunkPattern.branchGroupIndex;
		self.branchGroup = Utils.indexToObject(self.nodeId, self.branchGroupIndex);
		self.branches = {}
		for i=1, getNumOfChildren(self.branchGroup) do
			table.insert(self.branches, getChildAt(self.branchGroup, i-1));
		end;
		
		if visibleBranches ~= nil then
			local branches = Utils.splitString(";", visibleBranches);
			for k, v in pairs(branches) do
				if v == "-" then
					setVisibility(self.branches[k], false);
				end;
			end;
		end;
	end;

	-- cut marks
	self.cutMarks = {}
	local joints = treePattern.trunks[self.trunkIndex].joints;
	for a=1, table.getn(joints) do
		local jointPattern = joints[a];	
		local markNode = Utils.indexToObject(self.nodeId, jointPattern.markNodeIndex);
		setVisibility(markNode, false);
		self.cutMarks[a] = markNode;
		
		if treePattern.trunks[self.trunkIndex].cutMeSignIndex ~= nil then
			self.cutMeSign = Utils.indexToObject(self.nodeId, treePattern.trunks[self.trunkIndex].cutMeSignIndex);
			setVisibility(self.cutMeSign, true);
		end;
	end;
	
	if trunkPattern.visNodeIndex ~= nil then
		self.visNode = Utils.indexToObject(self.nodeId, trunkPattern.visNodeIndex);
		self.visNodePos = { getTranslation(self.visNode) };
		self.visRefNode = Utils.indexToObject(self.nodeId, trunkPattern.visRefNodeIndex);
		
		local p1 = { getWorldTranslation(self.visRefNode) };
		local p2 = { getWorldTranslation(self.visNode) };
		
		local lp1 = { worldToLocal( getParent(self.visNode), p1[1],p1[2],p1[3] ) };
		local lp2 = { worldToLocal( getParent(self.visNode), p2[1],p2[2],p2[3] ) };
		
		local lpd = { lp1[1]-lp2[1], lp1[2]-lp2[2], lp1[3]-lp2[3] };
		
		self.visNodeRefLength = Utils.vector3Length( lpd[1], lpd[2], lpd[3] );
		self.visNodeRefDir = { lpd[1], lpd[2], lpd[3] };
	end;

	self.isAttached = false;
	self.isTrunk = true;
	self.isRoot = trunkPattern.isRoot;
	self.sellPrice = trunkPattern.sellPrice;
	self.woodChipAmount = trunkPattern.woodChipAmount;
	self.collisionMaskBackup = getCollisionMask(self.nodeId);
	
	self.centerOfMass = { unpack(trunkPattern.centerOfMass) };
	
	self.cuttingTime = 0;
	self.cuttingTimeDuration = 2000;
	self.cuttingDir = {};	

	g_currentMission:addItemToSave(self);
	
	if saveHelpId ~= nil then
		g_currentMission.harvestableTrees.trunksForJointLoadHelp[saveHelpId] = self;
	end;
	self.saveHelpId = saveHelpId;
	
	-- cut sound
	if treePattern.cutSound ~= nil then
		self.cutSoundSource = createAudioSource("treeCutSound", treePattern.cutSound.filename, treePattern.cutSound.radius, treePattern.cutSound.innerRadius, treePattern.cutSound.volume, 1);
		self.cutSoundSample = getAudioSourceSample(self.cutSoundSource);
		setSamplePitch(self.cutSoundSample, treePattern.cutSound.pitch);
		self.cutSoundVolume = treePattern.cutSound.volume;
		link(self.nodeId, self.cutSoundSource);
		setVisibility(self.cutSoundSource, false);
	end;
	-- impact sound
	if treePattern.impactSound ~= nil then
		self.impactSoundSource = createAudioSource("treeImpactSound", treePattern.impactSound.filename, treePattern.impactSound.radius, treePattern.impactSound.innerRadius, treePattern.impactSound.volume, 1);
		self.impactSoundSample = getAudioSourceSample(self.impactSoundSource);
		setSamplePitch(self.impactSoundSample, treePattern.impactSound.pitch);
		self.impactSoundVolume = treePattern.impactSound.volume;
		link(self.nodeId, self.impactSoundSource);
		setVisibility(self.impactSoundSource, false);
	end;
end;

function Trunk:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
	local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, key.."#position"));
	local rx, ry, rz = Utils.getVectorFromString(getXMLString(xmlFile, key.."#rotation"));
	
	local treeType = getXMLString(xmlFile, key.."#treeType");
	local treeIndex = getXMLInt(xmlFile, key.."#treeIndex");
	local trunkIndex = getXMLInt(xmlFile, key.."#trunkIndex");
	local saveHelpId = getXMLInt(xmlFile, key.."#saveHelpId");
	local visibleBranches = getXMLString(xmlFile, key.."#visibleBranches");
	local activeMarksString = getXMLString(xmlFile, key.."#activeMarks");	
	
	local typeIndex = TreeManager.treeTypes[treeType].num;
	
	local nodeId = g_currentMission.harvestableTrees:getTrunkNodeId(typeIndex, treeIndex, trunkIndex);
	
	if nodeId ~= nil then
		self:load(nodeId, x, y, z, rx, ry, rz, typeIndex, treeIndex, trunkIndex, visibleBranches, saveHelpId);
		if activeMarksString ~= nil and activeMarksString ~= "" then
			local activeMarks = Utils.splitString(";", activeMarksString);
			for k, v in pairs(activeMarks) do
				if v == "+" then -- otherwise it is invisible anyway
					setVisibility(self.cutMarks[k], true);
				end;
			end;
		end;
	end;
	
	if self.cutMeSign ~= nil then
		local cutMeSignActive = Utils.getNoNil(getXMLBool(xmlFile, key.."#cutMeSignActive"), false);
		setVisibility(self.cutMeSign, cutMeSignActive);
	end;
	return true;
end;

function Trunk:getSaveAttributesAndNodes(nodeIdent)
	if not g_currentMission.harvestableTrees.trunkIdsSet then -- give every trunk a valid saveHelpId
		g_currentMission.harvestableTrees.trunkIdsSet = true;
		local id = 1;
		for k, trunk in pairs(g_currentMission.treeManager.nodeIdToTrunk) do
			trunk.saveHelpId = id;
			id = id + 1;
		end;
	end;
	
	local attributes, nodes = "", "";
	local x, y, z = getTranslation(self.nodeId);
	local rx, ry, rz = getRotation(self.nodeId);
	local treeType = TreeManager.treeTypes[self.typeIndex].treeType;
	attributes = ' position="'..tostring(x)..' '..tostring(y)..' '..tostring(z)..'" rotation="'..tostring(rx)..' '..tostring(ry)..' '..tostring(rz)..'"';
	attributes = attributes ..' treeType="'..treeType..'" treeIndex="'..tostring(self.treeIndex)..'" trunkIndex="'..tostring(self.trunkIndex)..'"';
	if self.branches ~= nil then
		local branchesStr = "";
		for a=1, table.getn(self.branches) do
			if getVisibility(self.branches[a]) then
				branchesStr = branchesStr .. "+;";
			else
				branchesStr = branchesStr .. "-;";
			end;
		end;
		attributes = attributes .. ' visibleBranches="' .. branchesStr .. '"';
	end;
	
	local marksActiveString = "";
	for a=1, table.getn(self.cutMarks) do
		if getVisibility(self.cutMarks[a]) then
			marksActiveString = marksActiveString .. "+;";
		else
			marksActiveString = marksActiveString .. "-;";
		end;
	end;
	attributes = attributes .. ' activeMarks="'..marksActiveString..'"';
	
	if self.cutMeSign ~= nil then
		attributes = attributes .. ' cutMeSignActive="'..tostring(getVisibility(self.cutMeSign))..'"';
	end;
	
	attributes = attributes .. ' saveHelpId="'..tostring(self.saveHelpId)..'"';
	
	return attributes, nodes;
end;

function Trunk:readStream(streamId, connection)
	local typeIndex = streamReadInt8(streamId);
	local treeIndex = streamReadInt8(streamId);
	local trunkIndex = streamReadInt8(streamId);
	
	-- load trunk
	local nodeId = g_currentMission.harvestableTrees:getTrunkNodeId(typeIndex, treeIndex, trunkIndex);
	self:load(nodeId, 0, 0, 0, 0, 0, 0, typeIndex, treeIndex, trunkIndex, nil, nil);

	-- synchronize branches
	if streamReadBool(streamId) then
		for a=1, table.getn(self.branches) do
			setVisibility(self.branches[a], streamReadBool(streamId));
		end;
	end;
	
	-- synchronize marks
	for a=1, table.getn(self.cutMarks) do
		setVisibility(self.cutMarks[a], streamReadBool(streamId));
	end;
	
	-- synchronize cutMeSign
	if streamReadBool(streamId) then
		setVisibility(self.cutMeSign, streamReadBool(streamId));
	end;
	
	-- basically synchronizes position
	Trunk:superClass().readStream(self, streamId, connection);
	self.pos = {getTranslation(self.nodeId)};
	self.rot = {getRotation(self.nodeId)};
end;

function Trunk:writeStream(streamId, connection)
	streamWriteInt8(streamId, self.typeIndex);
	streamWriteInt8(streamId, self.treeIndex);
	streamWriteInt8(streamId, self.trunkIndex);

	-- here, the client calls Trunk:load()
	
	-- synchronize branches
	local hasBranches = self.branches ~= nil;
	streamWriteBool(streamId, hasBranches);
	if hasBranches then
		for a=1, table.getn(self.branches) do
			streamWriteBool(streamId, getVisibility(self.branches[a]));
		end;
	end;
	
	-- synchronize marks
	for a=1, table.getn(self.cutMarks) do
		streamWriteBool(streamId, getVisibility(self.cutMarks[a]));
	end;
	
	-- cutMeSign
	if self.cutMeSign ~= nil then
		streamWriteBool(streamId, true);
		streamWriteBool(streamId, getVisibility(self.cutMeSign));
	else
		streamWriteBool(streamId, false);
	end;
	
	-- basically synchronizes position
	Trunk:superClass().writeStream(self, streamId, connection);
end;

function Trunk:readUpdateStream(streamId, connection, dirtyMask)
	Trunk:superClass().readUpdateStream(self, streamId, connection, dirtyMask);
end;

function Trunk:writeUpdateStream(streamId, connection, dirtyMask)
	Trunk:superClass().writeUpdateStream(self, streamId, connection, dirtyMask);
end;

function Trunk:update(dt)
	-- the tree falls in the right direction
	if self.isServer then	
		if self.cuttingTime + self.cuttingTimeDuration > g_currentMission.time and g_currentMission.time > self.cuttingTimeDuration then
			local fac = dt/1000;
			fac = fac * ((g_currentMission.time-self.cuttingTime)/self.cuttingTimeDuration ) ; 
			addForce(self.nodeId, self.cuttingDir[1]*fac,self.cuttingDir[2]*fac,self.cuttingDir[3]*fac, 0,5,0, true);	
		end;
	end;
	Trunk:superClass().update(self, dt);
end;

function Trunk:updateTick(dt)
	Trunk:superClass().updateTick(self, dt);
end;

function Trunk:getDefaultRigidBodyType()
	local bodyType = getRigidBodyType(self.nodeId);
	if g_server ~= nil then
		return bodyType;
	else
		return "Kinematic";
	end;
end;

function Trunk:setBranchInvisible(branchNum) 
	setVisibility(self.branches[branchNum], false);

	if g_server ~= nil then
		g_server:broadcastEvent(SetBranchInvisibleEvent:new(self, branchNum), nil, nil, self); 
	end;
end;

function Trunk:setCuttingForce(dx,dy,dz)
	if self.isServer then
		self.cuttingTime = g_currentMission.time;
		self.cuttingDir = {dx,dy,dz};
		for i,joint in pairs(self.jointsToOtherTrunks) do
			if joint.trunk2 ~= nil and joint.trunk1 == self then
				joint.trunk2:setCuttingForce(dx,dy,dz);
			end;
		end;
	end;
end;

function Trunk:removeTrunkJoint(jointTableIndex, dx,dy,dz, noEventSend)
	if g_server ~= nil then
		g_server:broadcastEvent(RemoveTrunkJointEvent:new(self, jointTableIndex), nil, nil, self); 
	end;

	local treePattern = g_currentMission.harvestableTrees.harvestableTreesPattern[self.typeIndex][self.treeIndex];
	local jointPattern = treePattern.trunks[self.trunkIndex].joints[jointTableIndex];
	
	if self.cutMarks[jointTableIndex] ~= nil then
		setVisibility(self.cutMarks[jointTableIndex], false);
	end;
	
	-- cut sound
	if self.isRoot then
		if self.cutSoundSample ~= nil then
			setVisibility(self.cutSoundSource, true);
			playSample(self.cutSoundSample, 1, self.cutSoundVolume, 0);
			addTimer(getSampleDuration(self.cutSoundSample), "stopCutSound", self);
		end;
		
		if self.isServer then
			-- add impact callback
			self:addImpactCallback({});
		end;
	end;
	
	if self.isServer then	
		local joint;
		for a=1, table.getn(g_currentMission.harvestableTrees.treeJoints) do
			if g_currentMission.harvestableTrees.treeJoints[a].jointTableIndex == jointTableIndex and g_currentMission.harvestableTrees.treeJoints[a].trunk1 == self then
				joint = g_currentMission.harvestableTrees.treeJoints[a];
				break;
			end;
		end;
	
		if joint ~= nil then
			setPairCollision(self.nodeId, joint.trunk2.nodeId, true);
			
			removeJoint(joint.jointIndex);

			for a=1, table.getn(joint.trunk1.jointsToOtherTrunks) do
				if joint.trunk1.jointsToOtherTrunks[a] == joint then
					table.remove(joint.trunk1.jointsToOtherTrunks, a);
				end;
			end;		
			for a=1, table.getn(joint.trunk2.jointsToOtherTrunks) do
				if joint.trunk2.jointsToOtherTrunks[a] == joint then
					table.remove(joint.trunk2.jointsToOtherTrunks, a);
				end;
			end;		
			
			if dx ~= 0 or dy ~= 0 or dz ~= 0 then
				local trunk;
				if joint.trunk1.isRoot then	
					trunk = joint.trunk2;
				else
					trunk = joint.trunk1;
				end;
				trunk:setCuttingForce(dx,dy,dz);
			end;
			
			for a=1, table.getn(g_currentMission.harvestableTrees.treeJoints) do
				if g_currentMission.harvestableTrees.treeJoints[a] == joint then
					table.remove(g_currentMission.harvestableTrees.treeJoints, a);
				end;
			end;
		end;
	end;

	if self.cutMeSign ~= nil then
		setVisibility(self.cutMeSign, false);
	end;
end;

function Trunk:removeAllJointsToOtherTrunks()
	if self.isServer then
		for a=1, table.getn(self.jointsToOtherTrunks) do
			local joint = self.jointsToOtherTrunks[a];
			if trunk1 == self then
				self:removeTrunkJoint(joint.jointTableIndex, 0, 0, 0);
			elseif trunk2 == self then
				trunk1:removeTrunkJoint(joint.jointTableIndex, 0, 0, 0);
			end;
		end;
		self.jointsToOtherTrunks = {}
	else
		print("function Trunk:removeAllJointsToOtherTrunks() should be called only on server");
	end;
end;

function Trunk:setCollisionMaskForChopping() 
	setCollisionMask(self.nodeId, -2147483648); -- Bitflag 31
end;

function Trunk:resetCollisionMask()
	setCollisionMask(self.nodeId, self.collisionMaskBackup); 
end;

function Trunk:getMassOfAttachedTrees(lastTrunk)
	if self.nodeId == 0 then
		return 0;
	end;
	local mass = getMass(self.nodeId);
	for a=1, table.getn(self.jointsToOtherTrunks) do
		local joint = self.jointsToOtherTrunks[a];
		if joint.trunk1 == self then
			if joint.trunk2 ~= lastTrunk then
				mass = mass + joint.trunk2:getMassOfAttachedTrees(self);
			end;
		else
			if joint.trunk1 ~= lastTrunk then
				mass = mass + joint.trunk1:getMassOfAttachedTrees(self);
			end;
		end;
	end;
	return mass;
end;

function Trunk:addImpactCallback(relatedTrunks, lastTrunk)
	addContactReport(self.nodeId, 0.0001, "impactCallback", self);
	relatedTrunks[self.nodeId] = self;
	self.impactCallbackRelatedTrunks = relatedTrunks;

	for a=1, table.getn(self.jointsToOtherTrunks) do
		local joint = self.jointsToOtherTrunks[a];
		if joint.trunk1 == self then
			if joint.trunk2 ~= lastTrunk then
				joint.trunk2:addImpactCallback(relatedTrunks, self);
			end;
		else
			if joint.trunk1 ~= lastTrunk then
				joint.trunk1:addImpactCallback(relatedTrunks, self);
			end;
		end;
	end;
end;

function Trunk:impactCallback(objectId, otherObjectId, isStart, normalForce, tangentialForce)
	if isStart or normalForce > 0 or tangentialForce > 0 then
		if self.impactCallbackRelatedTrunks[objectId] ~= nil and self.impactCallbackRelatedTrunks[otherObjectId] == nil then
			for nodeId, trunk in pairs(self.impactCallbackRelatedTrunks) do
				trunk:setMarksVisible();
				removeContactReport(nodeId);
				self.impactCallbackRelatedTrunks[nodeId] = nil;
			end;
			self:startImpactSound();
		end;
	end;
end;

function Trunk:setVisScale(scale, bMove, noEventSend)
	SetTrunkVisScaleEvent.sendEvent(self, scale, bMove, noEventSend);
	setScale( self.visNode, 1, scale, 1);		

	local refNode = self.visNode;	
	
	if bMove then	
		setTranslation( self.visNode, self.visNodePos[1]+self.visNodeRefDir[1]*(1-scale), self.visNodePos[2]+self.visNodeRefDir[2]*(1-scale), self.visNodePos[3]+self.visNodeRefDir[3]*(1-scale) );	
		refNode = self.visRefNode;
	end;
	
	local x, y, z = unpack(self.centerOfMass);
	local xr, yr, zr = getWorldTranslation(refNode);
	local lxr, lyr, lzr = worldToLocal(self.nodeId, xr, yr, zr);
	local difY = y-lyr;
	setCenterOfMass(self.nodeId, x, y-(difY*(1-scale*0.8)), z);
end;

function Trunk:stopCutSound()
	setVisibility(self.cutSoundSource, false);
	return false;
end;

function Trunk:startImpactSound()
	self:setMarksVisible()
	
	setVisibility(self.impactSoundSource, true);
	playSample(self.impactSoundSample, 1, self.impactSoundVolume, 0);
	addTimer(getSampleDuration(self.impactSoundSample), "stopImpactSound", self);	
	-- broadcast impact to clients
	if self.isServer then
		g_server:broadcastEvent(ImpactSoundEvent:new(self), nil, nil);
	end;
	return false;
end;

function Trunk:setMarksVisible()
	for a=1, table.getn(self.cutMarks) do
		setVisibility(self.cutMarks[a], true);
	end;
	-- broadcast event
	if self.isServer then
		g_server:broadcastEvent(SetMarksVisibleEvent:new(self), nil, nil);
	end;
end;

function Trunk:stopImpactSound()
	setVisibility(self.impactSoundSource, false);
	return false;
end;