RealisticVehicle = {};

function RealisticVehicle.prerequisitesPresent(specializations) 
	return true;--SpecializationUtil.hasSpecialization(realistic, specializations);
end;


function RealisticVehicle:load(xmlFile) 

	self.isRealistic = true;
	
	
	self.realGetIsAiDriven = RealisticVehicle.realGetIsAiDriven;
	self.realForceAiDriven = false; -- for follow me mod and courseplay
	
	--20131218 - handle model that are set in the wrong direction => z axis toward the back of the vehicle instead of the front
	self.realInversedZAxis = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realInversedZAxis"), false);
	
	--self.realHandBrakeForce = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realHandBrakeForce"), 0);
	self.realMaxVehicleSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMaxVehicleSpeed"), 25);	
	self.realBrakeMaxMovingMass = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realBrakeMaxMovingMass"), 1);
	self.realBrakingDeceleration = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realBrakingDeceleration"), 5); -- deceleration in m/s2.  2 = very bad. 3 = bad. 4 = average. 5 = good. 7 = truck braking system
	--self.realMaxBrakeForce = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realMaxBrakeForce"), 10);-- 10 = very good brakeForce for a tractor	
	self.realSCX = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realSCX"), 0);
	
	self.realVehicleFlotationFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realVehicleFlotationFx"), 1); -- 1 = standard wheel compared to the vehicle mass - 0.75 = care wheels - 1.25 = duals - 1.4 = tracks
	
	self.realCanLockWheelsWhenBraking = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realCanLockWheelsWhenBraking"), true);
	self.realCanLockWheelsWhenBrakingFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realCanLockWheelsWhenBraking#randomFx"), 1);
	self.realLateralStiffDecreaseFunctionOfSteeringAngle = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realLateralStiffDecreaseFunctionOfSteeringAngle"), 0.1);-- to attenuate speed decrease in cornering
	self.realSteeringAngleFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realSteeringAngleFx"), 1);
	
	self.realIsTracked = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realIsTracked"), false);
	
	--self.realStillMassFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realStillMassFx"), 4);
	self.realStillStatePossible = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realStillStatePossible"), true);
	
	self.realVehicleSteeringAngle = getXMLFloat(xmlFile, "vehicle.realVehicleSteeringAngle");
	
	--20130917 - useful for "steeringMod" for gravity wagon with fixed front axle
	self.realNoSteeringAxleDamping = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.steeringAxleAngleScale#realNoSteeringAxleDamping"), false); -- see overrideWheelUtils => updateWheelSteeringAngle
		
	
	
	
	--self.realRollingResistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realRollingResistance"), 0.025); -- this value is overrided in realistic motorized
	self.realRollingResistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realRollingResistance"), 0.02); -- 20131217 - new default value
	
	
	self.realMeasuredMassForRollingResistance = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realMeasuredMassForRollingResistance"), false);
	self.realAntiRollForceFx = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.realAntiRollForceFx"), 1);
	
	self.realDisableCameraCollisionDetection = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.realDisableCameraCollisionDetection"), true);
	
	local mainComponentIndex = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.components#realMainComponent"), 1);
	--print("RealisticVehicle:load : " .. tostring(xmlFile) .. " / " .. tostring(mainComponentIndex));
	self.realMainComponent = self.components[mainComponentIndex];
	
		
	self.realVehicleName = Utils.getXMLI18N(xmlFile, "vehicle.name", "", "Unidentified Vehicle Object...", self.customEnvironment);
	
	
	self.realOnDesactivate = RealisticVehicle.realOnDesactivate;

	
	
	--self.realWheelGroup = {};
	self.wheelsWithAntiRoll = {};
	
	local nbWheelsWithBrake = 0;
	local i = 0;
	while true do
		local wheelnamei = string.format("vehicle.wheels.wheel(%d)", i);
		if self.wheels[i+1] == nil then
			break;
		end;
		
		--overide default stiffness value 
		--[[
		self.wheels[i+1].lateralStiffness = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralStiffness"), 0.5);
		self.wheels[i+1].longitudalStiffness = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalStiffness"), 0.5);
		
		if self.isServer then
			local wheel = self.wheels[i+1];
			setWheelShapeTireFunction(wheel.node, wheel.wheelShape, false, wheel.lateralExtremumSlip, wheel.lateralExtremumValue, wheel.lateralAsymptoteSlip, wheel.lateralAsymptoteValue, 1000000*wheel.lateralStiffness);
			setWheelShapeTireFunction(wheel.node, wheel.wheelShape, true, wheel.longitudalExtremumSlip, wheel.longitudalExtremumValue, wheel.longitudalAsymptoteSlip, wheel.longitudalAsymptoteValue, 1000000*wheel.longitudalStiffness);		
		end;]]
		
		self.wheels[i+1].spring = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#spring"), 0)*Vehicle.springScale;
		self.wheels[i+1].suspTravel = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#suspTravel"), 0);
		
		if self.wheels[i+1].suspTravel==0 then
			RealisticUtils.printWarning("RealisticVehicle:load - " .. tostring(self.realVehicleName), "wheel number "..tostring(i+1).." has a suspTravel of 0. This is not allowed for a moreRealistic vehicle.", false);
		end;
		
		self.wheels[i+1].damper = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#damper"), 0);
		
		self.wheels[i+1].brakeRatio = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#brakeRatio"), 0);
		self.wheels[i+1].hasHandBrake = Utils.getNoNil(getXMLBool(xmlFile, wheelnamei .. "#hasHandBrake"), self.wheels[i+1].brakeRatio>0);
		
		self.wheels[i+1].realTransYOffset = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#realTransYOffset"), 0); -- correction of the y position of the graphical wheel compared to the wheelshape
		
		--getting the position of the wheelshape "fixation" point (where the spring is attached)
		--needed for "WheelsUtil.realGetWheelShapePosition"
		local positionX, positionY, positionZ = getTranslation(self.wheels[i+1].repr);
		self.wheels[i+1].realAttachPoint = {x=positionX, y=positionY+self.wheels[i+1].deltaY, z=positionZ};
		
		self.wheels[i+1].realSteeringFx = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#steeringFx"), 1); -- just for realTouchingWheelsAvgSteeringAngle computing
		self.wheels[i+1].realContinousBrakeForceWhenNotActive = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#continousBrakeForceWhenNotActive"), 0); --  for virtual wheels on tools (prevents tools from "rolling" when detached with some speed)
		
		self.wheels[i+1].realAntiRollFx = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#antiRollFx"), 1);
		
		self.wheels[i+1].realTractionModeLateralStiffnessFx = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#tractionModeLateralStiffnessFx"), 4);
		
		self.wheels[i+1].realBoundSteeringAxle = Utils.getNoNil(getXMLBool(xmlFile, wheelnamei .. "#realBoundSteeringAxle"), false);
		
		self.wheels[i+1].realTirePressureFx = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#tirePressureFx"), 1);
		
		
		
		
		-- approx wheel mass = 450 * r^3
		self.wheels[i+1].realMass = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#realMass"), 0.45 * self.wheels[i+1].radius^3); -- Metric Ton
		
		if self.wheels[i+1].realAntiRollFx>0 then
			table.insert(self.wheelsWithAntiRoll, i+1);
		end;
		--[[
		self.wheels[i+1].realGroup = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#group"), 1);	
		if self.realWheelGroup[self.wheels[i+1].realGroup]==nil then
			self.realWheelGroup[self.wheels[i+1].realGroup] = {};
			self.realWheelGroup[self.wheels[i+1].realGroup].NumWheelsWithGroundContact = 0;
		end;
		self.realWheelGroup[self.wheels[i+1].realGroup].NumWheelsInGroup = Utils.getNoNil(self.realWheelGroup[self.wheels[i+1].realGroup].NumWheelsInGroup, 0) + 1;]]
		
		--self.wheels[i+1].realIsJointWheelShape = RealisticUtils.computeIsJointWheelShape(self, self.wheels[i+1]);
		
		--local alpha = getXMLFloat(xmlFile, wheelnamei .. "#alpha");
		--if self.wheels[i+1].realIsJointWheelShape and alpha==nil then
			--alpha = 196;
		--end;
		self.wheels[i+1].realAlpha = getXMLFloat(xmlFile, wheelnamei .. "#alpha");	
		
		self.wheels[i+1].realSupportedMass = 0;
		self.wheels[i+1].realSupportedMassS = 0;		
		self.wheels[i+1].realY = 0;
		
		self.wheels[i+1].realRainMode = false;
		
		--self.wheels[i+1].realSlipCoeff = 1;
		--self.wheels[i+1].realSlipCoeffTmp = 0;
		self.wheels[i+1].realLastSlip = 0;
		self.wheels[i+1].realAxleSpdAdd = 0;
		self.wheels[i+1].realPowerApplied = 0; -- Watt
		
		self.wheels[i+1].realLastWsAxleSpd = 0; -- last wheelshape axle speed (can be different from axleSpd, especially for motorized wheels)
		
		self.wheels[i+1].realPreviousHasGroundContact = true;
		
		self.wheels[i+1].realBrakeLocking = false;
		self.wheels[i+1].realPreviousBrakeLocking = false;
		self.wheels[i+1].realBrakeLockingFx = 1;
		self.wheels[i+1].realStoppedWithWheelShapeBrakeForce = false;
		--self.wheels[i+1].realWheelShapeAddBrake = 0;
		--self.wheels[i+1].realNullSpd = 1;
		--self.wheels[i+1].realBrakeForce = 0;
		
		self.wheels[i+1].realLastSteeringAngle = -1;  -- different from 0, since the first time we will check the lastSteeringAngle, it will be to set the stiffness and release the brake (new brakeforce = 0)  => see overrrideWheelUtils (WheelsUtil.realApplyWheelPhysics)
		self.wheels[i+1].realLastWheelShapeBrakeForce = 0;
		
		self.wheels[i+1].realStoppedWithNoGroundContact = false;
		
		self.wheels[i+1].realTransYOffset2 = 0; -- correction of the y position of the graphical wheel compared to the wheelshape, used in updateAnimatedWheelsYcorrection
		
		local wheelComponentIndex = RealisticUtils.getWheelComponentIndex(self, self.wheels[i+1]);
		self.wheels[i+1].realRootComponent = self.components[wheelComponentIndex];
		
		self.wheels[i+1].num = i+1;	
		
		if self.wheels[i+1].brakeRatio>0 then
			nbWheelsWithBrake = nbWheelsWithBrake + 1;
		end;
		
		i = i + 1;
	end;
	
	self.realNbWheels = i;
	self.realNbWheelsWithBrake = nbWheelsWithBrake;
	
	
	--check that number of wheels with antiroll is even	
	if math.mod(table.getn(self.wheelsWithAntiRoll), 2)~=0 then		
		RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .. "' - number of wheels with antiroll is odd. it should be even. please set the 'antiRollFx' parameter of the single wheel to '0'.");
	end;
	
	
	--we get the component with wheels because we need to correct their downforce 
	--computing the "vehicleSteeringAngle" too
	self.realComponentsWithWheels = {};
	local sumSteeringAngle, nbSteeringAngles = 0,0;
	for k, wheel in pairs(self.wheels) do
		wheel.realMaxSupportedMass = RealisticUtils.computeMassFromSpringExtensionV3(2 + 100*wheel.suspTravel, wheel.spring, wheel.realAlpha);
		local componentIndex = RealisticUtils.getWheelComponentIndex(self, wheel);
		
		if self.components[componentIndex].realWheels==nil then
			self.components[componentIndex].realWheels = {};			
		end;		
		table.insert(self.components[componentIndex].realWheels, wheel);
		
		if self.realComponentsWithWheels[componentIndex]==nil then
			self.realComponentsWithWheels[componentIndex] = self.components[componentIndex];
		end;
		
		if wheel.rotSpeed~=0 and wheel.rotMin~=nil and wheel.rotMax~=nil then
			sumSteeringAngle = sumSteeringAngle + math.abs(wheel.rotMin) + math.abs(wheel.rotMax);
			nbSteeringAngles = nbSteeringAngles + 2;
		end;
			
	end;
	
	if self.realVehicleSteeringAngle==nil then
		if nbSteeringAngles>0 then
			self.realVehicleSteeringAngle = math.deg(sumSteeringAngle / nbSteeringAngles);
		else
			self.realVehicleSteeringAngle = 0;
		end;
	end;
		
	
	local i1 = 0;
	self.realGenCuttingAreas = {};
	for k, cuttingArea in pairs(self.cuttingAreas) do
		i1 = i1 + 1;
		self.realGenCuttingAreas[i1] = {};
		local x,y,z = getTranslation(cuttingArea.start);
		self.realGenCuttingAreas[i1].start = {x=x, y=y, z=z};
		x,y,z = getTranslation(cuttingArea.width);
		self.realGenCuttingAreas[i1].width = {x=x, y=y, z=z};
		x,y,z = getTranslation(cuttingArea.height);
		self.realGenCuttingAreas[i1].height = {x=x, y=y, z=z};
	end;
	
	
	local numComponents = table.getn(self.components);
	for i=1, numComponents do
	
		if self.components[i].centerOfMass==nil then
			local x, y, z = getCenterOfMass(self.components[i].node);
			self.components[i].centerOfMass = {x,y,z};
		end;
		
		local namei = string.format("vehicle.components.component%d", i);
		local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, namei .. "#realTransWithMass"));		
		self.components[i].realTransWithMass = {x=Utils.getNoNil(x, 0), y=Utils.getNoNil(y, 0), z=Utils.getNoNil(z, 0)};
		xmax, ymax, zmax = Utils.getVectorFromString(getXMLString(xmlFile, namei .. "#realTransWithMassMax"));		
		self.components[i].realTransWithMassMax = {x=Utils.getNoNil(xmax, 100*self.components[i].realTransWithMass.x), y=Utils.getNoNil(ymax, 100*self.components[i].realTransWithMass.y), z=Utils.getNoNil(zmax, 100*self.components[i].realTransWithMass.z)};	
		
		if Utils.sign(self.components[i].realTransWithMass.x)~=Utils.sign(self.components[i].realTransWithMassMax.x) then
			RealisticUtils.printWarning("RealisticVehicle.load", tostring(self.realVehicleName) .. " - realTransWithMass and realTransWithMassMax should have the same sign. check the 'x' parameter.");
		end;
		if Utils.sign(self.components[i].realTransWithMass.y)~=Utils.sign(self.components[i].realTransWithMassMax.y) then
			RealisticUtils.printWarning("RealisticVehicle.load", tostring(self.realVehicleName) .. " - realTransWithMass and realTransWithMassMax should have the same sign. check the 'y' parameter.");
		end;
		if Utils.sign(self.components[i].realTransWithMass.z)~=Utils.sign(self.components[i].realTransWithMassMax.z) then
			RealisticUtils.printWarning("RealisticVehicle.load", tostring(self.realVehicleName) .. " - realTransWithMass and realTransWithMassMax should have the same sign. check the 'z' parameter.");			
		end;

		self.components[i].realMassWanted = Utils.getNoNil(getXMLFloat(xmlFile, namei .. "#realMassWanted"),0);
	end;
	
	
	
	--parsing the "attacherJoints" to override some default values
	local i=0;
	while true do
		local key = string.format("vehicle.attacherJoints.attacherJoint(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;		

		local index = getXMLString(xmlFile, key.. "#index");
		if index == nil then
			break;
		end;
		
		local object = Utils.indexToObject(self.components, index);
		if object ~= nil then
			for k, attacherJoint in pairs(self.attacherJoints) do
				if attacherJoint.jointTransform == object then
				
					--DURAL - 20131005 - loading the "realAutoDetaching" param
					attacherJoint.realAutoDetaching = Utils.getNoNil(getXMLBool(xmlFile, key.."#realAutoDetaching"), false);
				
					if attacherJoint.rotationNode ~= nil then
				
						--local realKeepGenLoweredRot = Utils.getNoNil(getXMLBool(xmlFile, key .. "#realKeepGenLoweredRot"), false);
						--override default moveTime and maxRotDistanceToGround
						attacherJoint.moveTime = Utils.getNoNil(getXMLFloat(xmlFile, key.."#moveTime"), 2)*1000;
						attacherJoint.maxRotDistanceToGround = Utils.getNoNil(getXMLFloat(xmlFile, key.."#maxRotDistanceToGround"), 0.2);
						
						--compute default minRot2
						local newMinRot2 = nil;
						local test = getXMLString(xmlFile, key.."#minRot2");
						--print(self.realVehicleName .. " test = " .. tostring(test));
						if test==nil then
							attacherJoint.minRot2 = nil; -- we set the minRot2 to nil so that we know we have to compute it (see the next parser just below, after computing "realPositiveRotationTowardGround")							
						end;
						
						local _, y, _ = Utils.getVectorFromString(getXMLString(xmlFile, key.."#maxTransLimit"));											
						attacherJoint.maxTransLimit[2] = math.abs(Utils.getNoNil(y, 0.5));						
						
						local _, _, z = Utils.getVectorFromString(getXMLString(xmlFile, key.."#maxRotLimit"));										
						attacherJoint.maxRotLimit[3] = math.rad(math.abs(Utils.getNoNil(z, 5)));
						
					end;
					
				end;
			end;
		end;
		
		
		i = i + 1;
	end;
	
	
	
	-- setting the var for implement lift height adjustment
	for k, attacherJoint in pairs(self.attacherJoints) do
		local length = 0;
		attacherJoint.realZeroRotHeightDistance = 0;
		attacherJoint.realDefaultLiftHeight = 0;
		if attacherJoint.rotationNode~=nil and attacherJoint.jointTransform~=nil then
			local x1,y1,z1 = getWorldTranslation(attacherJoint.rotationNode);
			local x2,y2,z2 = getWorldTranslation(attacherJoint.jointTransform);
			length = Utils.vector3Length(x1-x2,y1-y2,z1-z2);	
			
			--we want to determine if a positive angle raise or lower the joint
			-- 1. backup current rotation
			local rotX, rotY, rotZ = getRotation(attacherJoint.rotationNode);
			local backupRotation = {rotX, rotY, rotZ};
			
			--2. set a 0 rotation and measure attacher joint height
			setRotation(attacherJoint.rotationNode, 0,0,0);
			local x,y,z = getWorldTranslation(attacherJoint.jointTransform);
			local _,height0,_ = worldToLocal(self.realMainComponent.node, x,y,z);
			
			
			--3. set a 10 rotation and measure attacher joint height
			setRotation(attacherJoint.rotationNode, math.rad(10),0,0);
			local x,y,z = getWorldTranslation(attacherJoint.jointTransform);
			local _,height1,_ = worldToLocal(self.realMainComponent.node, x,y,z);
			
			--4. set the rotationnode back to normal
			setRotation(attacherJoint.rotationNode, unpack(backupRotation));
			
			--5. wath the rotating direction
			attacherJoint.realPositiveRotationTowardGround = false;
			local angle = attacherJoint.maxRot[1];
			if height1>height0 then
				--positive rotation toward the ground
				attacherJoint.realPositiveRotationTowardGround = true;
				angle = -angle;
			end;
			
			attacherJoint.realRotRotationOffset = attacherJoint.maxRot[1] + attacherJoint.maxRot2[1];
			
			if attacherJoint.realPositiveRotationTowardGround then
				attacherJoint.realZeroRotHeightDistance = length * math.sin(-attacherJoint.maxRot[1]); -- current y distance compared to a zero rotation angle
				if attacherJoint.minRot2 == nil then
					--computing a default minRot2
					attacherJoint.minRot2 = {-attacherJoint.minRot[1]+attacherJoint.minRotRotationOffset+attacherJoint.realRotRotationOffset, 0, 0}; 
				end;
			else
				attacherJoint.realZeroRotHeightDistance = length * math.sin(attacherJoint.maxRot[1]); -- current y distance compared to a zero rotation angle
				if attacherJoint.minRot2 == nil then
					--computing a default minRot2
					attacherJoint.minRot2 = {-attacherJoint.minRot[1]-attacherJoint.minRotRotationOffset+attacherJoint.realRotRotationOffset, 0, 0}; 
				end;
			end;
			
				
			attacherJoint.realDefaultLiftHeight = length*math.abs(math.sin(attacherJoint.maxRot[1]) - math.sin(attacherJoint.minRot[1]));
			
			
			attacherJoint.realBaseMoveTime = attacherJoint.moveTime;
		
			--backup default values
			attacherJoint.realDefaultMoveTime = attacherJoint.moveTime;
			
			attacherJoint.realDefaultMaxRot = {};
			attacherJoint.realDefaultMaxRot2 = {};
			attacherJoint.realDefaultMinRot = {};
			attacherJoint.realDefaultMinRot2 = {};
			attacherJoint.realDefaultMaxRotLimit = {};
			attacherJoint.realDefaultMinRotLimit = {};
			attacherJoint.realDefaultMaxTransLimit = {};
			attacherJoint.realDefaultMinTransLimit = {};
			for i=1,3 do
				attacherJoint.realDefaultMaxRot[i] = attacherJoint.maxRot[i];
				attacherJoint.realDefaultMaxRot2[i] = attacherJoint.maxRot2[i];
				attacherJoint.realDefaultMinRot[i] = attacherJoint.minRot[i];
				attacherJoint.realDefaultMinRot2[i] = attacherJoint.minRot2[i];
				attacherJoint.realDefaultMaxRotLimit[i] = attacherJoint.maxRotLimit[i];
				attacherJoint.realDefaultMinRotLimit[i] = attacherJoint.minRotLimit[i];
				attacherJoint.realDefaultMaxTransLimit[i] = attacherJoint.maxTransLimit[i];
				attacherJoint.realDefaultMinTransLimit[i] = attacherJoint.minTransLimit[i];
			end;
			
		end;
		attacherJoint.realBottomArmsLength = length;
		
	end;
	
	
	-- loading realAnimatedWheelsY
	self.realHasAnimatedWheelsYcorrection = false;	
	local i=0;
	while true do
		
		local key = string.format("vehicle.realAnimatedWheelsYcorrection.wheel(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;

		local wheelIndex = getXMLInt(xmlFile, key.."#index");
		
		if wheelIndex==nil then
			RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .."xml key : '" .. key.."#index' is not defined");
			break;
		end;
		
		--check wheel exists
		if self.wheels[wheelIndex]==nil then
			RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .."xml key : '" .. key.."#index' do not correspond to an existing wheel number");
			break;
		end;
		
		local animationName = getXMLString(xmlFile, key.."#animation");
		if animationName==nil then
			RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .."xml key : '" .. key.."#animation' is not defined");
			break;
		end;
		
		--check if the animation if present
		local animationFound = false;
		local i2 = 0;
		while true do
			local key2 = string.format("vehicle.animations.animation(%d)", i2);
			if not hasXMLProperty(xmlFile, key2) then
				break;
			end;
			local name = getXMLString(xmlFile, key2.."#name");
			if name == animationName then
				animationFound = true;
			end;
			i2 = i2 + 1;
		end;
		
		if not animationFound then
			RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .."xml key : '" .. key.."#animation' do not correspond to an existing animation");
			break;
		end;		
		
		local startYinc = getXMLFloat(xmlFile, key.."#startYinc");
		if startYinc==nil then
			RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .."xml key : '" .. key.."#startYinc' is not defined");
			break;
		end;
		
		local endYinc = getXMLFloat(xmlFile, key.."#endYinc");
		if endYinc==nil then
			RealisticUtils.printWarning("RealisticVehicle.load", self.realVehicleName .."xml key : '" .. key.."#endYinc' is not defined");
			break;
		end;
		
		
		--everything seems ok, adding the "realAnimatedWheelsHeight.wheel"
		if self.realAnimatedWheelsYcorrections == nil then
			self.realAnimatedWheelsYcorrections = {};
		end;
		
		local entry = {wheelIndex=wheelIndex, animationName=animationName, startYinc=startYinc, endYinc=endYinc};
		table.insert(self.realAnimatedWheelsYcorrections, entry);
		
		self.realHasAnimatedWheelsYcorrection = true;
		
		i = i + 1;
	
	end;
 
	
	self.realEmptyMass = 0;
	
	self.realMainComponentMass = 0;
	self.realMainComponentCurrentMass = 0;
	self.realComponentsCurrentMass = 0;
	self.realLoadMass = 0;	-- old realisticTrailer.lua
	self.realTotalMovingMassOnTheWheels = 0;
	self.realTotalMassOnTheWheels = 0;
	
	--self.realMassCoeff = 1;
	self.realActiveStillMassFx = 1;
	
	--self.realMassCorrectionFx = 1; --math.max(1, self.realNbWheels);
	
	
	--self.realTempStillMass = 0;
	
	
	self.realSecondUpdateTime = false;
	self.realInitialized = false;
	--self.realStill = false;
	
	self.realGroundSpeed = 0;
	self.real2dGroundSpeed = 0;
	self.realMovingDirection = 0;
		
	self.realVelX = 0;
	self.realVelY = 0;
	self.realVelZ = 0;
	
	
	self.realYCosValue = 1;
	
	self.realNumTouchingWheels = 0;
	self.realNumTouchingDriveWheels = 0;
	self.realTouchingWheelsAvgSteeringAngle = 0;
	self.realLastNumTouchingWheels = -1;
	self.realAvgTouchingWheelSpd = 0;	
	self.realAvgMotorizedWheelSpd = 0;
	self.realAvgTouchingMotorizedWheelSpd = 0;
	self.realAvgTouchingMotorizedWheelSlipS = 0;
	self.realAvgTouchingMotorizedWheelSlipS2 = 0;  -- increase slower, decrease faster
	
	
	self.realComX, self.realComY, self.realComZ = getCenterOfMass(self.realMainComponent.node);
	--print("test getcenterofmass : " .. self.realComX .. " " .. self.realComY .. " " .. self.realComZ);
	
	self.realCurrentForceNeeded = 0;
	self.realCurrentPowerConsumption = 0;
	
	
	self.realActiveBefore = true;
	self.realIsActive = false;
	self.realForceIsActive = false;
	
		
	self.realCurrentMaxDeceleration = 100;
	self.realMaxEC = self.realBrakeMaxMovingMass * (self.realMaxVehicleSpeed/3.6)^2;
	
	self.realBrakePedal = 0;
	self.realIsBraking = false; --see overrideAttachable for attachable, see owerrideWheelUtils for motorized
	self.realBrakingSystemOverloaded = false;
	
	self.realVehicleOnField = false;
	
	self.getTooFastSpeedToDisplay = RealisticVehicle.getTooFastSpeedToDisplay;	
	
	self.getImplementsNeeds = RealisticVehicle.getImplementsNeeds;
	
	self.realWheelsGraphicsUpdated = false; -- info for mods to know when the graphics of wheels have been effectively updated
	
	self.realExtraMassGetterFunctions = nil;
	
	self.realPsStopped = true;
	
	
	
	
	self.realVehicleDirtyFlag = self:getNextDirtyFlag();
	
	
	
	self.realDebugRenderingInitialized = false;
	
	if Vehicle.debugRendering and self.isServer then	
		RealisticVehicle.realInitDebugRendering(self);
	end;
  
end;


function RealisticVehicle:postLoad(xmlFile) 
	--local xmlFile = loadXMLFile("Tmp", self.configFileName);
	
	for i=1, table.getn(self.wheels) do
		local wheel = self.wheels[i];
		local wheelnamei = string.format("vehicle.wheels.wheel(%d)", wheel.xmlIndex);
		RealisticVehicle.loadDynamicWheelDataFromXML(self, xmlFile, wheelnamei, wheel);
	end;
	
end;

function RealisticVehicle:delete()    
end;

function RealisticVehicle:mouseEvent(posX, posY, isDown, isUp, button)
end;

function RealisticVehicle:keyEvent(unicode, sym, modifier, isDown)
end;





function RealisticVehicle:readUpdateStream(streamId, timestamp, connection)
	if connection:getIsServer() then
		if streamReadBool(streamId) then
			self.realIsActive = streamReadBool(streamId);	
			self.realForceIsActive = streamReadBool(streamId);
		end;
	end;
end;
 
function RealisticVehicle:writeUpdateStream(streamId, connection, dirtyMask)
	if not connection:getIsServer() then
		if streamWriteBool(streamId, bitAND(dirtyMask, self.realVehicleDirtyFlag) ~= 0) then
			streamWriteBool(streamId, self.realIsActive);
			streamWriteBool(streamId, self.realForceIsActive);
		end;
	end;
end;




function RealisticVehicle:update(dt)


	if self.isServer and self.realInitialized then		
				
		RealisticVehicle.updateVehiclePosition(self);
	
		--self.realIsActive = self.isActive or self.realVelX>0.1 or self.realVelZ>0.1;
		--self.realIsActive = self.isActive or self.realGroundSpeed>0.1;	
				
		self.realForceIsActive = self.realGroundSpeed>0.1 or self.time<RealisticGlobalListener.initTime or self:realGetIsAiDriven(); -- hirable uses "forceIsActive" param. so we must take this into account when updating "realForceIsActive"
		
		if self.realIsActive and not (self.isActive or self.realForceIsActive) then
			self:realOnDesactivate();
		end;
		
		self.realIsActive = self.isActive or self.realForceIsActive;	
		
		self:raiseDirtyFlags(self.realVehicleDirtyFlag);
		
		if self.realIsActive then		
			RealisticVehicle.updateWheelsParams(self);							
		end;
		
		
		if self.realForceIsActive then				
			--print("RealisticVehicle is active : " .. self.realEmptyMass);
			
			if not self.realActiveBefore then
				RealisticVehicle.setStillState(self, false);
				self.realActiveBefore = true;				
				--print("RealisticVehicle : activate. empty mass=" .. self.realEmptyMass);
			end;			
			
			RealisticVehicle.updateCorrectionDownForce(self, dt/1000);			
			RealisticVehicle.updateVehicleExternalForces(self, dt/1000);					
		else			
			if self.realActiveBefore then
				RealisticVehicle.setStillState(self, true)
				self.realActiveBefore = false;				
			end;			
		end;
		
	end;--isServer	
	
	
	self.forceIsActive = self.realForceIsActive;-- we want the wheels to continue spinning when exiting a moving vehicle (see vehicle.lua, update function,  calling WheelsUtil.updateWheelsGraphics)
	
	--for cultivator / plough / seeder etc => avoid the "particle system" to run indefinitely when exiting a vehicle with a working implement (or detaching an implement while it is running)
	if self.isClient then
		if self.isActive then
			self.realPsStopped = false;
		else
			if not self.realPsStopped then
				if self.realOnDesactivatePS~=nil then
					self:realOnDesactivatePS();	
				end;
				self.realPsStopped = true;
			end;
		end;
	end;
	
	--[[
	if self.forceIsActive==true and self.realForceIsActive==false then
		--switching forceIsActive from true to false
		--for cultivator / plough / seeder etc => avoid the "particle system" to run indefinitely when exiting a vehicle with a working implement (or detaching an implement while it is running)
		
		print(self.time .. " RealisticVehicle:update - trying to call realOnDesactivatePS");
		if self.realOnDesactivatePS~=nil then			
			self.forceIsActive = false;
			if self.isClient and not self:getIsActive() then
				print(self.time .. " RealisticVehicle:update - calling realOnDesactivatePS");
				self:realOnDesactivatePS();				
			end;
			--print(self.time .. " desactivate - " .. tostring(self.realVehicleName));
		end;
	end;]]
	

end;

function RealisticVehicle:updateTick(dt)

	if self.isServer then

		if Vehicle.debugRendering and not self.realDebugRenderingInitialized then
			RealisticVehicle.realInitDebugRendering(self);
		end;

		if self.realSecondUpdateTime and not self.realInitialized then
				
			--if self.realEmptyMass==0 then		
		
			for k, component in pairs(self.components) do
				--local realMass = getUserAttribute(component.node, "realMass");
				
				if component.realMassWanted~=0 then
					realMass = component.realMassWanted;
					setMass(component.node, realMass);						
				else
					realMass = getMass(component.node);
					component.realMassWanted = realMass;
				end;			
				if self.realMainComponentMass==0 and component.node==self.realMainComponent.node then
					self.realMainComponentMass = realMass;					
					self.realMainComponentCurrentMass = realMass;					
				end;
				self.realEmptyMass = self.realEmptyMass + realMass;
				
				--setCenterOfMass(component.node, 0, 0.5, 0.5);
			end;	
			
			--RealisticVehicle.setMainComponentMass(self, self.realMainComponentMass);	
			RealisticVehicle.updateVehicleMass(self, false);
			RealisticVehicle.updateAnimatedWheelsYcorrection(self);			
			
			self.realInitialized = true;
				
			--end;
			
		end;
		
		
		
		if self.realInitialized then

			
			if self.realIsActive then
				-- update realCurrentMaxDeceleration and realBrakingSystemOverloaded			
				--local ec = math.max(1, self.realTotalMovingMassOnTheWheels * self.realGroundSpeed^2);
				local ec = math.max(1, self.realTotalMassOnTheWheels * self.realGroundSpeed^2);
				local brakeFx = math.min(1, self.realMaxEC / ec)^0.5;
				self.realBrakingSystemOverloaded = brakeFx<1;
				self.realCurrentMaxDeceleration = self.realBrakingDeceleration * brakeFx * RealisticGlobalListener.realRainFx2^0.8; -- take into account dry/wet ground after rain
				
				RealisticVehicle.updateAnimatedWheelsYcorrection(self);
			end;
		
			if self.isActive then
				--if Vehicle.debugRendering and self.realDebugRenderingInitialized then
					RealisticVehicle.updateYCosAngle(self);	
				--end;				
				RealisticVehicle.updateVehicleMass(self, false);			
			end;			
			
		end;
		
		self.realSecondUpdateTime = true;
			
		
		
	end;--end server
	
end;

function RealisticVehicle:draw() 
	
	--draw is called only when the vehicle is active
	
	local hasTooFastMsgToDisplay, tooFastSpeedToDisplay = self:getTooFastSpeedToDisplay();
	--print(self.time .. "RealisticVehicle:draw hasTooFastMsgToDisplay / tooFastSpeedToDisplay = " .. tostring(hasTooFastMsgToDisplay) .. " / " .. tostring(tooFastSpeedToDisplay));
	if hasTooFastMsgToDisplay then			
		setTextBold(false);
		setTextColor(1, 1, 1, 1);
		local msg = g_i18n:getText("Dont_drive_to_fast") .. "\n\n" .. string.format(g_i18n:getText("Cruise_control_levelN"), string.format(":\n%2d " .. g_i18n:getText("speedometer"), g_i18n:getSpeed(tooFastSpeedToDisplay)));
		--print(self.time .. "RealisticVehicle:draw message = " .. msg); 
		g_currentMission:addWarning(msg, 0.07+0.022, 0.019+0.029);		
	end;
	
	
	
end;
















function RealisticVehicle.manageTestButtons(self)
	
	if Vehicle.debugRendering and self.realDebugRenderingInitialized and self.isControlled then
		if self:getIsActiveForInput() and InputBinding.hasEvent(InputBinding.REALISTIC_TEST_PLUS) then
			self.realTestValue = self.realTestValue + 1;			
		end;
		if self:getIsActiveForInput() and InputBinding.hasEvent(InputBinding.REALISTIC_TEST_MINUS) then
			self.realTestValue = self.realTestValue - 1;			
		end;
	end;
	
end;

function RealisticVehicle.manageTest2Buttons(self)
	
	if Vehicle.debugRendering and self.realDebugRenderingInitialized and self.isControlled then
		if self:getIsActiveForInput() and InputBinding.hasEvent(InputBinding.REALISTIC_TEST2_PLUS) then
			self.realTest2Value = self.realTest2Value + 1;
			self.components[1].centerOfMass[3] = self.components[1].centerOfMass[3] + 1/20;
			RealisticVehicle.updateVehicleMass(self, true);
		end;
		if self:getIsActiveForInput() and InputBinding.hasEvent(InputBinding.REALISTIC_TEST2_MINUS) then
			self.realTest2Value = self.realTest2Value - 1;
			self.components[1].centerOfMass[3] = self.components[1].centerOfMass[3] - 1/20;
			RealisticVehicle.updateVehicleMass(self, true);
		end;			
	end;
	
end;




function RealisticVehicle.setStillState(self, isStill)

	self.realActiveStillMassFx = 1;
	
	for c, component in pairs(self.realComponentsWithWheels) do
	
		local x,y,z;
		local mass;
	
		if component.node == self.realMainComponent.node then
			x,y,z = self.realComX, self.realComY, self.realComZ;
			mass = self.realMainComponentCurrentMass;
		else
			x,y,z = component.centerOfMass[1], component.centerOfMass[2], component.centerOfMass[3];
			mass = component.realMassWanted;
		end;	
	
		if not isStill then
			-- the vehicle is moving. the correction downforce manage the true mass function of wheel ground contact
			setMass(component.node, mass);
			setCenterOfMass(component.node, x, y, z);		
		
		else
			--the vehicle is still => we set the mass according to the wheel ground contact (no correction downforce is applyied)
		
			local numWheelsWithGroundContact = 0;			
			for k, wheel in pairs(component.realWheels) do
				--print(self.time .. " RealisticVehicle.updateCorrectionDownForce  k / wheel / hasgroundcontact = " .. tostring(k) .. " / " .. tostring(wheel) .. " / " .. tostring(wheel.hasGroundContact));
				if wheel.hasGroundContact then
					numWheelsWithGroundContact = numWheelsWithGroundContact + 1;
				end;	
			end;
			
			--print(self.time .. " RealisticVehicle.setStillState  name / component / numWheelsWithGroundContact = " .. tostring(self.realVehicleName) .. " / " .. tostring(c) .. " / " .. tostring(numWheelsWithGroundContact));
							
			numWheelsWithGroundContact = math.max(1, numWheelsWithGroundContact)
			if component.node == self.realMainComponent.node then
				self.realActiveStillMassFx = math.max(1, numWheelsWithGroundContact);
			end;
			
			setMass(component.node, numWheelsWithGroundContact*mass);
			setCenterOfMass(component.node, x, y, z);
			
		
		end; -- istill
	
	end; -- for each component



end;





function RealisticVehicle.manageAntiRollBar(self, dt, wheel1, wheel2)

	if self.realAntiRollForceFx==0 or self.realGroundSpeed<0.1 then
		return;
	end;

	--local antiRollFx = 0.25*wheel1.spring;
	--local antiRollFx = RealisticGlobalListener.antiRollForceTuning * self.realAntiRollForceFx * self.realGroundSpeed^1.5 * self.realComponentsCurrentMass;
	local antiRollFx = 2 * RealisticGlobalListener.antiRollForceTuning * self.realAntiRollForceFx * self.realGroundSpeed * self.realComponentsCurrentMass;

	local wheel1Travel = 1.0;
	if wheel1.hasGroundContact then
		--local x, y, z = getWheelShapePosition(wheel1.node, wheel1.wheelShape);
		wheel1Travel = math.max(0, wheel1.suspTravel - (wheel1.netInfo.y-wheel1.netInfo.yMin))/wheel1.suspTravel;
	end;
	
	local wheel2Travel = 1.0;
	if wheel2.hasGroundContact then
		--local x, y, z = getWheelShapePosition(wheel2.node, wheel2.wheelShape);
		wheel2Travel = math.max(0, wheel2.suspTravel - (wheel2.netInfo.y-wheel2.netInfo.yMin))/wheel2.suspTravel;
	end;
	
	--if self.isEntered then
	--	print(self.time .. " manageAntiRollBar - wheel1Travel / wheel2Travel = " .. tostring(wheel1Travel) .. " / " .. tostring(wheel2Travel));
	--end;
	

	local antiRollForce = antiRollFx * (wheel1Travel - wheel2Travel);
	
	if antiRollForce~=0 then	
		if wheel1.hasGroundContact then
			addForce(self.realMainComponent.node, 0, -antiRollForce*dt, 0, wheel1.netInfo.x, wheel1.netInfo.y, wheel1.netInfo.z, true);
		end;
		
		if wheel2.hasGroundContact then
			addForce(self.realMainComponent.node, 0, antiRollForce*dt, 0, wheel2.netInfo.x, wheel2.netInfo.y, wheel2.netInfo.z, true);
		end;		
	end;
	
end;





function RealisticVehicle.updateCorrectionDownForce(self, dt)

		
	for c, component in pairs(self.realComponentsWithWheels) do
	
		local numWheelsWithGroundContact = 0;
		
		--print(self.time .. " RealisticVehicle.updateCorrectionDownForce  c / component = " .. tostring(c) .. " / " .. tostring(component));
		
		for k, wheel in pairs(component.realWheels) do
			--print(self.time .. " RealisticVehicle.updateCorrectionDownForce  k / wheel / hasgroundcontact = " .. tostring(k) .. " / " .. tostring(wheel) .. " / " .. tostring(wheel.hasGroundContact));
			if wheel.hasGroundContact then
				numWheelsWithGroundContact = numWheelsWithGroundContact + 1;
			end;	
		end;
		
		local x,y,z;
		local mass;
	
		if component.node == self.realMainComponent.node then
			x,y,z = self.realComX, self.realComY, self.realComZ;
			mass = self.realMainComponentCurrentMass;
		else
			x,y,z = component.centerOfMass[1], component.centerOfMass[2], component.centerOfMass[3];
			mass = component.realMassWanted;
		end;
		
		if numWheelsWithGroundContact>1 then			
			--correction downforce for components on wheelshape
			local suppDownForce2 = (1-1/numWheelsWithGroundContact)*mass*RealisticGlobalListener.gravity*dt;			
			addForce(self.realMainComponent.node, 0, -suppDownForce2, 0, x, y, z, true);
		end;			
	
	end;
	
	local numAntiRollWheels = table.getn(self.wheelsWithAntiRoll);
	if numAntiRollWheels>1 then
		local i = 1;
		while i<numAntiRollWheels do
		
			local wheeli1, wheeli2 = self.wheelsWithAntiRoll[i], self.wheelsWithAntiRoll[i+1];
			RealisticVehicle.manageAntiRollBar(self, dt, self.wheels[wheeli1], self.wheels[wheeli2]);
		
			i = i + 2;
		end;
	end;
	
end;


function RealisticVehicle.setMainComponentMass(self, newMass, comTrans)

	local component = self.realMainComponent;
	
	self.realMainComponentCurrentMass = newMass;
	setMass(component.node, self.realMainComponentCurrentMass*self.realActiveStillMassFx);
	
	if component.centerOfMass ~= nil then
		local newComX = component.centerOfMass[1];
		local newComY = component.centerOfMass[2];
		local newComZ = component.centerOfMass[3];		
		
		local addMass = self.realMainComponentCurrentMass-self.realMainComponentMass;
		
		--print(self.time .. " RealisticVehicle.setMainComponentMass - setting mass of vehicle : " .. tostring(self.realVehicleName) .. " new mass = " .. tostring(newMass) .. " fillMass = " .. tostring(addMass));
		
		if component.realTransWithMass.x~=0 then			
			if component.realTransWithMass.x<0 then
				newComX = newComX + math.max(component.realTransWithMassMax.x, component.realTransWithMass.x*addMass);			
			else
				newComX = newComX + math.min(component.realTransWithMassMax.x, component.realTransWithMass.x*addMass);
			end;
		end;
		
		if component.realTransWithMass.y~=0 then
			if component.realTransWithMass.y<0 then
				newComY = newComY + math.max(component.realTransWithMassMax.y, component.realTransWithMass.y*addMass);			
			else
				newComY = newComY + math.min(component.realTransWithMassMax.y, component.realTransWithMass.y*addMass);
			end;
		end;
		
		if component.realTransWithMass.z~=0 then
			if component.realTransWithMass.z<0 then
				newComZ = newComZ + math.max(component.realTransWithMassMax.z, component.realTransWithMass.z*addMass);			
			else
				newComZ = newComZ + math.min(component.realTransWithMassMax.z, component.realTransWithMass.z*addMass);
			end;
		end;
		
		
		--extraMass com trans
		if comTrans~=nil then
			--print("RealisticVehicle.setComponent1Mass : comTrans = " .. tostring(comTrans.x) .. " / " .. tostring(comTrans.y) .. " / " .. tostring(comTrans.z));
			newComX = newComX + comTrans.x;
			newComY = newComY + comTrans.y;
			newComZ = newComZ + comTrans.z;
		end;
		
		--print("setting zmass = " .. component.realZIncWithMassMax .. " " .. component.realZIncWithMass .. " " .. addMass .. " " .. newComZ);
		setCenterOfMass(component.node, newComX, newComY, newComZ);
		self.realComX = newComX;
		self.realComY = newComY;
		self.realComZ = newComZ;
	end;
	
	
	
end;


function RealisticVehicle.updateVehicleMass(self, force)
		
	local extraMass, extraMassComTrans = RealisticVehicle.getExtraMassSum(self);
		
	local newVariableMass = Utils.getNoNil(self.realGrainTankFillMass, 0) + Utils.getNoNil(self.realFillableFillMass, 0) + Utils.getNoNil(self.realFuelTankFillMass, 0) + extraMass;
	local newComponentsMass = self.realEmptyMass + newVariableMass;
	if force or self.realComponentsCurrentMass ~= newComponentsMass then
			
					
		self.realComponentsCurrentMass = newComponentsMass;
		
		--print("RealisticVehicle : setting mass - " .. self.realComponent1CurrentMass);
		RealisticVehicle.setMainComponentMass(self, self.realMainComponentMass+newVariableMass, extraMassComTrans);
		
		
	end;		
			
	self.realTotalMovingMassOnTheWheels = self.realTotalMassOnTheWheels;
	
	if Vehicle.debugRendering and self.realDebugRenderingInitialized then
		self.realTotalMovingMass = self.realComponentsCurrentMass + self.realLoadMass;
	end;
	
	for k, implement in pairs(self.attachedImplements) do				
		self.realTotalMovingMassOnTheWheels = self.realTotalMovingMassOnTheWheels + Utils.getNoNil(implement.object.realTotalMovingMassOnTheWheels, 0);		
		if Vehicle.debugRendering and self.realDebugRenderingInitialized then
			local currImplMass = Utils.getNoNil(implement.object.realComponentsCurrentMass, 0);
			self.realTotalMovingMass = self.realTotalMovingMass + Utils.getNoNil(implement.object.realTotalMovingMass, currImplMass);
		end;		
	end;
	
end;


function RealisticVehicle.updateWheelsParams(self)
	
	local numTouching = 0;
	local nbWheels = 0;
	local sumDriveWheelsSupportedMass = 0;
	local sumWheelsSupportedMass = 0;
	
	local nbDriveWheels = 0;
	local nbTouchingDriveWheels = 0;
	
	local sumSteeringAngle = 0;
	
	local avgSpd = 0;
	local avgTouchingSpd = 0;
	local avgTouchingSpd2 = 0;
	local wheelSpd = 0;
	local avgWheelSlip = 0;

	
	for k, wheel in pairs(self.wheels) do
		nbWheels = nbWheels + 1;
		wheel.realSupportedMass = 0;
		wheelSpd = 0;
		
		wheel.realPreviousHasGroundContact = wheel.hasGroundContact;
		
		WheelsUtil.updateWheelHasGroundContact(wheel);		
		
		if wheel.hasGroundContact then	
			wheelSpd = wheel.axleSpeed * wheel.radius;
			wheel.realSupportedMass = WheelsUtil.getMassOnTheWheelV3(self, wheel);
			numTouching = numTouching + 1;	
			sumSteeringAngle = sumSteeringAngle + wheel.steeringAngle*wheel.realSteeringFx;
			avgTouchingSpd = avgTouchingSpd + wheelSpd;
			sumWheelsSupportedMass = sumWheelsSupportedMass + wheel.realSupportedMass;
		end;

		if wheel.driveMode>=self.requiredDriveMode then	
			nbDriveWheels = nbDriveWheels + 1;
			sumDriveWheelsSupportedMass = sumDriveWheelsSupportedMass + wheel.realSupportedMass;
			avgSpd = avgSpd + wheel.axleSpeed * wheel.radius;
			if wheel.hasGroundContact then
				nbTouchingDriveWheels = nbTouchingDriveWheels + 1;	
				avgTouchingSpd2 = avgTouchingSpd2 + wheelSpd;	
				avgWheelSlip = avgWheelSlip + wheel.realLastSlip;
			end;
		end;

		wheel.realSupportedMassS = 0.96*wheel.realSupportedMassS + 0.04*wheel.realSupportedMass;		
						
	end;
	
	self.realNumTouchingWheels = numTouching;
	self.realNbMotorizedWheels = nbDriveWheels;	
	if numTouching==0 then
		self.realTouchingWheelsAvgSteeringAngle = 0;
	else
		self.realTouchingWheelsAvgSteeringAngle = self.realSteeringAngleFx * sumSteeringAngle / numTouching;
		avgTouchingSpd = avgTouchingSpd / numTouching;
		self.realAvgTouchingWheelSpd = avgTouchingSpd;
	end;
	self.realTotalMassOnMotorizedWheels = sumDriveWheelsSupportedMass;
	self.realTotalMassOnTheWheels = sumWheelsSupportedMass;
	
	if nbDriveWheels>0 then
		avgSpd = avgSpd / nbDriveWheels;
	end;
	self.realAvgMotorizedWheelSpd = avgSpd;
	self.realNumTouchingDriveWheels = nbTouchingDriveWheels;
	if nbTouchingDriveWheels>0 then
		avgTouchingSpd2 = avgTouchingSpd2 / nbTouchingDriveWheels;
		avgWheelSlip = avgWheelSlip / nbTouchingDriveWheels;
	end;
	self.realAvgTouchingMotorizedWheelSpd = avgTouchingSpd2;
	self.realAvgTouchingMotorizedWheelSlipS = self.realAvgTouchingMotorizedWheelSlipS*0.98 + avgWheelSlip*0.02;
	
	if avgWheelSlip>self.realAvgTouchingMotorizedWheelSlipS2 then
		self.realAvgTouchingMotorizedWheelSlipS2 = self.realAvgTouchingMotorizedWheelSlipS2*0.995 + avgWheelSlip*0.005;
	else
		self.realAvgTouchingMotorizedWheelSlipS2 = self.realAvgTouchingMotorizedWheelSlipS2*0.95 + avgWheelSlip*0.05;
	end;
	
	
end;



function RealisticVehicle.updateVehiclePosition(self)
	
	--print(self.time .. " RealisticVehicle.updateVehiclePosition - starting - realMainComponent = " .. tostring(self.realMainComponent.node));
	
	local velX, velY, velZ = getLinearVelocity(self.realMainComponent.node);
	
	--checking for NAN (not a number) result  SHOULD NEVER HAPPEN
	if velX~=velX or velY~=velY or velZ~=velZ then
		RealisticUtils.printWarning("RealisticVehicle.updateVehiclePosition", self.realVehicleName .. " - getLinearVelocity - impossible result returned. velx/vely/velz="..tostring(velX).."/"..tostring(velY).."/"..tostring(velZ), true);
	else
		self.realVelX, self.realVelY, self.realVelZ = velX, velY, velZ;
	end;
	self.realGroundSpeed = Utils.vector3Length(self.realVelX, self.realVelY, self.realVelZ);
	
	
	--[[
	if self.isControlled then
		--print(self.time .. " " .. self.realVehicleName .. " RealisticVehicle.updateVehiclePosition - 1 velX / velY / velZ = " .. tostring(self.realVelX) .. " / " .. tostring(self.realVelY) .. " / " .. tostring(self.realVelZ));
		--self.realVelX, self.realVelY, self.realVelZ = getLinearVelocity(self.components[1].node);
		print(self.time .. " type velX = " .. type(self.realVelX));
		print(self.time .. " " .. self.realVehicleName .. " RealisticVehicle.updateVehiclePosition -  2 velX / velY / velZ = " .. tostring(self.realVelX) .. " / " .. tostring(self.realVelY) .. " / " .. tostring(self.realVelZ));
		
		print(self.time .. " " .. self.realVehicleName .. " realGroundSpeed>0 = " .. tostring(self.realGroundSpeed>0));
	end;]]
	
	--if self.realGroundSpeed>=0 then
	
	local dx, dy, dz = worldDirectionToLocal(self.realMainComponent.node, self.realVelX, self.realVelY, self.realVelZ);
	--print(self.time .. " RealisticVehicle.updateVehiclePosition - dx / dy / dz = " .. tostring(dx) .. " / " .. tostring(dy) .. " / " .. tostring(dz));
	self.real2dGroundSpeed = Utils.vector2Length(dx, dz);
	
	self.realMovingDirection = 0;
	if dz>0.0001 then
		if self.realInversedZAxis then
			self.realMovingDirection = -1;
		else
			self.realMovingDirection = 1;
		end;
	elseif dz<0.0001 then
		if self.realInversedZAxis then
			self.realMovingDirection = 1;
		else
			self.realMovingDirection = -1;
		end;
	end;	
	
	--[[else -- realGroundSpeed<0 -> adnormal, should never happen. (test case : attaching the Same Explorer 105 on the Amazone Cayena 6001) => it was applying a 0 force in realisticTool that was causing this issue...
	
		--print(self.time .. " " .. self.realVehicleName .. " type = " .. tostring(getRigidBodyType(self.realMainComponent.node)));
		print(self.time .. " " .. self.realVehicleName .. " RealisticVehicle.updateVehiclePosition -  2 velX / velY / velZ = " .. tostring(self.realVelX) .. " / " .. tostring(self.realVelY) .. " / " .. tostring(self.realVelZ));
		local bx, by, bz = getWorldTranslation(self.realMainComponent.node);
		print(self.time .. " " .. self.realVehicleName .. " WorldTranslation x,y,z = " .. tostring(bx) .. " / " .. tostring(by) .. " / " .. tostring(bz));
		self.realVelX = 0;
		self.realVelY = 0;
		self.realVelZ = 0;
		self.realGroundSpeed = 0;
		self.real2dGroundSpeed = 0;		
		self.realMovingDirection = 0;
		
	end;	]]
	
end;





function RealisticVehicle.updateVehicleExternalForces(self, dt)

	--local totalForce = 0;	
	local spd = self.realGroundSpeed;	
	
	--we add the rolling force only if the "resistanceForce" (for realistictool) is not in use (no ground contact)
	if spd>0.02 and self.realCurrentForceNeeded<0.1 then
		
		--local forceFx = 1;
		local rr = RealisticVehicle.getCurrentRollingResistance(self);
		local uAir = 1.22;		
		
		local rrForce = 0;
		
		if self.realMeasuredMassForRollingResistance or self.realTractionMode then			
			--now using real measured mass
			if self.realNumTouchingWheels>0 and math.abs(self.realAvgTouchingWheelSpd)>0.1 then
				RealisticVehicle.applyRollingResistance(self, spd, rr, self.realTotalMassOnTheWheels, dt);	
				--print(self.time .. " / " .. self.realVehicleName .. " -> traction Mode rr, mass = " .. tostring(self.realTotalMassOnTheWheels));
			end;
		else
			if self.realNbWheels==0 or self.realNumTouchingWheels==0 then -- vehicle has no wheel touching the ground. example : weight, cultivator
				--we add rolling resistance only if the vehicle is attached and the attacherVehicle is rolling(and not working, since in working mode we take into account the measured mass on the wheels)			
				if self.attacherVehicle~=nil and self.attacherVehicle.isRealistic and not self.attacherVehicle.realTractionMode and math.abs(self.attacherVehicle.realAvgTouchingWheelSpd)>0.1 then
					RealisticVehicle.applyRollingResistance(self, spd, rr, self.realComponentsCurrentMass + self.realLoadMass, dt);
					--print(self.time .. " / " .. self.realVehicleName .. " -> no wheel touching for rr and attacher vehicle not in tractionmode, mass = " .. tostring(self.realComponentsCurrentMass + self.realLoadMass));
				end;				
			else --vehicle has touching wheels that are actually rolling (do not apply rr for working equipment, because there is already the "resistance" of the tool to manage all resistances for it)
				if math.abs(self.realAvgTouchingWheelSpd)>0.1 and not self.realIsWorking then
					--print(self.time .. " " .. self.realVehicleName .. " - applying rr = " .. tostring(rr) .. " - mass = " .. tostring(self.realComponentsCurrentMass + self.realLoadMass));
					RealisticVehicle.applyRollingResistance(self, spd, rr, self.realComponentsCurrentMass + self.realLoadMass, dt);
					--print(self.time .. " / " .. self.realVehicleName .. " -> touching wheels for rr, mass = " .. tostring(self.realComponentsCurrentMass + self.realLoadMass));
				end;
			end;
		end;
		
		
		local SCx = RealisticVehicle.getCurrentSCx(self);
		
		if SCx>0 then
			local aForce = RealisticGlobalListener.airResistanceTuning * 0.001 * 0.5*uAir*SCx*spd^2; -- 0.001 because addForce is in KN
			--totalForce = totalForce + aForce;		
			aForce = aForce * dt / spd; -- divide by spd because we are multiplying by realVelX, realVelY and realVelZ
			addForce(self.realMainComponent.node, -aForce*self.realVelX, -aForce*self.realVelY, -aForce*self.realVelZ, self.realComX, self.realComY, self.realComZ, true);
		end;
		
		--print(self.time .. " name / totalForce = " .. tostring(self.realVehicleName) .. " / " .. tostring(totalForce));
		
	end;	
	
	--if Vehicle.debugRendering and self.realDebugRenderingInitialized and self.isControlled then		
	--	renderText(0.2, 0.16, 0.025, string.format("resistance force : %1.4f", totalForce));		
	--end;
	
end;









function RealisticVehicle.getTooFastSpeedToDisplay(self)
	local minSpeedLimit = 99999;
	for _,v in pairs(self.attachedImplements) do
		if v.object.realGetIsWorkingTooFast~=nil then
			local workingTooFast, speedLimit = v.object:realGetIsWorkingTooFast();	
			if workingTooFast then
				minSpeedLimit = math.min(minSpeedLimit, speedLimit);
			end;
		end;
	end;
	
	if self.realGetIsWorkingTooFast~=nil then
		local workingTooFast, speedLimit = self:realGetIsWorkingTooFast();	
		if workingTooFast then
			minSpeedLimit = math.min(minSpeedLimit, speedLimit);
		end;
	end;
	
	return minSpeedLimit<99999, minSpeedLimit;
end;




function RealisticVehicle.getCurrentSCx(self)

	if self.attacherVehicle~=nil then -- only apply SCx res force to main vehicle
		return 0;
	else
		--get the highest SCx between the "vehicle" and its implements
		local currentScx = self.realSCX;
		for k,implement in pairs(self.attachedImplements) do
			if implement.object.isRealistic then
				currentScx = math.max(currentScx, implement.object.realSCX);
			end;
		end;
		return currentScx;
	end;
	
end;

function RealisticVehicle.getCurrentRollingResistance(self)

	if self.attacherVehicle==nil then
		return self.realRollingResistance;
	end;
	
	if self.attacherVehicle.isRealistic and (self.realNbWheels==0 or self.realNumTouchingWheels==0) then
		return RealisticVehicle.getCurrentRollingResistance(self.attacherVehicle);
	else
		return self.realRollingResistance;
	end;
	
end;


function RealisticVehicle:registerExtraMassGetter(getterFunction)

	if self.realExtraMassGetterFunctions==nil then
		self.realExtraMassGetterFunctions = {};
	end;

	local num = table.getn(self.realExtraMassGetterFunctions);
	self.realExtraMassGetterFunctions[num+1] = getterFunction;

end;

function RealisticVehicle:getExtraMassSum()

	if self.realExtraMassGetterFunctions==nil then
		return 0;
	end;
	
	local extraMassSum = 0;
	local extraMassComTrans = {x=0, y=0, z=0};
	
	for i = 1, table.getn(self.realExtraMassGetterFunctions) do
		if self.realExtraMassGetterFunctions[i]~=nil then
			local extraMass, comTrans = self.realExtraMassGetterFunctions[i](self);
			extraMassSum = extraMassSum + extraMass;
			if comTrans~=nil then
				extraMassComTrans.x = extraMassComTrans.x + comTrans.x;
				extraMassComTrans.y = extraMassComTrans.y + comTrans.y;
				extraMassComTrans.z = extraMassComTrans.z + comTrans.z;
			end;
		end;
	end;
	
	return extraMassSum, extraMassComTrans;

end;




function RealisticVehicle:getImplementsNeeds()

	local toolNeedTraction = Utils.getNoNil(self.realToolNeedTraction, false);
	local powerConsumption = Utils.getNoNil(self.realCurrentPowerConsumption, 0);
	local forceNeeded = Utils.getNoNil(self.realCurrentForceNeeded, 0);
	local toolNeedEngineRev = Utils.getNoNil(self.realToolNeedEngineRev, false);
	
	--print(self.time .. " name = " .. tostring(self.realVehicleName) .. " - toolNeedTraction / powerConsumption / forceNeeded = " .. tostring(toolNeedTraction) .. " / " .. tostring(powerConsumption) .. " / " .. tostring(forceNeeded));
	
	for k, implement in pairs(self.attachedImplements) do
		if implement.object.isRealistic then
			--print(self.time .. " name = " .. tostring(implement.object.realVehicleName) .. " - getimplementsNeeds = " .. tostring(implement.object.getImplementsNeeds));
			--local iNeedTraction, iPowerConsumption, iForceNeeded;
			local iNeedTraction, iPowerConsumption, iForceNeeded, iNeedEngineRev = implement.object:getImplementsNeeds();
			toolNeedTraction = toolNeedTraction or iNeedTraction;
			powerConsumption = powerConsumption + iPowerConsumption;
			forceNeeded = forceNeeded + iForceNeeded;
			toolNeedEngineRev = toolNeedEngineRev or iNeedEngineRev;
		end;
	end;
	
	if powerConsumption<0.001 then
		powerConsumption = 0;
	end;
	
	return toolNeedTraction, powerConsumption, forceNeeded, toolNeedEngineRev;

end;





function RealisticVehicle.updateYCosAngle(self)

	local x, yCosValue, z = localDirectionToWorld(self.realMainComponent.node, 0, 1, 0);
	self.realYCosValue = Utils.clamp(yCosValue, 0, 1); -- only for rolling resistance correction
	
end;



function RealisticVehicle.applyRollingResistance(self, velocity, rrFx, mass, dt)

	local rrForce = self.realYCosValue * RealisticGlobalListener.rollingResistanceTuning * rrFx * mass * RealisticGlobalListener.gravity;	
	if rrForce~= 0 then	
		rrForce = rrForce * RealisticVehicle.getPositionRollingResistanceFactor(self);		
		
		rrForce = rrForce * dt / velocity; -- divide by velocity because we are multiplying by VelX, VelY and VelZ to turn the force toward the moving direction
		
		
		--print(self.time .. " name = " .. tostring(self.realVehicleName) .. " applying rolling resistance = " .. tostring(rrForce));
		
		addForce(self.realMainComponent.node, -rrForce*self.realVelX, -rrForce*self.realVelY, -rrForce*self.realVelZ, self.realComX, 0, self.realComZ, true);
		
	end;
end;


function RealisticVehicle.updateAnimatedWheelsYcorrection(self)
			
	if self.realHasAnimatedWheelsYcorrection and self.getAnimationTime~=nil then
	
		for k,animatedWheel in pairs(self.realAnimatedWheelsYcorrections) do
			--if self:getIsAnimationPlaying(animatedWheel.animationName) then
				local newYinc = RealisticUtils.linearFx(self:getAnimationTime(animatedWheel.animationName), animatedWheel.endYinc, animatedWheel.startYinc); -- getAnimationTime==1 when starting and 0 when stopping
				self.wheels[animatedWheel.wheelIndex].realTransYOffset2 = newYinc;
			--end;
		end;		
	
	end;

end;


function RealisticVehicle.realGetIsAiDriven(self)

	if self.realForceAiDriven then
		--print(self.time .. " RealisticVehicle.realGetIsAiDriven return true");
		return true;
	else
		--print(self.time .. " RealisticVehicle.realGetIsAiDriven return getIsHired");
		return self:getIsHired();
	end;

end;








function RealisticVehicle.realInitDebugRendering(self)

	

	self.realTestValue = 0;
	self.realTest2Value = 0;
	
	self.realTestTractionForce = 0;
	
	
	self.realDebugBrakeActivated = false;
	self.realDebugAcc = 0;
	
	
			
	self.realTotalMovingMass = 0;
	
	self.realDebugAcc = 0;
	
	
	
	-- __DURAL -> DEBUG
	self.testDistanceStarted = false;
	self.testFinBlocage = false;
	self.testTimer = 0;
	self.testSpeed = 0;
	self.testDist1 = 0;
	self.testDist2 = 0;
	self.testAcc = 0;
	self.testLastAcceleration = 0;	
	self.realDebugDt = 16.667;
	
	
	for _, wheel in pairs(self.wheels) do			
		wheel.realDebugActMotorTorque = 0;
		wheel.realDebugStiffCoeff = 0;
		wheel.realDebugBrakeForce = 0;
		wheel.realDebugWheelForce = 0;
		wheel.realDebugMaxPossibleForce = 0;
		--wheel.realDebugCurrentForce = 0;		
		wheel.realDebugMotorBrakeForce = 0;
	end;
	
	--self.realDebugSpdFx = 0;
	--self.realDebugMotorFx = 0;
	
	
	self.realDebugRenderingInitialized = true;

end;









--*********************************** DURAL **************************************
--** modification to override default values and manage different rotMin & rotMax but with the same time to turn the wheel fully
function RealisticVehicle.loadDynamicWheelDataFromXML(self, xmlFile, wheelnamei, wheel)



	--override default values
	if SpecializationUtil.hasSpecialization(Steerable, self.specializations) then
		wheel.lateralExtremumSlip = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralExtremumSlip"), 0);
		wheel.lateralExtremumValue = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralExtremumValue"), 0.05);
		wheel.lateralAsymptoteSlip = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralAsymptoteSlip"), 1);
		wheel.lateralAsymptoteValue = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralAsymptoteValue"), 0.025);	
	else
		wheel.lateralExtremumSlip = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralExtremumSlip"), 0.2);
		wheel.lateralExtremumValue = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralExtremumValue"), 0.04);
		wheel.lateralAsymptoteSlip = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralAsymptoteSlip"), 1);
		wheel.lateralAsymptoteValue = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralAsymptoteValue"), 0.02);
	end;
	
	wheel.lateralStiffness = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#lateralStiffness"), 0.5);

	wheel.longitudalExtremumSlip = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalExtremumSlip"), 1);
	wheel.longitudalExtremumValue = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalExtremumValue"), 0.02);
	wheel.longitudalAsymptoteSlip = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalAsymptoteSlip"), 2.0);
	wheel.longitudalAsymptoteValue = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalAsymptoteValue"), 0.01);
	wheel.longitudalStiffness = Utils.getNoNil(getXMLFloat(xmlFile, wheelnamei .. "#longitudalStiffness"), 0.5);
	

	if self.isServer then
		setWheelShapeTireFunction(wheel.node, wheel.wheelShape, false, wheel.lateralExtremumSlip, wheel.lateralExtremumValue, wheel.lateralAsymptoteSlip, wheel.lateralAsymptoteValue, 1000000*wheel.lateralStiffness);
		setWheelShapeTireFunction(wheel.node, wheel.wheelShape, true, wheel.longitudalExtremumSlip, wheel.longitudalExtremumValue, wheel.longitudalAsymptoteSlip, wheel.longitudalAsymptoteValue, 1000000*wheel.longitudalStiffness);
	end;

	--wheel.maximalLongitudialTireForceSlip = WheelsUtil.findSlipWithMaximalTireForce(wheel.longitudalExtremumSlip, wheel.longitudalExtremumValue, wheel.longitudalAsymptoteSlip, wheel.longitudalAsymptoteValue);

	
	-- for realistic vehicle, adding minRotSpeed and maxRotSpeed
	wheel.realRotMaxSpeed = 0;
	wheel.realRotMinSpeed = 0;
	
	--print("RealisticVehicle.loadDynamicWheelDataFromXML - wheel.rotMax="..tostring(wheel.rotMax) .. " / wheel.rotMin="..tostring(wheel.rotMin) .. " /  wheel.rotSpeed="..tostring(wheel.rotSpeed));
	
	if wheel.rotMax~=0 and wheel.rotMin~=0 and wheel.rotSpeed~=0 then	
	
		if math.abs(wheel.rotMax)>math.abs(wheel.rotMin) then
			wheel.realRotMaxSpeed = wheel.rotSpeed;
			wheel.realRotMinSpeed = math.abs(wheel.rotMin/wheel.rotMax)*wheel.rotSpeed;
		else
			wheel.realRotMinSpeed = wheel.rotSpeed;
			wheel.realRotMaxSpeed = math.abs(wheel.rotMax/wheel.rotMin)*wheel.rotSpeed;
		end;
		
		if wheel.rotSpeed<0 then
			local tmp = wheel.realRotMaxSpeed;
			wheel.realRotMaxSpeed = wheel.realRotMinSpeed;
			wheel.realRotMinSpeed = tmp;
		end;
	
	end;
	
	--override defaut ai rotate speed	
	self.aiSteeringSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.aiSteeringSpeed"), self.maxRotTime)*0.001;
	
end





function RealisticVehicle.getPositionRollingResistanceFactor(self)

	self.realVehicleOnField = false;

	local fx = 1;
	if self.realTractionMode then
		fx = 1.25;	
	end;
	
	--[[
	if self.realTractionMode then
		fx = 2.5;		
		if self.realGroundSpeed>0.3 then -- 0.3 = 1.08kph
			--additionnal rolling resistance if slipping -- take into account wheels floation factor
			fx = fx+4*math.max(0, self.realAvgTouchingMotorizedWheelSlipS-0.12)/self.realMotorizedFlotationFx; -- on field rolling resistance factor	-> no malus under 12% of slip			
		end;
		--increased rr should take into account mass on motorized wheel
		if self.realTotalMassOnTheWheels>0 then
			fx = 2 + (fx - 2)*self.realTotalMassOnMotorizedWheels/self.realTotalMassOnTheWheels; -- fx*massonmoto/totalmass + 2*(totalmass-massonmoto)/totalmass
		end;
	else]]
	local terrainDetailRequiredMask = bitOR(0, 2^g_currentMission.cultivatorChannel);
	terrainDetailRequiredMask = bitOR(terrainDetailRequiredMask, 2^g_currentMission.ploughChannel);
	terrainDetailRequiredMask = bitOR(terrainDetailRequiredMask, 2^g_currentMission.sowingChannel);
	terrainDetailRequiredMask = bitOR(terrainDetailRequiredMask, 2^g_currentMission.sowingWidthChannel);--potatoes dams		
	local x,_,z = getWorldTranslation(self.realMainComponent.node);
	local detailId = g_currentMission.terrainDetailId;
	--local x,z, widthX,widthZ, heightX,heightZ = Utils.getXZWidthAndHeight(detailId, x1, z1, x1, z1, x1, z1);
	
	--print(self.time .. " position X,Z = " .. tostring(x1) .. " / " .. tostring(z1) .. " result x,z = " .. tostring(x) .. " / " .. tostring(z));
	--print(self.time .. " widthX,widthZ, heightX,heightZ = " .. tostring(widthX) .. " / " .. tostring(widthZ) .. " / " .. tostring(heightX) .. " / " .. tostring(heightZ));

	setDensityCompareParams(detailId, "greater", 0, 0, terrainDetailRequiredMask, 0);

	_,area,_ = getDensityParallelogram(detailId, x, z, 0, 0, 0, 0, g_currentMission.terrainDetailAIFirstChannel, g_currentMission.terrainDetailAINumChannels);
	
		
	
	
	if area>0 then
	
		self.realVehicleOnField = true;
	
		fx = 2*fx + 2*RealisticGlobalListener.realRainFx/self.realVehicleFlotationFx; -- on field rolling resistance factor
		
		if self.realIsMotorized and self.realGroundSpeed>0.3 and self.realAvgTouchingMotorizedWheelSlipS2>0.12 then -- 0.3 = 1.08kph
			--additionnal rolling resistance if slipping, because the wheel dig the ground -- take into account wheels floation factor
			local fx2 = 2*math.max(0, self.realAvgTouchingMotorizedWheelSlipS2-0.12)/self.realVehicleFlotationFx; -- on field rolling resistance factor	-> no malus under 12% of slip
						
			--increased rr should take into account mass on motorized wheel
			if self.realTotalMassOnTheWheels>0 then
				fx2 = fx2*self.realTotalMassOnMotorizedWheels/self.realTotalMassOnTheWheels;
			end;			
			fx = fx+fx2;
			
		end;
		
		--add some randomness in field
		fx = fx * (0.9 + 0.2*math.random());
		
	end;
	
	

	--reset compareParams
	setDensityCompareParams(detailId, "greater", -1);
	--end;
	
	return fx;
	
end;



function RealisticVehicle:realOnDesactivate()
	
	--since we don't update wheel graphic anymore, set the axlespeed variable to 0 (avoid that motorized wheel with no ground contact spin when entering a stopped vehicle)
	for k,wheel in pairs(self.wheels) do
		wheel.axleSpeed = 0;
	end;

end;




function RealisticVehicle.realSetBackAttacherJointValues(vehicleAttacherJoint)

	vehicleAttacherJoint.moveTime = vehicleAttacherJoint.realDefaultMoveTime;
	
	if vehicleAttacherJoint.rotationNode ~= nil and vehicleAttacherJoint.jointTransform~=nil then
		for i=1,3 do
			vehicleAttacherJoint.maxRot[i] = vehicleAttacherJoint.realDefaultMaxRot[i];
			vehicleAttacherJoint.maxRot2[i] = vehicleAttacherJoint.realDefaultMaxRot2[i];
			vehicleAttacherJoint.minRot[i] = vehicleAttacherJoint.realDefaultMinRot[i];
			vehicleAttacherJoint.minRot2[i] = vehicleAttacherJoint.realDefaultMinRot2[i];
			vehicleAttacherJoint.maxRotLimit[i] = vehicleAttacherJoint.realDefaultMaxRotLimit[i];
			vehicleAttacherJoint.minRotLimit[i] = vehicleAttacherJoint.realDefaultMinRotLimit[i];
			vehicleAttacherJoint.maxTransLimit[i] = vehicleAttacherJoint.realDefaultMaxTransLimit[i];
			vehicleAttacherJoint.minTransLimit[i] = vehicleAttacherJoint.realDefaultMinTransLimit[i];
		end;
	end;
	
	--print("setting back attacher joint values - transLimit y = " .. tostring(vehicleAttacherJoint.maxTransLimit[2]));
	
end;



function RealisticVehicle.realSetAttacherJointValuesFromImplement(vehicleAttacherJoint, implementAttacherJoint)


	--[[
	local d0 = attacherJoint.realZeroRotHeightDistance;
	local dg0 = groundRaycastResult.groundDistance;
	local dg1 = attacherJoint.maxRotDistanceToGround;
	
	local suggestedValue = math.deg(math.asin((d0+dg0-dg1)/attacherJoint.realBottomArmsLength));
	if attacherJoint.realPositiveRotationTowardGround then
		suggestedValue = -suggestedValue;
	end;]]



	--setting the new maxRot
	local sinRotLowered = (vehicleAttacherJoint.realZeroRotHeightDistance+vehicleAttacherJoint.maxRotDistanceToGround-implementAttacherJoint.lowerDistanceToGround)/vehicleAttacherJoint.realBottomArmsLength;
	if vehicleAttacherJoint.realPositiveRotationTowardGround then
		sinRotLowered = -sinRotLowered;
	end;
	sinRotLowered = Utils.clamp(sinRotLowered, -1, 1);
	local newRotLowered = math.asin(sinRotLowered);			
	if vehicleAttacherJoint.maxRot[1]>vehicleAttacherJoint.minRot[1] then
		newRotLowered = Utils.clamp(newRotLowered, vehicleAttacherJoint.minRot[1], vehicleAttacherJoint.maxRot[1]);
	else
		newRotLowered = Utils.clamp(newRotLowered, vehicleAttacherJoint.maxRot[1], vehicleAttacherJoint.minRot[1]);
	end;			
	--print(RealisticGlobalListener.realMissionTime .. string.format(" ZeroRotHDist=%1.2f / ILowerDistToGrnd=%1.2f / HitchLoweredDistToGrnd=%1.2f / ArmsLength=%1.2f / newRotLowered=%1.2f",vehicleAttacherJoint.realZeroRotHeightDistance,implementAttacherJoint.lowerDistanceToGround,vehicleAttacherJoint.maxRotDistanceToGround,vehicleAttacherJoint.realBottomArmsLength,math.deg(newRotLowered)));

	
	--setting the new minRot
	local sinRotRaised = (vehicleAttacherJoint.realZeroRotHeightDistance+vehicleAttacherJoint.maxRotDistanceToGround-implementAttacherJoint.upperDistanceToGround)/vehicleAttacherJoint.realBottomArmsLength;
	if vehicleAttacherJoint.realPositiveRotationTowardGround then
		sinRotRaised = -sinRotRaised;
	end;
	sinRotRaised = Utils.clamp(sinRotRaised, -1, 1);
	local newRotRaised = math.asin(sinRotRaised);
	if vehicleAttacherJoint.maxRot[1]>vehicleAttacherJoint.minRot[1] then
		newRotRaised = Utils.clamp(newRotRaised, vehicleAttacherJoint.minRot[1], vehicleAttacherJoint.maxRot[1]);
	else
		newRotRaised = Utils.clamp(newRotRaised, vehicleAttacherJoint.maxRot[1], vehicleAttacherJoint.minRot[1]);
	end;	
	--print(RealisticGlobalListener.realMissionTime .. string.format(" ZeroRotHDist=%1.2f / IUpperDistToGrnd=%1.2f / HitchRaisedDistToGrnd=%1.2f / ArmsLength=%1.2f / newRotRaised=%1.2f",vehicleAttacherJoint.realZeroRotHeightDistance,implementAttacherJoint.upperDistanceToGround,vehicleAttacherJoint.minRotDistanceToGround,vehicleAttacherJoint.realBottomArmsLength,math.deg(newRotRaised)));


	
	
	--setting the new maxRot2
	if vehicleAttacherJoint.realPositiveRotationTowardGround then
		vehicleAttacherJoint.maxRot2[1] = -newRotLowered + vehicleAttacherJoint.realRotRotationOffset + implementAttacherJoint.realWantedLoweredRot2;
	else
		vehicleAttacherJoint.maxRot2[1] = -newRotLowered + vehicleAttacherJoint.realRotRotationOffset - implementAttacherJoint.realWantedLoweredRot2;
	end;
	--print(self.time .. string.format(" maxRot2 = %1.2f / realRotRotationOffset = %1.2f / realWantedLoweredRot2 = %1.2f",math.deg(vehicleAttacherJoint.maxRot2[1]), math.deg(vehicleAttacherJoint.realRotRotationOffset), math.deg(object.attacherJoint.realWantedLoweredRot2)));
	
	--setting the new minRot2
	local newMinRot2 = 0;
	if implementAttacherJoint.realWantedRaisedRot2 ~= nil then
		if vehicleAttacherJoint.realPositiveRotationTowardGround then
			newMinRot2 =  -newRotRaised + implementAttacherJoint.realWantedRaisedRot2;
		else
			newMinRot2 =  -newRotRaised - implementAttacherJoint.realWantedRaisedRot2;
		end;
	else -- using the realWantedMinRotInc
		--computing the height difference between the lowered and the raised position
		-- al-kashi / cosinus law 
		--[[print(self.time .. string.format(" armLen = % 1.3f / maxRot = %1.2f / minRot = %1.2f ",vehicleAttacherJoint.realBottomArmsLength, math.deg(vehicleAttacherJoint.maxRot[1]), math.deg(vehicleAttacherJoint.minRot[1])));
		local heightDiff = vehicleAttacherJoint.realBottomArmsLength*(2-2*math.cos(math.abs(vehicleAttacherJoint.maxRot[1]-vehicleAttacherJoint.minRot[1])))^0.5;
		
		newMinRot2 = newMinRot2 + heightDiff * object.attacherJoint.realWantedMinRotInc;
		if vehicleAttacherJoint.maxRot[1]>vehicleAttacherJoint.minRot[1] then
			newMinRot2 =  -vehicleAttacherJoint.maxRot[1] + newMinRot2;
		else
			newMinRot2 =  -vehicleAttacherJoint.maxRot[1] - newMinRot2;
		end;
		print(self.time .. " attachImplement - heightDiff = " .. tostring(heightDiff) .. " / newMinRot2 = " .. math.deg(newMinRot2));
		]]
		
		--  inc to add by  lift	
		local angleDifference = math.abs(newRotLowered-newRotRaised);
		if vehicleAttacherJoint.realPositiveRotationTowardGround then
			newMinRot2 =  vehicleAttacherJoint.maxRot2[1] + math.deg(angleDifference)*implementAttacherJoint.realWantedRaisedRotInc;
		else
			newMinRot2 =  vehicleAttacherJoint.maxRot2[1] - math.deg(angleDifference)*implementAttacherJoint.realWantedRaisedRotInc;
		end;
		--print(RealisticGlobalListener.realMissionTime .. " attachImplement - angleDiff = " .. math.deg(angleDifference) .. " / newMinRot2 = " .. math.deg(newMinRot2) .. " / realWantedRaisedRotInc = " .. math.deg(implementAttacherJoint.realWantedRaisedRotInc));
		
	end;	
	
	
	
	vehicleAttacherJoint.minRot2[1] = newMinRot2;
	
	for i=1,3 do
	
		--setting new maxRotLimit
		vehicleAttacherJoint.maxRotLimit[i] = implementAttacherJoint.realWantedLoweredRotLimit[i];
		
		--setting new maxTransLimit
		vehicleAttacherJoint.maxTransLimit[i] = implementAttacherJoint.realWantedLoweredTransLimit[i];
		
		--setting new minRotLimit
		vehicleAttacherJoint.minRotLimit[i] = implementAttacherJoint.realWantedRaisedRotLimit[i];
		
		--setting new minTransLimit
		vehicleAttacherJoint.minTransLimit[i] = implementAttacherJoint.realWantedRaisedTransLimit[i];
		
	end;
	
	--setting new jointMoveTime
	local newAngleDifference = math.abs(newRotLowered-newRotRaised);
	if vehicleAttacherJoint.realDefaultLiftHeight>0 then
		--taking into consideration lifting height difference (included the y transLimit)
		-- al-kashi / cosinus law 
		local newHeightDiff = vehicleAttacherJoint.realBottomArmsLength*(2-2*math.cos(math.abs(newRotLowered-newRotRaised)))^0.5;
		newHeightDiff = newHeightDiff + vehicleAttacherJoint.maxTransLimit[2] - vehicleAttacherJoint.minTransLimit[2];
		vehicleAttacherJoint.realBaseMoveTime = vehicleAttacherJoint.realDefaultMoveTime * newHeightDiff / vehicleAttacherJoint.realDefaultLiftHeight;			
	else
		vehicleAttacherJoint.realBaseMoveTime = vehicleAttacherJoint.realDefaultMoveTime;
	end;
	
	vehicleAttacherJoint.maxRot[1] = newRotLowered;
	vehicleAttacherJoint.minRot[1] = newRotRaised;
	
	--print(self.time .. string.format("default time = %1.0f / new base time = %1.0f",vehicleAttacherJoint.realDefaultMoveTime, vehicleAttacherJoint.realBaseMoveTime));
	--print(RealisticGlobalListener.realMissionTime .. string.format(" new values - maxRot = %1.1f / maxRot2 = %1.1f / minRot = %1.1f / minRot2 = %1.1f", math.deg(vehicleAttacherJoint.maxRot[1]),math.deg(vehicleAttacherJoint.maxRot2[1]),math.deg(vehicleAttacherJoint.minRot[1]),math.deg(vehicleAttacherJoint.minRot2[1])));

end;





--*************************** DURAL ***********************************
--** more accurate version
function RealisticVehicle.consoleCommandAnalyze2(unusedSelf)

	if g_currentMission ~= nil and g_currentMission.controlledVehicle ~= nil and g_currentMission.controlledVehicle.isServer then
		local self = g_currentMission.controlledVehicle;

		print("Analyzing vehicle '"..self.configFileName.."'. Make sure vehicle is standing on a flat plane parallel to xz-plane");

		local groundRaycastResult = {
		  raycastCallback = function (self, transformId, x, y, z, distance)				
				--print("raycast all - transformid = '" .. tostring(transformId) .. "' / distance = " .. tostring(distance));				
				--print("         g_currentMission.terrainRootNode = '" .. tostring(g_currentMission.terrainRootNode) .. "'");
		  
				if transformId==g_currentMission.terrainRootNode then
					--print("        ground terrain found");
					self.groundDistance = distance;
					return false;
				end;
				--continue searching for terrain collision
				--raycastAll is multithreaded, and so, never reste the wanted value within the callback function
				--self.groundDistance = 0;
				return true;
		  end
		};
		
		
		for i, attacherJoint in ipairs(self.attacherJoints) do
		  
			--check if an implement is attached to this joint
			local implementIndex = self:getImplementIndexByJointDescIndex(i);
			if implementIndex ~= nil then	
				local implement = self.attachedImplements[implementIndex];
				if implement.object ~= nil then					
					-- display the implement attacher node distance to ground
					local iname = implement.realVehicleName;
					if iname==nil then
						iname = implement.object.typeDesc;
					end;
					local x,y,z = getWorldTranslation(implement.object.attacherJoint.node);
					groundRaycastResult.groundDistance = 0;					
					--print("--** calling raycast all ** --");
					raycastAll(x, y, z, 0, -1, 0, "raycastCallback", 4, groundRaycastResult);					
					print(" Attacher joint "..i.." has an implement attached '" .. iname .. "' -> its attacher distance to ground is : " .. groundRaycastResult.groundDistance);
				end;
			else
		  
				  local trx, try, trz = getRotation(attacherJoint.jointTransform);
				  setRotation(attacherJoint.jointTransform, unpack(attacherJoint.jointOrigRot));
				  if attacherJoint.rotationNode ~= nil or attacherJoint.rotationNode2 ~= nil then
					  local rx,ry,rz;
					  if attacherJoint.rotationNode ~= nil then
						  rx,ry,rz = getRotation(attacherJoint.rotationNode);
					  end
					  local rx2,ry2,rz2;
					  if attacherJoint.rotationNode2 ~= nil then
						  rx2,ry2,rz2 = getRotation(attacherJoint.rotationNode2);
					  end

					  -- test min rot
					  if attacherJoint.rotationNode ~= nil then
						  setRotation(attacherJoint.rotationNode, unpack(attacherJoint.minRot));
					  end
					  if attacherJoint.rotationNode2 ~= nil then
						  setRotation(attacherJoint.rotationNode2, unpack(attacherJoint.minRot2));
					  end
					  local x,y,z = getWorldTranslation(attacherJoint.jointTransform);
					  groundRaycastResult.groundDistance = 0;
					  raycastAll(x, y, z, 0, -1, 0, "raycastCallback", 4, groundRaycastResult);
					  --********************************************************************************  DURAL **********************
					  if math.abs(groundRaycastResult.groundDistance - attacherJoint.minRotDistanceToGround) > 0.005 then -- 5 mm
						  print(" Issue found: Attacher joint "..i.." has invalid minRotDistanceToGround. True value is: "..groundRaycastResult.groundDistance.. ". Value in XML: "..attacherJoint.minRotDistanceToGround);
					  end
					  local dx,dy,dz = localDirectionToWorld(attacherJoint.jointTransform, 0, 1, 0);
					  local angle = math.deg(math.acos(Utils.clamp(dy, -1, 1)));
					  local dxx,dxy,dxz = localDirectionToWorld(attacherJoint.jointTransform, 1, 0, 0);
					  if dxy < 0 then
						  angle = -angle;
					  end
					  
					   --********************************************************************************  DURAL **********************
					  if math.abs(angle-math.deg(attacherJoint.minRotRotationOffset)) > 0.2 then
						  print(" Issue found: Attacher joint "..i.." has invalid minRotRotationOffset. True value is: "..angle.. ". Value in XML: "..math.deg(attacherJoint.minRotRotationOffset));
					  end
					  
					  
					  
					  

					  -- test max rot
					  if attacherJoint.rotationNode ~= nil then
						  setRotation(attacherJoint.rotationNode, unpack(attacherJoint.maxRot));						 
					  end
					  if attacherJoint.rotationNode2 ~= nil then
						  setRotation(attacherJoint.rotationNode2, unpack(attacherJoint.maxRot2));						 
					  end
					  
					  
					  local x,y,z = getWorldTranslation(attacherJoint.jointTransform);
					  groundRaycastResult.groundDistance = 0;
					  raycastAll(x, y, z, 0, -1, 0, "raycastCallback", 4, groundRaycastResult);
					  
					  --********************************************************************************  DURAL **********************
					  if math.abs(groundRaycastResult.groundDistance - attacherJoint.maxRotDistanceToGround) > 0.005 then -- 5 mm
						--****************  compute a suggested value for maxRot
							if self.isRealistic then
							
								--local sinRotRaised = (vehicleAttacherJoint.realZeroRotHeightDistance+implementAttacherJoint.upperDistanceToGround-vehicleAttacherJoint.maxRotDistanceToGround)/vehicleAttacherJoint.realBottomArmsLength;
							
								local d0 = attacherJoint.realZeroRotHeightDistance;
								local dg0 = groundRaycastResult.groundDistance;
								local dg1 = attacherJoint.maxRotDistanceToGround;
								
								local suggestedValue = math.deg(math.asin((d0+dg0-dg1)/attacherJoint.realBottomArmsLength));
								if attacherJoint.realPositiveRotationTowardGround then
									suggestedValue = -suggestedValue;
								end;
								--local suggestedValue = math.deg(attacherJoint.maxRot[1]*(D+groundRaycastResult.groundDistance-attacherJoint.maxRotDistanceToGround)/D);
							
								--print("d0 value = " .. tostring(d0) .. " armLength = " .. attacherJoint.realBottomArmsLength);
								
								--local suggestedValue = math.deg(math.asin((D+groundRaycastResult.groundDistance-attacherJoint.maxRotDistanceToGround)/attacherJoint.realBottomArmsLength));
							
								--local suggestedValue = math.deg(math.asin((attacherJoint.realZeroRotHeightDistance+attacherJoint.maxRotDistanceToGround-groundRaycastResult.groundDistance)/attacherJoint.realBottomArmsLength));
								print(string.format(" Issue found: Attacher joint %d has invalid maxRotDistanceToGround. True value: %1.3f. Value in XML: %1.3f. You can set 'maxRot value to : %1.3f",i,groundRaycastResult.groundDistance,attacherJoint.maxRotDistanceToGround,suggestedValue));
							else
								print(" Issue found: Attacher joint "..i.." has invalid maxRotDistanceToGround. True value: "..groundRaycastResult.groundDistance.. ". Value in XML: "..attacherJoint.maxRotDistanceToGround);
							end;
					  end
					  
					  --[[
					  
					  if attacherJoint.rotationNode ~= nil then
						  --setRotation(attacherJoint.rotationNode, unpack(attacherJoint.maxRot));
						  setRotation(attacherJoint.rotationNode, 0,0,0);
						  --local rotX, rotY, rotZ = unpack(attacherJoint.maxRot);
						  --print(string.format("    testing maxRot - setting rotationNode rotation to : %1.2f %1.2f %1.2f", math.deg(rotX),math.deg(rotY), math.deg(rotZ)));
					  end
					  if attacherJoint.rotationNode2 ~= nil then
						  --setRotation(attacherJoint.rotationNode2, unpack(attacherJoint.maxRot2));
						  setRotation(attacherJoint.rotationNode2, 0,0,0);
						  --local rotX, rotY, rotZ = unpack(attacherJoint.maxRot2);
						  --print(string.format("    testing maxRot - setting rotationNode2 rotation to : %1.2f %1.2f %1.2f", math.deg(rotX),math.deg(rotY), math.deg(rotZ)));
					  end
					  
					  
					  
					  setRotation(attacherJoint.jointTransform, 0,0,0);--]]
					  local dx,dy,dz = localDirectionToWorld(attacherJoint.jointTransform, 0, 1, 0);
					  --[[
					  local d2x, d2y, d2z = localDirectionToWorld(self.realMainComponent.node, 0, 1, 0);
					  local d3x, d3y, d3z = localDirectionToWorld(attacherJoint.rotationNode, 0, 1, 0);
					  local d4x, d4y, d4z = localDirectionToWorld(getParent(attacherJoint.rotationNode), 0, 1, 0);
					  
					  print("     parent(rotNode) = " .. tostring(getParent(attacherJoint.rotationNode)) .. " / main component = " .. tostring(self.realMainComponent.node));
					  
					  print("     ---- world y 1 value = " .. tostring(dy) .. " / component y 1 value = " .. tostring(d2y) .. " / rotNode y 1 value = " .. tostring(d3y) .. " / parent(rotNode) y 1 value = " .. tostring(d4y));
					  ]]
					  
					  
--******************************  ONLY WORKS IF Y SCALE OF attacherJoint.jointTransform and its parents is 1 **************************************
					  local angle = math.deg(math.acos(Utils.clamp(dy, -1, 1)));
					  local dxx,dxy,dxz = localDirectionToWorld(attacherJoint.jointTransform, 1, 0, 0);
					  if dxy < 0 then
						  angle = -angle;
					  end
					  
					   --********************************************************************************  DURAL **********************
					  if math.abs(angle-math.deg(attacherJoint.maxRotRotationOffset)) > 0.2 then
						  print(" Issue found: Attacher joint "..i.." has invalid maxRotRotationOffset. True value is: "..angle.. ". Value in XML: "..math.deg(attacherJoint.maxRotRotationOffset));
					  end

					  -- reset rotations
					  if attacherJoint.rotationNode ~= nil then
						  setRotation(attacherJoint.rotationNode, rx,ry,rz);
					  end
					  if attacherJoint.rotationNode2 ~= nil then
						  setRotation(attacherJoint.rotationNode2, rx2,ry2,rz2);
					  end
				  end
				  setRotation(attacherJoint.jointTransform, trx, try, trz);
				end -- end implement in use with the joint
  
			end;   -- end for each attacherJoint
  
          return "Analyzed vehicle";
      end;
  
      return "Failed to reload vehicle from xml. Invalid controlled vehicle";
  end;
  
  
  addConsoleCommand("testvehicle", "Analyze vehicle V2", "RealisticVehicle.consoleCommandAnalyze2", nil);



