--
-- RainPlanes
--
-- @author:    	Xentro (www.fs-uk.com)(Marcus@Xentro.se)
-- @version:    v0.6
-- @date:       2013-04-14
-- 

RainPlane = {};

local RainPlane_mt = Class(RainPlane, Object);
InitObjectClass(RainPlane, "RainPlane");

function RainPlane.RainPlane(id)
    local instance = RainPlane:new(g_server ~= nil, g_client ~= nil);
	if instance:load(id) then
		g_currentMission:addOnCreateLoadedObject(instance);
		instance:register(true);
	else
		instance:delete();
	end;
end;
g_onCreateUtil.addOnCreateFunction("RainPlane", RainPlane.RainPlane);

function RainPlane:new(isServer, isClient)
    local self = Object:new(isServer, isClient, RainPlane_mt);
    self.className = "RainPlane";
	
	self.floodVehicles = {};
	
	self.rainPlaneDirtyFlag = self:getNextDirtyFlag();
	
    return self;
end;

function RainPlane:load(id)
	self.fillTime = 0;
	self.fillScale = Utils.getNoNil(getUserAttribute(id, "fillScale"), 3);
	self.maxFillTime = Utils.getNoNil(getUserAttribute(id, "maxTime"), 240); -- game minutes
	
	self.slipScale = Utils.getNoNil(getUserAttribute(id, "slipScale"), 0.8);
	
	self.activateFloodTime = Utils.getNoNil(getUserAttribute(id, "activateFloodTime"), self.maxFillTime / 2);
	local trigger = getUserAttribute(id, "triggerIndex");
	if trigger ~= nil and trigger ~= "" then
		self.triggerId = Utils.indexToObject(id, trigger);
		
		if self.triggerId ~= nil then
			self.isEnabled = true;
			addTrigger(self.triggerId, "triggerCallback", self);
		end;
	end;
	
	self.planeId = id;
	if self.planeId ~= nil then
		local _, minY, _ = getWorldTranslation(self.planeId);
		self.planeMinY = minY;
		self.planeMaxY = Utils.getNoNil(getUserAttribute(self.planeId, "planeMaxY"), minY + 1);
		
		local x,y,z = getTranslation(self.planeId);
		y = self.planeMinY + (self.fillTime / self.maxFillTime) * (self.planeMaxY - self.planeMinY);
		setTranslation(self.planeId, x, y, z);
	else
		return false;
    end;
	
	g_currentMission:addOnCreateLoadedObjectToSave(self)
	return true;
end;

function RainPlane:delete()
	g_currentMission:removeOnCreateLoadedObjectToSave(self)
	
	if self.triggerId ~= nil then
		removeTrigger(self.triggerId);
	end;
end;

function RainPlane:readStream(streamId, connection)
end;

function RainPlane:writeStream(streamId, connection)
end;

function RainPlane:readUpdateStream(streamId, timestamp, connection)
	if connection:getIsServer() then
		self.fillTime = streamReadFloat32(streamId);
		
		if self.planeId ~= nil then
			local x,y,z = getTranslation(self.planeId);
			y = self.planeMinY + (self.fillTime / self.maxFillTime) * (self.planeMaxY - self.planeMinY);
			setTranslation(self.planeId, x, y, z);
		end;
    end;
end;

function RainPlane:writeUpdateStream(streamId, connection, dirtyMask)
    if not connection:getIsServer() then
		streamWriteFloat32(streamId, self.fillTime);
    end;
end;

function RainPlane:loadFromAttributesAndNodes(xmlFile, tag)
	local rainLevel = getXMLFloat(xmlFile, tag .. "#rainLevel");
	if rainLevel ~= nil then
		self:updateMoving(rainLevel);
	end;
	return true;
end;

function RainPlane:getSaveAttributesAndNodes()
	local attributes = 'rainLevel="' .. self.fillTime .. '"';
	local nodes = "";
	return attributes, nodes;
end;

function RainPlane:update(dt)
	if self.isServer then
		local dtMinutes = dt / 60000 * g_currentMission.environment.timeScale;
		
		if g_currentMission.environment.currentRain ~= nil and g_currentMission.environment.currentRain.rainTypeId == "rain" and g_currentMission.environment.lastRainScale > 0.01 then
			self:updateMoving(self.fillTime + (dtMinutes * self.fillScale));
		elseif g_currentMission.environment.timeSinceLastRain > 0 and self.fillTime ~= 0 then
			local fillTime = self.fillTime - dtMinutes;
			self:updateMoving(fillTime);
		end;
	end;
end;

function RainPlane:updateTick(dt)
	if self.isServer then
		for vehicle in pairs(self.floodVehicles) do
			if self.fillTime > self.activateFloodTime then
				if not vehicle.isBroken then
					vehicle.isBroken = true;
					g_currentMission:onSunkVehicle(vehicle);
					
					if vehicle.isEntered then
						g_currentMission:onLeaveVehicle();
					end;
				else
					if vehicle.isFlooded == nil then
						vehicle.isFlooded = true;
					end;
				end;
			else
				if vehicle.isBroken and vehicle.isFlooded == nil then
					vehicle.isBroken = false;
				else
					vehicle.isFlooded = nil;
				end;
			end;
		end;
	end;
end;

function RainPlane:updateMoving(delta)
	self.fillTime = math.min(delta, self.maxFillTime);
	
	if self.fillTime < 0 then
		self.fillTime = 0;
	end;
	
	if self.planeId ~= nil then
		local x,y,z = getTranslation(self.planeId);
		y = self.planeMinY + (self.fillTime / self.maxFillTime) * (self.planeMaxY - self.planeMinY);
		setTranslation(self.planeId, x, y, z);
	end;
	
	if self.isServer then
		self:raiseDirtyFlags(self.rainPlaneDirtyFlag);
	end;
end;


function RainPlane:triggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	if self.isEnabled ~= nil and self.isEnabled then
		local vehicle = g_currentMission.nodeToVehicle[otherId];
		if vehicle ~= nil then
			if onEnter then
				if self.floodVehicles[vehicle] == nil then
					self.floodVehicles[vehicle] = true;
				end;
			elseif onLeave then
				if self.floodVehicles[vehicle] ~= nil then
					self.floodVehicles[vehicle] = nil;
				end;
			end;
		end;
	end;
end;