--
-- PdaPlugin_Prices
--
-- @author  Decker_MMIV - fs-uk.com, forum.farming-simulator.com, modhoster.com
-- @date    2013-03-21
--
-- @history
--  2013-March
--      v0.01   - Initial experiment
--  2013-April
--      v0.03   - Misc. changes.
--      v0.04   - Included with PdaMod.
--  2013-June
--      v0.06   - Better support for 'Great Demand' prices.
--              - Fixed aspect ratio on fruit-icons.
--      v0.07   - Page with reference prices for fill-types added.
--  2013-August
--      v0.08   - Do not include AlternativeTipTrigger in Price screen.
--  2013-October
--      v0.09   - Get station-names using loaded map's g_i18n object.
--  2014-May
--      v0.10   - Changed to use modPDAMOD.registerPlugin(...)
--  2014-June
--      v0.11   - Due to the NON-deterministic way that mods are loaded in multiplayer, the modPDAMOD.registerPlugin(...)
--                is replaced in favor of a hopefully more robust "create object modPDAMODplugins, and add yourself to it" method.
--      v0.12   - Attempted setting how fill-type prices should be shown in screen, with regards to quantity and unit-type.
--                This is for the animal mods that have to specify prices in "1000-liters", but want to show price only "per pcs."
--                The modPDAMOD.getFillTypePrice() will take this setting into account when calculating.
--      v0.14   - Not setting the fill-type quantity/unit-types anyway. Looks like MoreRealistic has some special values regarding
--                "animal fruit-types".
--      2.0.1   - Using http://semver.org/
--      2.0.2   - Fix: Tested on HM13 map which caused some bugs regarding fruit-icons.
--


PdaPlugin_Prices = {}
--
local modItem = ModsUtil.findModItemByModName(g_currentModName);
PdaPlugin_Prices.version = (modItem and modItem.version) and modItem.version or "?.?.?";
--
PdaPlugin_Prices.isInitialized = false;
PdaPlugin_Prices.lastUpdatedDayHour = -1;
-- 
PdaPlugin_Prices.fillTypeQuantityAndUnitTypes = {
--     ["pig"]        = {qty=1, unitType="pcs"}
--    ,["piglet"]     = {qty=1, unitType="pcs"}
--    ,["beef"]       = {qty=1, unitType="pcs"}
--    ,["cattle"]     = {qty=1, unitType="pcs"}
--    ,["calf"]       = {qty=1, unitType="pcs"}
--    ,["horse"]      = {qty=1, unitType="pcs"}
--    ,["cow"]        = {qty=1, unitType="pcs"}
--    ,["sheep"]      = {qty=1, unitType="pcs"}
--    ,["chicken"]    = {qty=1, unitType="pcs"}
--    -- TODO: Add more... - or even better; make it configurable in a .XML file!
}


-- Register this PDAMod plugin
getfenv(0)["modPDAMODplugins"] = getfenv(0)["modPDAMODplugins"] or {}
table.insert(getfenv(0)["modPDAMODplugins"], PdaPlugin_Prices)
--

function PdaPlugin_Prices:loadMap(xmlName)
end;

function PdaPlugin_Prices:deleteMap()
    --g_currentMission.environment:removeHourChangeListener(PdaPlugin_Prices); -- Not possible to remove this late.
    PdaPlugin_Prices.isInitialized = false;
    PdaPlugin_Prices.lastUpdatedDayHour = -1;
end;

function PdaPlugin_Prices.initialize(self, initStep)
    if initStep == 4 then
        -- Add parameter for how price-vs-quantity/unit should be shown in the PDAMod-Prices screen.
        for fillName,qtyUnit in pairs(PdaPlugin_Prices.fillTypeQuantityAndUnitTypes) do
            local fillDesc = Fillable.fillTypeNameToDesc[fillName];
            if fillDesc then 
                if fillDesc.showPriceForQuantity == nil then
                    fillDesc.showPriceForQuantity = qtyUnit.qty;
                end
                if fillDesc.showPriceForUnitType == nil then
                    fillDesc.showPriceForUnitType = qtyUnit.unitType;
                end
            end
        end
    
        -- Register ourself as a hour-change-listener
        g_currentMission.environment:addHourChangeListener(PdaPlugin_Prices);
    
        -- Compute some column positions and widths...
        local pda = g_currentMission.missionPDA;
        -- Some other mod apparently has the brilliant idea (NOT!) to constantly modify 'pdaRowSpacing' in its update() function! WTF?
        -- I'm looking at you PDAExtensionForageStorage.LUA and HagenstedtModified2013-map! >:-(
        PdaPlugin_Prices.pdaFontSize = pda.pdaHeight / 12
        PdaPlugin_Prices.pdaRowSpacing = PdaPlugin_Prices.pdaFontSize * 1.05
        --
        local xOffset = pda.pdaCol1 + getTextWidth(PdaPlugin_Prices.pdaFontSize, "MMMMMMMMMMMMMMM");
        local colSpacing = getTextWidth(PdaPlugin_Prices.pdaFontSize, "M");
        self.pdaColsHalfWidth = getTextWidth(PdaPlugin_Prices.pdaFontSize, "MMMMMMMM")/2;
        self.pdaCols = {}
        for c=1,3 do
            self.pdaCols[c] = xOffset + (self.pdaColsHalfWidth*2 + colSpacing) * (c-1);
        end;
        
        -- Build arrays of stations and fillnames
        PdaPlugin_Prices.recalculateFruitPrices();
        
        -- Use above to setup needed subpages for "pricesMultifruit" page
        for fillIdx=1,#PdaPlugin_Prices.FilltypeNames,3 do
            local maxRowsPerPage = 12-2  -- TODO - calculate this from pda-height vs fontsize + spacing
            for stationIdx=1,#PdaPlugin_Prices.StationPrices,maxRowsPerPage do
                modPDAMOD.registerPage(
                    "pricesMultifruit", nil, 
                    PdaPlugin_Prices.subPageDraw_MultiFruits, g_currentMission.missionPDA, {stationIdx,fillIdx,maxRowsPerPage}, 
                    nil, nil, nil, 
                    0
                );
            end;
        end;
        
        -- Add the 'Great Demands' subpage
        modPDAMOD.registerPage(
            "pricesMultifruit", nil, 
            PdaPlugin_Prices.subPageDraw_GreatDemands, g_currentMission.missionPDA, nil, 
            nil, nil, nil, 
            0
        );
        
        -- Create some filltype overlays, to make "pricesMultifruit" page more stylish
        PdaPlugin_Prices.FilltypeIcons = {}
        for _,ft in pairs(PdaPlugin_Prices.FilltypeNames) do
            local fillIcon = Fillable.fillTypeIndexToDesc[ft.fillType].hudOverlayFilename;
            if fillIcon ~= nil and fillIcon ~= "" then
                local overlayId = createImageOverlay(fillIcon)
                if overlayId ~= nil and overlayId ~= 0 then
                    PdaPlugin_Prices.FilltypeIcons[ft.fillType] = overlayId;
                    setOverlayColor(PdaPlugin_Prices.FilltypeIcons[ft.fillType], 1,1,1, 0.2); -- 20% opaque
                end
            end;
        end;
        
        -- ReferencePrices
        local numWidth = getTextWidth(PdaPlugin_Prices.pdaFontSize, "88888 ");
        PdaPlugin_Prices.refCols = {}
        PdaPlugin_Prices.refCols[1] = pda.pdaX + numWidth;
        PdaPlugin_Prices.refCols[2] = (pda.pdaX + pda.pdaWidth) / 2 + numWidth;
        PdaPlugin_Prices.refSpacing = getTextWidth(PdaPlugin_Prices.pdaFontSize, " ")/2;

        local numOfPrices = 0;
        for fillType=1,Fillable.NUM_FILLTYPES do
            local fillDesc = Fillable.fillTypeIndexToDesc[fillType];
            if fillDesc.pricePerLiter > 0 then
                numOfPrices = numOfPrices + 1;
                if numOfPrices > 9*2 then -- 9 rows, 2 cols.
                    numOfPrices = 1
                end
            end
            if numOfPrices == 1 then
                modPDAMOD.registerPage(
                    "pricesMultifruit", nil,
                    PdaPlugin_Prices.subPageDraw_ReferencePrices, g_currentMission.missionPDA, fillType, 
                    nil, nil, nil, 
                    0
                );
            end
        end;
    end
end;

--
function PdaPlugin_Prices:hourChanged()
    PdaPlugin_Prices.lastUpdatedDayHour = -1; -- Force recalculation
end

-- On clients, catch when event is received.
function PdaPlugin_Prices.GreatDemandsEventRun(self, superFunc, parm1)
    superFunc(self, parm1);
    PdaPlugin_Prices.lastUpdatedDayHour = -1; -- Force recalculation
end
GreatDemandsEvent.run = Utils.overwrittenFunction(GreatDemandsEvent.run, PdaPlugin_Prices.GreatDemandsEventRun);

--
function PdaPlugin_Prices.recalculateFruitPrices()
    local nowDayHour = g_currentMission.environment.currentDay*100 + g_currentMission.environment.currentHour;
    if PdaPlugin_Prices.lastUpdatedDayHour ~= nowDayHour then
        PdaPlugin_Prices.lastUpdatedDayHour = nowDayHour;
        --
        PdaPlugin_Prices.StationPrices = {};
        local acceptedFilltypes = {}
        for _,tipTrigger in pairs(g_currentMission.tipTriggers) do
            if (tipTrigger.appearsOnPDA and tipTrigger.isEnabled)
            and (tipTrigger.isExtendedTrigger == nil) -- AlternativeTipTrigger is excluded.
            then
                local fillTypesPrices = {}
                for fillType,accepted in pairs(tipTrigger.acceptedFillTypes) do
                    if accepted then
                        --
                        local priceMultiplier = 0;
                        if tipTrigger.priceMultipliers[fillType] ~= nil then
                            priceMultiplier = tipTrigger.priceMultipliers[fillType];
                        end;
                        --
                        local greatDemand = g_currentMission.economyManager:getCurrentGreatDemand(tipTrigger.stationName, fillType);
                        local greatDemandMultiplier = 1.0;
                        if greatDemand ~= nil then
                            greatDemandMultiplier = greatDemand.demandMultiplier;
                        end;
                        --
                        fillTypesPrices[fillType] = {
                            px   = math.ceil(modPDAMOD.getFillTypePrice(fillType) * priceMultiplier * greatDemandMultiplier)
                            ,gdm = greatDemandMultiplier
                        };
                        acceptedFilltypes[fillType] = true;
                    end
                end
                local nameI18N = tipTrigger.stationName or "Unknown";
                if g_i18n:hasText(nameI18N) then
                    nameI18N = g_i18n:getText(nameI18N);
                elseif g_currentMission.missionInfo and g_currentMission.missionInfo.map and g_currentMission.missionInfo.map.customEnvironment then
                    -- Try to find station-name from the map's own g_i18n object
                    local env0 = getfenv(0)
                    local mapMod = env0[g_currentMission.missionInfo.map.customEnvironment]
                    if mapMod ~= nil and mapMod.g_i18n ~= nil then
                        if mapMod.g_i18n:hasText(nameI18N) then
                            nameI18N = mapMod.g_i18n:getText(nameI18N);
                        end;
                    end
                end;
                table.insert(PdaPlugin_Prices.StationPrices, {name=tipTrigger.stationName, nameI18N=nameI18N, obj=tipTrigger, fillTypesPrices=fillTypesPrices });
            end;
        end;
        table.sort(PdaPlugin_Prices.StationPrices, function(l,r) 
            return (l.nameI18N):upper() < (r.nameI18N):upper(); 
        end);
        --
        PdaPlugin_Prices.FilltypeNames = {}
        for fillType,_ in pairs(acceptedFilltypes) do
            local nameI18N = Fillable.fillTypeIndexToDesc[fillType].nameI18N;
            if nameI18N == nil then
                nameI18N = Fillable.fillTypeIndexToDesc[fillType].name;
            end;
            if nameI18N == nil then
                nameI18N = "Unknown";
            end;
            table.insert(PdaPlugin_Prices.FilltypeNames, {fillType=fillType, nameI18N=nameI18N} );
        end;
        table.sort(PdaPlugin_Prices.FilltypeNames, function(l,r) 
            return (l.nameI18N):upper() < (r.nameI18N):upper(); 
        end);
    end
end;

function PdaPlugin_Prices.subPageDraw_MultiFruits(self, parm, origMissionPdaDrawFunc)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.

    PdaPlugin_Prices.recalculateFruitPrices();
    --
    self.hudPDABackgroundOverlay:render()
    --
    local row = 1;
    local topRow = self.pdaHeadRow - 0.01;
    
    local fillIdxStart = parm[2];
    local fillIdxEnd   = math.min(fillIdxStart + 2, #PdaPlugin_Prices.FilltypeNames);
    
    -- Render fruitnames
    setTextBold(true);
    setTextColor(1,1,1,1);
    setTextAlignment(RenderText.ALIGN_RIGHT);
    local iconW = (PdaPlugin_Prices.pdaColsHalfWidth * 2) * 0.9;
    local iconH = iconW * (4/3);
    local iconX = iconW / 2;
    local iconY = (topRow - PdaPlugin_Prices.pdaRowSpacing * 0) - iconH;
    local c = 1
    local y = topRow + PdaPlugin_Prices.pdaRowSpacing * 0.25
    for idx=fillIdxStart,fillIdxEnd do
        local fillType = PdaPlugin_Prices.FilltypeNames[idx].fillType;
        if PdaPlugin_Prices.FilltypeIcons[fillType] then
            renderOverlay(PdaPlugin_Prices.FilltypeIcons[fillType], PdaPlugin_Prices.pdaCols[c] - iconX, iconY, iconW, iconH);
        end
        --
        renderText(PdaPlugin_Prices.pdaCols[c] + PdaPlugin_Prices.pdaColsHalfWidth, y, PdaPlugin_Prices.pdaFontSize, PdaPlugin_Prices.FilltypeNames[idx].nameI18N);
        c = c + 1
    end;
    -- Render stationnames and fruit prices
    for stationIdx=parm[1],math.min(parm[1]+parm[3]-1, #PdaPlugin_Prices.StationPrices) do
        setTextBold(false);
        setTextColor(1,1,1,1);
        local station = PdaPlugin_Prices.StationPrices[stationIdx];
        setTextAlignment(RenderText.ALIGN_LEFT);
        renderText(self.pdaCol1, topRow - PdaPlugin_Prices.pdaRowSpacing * row, PdaPlugin_Prices.pdaFontSize, station.nameI18N);
        --        
        c = 1
        setTextAlignment(RenderText.ALIGN_RIGHT);
        for idx=fillIdxStart,fillIdxEnd do
            local fillType = PdaPlugin_Prices.FilltypeNames[idx].fillType;
            if station.fillTypesPrices[fillType] ~= nil then
                local price = station.fillTypesPrices[fillType].px;
                if price ~= nil and (price > 0 or price < 0) then
                    -- Color depending on Great Demand multiplier.
                    if (station.fillTypesPrices[fillType].gdm > 1.0) then
                        setTextColor(0.2,1,0.2,1);
                    elseif (station.fillTypesPrices[fillType].gdm < 1.0) then
                        setTextColor(1,0.2,0.2,1);
                    else
                        setTextColor(1,1,1,1);
                    end
                    renderText(PdaPlugin_Prices.pdaCols[c] + PdaPlugin_Prices.pdaColsHalfWidth, topRow - PdaPlugin_Prices.pdaRowSpacing * row, PdaPlugin_Prices.pdaFontSize, string.format("%d", price));
                end;
            end;
            c = c + 1
        end;
          
        row=row+1;
    end;
    --
    self.hudPDAFrameOverlay:render()
    --    
    modPDAMOD.drawTitle(g_i18n:getText("PDATitle4")); -- "Prices"
end

--
function PdaPlugin_Prices.subPageDraw_GreatDemands(self, parm, origMissionPdaDrawFunc)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.
    
    self.hudPDABackgroundOverlay:render()
    --
    setTextBold(true)
    renderText(self.pdaPricesCol[1], self.pdaHeadRow, PdaPlugin_Prices.pdaFontSize, g_i18n:getText("PDATitleGreatDemands"))
    setTextBold(false)
    local i18n = g_i18n
    if g_currentMission.missionInfo.customEnvironment ~= nil then
      i18n = _G[g_currentMission.missionInfo.customEnvironment].g_i18n
    end
    local hasGreatDemands = false
    local y = self.pdaHeadRow - PdaPlugin_Prices.pdaRowSpacing
    for _,greatDemand in ipairs(g_currentMission.economyManager.greatDemands) do
      if greatDemand.isValid and greatDemand.isRunning then
        hasGreatDemands = true
        local stationName = greatDemand.stationName
        if i18n:hasText(stationName) then
          stationName = i18n:getText(stationName)
        end
        renderText(self.pdaPricesCol[1], y, PdaPlugin_Prices.pdaFontSize, stationName)
        renderText(self.pdaPricesCol[2], y, PdaPlugin_Prices.pdaFontSize, Fillable.fillTypeIndexToDesc[greatDemand.fillTypeIndex].nameI18N .. string.format(" x%.1f", greatDemand.demandMultiplier))
        renderText(self.pdaPricesCol[3], y, PdaPlugin_Prices.pdaFontSize, string.format(g_i18n:getText("PDAGreatDemandTimeRemaining"), greatDemand.demandDuration))
        y = y - PdaPlugin_Prices.pdaRowSpacing
      end
    end
    if not hasGreatDemands then
      renderText(self.pdaPricesCol[1], self.pdaHeadRow - PdaPlugin_Prices.pdaRowSpacing * 2, PdaPlugin_Prices.pdaFontSize, g_i18n:getText("PDANoGreatDemands"))
    end
    --
    self.hudPDAFrameOverlay:render()
    --    
    modPDAMOD.drawTitle(g_i18n:getText("PDATitle4")); -- "Prices"
end

--
function PdaPlugin_Prices.subPageDraw_ReferencePrices(self, parm, origMissionPdaDrawFunc)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.

    self.hudPDABackgroundOverlay:render()
    --
    local topRow = self.pdaHeadRow - 0.01;
    --
    setTextColor(1,1,1,1);
    setTextBold(true);
    renderText(self.pdaX, topRow + PdaPlugin_Prices.pdaRowSpacing * 0.25, PdaPlugin_Prices.pdaFontSize, g_i18n:getText("RefPricesPer1000Units"));
    --
    setTextBold(false);
    local row = 1;
    local col = 1;
    for fillType=parm,Fillable.NUM_FILLTYPES do
        local price = modPDAMOD.getFillTypePrice(fillType)
        if price ~= nil then
            local fillDesc = Fillable.fillTypeIndexToDesc[fillType]
            local y = topRow - PdaPlugin_Prices.pdaRowSpacing * row;
        
            setTextAlignment(RenderText.ALIGN_RIGHT);
            renderText(PdaPlugin_Prices.refCols[col] - PdaPlugin_Prices.refSpacing, y, PdaPlugin_Prices.pdaFontSize, string.format("%.0f", price));
        
            setTextAlignment(RenderText.ALIGN_LEFT);
            renderText(PdaPlugin_Prices.refCols[col] + PdaPlugin_Prices.refSpacing, y, PdaPlugin_Prices.pdaFontSize, fillDesc.nameI18N);
            
            row = row + 1;
            if row > 9 then
              col = col + 1
              row = 1
            end
            if col > 2 then
                break;
            end;
        end
    end;
    --
    self.hudPDAFrameOverlay:render()
    --    
    modPDAMOD.drawTitle(g_i18n:getText("RefPrices"));
end


print(string.format("Script loaded: PdaPlugin_PricesMultiFruit.LUA (v%s)", PdaPlugin_Prices.version));
