--
-- Overcharge
-- Specialization for Overcharge
--
-- @author:		sven18koehler
--				fruktor
--				JoPi
-- @date:		19/06/14
-- @version:	v1.2
-- @history:	v1.0 - initial version
--				v1.1 - removed plane; added optional control through VariableTip; disabled discharge at tip trigger
--				v1.2 - converted to FS 2013; removed VariableTip, animationParts, hydraulics, triggerNode; fixed tipping at tip trigger; added value for definded overcharge fruits
--
-- free for noncommercial-usage

Overcharge = {};

function Overcharge.prerequisitesPresent(specializations)
	return true;
end;

function Overcharge:load(xmlFile)
	self.setSoundState = SpecializationUtil.callSpecializationsFunction("setSoundState");
	self.setUnloadingState = SpecializationUtil.callSpecializationsFunction("setUnloadingState");
	self.setVehicleRpmUp = SpecializationUtil.callSpecializationsFunction("setVehicleRpmUp");
	self.findTrailerRaycastCallback = Overcharge.findTrailerRaycastCallback;
	self.setFillLevel = Utils.prependedFunction(self.setFillLevel, Overcharge.setFillLevel);
	self.unloadingSounds = {};
	self.unloadingSounds.startSound = {};
	local startSound = getXMLString(xmlFile, "vehicle.unloadingSounds#startFile");
	if startSound ~= nil and startSound ~= "" then
		startSound = Utils.getFilename(startSound, self.baseDirectory);
		self.unloadingSounds.startSound.file = createSample("startSound");
		self.unloadingSounds.startSound.enabled = false;
		self.unloadingSounds.startSound.duration = getSampleDuration(self.unloadingSounds.startSound.file);
		loadSample(self.unloadingSounds.startSound.file, startSound, false);
	end;
	self.unloadingSounds.loopSound = {};
	local loopSound = getXMLString(xmlFile, "vehicle.unloadingSounds#loopFile");
	if loopSound ~= nil and loopSound ~= "" then
		loopSound = Utils.getFilename(loopSound, self.baseDirectory); 
		self.unloadingSounds.loopSound.file = createSample("loopSound");
		self.unloadingSounds.loopSound.enabled = false;
		loadSample(self.unloadingSounds.loopSound.file, loopSound, false);
	end;
	self.playLoopSoundTime = 0;
	self.pipe = {};
	self.pipe.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.pipe#index"));
	self.pipe.distance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipe#raycastDistance"), 7);
	self.allowOvercharge = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.pipe#allowOvercharge"), true);
	self.unloadingCapacity = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.unloadingCapacity"), 100);
	if self.isClient then
		self.overchargeParticleSystems = {};
		local i = 0;
		while true do
			local key = string.format("vehicle.overchargeParticleSystems.overchargeParticleSystem(%d)", i);
			local t = getXMLString(xmlFile, key .. "#type");
			if t == nil then
				break;
			end;
			local fillType = Fillable.fillTypeNameToInt[t];
			if fillType ~= nil then
				local currentPS = {};
				local particleNode = Utils.loadParticleSystem(xmlFile, currentPS, key, self.components, false, nil, self.baseDirectory);
				self.overchargeParticleSystems[fillType] = currentPS;
			end;
			i = i + 1;
		end;
	end;
	self.overchargeFillTypes = {};
	self.overchargeFillTypes[Fillable.FILLTYPE_UNKNOWN] = true;
	local fillTypes = Utils.splitString(" ", getXMLString(xmlFile, "vehicle.pipe#overchargeFillTypes"));
	if fillTypes ~= nil then
		for k,v in pairs(fillTypes) do
			local fillType = Fillable.fillTypeNameToInt[v];
			if fillType ~= nil then
				self.overchargeFillTypes[fillType] = true;
			end;
		end;
	end;
	self.saveMinRpm = 0;
	self.isUnloading = false;
	self.deactivateOnLeave = false;
	local x = g_currentMission.hudAttachmentOverlay.x;
	local y = g_currentMission.hudAttachmentOverlay.y;
	local width = g_currentMission.hudAttachmentOverlay.width;
	local height = g_currentMission.hudAttachmentOverlay.height;
	local filename = Utils.getFilename("hud/overcharge.png", self.baseDirectory);
	self.unloadingOverlay = Overlay:new("unloadingOverlay", filename, x, y, width, height);
	self.renderOverlay = false;
end;

function Overcharge:delete()
	if self.unloadingSounds.loopSound.file ~= nil then
		delete(self.unloadingSounds.loopSound.file);
	end;
	if self.unloadingSounds.startSound.file ~= nil then
		delete(self.unloadingSounds.startSound.file);
	end;
	if self.isClient then
		for _, particleSystem in pairs(self.overchargeParticleSystems) do
			Utils.deleteParticleSystem(particleSystem);
		end;
	end;
end;

function Overcharge:readStream(streamId, connection)
	local isUnloading = streamReadBool(streamId);
	self:setUnloadingState(isUnloading, true);
end;

function Overcharge:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isUnloading);
end;

function Overcharge:writeUpdateStream(streamId, connection, dirtyMask)
	if not connection:getIsServer() then
		streamWriteBool(streamId, self.allowOvercharge);
		streamWriteBool(streamId, self.renderOverlay);
	end;
end;

function Overcharge:readUpdateStream(streamId, timestamp, connection)
	if connection:getIsServer() then
		self.allowOvercharge = streamReadBool(streamId);
		self.renderOverlay = streamReadBool(streamId);
	end;
end;

function Overcharge:mouseEvent(posX, posY, isDown, isUp, button)
end;

function Overcharge:keyEvent(unicode, sym, modifier, isDown)
end;

function Overcharge:update(dt)
	if self.unloadingOverlay ~= nil then
		if g_currentMission.hudAttachmentOverlay.x ~= self.unloadingOverlay.x or g_currentMission.hudAttachmentOverlay.y ~= self.unloadingOverlay.y then
			self.unloadingOverlay:setPosition(g_currentMission.hudAttachmentOverlay.x, g_currentMission.hudAttachmentOverlay.y);
		end;
	end;
	if self:getIsActiveForInput() then
		if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA2) and self.allowOvercharge then
			self:setUnloadingState(not self.isUnloading);
		end;
	end;
end;

function Overcharge:updateTick(dt)
	if self.isServer then
		self.renderOverlay = false;
		local delta = 0;
		if self.isClient then
			if self.fillLevel > 0 then
				if self.pipe.node ~= nil then
					self.trailerFound = nil;
					local x,y,z = getWorldTranslation(self.pipe.node);
					raycastAll(x, y, z, 0, -0.5, 0, "findTrailerRaycastCallback", self.pipe.distance, self);
					if self.trailerFound ~= nil and self.overchargeFillTypes[self.currentFillType] then
						if self.trailerFound ~= nil and self.trailerFoundSupported then
							if self.isUnloading then
								self.trailerFound:resetFillLevelIfNeeded(self.currentFillType);
								local oldFillLevel = self.trailerFound:getFillLevel(self.currentFillType);
								delta = math.min(self.fillLevel, self.unloadingCapacity*dt/1000);
								self.trailerFound:setFillLevel(oldFillLevel + delta, self.currentFillType);
							else
								self.renderOverlay = true;
							end;
						else
							delta = 0;
						end;
					elseif not self.overchargeFillTypes[self.currentFillType] then
						self.trailerFound = nil;
					end;
					if delta ~= 0 then
						local fillType = self.currentFillType;
						local oldFillLevel = self.fillLevel;
						self:setFillLevel(oldFillLevel - delta, fillType);
					elseif delta == 0 and self.isUnloading then
						self:setUnloadingState(false);
					end;
				end;
			end;
		end;
	end;
	Utils.setEmittingState(self.overchargeParticleSystems[self.currentFillType], self.isUnloading);
	self:setSoundState(self.isUnloading);
	if self.attacherVehicle ~= nil then
		if self.attacherVehicle.motor ~= nil then
			self:setVehicleRpmUp(dt, self.isUnloading);
		end;
	end;
end;

function Overcharge:findTrailerRaycastCallback(transformId, x, y, z, distance)
	local trailer = g_currentMission.objectToTrailer[transformId];
	if trailer ~= nil and trailer ~= self then
		if trailer:allowFillType(self.currentFillType) and trailer.capacity ~= trailer.fillLevel and trailer.getAllowFillFromAir ~= nil and trailer:getAllowFillFromAir() then
			self.trailerFound = trailer;
			self.trailerFoundSupported = true;
		else
			if self.trailerFound == nil then
				self.trailerFound = trailer;
				self.trailerFoundSupported = false;
			end;
		end;
		return false;
	end;
	return true;
end;

function Overcharge:setUnloadingState(state, noEventSend)
	SetUnloadingEvent.sendEvent(self, state, noEventSend);
	self.isUnloading = state;
end;

function Overcharge:setSoundState(value)
	if value == true then
		if not self.unloadingSounds.startSound.enabled and self:getIsActiveForSound() and self.unloadingSounds.startSound.file ~= nil then
			playSample(self.unloadingSounds.startSound.file, 1, 0.35, 0);
			setSamplePitch(self.unloadingSounds.startSound.file , 1);
			self.unloadingSounds.startSound.enabled = true;
			self.playLoopSoundTime = self.time + self.unloadingSounds.startSound.duration;
		end;
		if self.playLoopSoundTime <= self.time and not self.unloadingSounds.loopSound.enabled and self:getIsActiveForSound() and self.unloadingSounds.loopSound.file ~= nil then
			playSample(self.unloadingSounds.loopSound.file, 0, 0.35, 0);
			setSamplePitch(self.unloadingSounds.loopSound.file, 1);
			self.unloadingSounds.loopSound.enabled  = true;
		end;
	else
		if self.unloadingSounds.loopSound.enabled and self.unloadingSounds.loopSound.file ~= nil then
			stopSample(self.unloadingSounds.loopSound.file);
			self.unloadingSounds.loopSound.enabled = false;
		end;
		if self.unloadingSounds.startSound.enabled and self.unloadingSounds.startSound.file ~= nil then
			stopSample(self.unloadingSounds.startSound.file);
			self.unloadingSounds.startSound.enabled = false;
		end;
	end;
end;

function Overcharge:setVehicleRpmUp(dt, isActive)
	if self.attacherVehicle ~= nil and self.saveMinRpm ~= 0 and
		self.attacherVehicle.motor ~= nil then
		if dt ~= nil then
			if isActive == true then
				self.attacherVehicle.motor.minRpm = math.max(self.attacherVehicle.motor.minRpm-(dt*2), -1200);
			else
				self.attacherVehicle.motor.minRpm = math.min(self.attacherVehicle.motor.minRpm+(dt*5), self.saveMinRpm);
			end;
		else
			self.attacherVehicle.motor.minRpm = self.saveMinRpm;
		end;
		if self.attacherVehicle.isMotorStarted then
			local fuelUsed = 0.0000011*math.abs(self.attacherVehicle.motor.minRpm);
			self.attacherVehicle:setFuelFillLevel(self.attacherVehicle.fuelFillLevel-fuelUsed);
			g_currentMission.missionStats.fuelUsageTotal = g_currentMission.missionStats.fuelUsageTotal + fuelUsed;
			g_currentMission.missionStats.fuelUsageSession = g_currentMission.missionStats.fuelUsageSession + fuelUsed;
		end;
	end;
end;

function Overcharge:draw()
	if self.isClient then
		if self.allowOvercharge then
			if self.isUnloading then
				g_currentMission:addHelpButtonText(g_i18n:getText("END_OVERCHARGING"), InputBinding.IMPLEMENT_EXTRA2);
			elseif self.renderOverlay then
				self.unloadingOverlay:render();
				g_currentMission:addHelpButtonText(g_i18n:getText("START_OVERCHARGING"), InputBinding.IMPLEMENT_EXTRA2);
			end;
		end;
	end;
end;

function Overcharge:setFillLevel(fillLevel, fillType, force)
	if self.isClient then
		if self.currentFillType ~= fillType or fillLevel <= 0 then
			self:setUnloadingState(false);
			Utils.setEmittingState(self.overchargeParticleSystems[self.currentFillType], false);
		end;
	end;
end;

function Overcharge:onAttach(attacherVehicle)
	self.attacherVehicle = attacherVehicle;
	if self.attacherVehicleCopy == nil then
		self.attacherVehicleCopy = self.attacherVehicle;
	end;
	if self.attacherVehicle.motor ~= nil then
		self.saveMinRpm = self.attacherVehicle.motor.minRpm;
	else
		if self.attacherVehicle.saveMinRpm ~= nil then
			self.saveMinRpm = self.attacherVehicle.saveMinRpm;
		else
			self.attacherVehicle.saveMinRpm  = 100;
		end;
	end;
end;

function Overcharge:onDetach()
	if self.deactivateOnDetach then
		Overcharge.onDeactivate(self);
	else
		Overcharge.onDeactivateSounds(self);
	end;
	for k, steerable in pairs(g_currentMission.steerables) do
		if self.attacherVehicleCopy == steerable then
			steerable.motor.minRpm = self.saveMinRpm;
			self.attacherVehicleCopy = nil;
		end;
	end;
end;

function Overcharge:onLeave()
	if self.deactivateOnLeave then
		Overcharge.onDeactivate(self);
	else
		Overcharge.onDeactivateSounds(self);
	end;
end;

function Overcharge:onDeactivate()
	self:setSoundState(false);
	self:setVehicleRpmUp(nil, false);
	for _, particleSystem in pairs(self.overchargeParticleSystems) do
		Utils.setEmittingState(particleSystem, false);
	end;
	Overcharge.onDeactivateSounds(self);
end;

function Overcharge:onDeactivateSounds()
	if self.isClient then
		self:setSoundState(false);
	end;
end;

--
-- SetUnloadingEvent
-- Networkevent for SetUnloading
--
-- @author:		sven18koehler
-- @date:		15/12/10
-- @version:	v1.0
-- @history:	v1.0 - initial version
--
-- Copyright (C) sven18koehler

SetUnloadingEvent = {};
SetUnloadingEvent_mt = Class(SetUnloadingEvent, Event);
InitEventClass(SetUnloadingEvent, "SetUnloadingEvent");

function SetUnloadingEvent:emptyNew()
	local self = Event:new(SetUnloadingEvent_mt);
	self.className = "SetUnloadingEvent";
	return self;
end;

function SetUnloadingEvent:new(object, isUnloading)
	local self = SetUnloadingEvent:emptyNew()
	self.object = object;
	self.isUnloading = isUnloading;
	return self;
end;

function SetUnloadingEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	self.isUnloading = streamReadBool(streamId);
	self.object = networkGetObject(id);
	self:run(connection);
end;

function SetUnloadingEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.object));
	streamWriteBool(streamId, self.isUnloading);
end;

function SetUnloadingEvent:run(connection)
	self.object:setUnloadingState(self.isUnloading, true);
	if not connection:getIsServer() then
		g_server:broadcastEvent(SetUnloadingEvent:new(self.object, self.isUnloading), nil, connection, self.object);
	end;
end;

function SetUnloadingEvent.sendEvent(vehicle, isUnloading, noEventSend)
	if isUnloading ~= vehicle.isUnloading then
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				g_server:broadcastEvent(SetUnloadingEvent:new(vehicle, isUnloading), nil, nil, vehicle);
			else
				g_client:getServerConnection():sendEvent(SetUnloadingEvent:new(vehicle, isUnloading));
			end;
		end;
	end;
end;