--
-- ESLimiter
-- Specialization for ESLimiter
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date:		05/11/10
-- @version:	v2.2
-- @history:	v1.0 - initial implementation
--				v2.0 - convert to LS2011 and some bugfixes
--				v2.1 - Network-Fixes
--				v2.2 - Bugfixes and new huds
--

ESLimiter = {};

function ESLimiter.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Motorized, specializations);
end;

function ESLimiter:load(xmlFile)

	self.setNewLimit = SpecializationUtil.callSpecializationsFunction("setNewLimit");

	self.ESLimiter = {};
	self.ESLimiter.maxRPM = {};
	self.ESLimiter.orgMaxRPM = {};
	self.ESLimiter.percentage = {};
		
	for k,_ in pairs(self.motor.maxRpm) do	
		local percentage = string.format("%d",(self.motor.maxRpm[k] * 100 / self.motor.maxRpm[3]));
		self.ESLimiter.percentage[k+1] = tonumber(percentage);
		self.ESLimiter.maxRPM[k+1] = self.motor.maxRpm[3];
		self.ESLimiter.orgMaxRPM[k+1] = self.motor.maxRpm[k];
	end;
	-- set limits for reverse
	self.ESLimiter.maxRPM[1] = self.ESLimiter.maxRPM[2];
	self.ESLimiter.percentage[1] = self.ESLimiter.percentage[2];	
	self.ESLimiter.default = self.motor.maxRpm[3];
	
	self.isLimiterActive = false;
	self.ESLimiter.isButtonPressed = false;
	self.doReset = false;
	
	self.ESLimiter.xPos = Utils.getNoNil(getXMLFloat(xmlFile,  "vehicle.ESLimiter#posX"), 0.7543);
	self.ESLimiter.yPos = Utils.getNoNil(getXMLFloat(xmlFile,  "vehicle.ESLimiter#posY"), 0.1712);
	self.ESLimiter.hudBarWidth = 0.205;
	self.ESLimiter.overlay = Overlay:new("hudESLOverlay", Utils.getFilename("Images/ESLimiter_hud.dds", self.baseDirectory), self.ESLimiter.xPos, self.ESLimiter.yPos, 0.2371, 0.039525);
	self.ESLimiter.overlayBg = Overlay:new("hudESLBackground", Utils.getFilename("Images/ESLimiter_bg.dds", self.baseDirectory), self.ESLimiter.xPos, self.ESLimiter.yPos, 0.2371, 0.039525);
	self.ESLimiter.overlayBar = Overlay:new("hudESLbar", Utils.getFilename("Images/ESLimiter_bar.dds", self.baseDirectory), self.ESLimiter.xPos + 0.123, self.ESLimiter.yPos + 0.0085714, 0.208, 0.02190);
	self.ESLimiter.lock = Overlay:new("hudESLlock", Utils.getFilename("Images/locked_icon.png", self.baseDirectory), self.ESLimiter.xPos + 0.12581, self.ESLimiter.yPos + 0.0095, 0.015, 0.02);
end;

function ESLimiter:delete()
end;

function ESLimiter:readStream(streamId, connection)
	for level,_ in pairs(self.ESLimiter.percentage) do
		local percentage = streamReadInt8(streamId);
		self:setNewLimit(level, percentage, true);
	end;
end;

function ESLimiter:writeStream(streamId, connection)
	for _,percentage in pairs(self.ESLimiter.percentage) do
		streamWriteInt8(streamId, percentage);
	end;
end;

function ESLimiter:mouseEvent(posX, posY, isDown, isUp, button)
end;

function ESLimiter:keyEvent(unicode, sym, modifier, isDown)
end;

function ESLimiter:update(dt)
	if self:getIsActive() then
		local level = 0;
		if self.movingDirection < 0 then
			level = 1;
		else
			if self.motor.speedLevel > 0 then
				level = self.motor.speedLevel + 1;
			end;
		end;
		
		if self:getIsActiveForInput() and level ~= 0 then
			local percentage = self.ESLimiter.percentage;
			if InputBinding.isPressed(InputBinding.ESLimiter_PLUS) and percentage[level] < 100 then					
				percentage[level] = percentage[level] + 0.5;
				self:setNewLimit(level, percentage[level], true);
				self.ESLimiter.isButtonPressed = true;
			elseif InputBinding.isPressed(InputBinding.ESLimiter_MINUS) and percentage[level] > 10 then
				percentage[level] = percentage[level] - 0.5;					
				self:setNewLimit(level, percentage[level], true);
				self.ESLimiter.isButtonPressed = true;
			else
				if self.ESLimiter.isButtonPressed then
					-- only send event to other players if key was released -> performance issue
					self:setNewLimit(level, percentage[level]);
					self.ESLimiter.isButtonPressed = false;
				end;	
			end;
		end;
		
		if self.isAIThreshing == nil or self.isAIThreshing == false then
			if level ~= 0 then
				local newRpm = tonumber(string.format("%d",(self.ESLimiter.percentage[level] * self.ESLimiter.maxRPM[level] / 100)));
				if self.movingDirection >= 0 then
					self.motor.maxRpm[level - 1] = newRpm;
				else
					-- backward
					self.motor.maxRpm[3] = newRpm;
				end;
			else
				-- reset to default value if no level is set
				self.motor.maxRpm[3] = self.ESLimiter.default;
			end;
			self.doReset = true;
		else
			if self.doReset then
				for k,_ in pairs(self.motor.maxRpm) do
					self.motor.maxRpm[k] = self.ESLimiter.orgMaxRPM[k+1];
				end;
				self.doReset = false;
			end;
		end;
	end;
end;

function ESLimiter:draw()	
	local text = "Disabled";
	local currentLevel = self.motor.speedLevel;
	local currentLevelText = 0;
	if self.movingDirection < 0 then
		currentLevel = 1;
		currentLevelText = "R";
	else
		if currentLevel ~= 0 then
			currentLevel = currentLevel + 1;
			currentLevelText = currentLevel - 1;
		end;
	end;
	local percentage = 0;
	if currentLevel ~= 0 then
		percentage = tonumber(string.format("%d", self.ESLimiter.percentage[currentLevel]));
		if percentage < 100 then
			percentage = " " .. percentage;
		end;
		text = g_i18n:getText("ESLimiter_level") .. currentLevelText .. ":" .. percentage .. "% ";	
	end;
	self.ESLimiter.overlayBg:render();
	
	self.ESLimiter.overlayBar.width = self.ESLimiter.hudBarWidth * math.min(1, percentage/195);	
    setOverlayUVs(self.ESLimiter.overlayBar.overlayId, 0, 0.05, 0, 1, math.min(1, percentage/195), 0.05, math.min(1, percentage/195), 1);
	self.ESLimiter.overlayBar:render();
	
	self.ESLimiter.overlay:render();
	setTextBold(true);
	setTextAlignment(RenderText.ALIGN_LEFT);
	setTextColor(0, 0, 0, 1);
    renderText(self.ESLimiter.xPos + 0.136, self.ESLimiter.yPos + 0.009, 0.02, text);	
	setTextColor(1, 1, 1, 1);
	renderText(self.ESLimiter.xPos + 0.136, self.ESLimiter.yPos + 0.012, 0.02, text);	
	
	if self.isAIThreshing then
		self.ESLimiter.lock:render();
	end;
	
	g_currentMission:addExtraPrintText(string.format(g_i18n:getText("ESLimiter_controls"), self.typeDesc) .. " " .. InputBinding.getKeyNamesOfDigitalAction(InputBinding.ESLimiter_PLUS) .. "/" .. InputBinding.getKeyNamesOfDigitalAction(InputBinding.ESLimiter_MINUS));
end;

function ESLimiter:setNewLimit(level, percentage, noEventSend)	
	ESLimiterEvent.sendEvent(self, level, percentage, noEventSend);	
	self.ESLimiter.percentage[level] = percentage;
end;


function ESLimiter:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#limiterValues");		
		if valueStr ~= nil then
			local found = string.find(valueStr, ";");
			local index = 1;
			while found ~= nil do
				local value = string.sub(valueStr, 0, found-1);
				valueStr = string.sub(valueStr, found+1);
				found = string.find(valueStr, ";");
				self:setNewLimit(index, tonumber(value), true);
				if found == nil then
					self:setNewLimit(index+1, tonumber(value), true);
				end;
				index = index + 1;	
			end;
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function ESLimiter:getSaveAttributesAndNodes(nodeIdent)
	local values = "";	
	for k,v in pairs(self.ESLimiter.percentage) do
		if k ~= 1 then
			values = values .. ";";
		end;
		values = values .. string.format("%d", v);
	end;
	
    local attributes = 'limiterValues="'.. values .. '"';
    return attributes, nil;
end;