--
-- BuyableFloatationWheels
-- Specialization for Buyable Floatation Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v1.0
-- @date  		30/10/10
-- @history:	v1.0 - Initial version
--
-- free for noncommerical-usage
--

BuyableFloatationWheels = {};

function BuyableFloatationWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableFloatationWheels:load(xmlFile)

	self.floatationWheelTriggerCallback = BuyableFloatationWheels.floatationWheelTriggerCallback;
	self.wheelDelete = BuyableFloatationWheels.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.floatationWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.floatationWheels#activationTrigger"));
	
	addTrigger(self.usingTrigger, "floatationWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.floatationWheels#belts"));
	self.part = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.floatationWheels#part"));	
	
    self.buyableWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.floatationWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.floatationWheelPos = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#floatationWheelPos"));
		wheel.floatationWheel = getChildAt(wheel.floatationWheelPos, 0);
		wheel.standardWheel = nil;
		wheel.driveNode = nil;
		wheel.standardWheelPos = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#standardWheelPos"));
		table.insert(self.buyableWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.floatationWheelUser = nil;
end;

function BuyableFloatationWheels:delete()
	if self.floatationWheelUser ~= nil then
		self.floatationWheelUser:onDisassembling(true);
		self:disassembleWheels();
		self.buyableWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableFloatationWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableFloatationWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local floatationWheelUser = networkGetObject(id);
		if floatationWheelUser ~= nil then
			self:assembleWheels(floatationWheelUser, true);
		end;
	end;
end;

function BuyableFloatationWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.floatationWheelUser ~= nil then
		idToWrite = networkGetObjectId(self.floatationWheelUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableFloatationWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableFloatationWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableFloatationWheels:update(dt)
end;

function BuyableFloatationWheels:updateTick(dt)

	-- try to load saved floatationWheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then				
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;
end;

function BuyableFloatationWheels:draw()	
end;

function BuyableFloatationWheels:floatationWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter then		
			if vehicle.rootNode == otherId then
				if vehicle.buyableFloatationWheels ~= nil then
					if vehicle.buyableFloatationWheels.checkString == self.checkString then
						if self.floatationWheelUser ~= nil then
							if self.floatationWheelUser == vehicle then
								vehicle.buyableFloatationWheels.wheelsInRange = self;
							end;
						else
							vehicle.buyableFloatationWheels.wheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.buyableFloatationWheels ~= nil then
					vehicle.buyableFloatationWheels.wheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableFloatationWheels:assembleWheels(vehicle, noEventSend)
	BuyableFloatationWheelsAttachEvent.sendEvent(self, vehicle, noEventSend);
	
	if self.belts ~= nil then
		setVisibility(self.belts, true);
		setVisibility(self.part, true);
	end;	

	if self.floatationWheelUser == nil then
		if vehicle.buyableFloatationWheels ~= nil then
			self.floatationWheelUser = vehicle;	
			for k,wheel in pairs(vehicle.wheels) do
				for l,floatationWheel in pairs(self.buyableWheels) do 
					if k == l then	
						floatationWheel.standardWheel = getChildAt(wheel.driveNode, 0);	
						floatationWheel.driveNode = wheel.driveNode;
						link(wheel.driveNode, floatationWheel.floatationWheel);	
						link(floatationWheel.standardWheelPos, floatationWheel.standardWheel);
						break;
					end;
				end;
			end;	
			self.floatationWheelUser:onAssembling(self);
		end;
	end;
end;

function BuyableFloatationWheels:disassembleWheels(noEventSend)
	BuyableFloatationWheelsDetachEvent.sendEvent(self, noEventSend);
	
	if self.belts ~= nil then
		setVisibility(self.belts, false);
		setVisibility(self.part, false);		
	end;

	self.floatationWheelUser:onDisassembling();
	
	for _, floatationWheel in pairs(self.buyableWheels) do			
		link(floatationWheel.driveNode, floatationWheel.standardWheel);	
		link(floatationWheel.floatationWheelPos, floatationWheel.floatationWheel);
		floatationWheel.standardWheel = nil;
		floatationWheel.driveNode = nil;
	end;	
	self.floatationWheelUser = nil;	
end;

function BuyableFloatationWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");		
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};			
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableFloatationWheels:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	
	if self.floatationWheelUser ~= nil then
		local x,y,z = getWorldTranslation(self.floatationWheelUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;
	
    return attributes, nil;
end;

BuyableFloatationWheelsAttachEvent = {};
BuyableFloatationWheelsAttachEvent_mt = Class(BuyableFloatationWheelsAttachEvent, Event);

InitEventClass(BuyableFloatationWheelsAttachEvent, "BuyableFloatationWheelsAttachEvent");

function BuyableFloatationWheelsAttachEvent:emptyNew()
    local self = Event:new(BuyableFloatationWheelsAttachEvent_mt);
    return self;
end;

function BuyableFloatationWheelsAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableFloatationWheelsAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableFloatationWheelsAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableFloatationWheelsAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableFloatationWheelsAttachEvent:run(connection)
	self.vehicle:assembleWheels(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableFloatationWheelsAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableFloatationWheelsAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableFloatationWheelsAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableFloatationWheelsAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableFloatationWheelsDetachEvent = {};
BuyableFloatationWheelsDetachEvent_mt = Class(BuyableFloatationWheelsDetachEvent, Event);

InitEventClass(BuyableFloatationWheelsDetachEvent, "BuyableFloatationWheelsDetachEvent");

function BuyableFloatationWheelsDetachEvent:emptyNew()
    local self = Event:new(BuyableFloatationWheelsDetachEvent_mt);
    return self;
end;

function BuyableFloatationWheelsDetachEvent:new(vehicle)
    local self = BuyableFloatationWheelsDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableFloatationWheelsDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableFloatationWheelsDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableFloatationWheelsDetachEvent:run(connection)
	self.vehicle:disassembleWheels(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableFloatationWheelsDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;


function BuyableFloatationWheelsDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableFloatationWheelsDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableFloatationWheelsDetachEvent:new(vehicle));
		end;
	end;
end;


