--
-- InteractiveComponent Interface
-- Specifies an interactive component
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - converted to ls2011
--

InteractiveComponentInterface = {};

function InteractiveComponentInterface:new(node, highlight, name, mark, xRot, yRot, zRot, size, onMessage, offMessage, infobar, mt)

	local mTable = mt;
	if mTable == nil then
		mTable = Class(InteractiveComponentInterface);
	end;
    local instance = {};
    setmetatable(instance, mTable);
	
	instance.node = node;
	instance.highlight = highlight;
	if highlight ~= nil then
		instance.scaleX, instance.scaleY, instance.scaleZ = getScale(instance.highlight);
	end;
	instance.name = name;
	instance.mark = mark;
	instance.xRot = xRot;
	instance.yRot = yRot;
	instance.zRot = zRot;
	if xRot == nil and yRot == nil and zRot == nil then
		setVisibility(mark,false);
	else
		if xRot ~= nil then
			instance.xRot = Utils.degToRad(xRot);
		end;
		if yRot ~= nil then
			instance.yRot = Utils.degToRad(yRot);
		end;
		if zRot ~= nil then
			instance.zRot = Utils.degToRad(zRot);
		end;
	end;
	instance.scale = 0.01;
	instance.size = size;
	instance.isActive = true;
	instance.isMouseOver = false;
	instance.isOpen = false;
	instance.onMessage = Utils.getNoNil(onMessage, g_i18n:getText("ic_component_open"));
	instance.offMessage = Utils.getNoNil(offMessage, g_i18n:getText("ic_component_close"));
	instance.synch = true;

	instance.infoBarX = infobar.posX;
	instance.infoBarY = infobar.posY;
	instance.infoBarWidth = infobar.width;		
	instance.infoBar = infobar.hud;
	
	return instance;	
end;


function InteractiveComponentInterface:delete()
end;

function InteractiveComponentInterface:mouseEvent(posX, posY, isDown, isUp, button)
end;

function InteractiveComponentInterface:keyEvent(unicode, sym, modifier, isDown)
end;

function InteractiveComponentInterface:update(dt)	
	if self.isActive then
		if self.highlight ~= nil then
			if self.isMouseOver and self.highlight ~= nil then	
				self.scale = self.scale - 0.0006 * dt;
				setScale(self.highlight, self.scaleX + self.scale, self.scaleY, self.scaleZ + self.scale);				
				if self.scaleX + self.scale <= 0.95 then
					self.scale = 0.05;
				end;				
			end;
		else
			if self.xRot ~= nil or self.yRot ~= nil or self.zRot ~= nil then
				--print(".");
				local rx,ry,rz = getRotation(self.mark);
				local nv = {};
				if self.xRot ~= nil then 
					nv = Utils.getMovedLimitedValues({rx}, {0}, {self.xRot}, 1, 200, dt, self.isOpen);
					setRotation(self.mark, nv[1], ry, rz);
				elseif self.yRot ~= nil then
					nv = Utils.getMovedLimitedValues({ry}, {0}, {self.yRot}, 1, 200, dt, self.isOpen);
					setRotation(self.mark, rx, nv[1], rz);
				elseif self.zRot ~= nil then
					nv = Utils.getMovedLimitedValues({rz}, {0}, {self.zRot}, 1, 200, dt, self.isOpen);
					setRotation(self.mark, rx, ry, nv[1]);
				end;				
			end;
		end;
	end;
end;

function InteractiveComponentInterface:draw()
	if self.isMouseOver then
		self.infoBar:render();
		setTextBold(true);
		setTextColor(1,1,1,1);
		setTextAlignment(RenderText.ALIGN_LEFT);
		if self.isOpen then
			renderText(self.infoBarX + 0.016, self.infoBarY + 0.017, 0.018, string.format(self.offMessage, self.name));
		else
			renderText(self.infoBarX + 0.016, self.infoBarY + 0.017, 0.018, string.format(self.onMessage, self.name));
		end;
		setTextAlignment(RenderText.ALIGN_LEFT);
		setTextBold(false);
	end;
end;

function InteractiveComponentInterface:doAction(noEventSend, forceValue)
	if forceValue ~= nil then
		self.isOpen = forceValue;
	else
		self.isOpen = not self.isOpen;
	end;
end;

function InteractiveComponentInterface:onEnter(dt)
	self.isMouseOver = true;
end;

function InteractiveComponentInterface:onExit(dt)
	self.isMouseOver = false;
end;

function InteractiveComponentInterface:setActive(isActive)
	self.isActive = isActive;
end;

function InteractiveComponentInterface:setVisible(isVisible)
	if self.mark ~= nil then
		if self.highlight ~= nil then
			setVisibility(self.mark, isVisible);
		end;
	end;
end;