--
-- AnimatedFrontloader
--
--
-- @author  Manuel Leithner
-- @date  04/05/12
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.

AnimatedFrontloader = {};

function AnimatedFrontloader.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Cylindered, specializations) and SpecializationUtil.hasSpecialization(MouseControlsVehicle, specializations) and SpecializationUtil.hasSpecialization(AnimatedVehicle, specializations);
end;

function AnimatedFrontloader:load(xmlFile)

	self.releaseShovel = SpecializationUtil.callSpecializationsFunction("releaseShovel");

	local jointIndex = getXMLInt(xmlFile, "vehicle.animatedFrontloader#attacherJointIndex");
	if jointIndex ~= nil then
		local reset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.animatedFrontloader#resetTime"), 1) * 1000;
		local limits = Utils.getRadiansFromString(getXMLString(xmlFile, "vehicle.animatedFrontloader#limits"), 2);
		local releaseAnim = Utils.getNoNil(getXMLString(xmlFile, "vehicle.animatedFrontloader#releaseAnim"), "releaseAnim");
		local fixAnim = Utils.getNoNil(getXMLString(xmlFile, "vehicle.animatedFrontloader#fixAnim"), "fixAnim");
		local joint = self.attacherJoints[jointIndex+1];
		if joint ~= nil then
			self.animatedFrontloader = {jointIndex=jointIndex+1, joint=joint, resetTime=reset, limits=limits, curLimits={0, 0}, releaseAnim=releaseAnim, fixAnim=fixAnim, isReleased=false, doLimiting=false, attachedTool=nil};
		else
			print("AnimatedFrontloader: attacherJointIndex " .. jointIndex .. " not found");
		end;
	end;
end;

function AnimatedFrontloader:delete()
end;

function AnimatedFrontloader:readStream(streamId, connection)
	local isReleased = streamReadBool(streamId);
	self:releaseShovel(isReleased, true);
end;

function AnimatedFrontloader:writeStream(streamId, connection)
	streamWriteBool(streamId, self.animatedFrontloader.isReleased);
end;

function AnimatedFrontloader:mouseEvent(posX, posY, isDown, isUp, button)
end;

function AnimatedFrontloader:keyEvent(unicode, sym, modifier, isDown)
end;

function AnimatedFrontloader:update(dt)
	if self:getIsActive() then
		if self.animatedFrontloader ~= nil then
			if self.animatedFrontloader.attachedTool ~= nil then
				if self.animatedFrontloader.attachedTool.isSelected then
					if InputBinding.hasEvent(InputBinding.DUMP_FRONTLOADER_TOOL) then
						
						self:releaseShovel(not self.animatedFrontloader.isReleased);
					end;
				end;
				
				if not self.animatedFrontloader.isReleased then
					if self.isServer then
						if self.animatedFrontloader.doLimiting then
							local curLimits = Utils.getMovedLimitedValues(self.animatedFrontloader.curLimits, {0,0}, self.animatedFrontloader.limits, 2, self.animatedFrontloader.resetTime, dt, false);
							local joint = self.animatedFrontloader.joint.jointIndex;
							setJointRotationLimit(joint, 2, true, curLimits[1], curLimits[2]);
							self.animatedFrontloader.curLimits = curLimits;
							
							if curLimits[1] == self.animatedFrontloader.limits[1] and curLimits[2]==self.animatedFrontloader.limits[2] then
								self.animatedFrontloader.doLimiting = false;
							end;
						end;
					end;
				else
					if self.animatedFrontloader.doLimiting then					
						if self.isServer then
							if self.animatedFrontloader.isReleased then
								local joint = self.animatedFrontloader.joint.jointIndex;
								setJointRotationLimit(joint, 2, true, self.animatedFrontloader.limits[1], self.animatedFrontloader.limits[2]);
								self.animatedFrontloader.curLimits[1] = self.animatedFrontloader.limits[1];
								self.animatedFrontloader.curLimits[2] = self.animatedFrontloader.limits[2];
							end;
						end;
						self.animatedFrontloader.doLimiting = false;
					end;
				end;
			end;
		end;
	end;
end;

function AnimatedFrontloader:updateTick(dt)
end;

function AnimatedFrontloader:draw()
	if self.animatedFrontloader ~= nil then
		local loader = self.animatedFrontloader;
		if loader.attachedTool ~= nil and loader.attachedTool.isSelected then
			if loader.isReleased then
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("FIX_FRONTLOADER_TOOL"), loader.attachedTool.typeDesc), InputBinding.DUMP_FRONTLOADER_TOOL);
			else
				g_currentMission:addHelpButtonText(string.format(g_i18n:getText("RELEASE_FRONTLOADER_TOOL"), loader.attachedTool.typeDesc), InputBinding.DUMP_FRONTLOADER_TOOL);
			end;
		end;
	end;
end;

function AnimatedFrontloader:attachImplement(implement)
	local loader = self.animatedFrontloader;
	if implement.jointDescIndex == loader.jointIndex then
		loader.attachedTool = implement.object;
	end;
end;

function AnimatedFrontloader:detachImplement(implementIndex)
	local loader = self.animatedFrontloader;
	local implement = self.attachedImplements[implementIndex];
	if implement.jointDescIndex == loader.jointIndex then
		loader.attachedTool = nil;
		self.animatedFrontloader.isReleased = false;
		self.animatedFrontloader.doLimiting = false;
		self.animatedFrontloader.curLimits = {0,0};
		self:playAnimation(self.animatedFrontloader.fixAnim, 1, nil, true);
	end;
end;

function AnimatedFrontloader:releaseShovel(isReleased, noEventSend)
	if self.animatedFrontloader ~= nil then
		AnimatedFrontloaderEvent.sendEvent(self, isReleased, noEventSend);
		self.animatedFrontloader.isReleased = isReleased;
		self.animatedFrontloader.doLimiting = true;
		if isReleased then
			self:playAnimation(self.animatedFrontloader.releaseAnim, 1, nil, true);
		else
			self:playAnimation(self.animatedFrontloader.fixAnim, 1, nil, true);
		end;
	end;
end;

--
-- AnimatedFrontloaderEvent
-- Networkevent for ChopperSwitcher
--
-- @author  	Manuel Leithner
-- @version 	v1.0
-- @date  		14/11/12
-- @history:	v1.0 - Initial version
--
AnimatedFrontloaderEvent = {};
AnimatedFrontloaderEvent_mt = Class(AnimatedFrontloaderEvent, Event);

InitEventClass(AnimatedFrontloaderEvent, "AnimatedFrontloaderEvent");

function AnimatedFrontloaderEvent:emptyNew()
    local self = Event:new(AnimatedFrontloaderEvent_mt);
    return self;
end;

function AnimatedFrontloaderEvent:new(vehicle, isReleased)
    local self = AnimatedFrontloaderEvent:emptyNew()
    self.vehicle = vehicle;
	self.isReleased = isReleased;
    return self;
end;

function AnimatedFrontloaderEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	self.isReleased = streamReadBool(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function AnimatedFrontloaderEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));	
	streamWriteBool(streamId, self.isReleased);
end;

function AnimatedFrontloaderEvent:run(connection)
	self.vehicle:releaseShovel(self.isReleased, true);
	if not connection:getIsServer() then
		g_server:broadcastEvent(AnimatedFrontloaderEvent:new(self.vehicle, self.isReleased), nil, connection, self.vehicle);
	end;
end;

function AnimatedFrontloaderEvent.sendEvent(vehicle, isReleased, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(AnimatedFrontloaderEvent:new(vehicle, isReleased), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(AnimatedFrontloaderEvent:new(vehicle, isReleased));
		end;
	end;
end;
