--
-- ESLimiter
-- Specialization for ESLimiter
--
-- SFM-Modding
-- @author      Manuel Leithner
-- @date:		04/10/11
-- @version:	v2.2
--
-- @history:	v1.0 - initial implementation
--				v2.0 - convert to LS2011 and some bugfixes
--				v2.1 - Network-Fixes
--              v2.2 - fixed localization of disabled text, now reads from i10 table
--
-- @Modified-by: Robert Hostens
-- @Modified-date: 2011/04/10
--
-- Notes:  Add the following constants to your localization table
--
-- ESLIMIT_DISABLED - Text displayed when script is disabled
-- ESLIMIT_DECREASE - text displayed to decrease cruise RPM
-- ESLIMIT_INCREASE - text displayed to increase cruise RPM
-- ESLIMIT_LEVEL    - Ex 'Level n: x%'
-- ESLIMIT_CONTROLS - text prepended to additional help text displaying keybinds
--
-- Add the following keybinds to your modDesc xml
--
-- ESLIMIT_INCREASE - key to increase current cruise RPM Limit
-- ESLIMIT_DECREASE - key to increase current cruise RPM Limit
--


ESLimiter = {};

function ESLimiter.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Motorized, specializations);
end;

function ESLimiter:load(xmlFile)
	self.setNewLimit = SpecializationUtil.callSpecializationsFunction("setNewLimit");

	self.ESLimiter = {};
	self.ESLimiter.xPos = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.ESLimiter#posX"), 0.7543);
	self.ESLimiter.yPos = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.ESLimiter#posY"), 0.2350);
	self.ESLimiter.maxRPM = {};
	self.ESLimiter.percentage = {};
	
	for k, v in pairs(self.motor.maxRpm) do
		local percentage = string.format("%d",(self.motor.maxRpm[k] * 100 / self.motor.maxRpm[3]));
		self.ESLimiter.maxRPM[k] = self.motor.maxRpm[3];
		self.ESLimiter.percentage[k] = tonumber(percentage);
	end;
	
	self.lastButtonDown = nil;
	self.isLimiterActive = false;
end;

function ESLimiter:delete()
end;

function ESLimiter:readStream(streamId, connection)
	for level, _ in pairs(self.ESLimiter.percentage) do
		local percentage = streamReadInt8(streamId);
		self:setNewLimit(level, percentage, false, true);
	end;
end;

function ESLimiter:writeStream(streamId, connection)
	for _, percentage in pairs(self.ESLimiter.percentage) do
		streamWriteInt8(streamId, percentage);
	end;
end;

function ESLimiter:mouseEvent(posX, posY, isDown, isUp, button)
end;

function ESLimiter:keyEvent(unicode, sym, modifier, isDown)
end;

function ESLimiter:update(dt)
	if self:getIsActive() then
		--if self:getIsActiveForInput() then
			local currentLevel = self.motor.speedLevel;
		
			if self.movingDirection < 0 then
				currentLevel = 1;
			end;
			
			if currentLevel ~= 0 then
				local percentage = self.ESLimiter.percentage;
				if InputBinding.isPressed(InputBinding.ESLIMIT_INCREASE) and percentage[currentLevel] < 100 then
					self.lastButtonDown = InputBinding.ESLIMIT_INCREASE;
					percentage[currentLevel] = percentage[currentLevel] + 1;
					-- don't send event to other players
					self:setNewLimit(currentLevel, percentage[currentLevel], false, true);
				elseif InputBinding.isPressed(InputBinding.ESLIMIT_DECREASE) and percentage[currentLevel] > 10 then
					percentage[currentLevel] = percentage[currentLevel] - 1;
					self.lastButtonDown = InputBinding.ESLIMIT_DECREASE;
					-- don't send event to other players
					self:setNewLimit(currentLevel, percentage[currentLevel], false, true);
				else
					if self.lastButtonDown ~= nil then
						-- only send event to other players if key was released -> performance issue
						self:setNewLimit(currentLevel, percentage[currentLevel], false);
						self.lastButtonDown = nil;
					end;	
				end;
				
				self.isLimiterActive = true;
			else
				if self.isLimiterActive then
					self:setNewLimit(3, 100, true);
					self.isLimiterActive = false;
				end;
				
				self.motor.maxRpm[3] = self.ESLimiter.maxRPM[3];
			end;
		--end;
	end;
end;

function ESLimiter:draw()
	local currentLevel = self.motor.speedLevel;
	local text = Motorized.ESLIMIT_DISABLED
	
	if self.movingDirection < 0 then
		currentLevel = 1;
	end;

	if currentLevel ~= 0 then
		local percentage = tonumber(string.format("%d", self.ESLimiter.percentage[currentLevel]));
	
		text = Motorized.ESLIMIT_LEVEL .. currentLevel .. ": " .. percentage .. "%";	
	end;
	
	Motorized.ESLimiter_hud:render();
	setTextBold(true);
	setTextAlignment(RenderText.ALIGN_LEFT);
	renderText(self.ESLimiter.xPos + 0.125, self.ESLimiter.yPos + 0.0085, 0.02, text);
	g_currentMission:addHelpButtonText(Motorized.ESLIMIT_INCREASE, InputBinding.ESLIMIT_INCREASE);
	g_currentMission:addHelpButtonText(Motorized.ESLIMIT_DECREASE, InputBinding.ESLIMIT_DECREASE);
end;

function ESLimiter:setNewLimit(level, percentage, doRestoreLimit, noEventSend)	
	ESLimiterEvent.sendEvent(self, level, percentage, doRestoreLimit, noEventSend);	
	local currentRPM = self.ESLimiter.maxRPM;

	if not doRestoreLimit then
		self.ESLimiter.percentage[level] = percentage;
	
		if self.movingDirection > 0 then
			self.motor.maxRpm[level] = tonumber(string.format("%d",(percentage * currentRPM[level] / 100)));
		else
			self.motor.maxRpm[3] = tonumber(string.format("%d",(percentage * currentRPM[level] / 100)));
		end;
	else
		self.motor.maxRpm[3] = self.ESLimiter.maxRPM[3];
	end;
end;


function ESLimiter:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#limiterValues");		
		if valueStr ~= nil then
			local found = string.find(valueStr, ";");
			local index = 1;
	
			while found ~= nil do
				local value = string.sub(valueStr, 0, found-1);
				valueStr = string.sub(valueStr, found+1);
				found = string.find(valueStr, ";");
				self.ESLimiter.percentage[index] = tonumber(value);
			
				if found == nil then
					self.ESLimiter.percentage[index+1] = tonumber(valueStr);
				end;
				
				index = index + 1;	
			end;
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function ESLimiter:getSaveAttributesAndNodes(nodeIdent)
	local values = "";	
	
	for k, v in pairs(self.ESLimiter.percentage) do
		if k ~= 1 then
			values = values .. ";";
		end;
		
		values = values .. string.format("%d", v);
	end;
	
    local attributes = 'limiterValues="'.. values .. '"';
    return attributes, nil;
end;
-----------------------------------------------------------------------

ESLimiterEvent = {};
ESLimiterEvent_mt = Class(ESLimiterEvent, Event);

InitEventClass(ESLimiterEvent, "ESLimiterEvent");

function ESLimiterEvent:emptyNew()
    local self = Event:new(ESLimiterEvent_mt);
    self.className="ESLimiterEvent";
    return self;
end;

function ESLimiterEvent:new(vehicle, level, percentage, doRestoreLimit)
    local self = ESLimiterEvent:emptyNew()
    self.vehicle = vehicle;
	self.level = level;
	self.percentage = percentage;
	self.doRestoreLimit = doRestoreLimit;
    return self;
end;

function ESLimiterEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	self.level = streamReadInt8(streamId);
	self.percentage = streamReadInt8(streamId);
	self.doRestoreLimit = streamReadBool(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function ESLimiterEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));	
	streamWriteInt8(streamId, self.level);
	streamWriteInt8(streamId, self.percentage);
	streamWriteBool(streamId, self.doRestoreLimit);
end;

function ESLimiterEvent:run(connection)
	self.vehicle:setNewLimit(self.level, self.percentage, self.doRestoreLimit, true);
	if not connection:getIsServer() then
		--g_server:broadcastEvent(ESLimiterEvent:new(self.vehicle, self.level, self.percentage, self.doRestoreLimit), nil, connection, self.object);
	end;
end;

function ESLimiterEvent.sendEvent(vehicle, level, percentage, doRestoreLimit, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			-- no need to broadcast the event: only the server need the information
			--g_server:broadcastEvent(ESLimiterEvent:new(vehicle, level, percentage, doRestoreLimit), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(ESLimiterEvent:new(vehicle, level, percentage, doRestoreLimit));
		end;
	end;
end;