--
-- Limitator
-- Specialization for Limitator
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date:		13/11/12
-- @version:	v3.0
-- @history:	v1.0 - initial implementation
--				v2.0 - convert to LS2011 and some bugfixes
--				v2.1 - Network-Fixes
--				v2.2 - Bugfixes and new huds
--				v3.0 - LS13 & integration of OreFunctionare
--
-- free for noncommerical-usage
--

Limitator = {};

function Limitator.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Motorized, specializations);
end;

function Limitator:load(xmlFile)

	self.setNewLimit = SpecializationUtil.callSpecializationsFunction("setNewLimit");
	self.addOperatingTime = SpecializationUtil.callSpecializationsFunction("addOperatingTime");
	
	self.Limitator = {};
	self.Limitator.maxRPM = {};
	self.Limitator.orgMaxRPM = {};
	self.Limitator.percentage = {};
		
	for k,_ in pairs(self.motor.maxRpm) do	
		local percentage = string.format("%d",(self.motor.maxRpm[k] * 100 / self.motor.maxRpm[3]));
		self.Limitator.percentage[k+1] = tonumber(percentage);
		self.Limitator.maxRPM[k+1] = self.motor.maxRpm[3];
		self.Limitator.orgMaxRPM[k+1] = self.motor.maxRpm[k];
	end;
	-- set limits for reverse
	self.Limitator.maxRPM[1] = self.Limitator.maxRPM[2];
	self.Limitator.percentage[1] = self.Limitator.percentage[2];	
	self.Limitator.default = self.motor.maxRpm[3];
	
	self.isLimiterActive = false;
	self.Limitator.isButtonPressed = false;
	
	self.Limitator.xPos = Utils.getNoNil(getXMLFloat(xmlFile,  "vehicle.Limitator#posX"), 0.7543);
	self.Limitator.yPos = Utils.getNoNil(getXMLFloat(xmlFile,  "vehicle.Limitator#posY"), 0.1715);
	self.Limitator.yOffset = 0.014;
	
	self.Limitator.hudBarWidth = 0.205;
	self.Limitator.overlay = Overlay:new("hudESLOverlay", Utils.getFilename("texturi/ESL/Limitator_hud.png", self.baseDirectory), self.Limitator.xPos, self.Limitator.yPos, 0.2371, 0.039525);
	self.Limitator.overlayBg = Overlay:new("hudESLBackground", Utils.getFilename("texturi/ESL/Limitator_bg.png", self.baseDirectory), self.Limitator.xPos, self.Limitator.yPos, 0.2371, 0.039525);
	self.Limitator.overlayBar = Overlay:new("hudESLbar", Utils.getFilename("texturi/ESL/Limitator_bar.png", self.baseDirectory), self.Limitator.xPos + 0.123, self.Limitator.yPos + 0.01, 0.21, 0.02190);
	
	self.Limitator.operatingTime = 0;	
	self.lastLevelUpdate = 0;
	self.lastLevel = 0;
end;

function Limitator:delete()
end;

function Limitator:readStream(streamId, connection)
	for level,_ in pairs(self.Limitator.percentage) do
		local percentage = streamReadInt16(streamId)/10;
		self:setNewLimit(level, percentage, true);
	end;
	
	self:addOperatingTime(streamReadFloat32(streamId), true);
end;

function Limitator:writeStream(streamId, connection)
	for _,percentage in pairs(self.Limitator.percentage) do
		streamWriteInt16(streamId, percentage*10);
	end;
	
	streamWriteFloat32(streamId, self.Limitator.operatingTime);
end;

function Limitator:mouseEvent(posX, posY, isDown, isUp, button)
end;

function Limitator:keyEvent(unicode, sym, modifier, isDown)
end;

function Limitator:update(dt)
	if (self:getIsActive() and self.isMotorStarted) or self.isHired then 
		self:addOperatingTime(dt);
	end;
	
	if self:getIsActive() then
		local level = 0;
		if self.movingDirection < 0 then
			level = 1;
		else
			if self.motor.speedLevel > 0 then
				level = self.motor.speedLevel + 1;
			end;
		end;
		
		if self.isHired then
			if self.attachedCutters ~= nil then
				level = 2;
				for cutter,implement in pairs(self.attachedCutters) do
					if Cutter.getUseLowSpeedLimit(cutter) then
						level = 1;
						break;
					end;
				end;
			else
				level = 1;
			end;
			level = level + 1;
		end;
		
		if self.lastLevel == level then
			self.lastLevelUpdate = self.lastLevelUpdate + dt;
		else	
			self.lastLevelUpdate = 0;
		end;
		self.lastLevel = level;
		
		if level ~= 0 and self.isClient and self:getIsActiveForInput(false) and not self:hasInputConflictWithSelection() then
			local percentage = self.Limitator.percentage;
			if InputBinding.isPressed(InputBinding.Limitator_PLUS) and percentage[level] < 100 then					
				percentage[level] = percentage[level] + 0.25;
				self:setNewLimit(level, percentage[level], true);
				self.Limitator.isButtonPressed = true;
			elseif InputBinding.isPressed(InputBinding.Limitator_MINUS) and percentage[level] > 10 then
				percentage[level] = percentage[level] - 0.25;					
				self:setNewLimit(level, percentage[level], true);
				self.Limitator.isButtonPressed = true;
			else
				if self.Limitator.isButtonPressed then
					-- only send event to other players if key was released -> performance issue
					self:setNewLimit(level, percentage[level]);
					self.Limitator.isButtonPressed = false;
				end;	
			end;
		end;
		
		if level ~= 0 then
			local newRpm = tonumber(string.format("%d",(self.Limitator.percentage[level] * self.Limitator.maxRPM[level] / 100)));
			if self.movingDirection >= 0 then
				self.motor.maxRpm[level - 1] = newRpm;
			else
				-- backward
				self.motor.maxRpm[3] = newRpm;
			end;
		else
			-- reset to default value if no level is set
			self.motor.maxRpm[3] = self.Limitator.default;
		end;
	end;
end;

function Limitator:draw()
	local text = "Dezactivat";
	local currentLevel = self.motor.speedLevel;
	local currentLevelText = 0;
	local percentage = 0;
	
	if self.movingDirection < 0 then
		currentLevel = 1;
		currentLevelText = "R";
	else
		if self.isHired then
			if self.attachedCutters ~= nil then
				currentLevel = 2;
				for cutter,implement in pairs(self.attachedCutters) do
					if Cutter.getUseLowSpeedLimit(cutter) then
						currentLevel = 1;
						break;
					end;
				end;
			else
				currentLevel = 1;
			end;
		end;
	
		if currentLevel ~= 0 then
			currentLevel = currentLevel + 1;
			currentLevelText = currentLevel - 1;
		end;
	end;
	
	if currentLevel ~= 0 and (self.lastLevelUpdate > 500 or self.movingDirection > 0) then
		percentage = tonumber(string.format("%d", self.Limitator.percentage[currentLevel]));
		if percentage < 100 then
			percentage = " " .. percentage;
		end;
		text = g_i18n:getText("Limitator_level") .. currentLevelText .. ":" .. percentage .. "% ";		
		g_currentMission:addExtraPrintText(string.format(g_i18n:getText("Limitator_controls"), self.typeDesc) .. " " .. InputBinding.getKeyNamesOfDigitalAction(InputBinding.Limitator_PLUS) .. "/" .. InputBinding.getKeyNamesOfDigitalAction(InputBinding.Limitator_MINUS));
	end;
	
	self.Limitator.overlayBg:render();
	
	self.Limitator.overlayBar.width = self.Limitator.hudBarWidth * math.min(1, percentage/190);	
    setOverlayUVs(self.Limitator.overlayBar.overlayId, 0, 0.05, 0, 1, math.min(1, percentage/190), 0.05, math.min(1, percentage/190), 1);
	self.Limitator.overlayBar:render();
	
	self.Limitator.overlay:render();
	setTextBold(true);
	setTextAlignment(RenderText.ALIGN_LEFT);
	setTextColor(0, 0, 0, 1);
    renderText(self.Limitator.xPos + 0.139, self.Limitator.yPos + 0.008, 0.02, text);	
	setTextColor(1, 1, 1, 1);
	renderText(self.Limitator.xPos + 0.139, self.Limitator.yPos + 0.010, 0.02, text);	
	
	local OreFunctionare = math.floor((self.Limitator.operatingTime / 1000 / 60 / 60)*10) / 10;
	local hours = math.floor(OreFunctionare);
	local minutes = tostring(math.floor((OreFunctionare - hours)*10));
	setTextAlignment(RenderText.ALIGN_RIGHT);
	setTextBold(true);	
	setTextColor(0, 0, 0, 1);
    renderText(self.Limitator.xPos+0.062, self.Limitator.yPos + 0.008, 0.02, tostring(hours) .. ",");	
	setTextColor(1,1,1,1);
	renderText(self.Limitator.xPos+0.062, self.Limitator.yPos + 0.010, 0.02, tostring(hours) .. ",");
	setTextColor(0, 0, 0, 1);
    renderText(self.Limitator.xPos+0.071, self.Limitator.yPos + 0.008, 0.02, minutes);
	setTextColor(0.95,0,0,1);
	renderText(self.Limitator.xPos+0.071, self.Limitator.yPos + 0.010, 0.02, minutes);
	setTextColor(0, 0, 0, 1);
    renderText(self.Limitator.xPos+0.08, self.Limitator.yPos + 0.008, 0.02, "h");
	setTextColor(1,1,1,1);
	renderText(self.Limitator.xPos+0.08, self.Limitator.yPos + 0.010, 0.02, "h");
	setTextAlignment(RenderText.ALIGN_LEFT);	
end;

function Limitator:setNewLimit(level, percentage, noEventSend)	
	LimitatorEvent.sendEvent(self, level, percentage, noEventSend);	
	self.Limitator.percentage[level] = percentage;
end;

function Limitator:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
	local valueStr = getXMLString(xmlFile, key.."#limiterValues");		
	if valueStr ~= nil then
		local found = string.find(valueStr, ";");
		local index = 1;
		while found ~= nil do
			local value = string.sub(valueStr, 0, found-1);
			valueStr = string.sub(valueStr, found+1);
			found = string.find(valueStr, ";");
			self:setNewLimit(index, tonumber(value), true);
			if found == nil then
				value = tonumber(valueStr);
				self:setNewLimit(index+1, tonumber(value), true);
			end;
			index = index + 1;	
		end;
	end;
	
	local operatingTime = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#operatingTime"), self.Limitator.operatingTime) * 1000;
	self:addOperatingTime(operatingTime);

    return BaseMission.VEHICLE_LOAD_OK;
end;

function Limitator:getSaveAttributesAndNodes(nodeIdent)
	local values = "";	
	for k,v in pairs(self.Limitator.percentage) do
		if k ~= 1 then
			values = values .. ";";
		end;
		values = values .. string.format("%d", v);
	end;
	
    local attributes = 'limiterValues="'.. values .. '" operatingTime="' .. tostring((self.Limitator.operatingTime / 1000)) ..'"';
    return attributes, nil;
end;

function Limitator:addOperatingTime(addTime)
	
	local time = math.max(Utils.getNoNil(addTime, 0), 0);	
	self.Limitator.operatingTime = self.Limitator.operatingTime + time;	
	
	if self.Limitator.operatingTime > (99999.9 * 1000 * 60 * 60) then
		self.Limitator.operatingTime = 0;
	end;
end;


--
-- LimitatorEvent
-- Networkevent for Limitator
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date:		14/12/11
-- @version:	v1.0
-- @history:	v1.0 - initial implementation
--
LimitatorEvent = {};
LimitatorEvent_mt = Class(LimitatorEvent, Event);

InitEventClass(LimitatorEvent, "LimitatorEvent");

function LimitatorEvent:emptyNew()
    local self = Event:new(LimitatorEvent_mt);
    return self;
end;

function LimitatorEvent:new(vehicle, level, percentage)
    local self = LimitatorEvent:emptyNew()
    self.vehicle = vehicle;
	self.level = level;
	self.percentage = percentage;
    return self;
end;

function LimitatorEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	self.level = streamReadInt8(streamId);
	self.percentage = streamReadInt16(streamId)/10;
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function LimitatorEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));	
	streamWriteInt8(streamId, self.level);
	streamWriteInt16(streamId, self.percentage*10);
end;

function LimitatorEvent:run(connection)
	self.vehicle:setNewLimit(self.level, self.percentage, true);
end;

function LimitatorEvent.sendEvent(vehicle, level, percentage, noEventSend)
	
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(LimitatorEvent:new(vehicle, level, percentage), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(LimitatorEvent:new(vehicle, level, percentage));
		end;
	end;
end;