--
-- ChopperSwitcher
-- Specialization for ChopperSwitcher
--
-- @author  	Manuel Leithner
-- @version 	v1.0
-- @date  		13/05/11
-- @history:	v1.0 - Initial version
--

ChopperSwitcher = {};

ChopperSwitcher.CHOPPER_TRANSPORT = 0;
ChopperSwitcher.CHOPPER_STRAW = 1;
ChopperSwitcher.CHOPPER = 2;

function ChopperSwitcher.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(GrainTankAnimation, specializations);
end;

function ChopperSwitcher:load(xmlFile)

	self.setIsStrawActive = SpecializationUtil.callSpecializationsFunction("setIsStrawActive");
	self.setChopperState = SpecializationUtil.callSpecializationsFunction("setChopperState");
	self.setTransportState = SpecializationUtil.callSpecializationsFunction("setTransportState");
	
	self.strawParticleSystemBackup = self.strawParticleSystems;
	self.cuttingAreasBackup = self.strawAreas;
	self.isStrawActive = true;
	self.toggleChopperWarningTime = 0;
	self.isLoadingFromSaveGame = false;
	self.chopperState = ChopperSwitcher.CHOPPER_TRANSPORT;
	self:setIsStrawActive(false, true);
	self:setTransportState(true);
end;

function ChopperSwitcher:delete()
	self.strawParticleSystems = self.strawParticleSystemBackup;
end;

function ChopperSwitcher:readStream(streamId, connection)
	local isStrawActive = streamReadBool(streamId);
	self:setIsStrawActive(isStrawActive, true);
end;

function ChopperSwitcher:writeStream(streamId, connection)
	streamWriteBool(streamId, self.isStrawActive);
end;

function ChopperSwitcher:mouseEvent(posX, posY, isDown, isUp, button)
end;

function ChopperSwitcher:keyEvent(unicode, sym, modifier, isDown)
end;

function ChopperSwitcher:update(dt)
	if self:getIsActive() then		
		if self:getIsActiveForInput() then
			if InputBinding.hasEvent(InputBinding.TOGGLE_CHOPPER) then
				if self.isGraintankExpanded then
					if self.lastValidFruitType ~= FruitUtil.FRUITTYPE_UNKNOWN then
						local fruitDesc = FruitUtil.fruitIndexToDesc[self.lastValidFruitType];
						if fruitDesc.hasStraw then
							self:setIsStrawActive(not self.isStrawActive);
						else
							self.toggleChopperWarningTime = self.time + 2000;
						end;
					else
						self:setIsStrawActive(not self.isStrawActive);				
					end;
				end;
			end;
		end;		
		
		if self.lastValidFruitType ~= FruitUtil.FRUITTYPE_UNKNOWN then
			local fruitDesc = FruitUtil.fruitIndexToDesc[self.lastValidFruitType];
			if not fruitDesc.hasStraw then
				if self.isStrawActive then
					self:setIsStrawActive(false);
				end;			
			end;
		end;
	end;
end;

function ChopperSwitcher:draw()

	local fruitDesc = FruitUtil.fruitIndexToDesc[self.lastValidFruitType];	
	if self.lastValidFruitType == FruitUtil.FRUITTYPE_UNKNOWN or fruitDesc.hasStraw then	
		if self.isGraintankExpanded then
			if self.isStrawActive then
				g_currentMission:addHelpButtonText(g_i18n:getText("ENABLE_CHOPPER"), InputBinding.TOGGLE_CHOPPER);
			else	
				g_currentMission:addHelpButtonText(g_i18n:getText("DISABLE_CHOPPER"), InputBinding.TOGGLE_CHOPPER);
			end;
		end;
	end;
	
	if self.toggleChopperWarningTime > self.time then
		g_currentMission:addWarning(g_i18n:getText("COULD_NOT_TOGGLE_CHOPPER"), 0.018, 0.033);
	end;
end;

function ChopperSwitcher:setTransportState(isTransportState)
	if isTransportState then
		self:setChopperState(ChopperSwitcher.CHOPPER_TRANSPORT);
	else
		if self.isStrawActive then
			self:setChopperState(ChopperSwitcher.CHOPPER_STRAW);
		else	
			self:setChopperState(ChopperSwitcher.CHOPPER);
		end;
	end;
end;

function ChopperSwitcher:setChopperState(state)
	if state ~= self.chopperState then
		local animTimeChopper = nil;
		local animTimeStraw = nil;
		if self:getIsAnimationPlaying("openStraw") then			
			animTimeStraw = self:getAnimationTime("openStraw");
		elseif self:getIsAnimationPlaying("openChopper") then			
			animTimeChopper = self:getAnimationTime("openChopper");
		end;
		self:stopAnimation("openStraw", true);
		self:stopAnimation("openChopper", true);
		local direction = 1;
		
		if state == ChopperSwitcher.CHOPPER_TRANSPORT then
			self:playAnimation("openStraw", 1*direction, animTimeStraw, true);
			self.chopperState = state;
		elseif state == ChopperSwitcher.CHOPPER_STRAW then
			if self.chopperState == ChopperSwitcher.CHOPPER or self.chopperState == ChopperSwitcher.CHOPPER_TRANSPORT then
				direction = -1;
			end;
			self:playAnimation("openStraw", 1*direction, animTimeStraw, true);
		elseif state == ChopperSwitcher.CHOPPER then
			if self.chopperState == ChopperSwitcher.CHOPPER_STRAW or self.chopperState == ChopperSwitcher.CHOPPER_TRANSPORT then
				direction = -1;
			end;
			self:playAnimation("openChopper", -1*direction, animTimeChopper, true);
		end;
		self.chopperState = state;
	end;
end;

function ChopperSwitcher:setIsStrawActive(isStrawActive, noEventSend)	
	ChopperSwitchEvent.sendEvent(self, isStrawActive, noEventSend);	
	self.isStrawActive = isStrawActive;
	
	if isStrawActive then
		if self.isGraintankExpanded then
			self:setChopperState(ChopperSwitcher.CHOPPER_STRAW);
		end;
		
		self.strawAreas = self.cuttingAreasBackup;
		self.strawParticleSystems = self.strawParticleSystemBackup;
		for fruit, chopperParticleSystem in pairs(self.chopperParticleSystems) do
			for strawFruit, strawParticleSystem in pairs(self.strawParticleSystems) do 
				if strawFruit == fruit then
					Utils.setEmittingState(strawParticleSystem, Utils.getNoNil(chopperParticleSystem.isEmitting, false));
					break;
				end;
			end;
			Utils.setEmittingState(chopperParticleSystem, false);
		end;		
	else
		if self.isGraintankExpanded then
			self:setChopperState(ChopperSwitcher.CHOPPER);
		end;
		
		self.strawAreas = {};
		for strawFruit, strawParticleSystem in pairs(self.strawParticleSystems) do
			for fruit, chopperParticleSystem in pairs(self.chopperParticleSystems) do 
				if strawFruit == fruit then
					Utils.setEmittingState(chopperParticleSystem, Utils.getNoNil(strawParticleSystem.isEmitting, false));
					break;
				end;
			end;
			Utils.setEmittingState(strawParticleSystem, false);
		end;
		self.strawParticleSystems = self.chopperParticleSystems;
	end;
	
	if self.isLoadingFromSaveGame then
		self.isLoadingFromSaveGame = false;
	end;
end;

function ChopperSwitcher:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)	
	if not resetVehicles then
		local isStrawActive = Utils.getNoNil(getXMLBool(xmlFile, key .. "#isStrawActive"), false);
		self.isLoadingFromSaveGame = true;
		self:setIsStrawActive(isStrawActive, true);
	end; 
    return BaseMission.VEHICLE_LOAD_OK;
end;

function ChopperSwitcher:getSaveAttributesAndNodes(nodeIdent)
	local attributes = 'isStrawActive="' .. tostring(self.isStrawActive) ..'"';
	return attributes, nil;
end;



--
-- ChopperSwitchEvent
-- Networkevent for ChopperSwitcher
--
-- @author  	Manuel Leithner
-- @version 	v1.0
-- @date  		14/11/12
-- @history:	v1.0 - Initial version
--
ChopperSwitchEvent = {};
ChopperSwitchEvent_mt = Class(ChopperSwitchEvent, Event);

InitEventClass(ChopperSwitchEvent, "ChopperSwitchEvent");

function ChopperSwitchEvent:emptyNew()
    local self = Event:new(ChopperSwitchEvent_mt);
    self.className="ChopperSwitchEvent";
    return self;
end;

function ChopperSwitchEvent:new(vehicle, isStrawActive)
    local self = ChopperSwitchEvent:emptyNew()
    self.vehicle = vehicle;
	self.isStrawActive = isStrawActive;
    return self;
end;

function ChopperSwitchEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	self.isStrawActive = streamReadBool(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function ChopperSwitchEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));	
	streamWriteBool(streamId, self.isStrawActive);
end;

function ChopperSwitchEvent:run(connection)
	self.vehicle:setIsStrawActive(self.isStrawActive, true);
	if not connection:getIsServer() then
		g_server:broadcastEvent(ChopperSwitchEvent:new(self.vehicle, self.isStrawActive), nil, connection, self.vehicle);
	end;
end;

function ChopperSwitchEvent.sendEvent(vehicle, isStrawActive, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(ChopperSwitchEvent:new(vehicle, isStrawActive), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(ChopperSwitchEvent:new(vehicle, isStrawActive));
		end;
	end;
end;