_ = [[
@title:			BunkerSilosHud
@description:	A hud that shows the contents of all silos (BGA and cow), including the distribution inside each silo, plus the fill levels of all liquid manure tanks.
@author:		Jakob Tischler
@project start:	13 Jan 2013
@date:			02 Mar 2014
@version:		0.99
@changelog:		0.98:	* initial release
				0.99:	* add safety check against impromperly created BunkerSilo triggers (w/o movingPlanes) - you know who you are!
						* add warning sign if silo is fully fermented but the cover plane hasn't been removed yet
						* fix scrolling functionality/zooming inhibit for vehicles with 'InteractiveControl' spec

@contact:		jakobgithub -tt- gmail -dot- com
@note:			Modification, upload, distribution or any other change without the author's written permission is not allowed.
@thanks:		bassaddict, for valuable performance input
				Peter van der Veen and Claus G. Pedersen, for testing the English version
				upsidedown, for the camera movement function and testing
				mor2000 for multiplayer testing


TODO:
1) how to render the effects overlay OVER the rendered text (z-index)? As of right now, the text, even though rendered before, is still rendered above the effects overlay.
]]

bsh = {};
bsh.modDir = g_currentModDirectory;
bsh.modName = g_currentModName;
bsh.imgDir = bsh.modDir .. 'res/';
bsh.hasAppendedSaveFn = false;
bsh.hasMouseCursorActive = false;
bsh.canScroll = false;

bsh.HUDSTATE_INTERACTIVE = 1;
bsh.HUDSTATE_ACTIVE = 2;
bsh.HUDSTATE_CLOSED = 3;

local modItem = ModsUtil.findModItemByModName(bsh.modName);
if modItem and modItem.version and modItem.author then
	bsh.version, bsh.author = modItem.version, modItem.author;
else
	bsh.version, bsh.author = '0.0', 'Jakob Tischler';
end;

function bsh:loadMap(name)
	if (self.initialized) then
		return;
	end;

	self.input = {
		hud = {
			-- action = self:getKeyIdOfAction(InputBinding.BUNKERSILOS_HUD);
			modifier = self:getKeyIdOfModifier(InputBinding.BUNKERSILOS_HUD);
		};
	};
	self.helpButtonText = g_i18n:getText('BUNKERSILOS_SHOWHUD');

	self.gui = {
		hudState = bsh.HUDSTATE_CLOSED;

		width  = self:getFullPixelX(512/1920); -- width = 512/1080 / g_screenAspectRatio;
		height = self:getFullPixelY(512/1080);

		hPadding = 46/1920; -- hPadding = 46/1080 / g_screenAspectRatio;
		vPadding =  6/1080;

		hMargin = 21/1920; -- hMargin = 21/1080 / g_screenAspectRatio;
		vMargin = 12/1080;

		gfxWidth  = 512/1080 / g_screenAspectRatio; -- gfxWidth  = 512/1920;
		gfxHeight = 291/1080;

		fontSize = 0.018;

		buttonGfxWidth  = 32/1080 / g_screenAspectRatio; -- buttonGfxWidth = 32/1920;
		buttonGfxHeight = 32/1080;

		arrowGfxWidth  = 16/1080 / g_screenAspectRatio; -- arrowGfxWidth = 16/1920;
		arrowGfxHeight = 32/1080;

		arrowContentWidth  = 16/1080 / g_screenAspectRatio; -- arrowContentWidth = 16/1920;
		arrowContentHeight = 20/1080;

		closeContentWidth  = 24/1080 / g_screenAspectRatio; -- closeContentWidth = 24/1920;
		closeContentHeight = 24/1080;
	};

	-- self.gui.baseX = self.gui.hMargin;
	-- self.gui.baseY = 0.05;
	self.gui.baseX = self:getFullPixelX(1 - self.gui.width - self.gui.hMargin);
	self.gui.baseY = self:getFullPixelY(265/1080 + self.gui.vMargin + 30/1080); --NOTE: add 30px so it doesn't overlap with Courseplay's hud


	self.gui.contentMinX = self:getFullPixelX(self.gui.baseX + self.gui.hPadding);
	self.gui.contentMaxX = self:getFullPixelX(self.gui.baseX + self.gui.width - self.gui.hPadding);
	self.gui.textMinX = self.gui.contentMinX - 0.0025;
	self.gui.textMaxX = self.contentMaxX;

	self.gui.fontSizeTitle = self.gui.fontSize * 1.15;

	self.gui.boxAreaWidth = self.gui.width - (2 * self.gui.hPadding);
	self.gui.boxMargin = 0.002;
	self.gui.boxMaxHeight = 45/1080; -- self.gui.boxMaxHeight = self.gui.fontSize * 1.75;

	self.gui.upperLineY = self:getFullPixelY(self.gui.baseY + 242/1080);

	self.gui.lines = {};
	-- Silos
	self.gui.lines[1] = self:getFullPixelY(self.gui.baseY + 209/1080);
	self.gui.lines[2] = self.gui.lines[1] - self.gui.fontSizeTitle;
	self.gui.lines[3] = self.gui.lines[2] - self.gui.fontSize;
	self.gui.lines[4] = self.gui.lines[3] - self.gui.fontSize;
	self.gui.lines[5] = self:getFullPixelY(self.gui.baseY + 101/1080); -- self.gui.lines[5] = self.gui.lines[4] - self.gui.boxMaxHeight;
	-- Liquid manure tanks
	self.gui.lines[6] = self.gui.lines[5] - self.gui.fontSize * 1.5 - 2/1080;
	self.gui.lines[7] = self.gui.lines[6] - self.gui.fontSizeTitle * 1.05;
	self.gui.lines[8] = self.gui.lines[7] - self.gui.fontSize;

	-- Liquid manure tank bar graph
	self.gui.tankBarHeight = self.gui.fontSize * 0.75;
	self.gui.tankBarBorderWidth = 0.002;
	self.gui.tankBarMinX = self.gui.contentMinX + (self.gui.tankBarBorderWidth * 2);
	self.gui.tankBarMaxX = self.gui.contentMaxX - (self.gui.tankBarBorderWidth * 2);
	self.gui.tankBarMaxWidth = self.gui.tankBarMaxX - self.gui.tankBarMinX;

	self.gui.colors = {
		default 			= self:rgba(33, 48, 24, 1.0),
		defaultHover		= self:rgba(33, 48, 24, 2/3),
		clicked 			= self:rgba(33, 48, 24, 1/3),
		defaultNonCompacted	= self:rgba(33, 48, 24, 0.5),
		text 				= self:rgba(51, 56, 48, 1.0),
		rotten 				= self:rgba(73, 47, 42, 1.0),
		rottenNonCompacted	= self:rgba(73, 47, 42, 0.5)
	};

	self.gui.background = Overlay:new('bsh_background', Utils.getFilename('bsh_bg.png',     bsh.imgDir), self.gui.baseX, self.gui.baseY, self.gui.width, self.gui.height);
	self.gui.effects =    Overlay:new('bsh_fx',         Utils.getFilename('bsh_bg_fx.png',  bsh.imgDir), self.gui.baseX, self.gui.baseY, self.gui.width, self.gui.height);

	self.gui.mouseWheel = Overlay:new('bsh_mouse',      Utils.getFilename('mouseWheel.png', bsh.imgDir), self.gui.contentMinX, self.gui.upperLineY, self.gui.closeContentWidth * .6, self.gui.closeContentHeight * 1.2);
	self:setOverlayColor(self.gui.mouseWheel, 'default');

	--					   x1					 x2					   y1				  y2
	self.gui.silosArea = { self.gui.contentMinX, self.gui.contentMaxX, self.gui.lines[5], self.gui.lines[1] + self.gui.arrowContentHeight };
	self.gui.tankArea  = { self.gui.contentMinX, self.gui.contentMaxX, self.gui.lines[8], self.gui.lines[6] + self.gui.arrowContentHeight };


	self.gui.buttons = {};
	self.gui.closeHudButton = self:registerButton('bsh_close.png', 'setHudState', bsh.HUDSTATE_CLOSED, self.gui.contentMaxX-self.gui.closeContentWidth, self.gui.upperLineY, self.gui.buttonGfxWidth, self.gui.buttonGfxHeight, self.gui.closeContentWidth, self.gui.closeContentHeight);

	self.gui.warning = Overlay:new('bsh_warning', Utils.getFilename('warning.png', bsh.imgDir), self.gui.contentMinX + self.gui.closeContentWidth, self.gui.upperLineY, self.gui.closeContentWidth, self.gui.closeContentHeight);
	self:setOverlayColor(self.gui.warning, 'default');
	self.displayWarning = false;
	self.blinkLength = 500; -- in ms

	self.bunkerStates = {
		[0] = g_i18n:getText('BUNKERSILOS_STATE_0'), -- to be filled
		[1] = g_i18n:getText('BUNKERSILOS_STATE_1'), -- fermentation
		[2] = g_i18n:getText('BUNKERSILOS_STATE_2')  -- done/silage
	};

	local ce = g_currentMission.missionInfo.customEnvironment;
	if ce and _G[ce] and _G[ce].g_i18n and _G[ce].g_i18n.getText then
		self.mapI18n = _G[ce].g_i18n;
	end;

	local langNumData = {
		br = { '.', ',' },
		cz = { ' ', ',' },
		de = { '.', ',' },
		en = { ',', '.' },
		es = { '.', ',' },
		fr = { ' ', ',' },
		it = { '.', ',' },
		jp = { ',', '.' },
		pl = { ' ', ',' },
		ru = { ' ', ',' }
	};
	self.numberSeparator = '\'';
	self.numberDecimalSeparator = '.';
	if g_languageShort and langNumData[g_languageShort] then
		self.numberSeparator        = langNumData[g_languageShort][1];
		self.numberDecimalSeparator = langNumData[g_languageShort][2];
	end;

	self.silos = {};
	self.tempSiloTriggers = {};
	self.siloTriggerIdToIdx = {};

	self.tanks = {};
	self.tempTankTriggers = {};
	self.tanksIdentifiedById = {};

	local bunkerSiloIdx = 0;
	local liquidManureIdx = 0;
	local farmLiquidManureIdx = nil;

	for k,v in pairs(g_currentMission.tipTriggers) do
		if g_currentMission.tipTriggers[k] ~= nil then
			local t = g_currentMission.tipTriggers[k];

			-- Silos
			if t.bunkerSilo ~= nil and t.bunkerSilo.movingPlanes ~= nil then
				bunkerSiloIdx = bunkerSiloIdx + 1;
				self.tempSiloTriggers[bunkerSiloIdx] = v;

				local siloTable = {
					bunkerSiloIdx = bunkerSiloIdx;
					bunkerSiloNum = bunkerSiloIdx;
					triggerId = t.triggerId;
					state = t.bunkerSilo.state;
					stateText = self.bunkerStates[tonumber(t.bunkerSilo.state)];
					fillLevel = t.bunkerSilo.fillLevel;
					fillLevelFormatted = self:formatNumber(t.bunkerSilo.fillLevel, 0);
					fillLevelPct = 0;
					fillLevelPctFormatted = '0';
					capacity = t.bunkerSilo.capacity;
					capacityFormatted = self:formatNumber(t.bunkerSilo.capacity, 0);
					toFillFormatted = '0';
					compactedFillLevel = t.bunkerSilo.compactedFillLevel;
					compactPct = 0;
					compactPctFormatted = '0';
					fermentingTime = t.bunkerSilo.fermentingTime;
					fermentingDuration = t.bunkerSilo.fermentingDuration;
					fermentationPct = 0;
					fermentationPctFormatted = '0';
					movingPlanesNum = #t.bunkerSilo.movingPlanes;
					movingPlanes = {};
					rottenFillLevel = 0;
					rottenFillLevelPctFormatted = '0';
					name = ('%s %i'):format(g_i18n:getText('BUNKERSILOS_SILO'), bunkerSiloIdx);
				};
				siloTable.boxWidth = (self.gui.boxAreaWidth - (siloTable.movingPlanesNum-1)*self.gui.boxMargin) / siloTable.movingPlanesNum;

				for i=1, siloTable.movingPlanesNum do
					local movingPlane = {
						capacity = t.bunkerSilo.movingPlanes[i].capacity;
						fillLevel = t.bunkerSilo.movingPlanes[i].fillLevel;
						boxX = self.gui.baseX + self.gui.hPadding + (i-1)*(siloTable.boxWidth + self.gui.boxMargin);
					};

					local imgPath = 'dataS2/menu/white.png';
					-- local imgPath = Utils.getFilename('movingPlaneBox_1x1.png', bsh.imgDir);
					movingPlane.bshOverlay = Overlay:new(('silo_%d_box_%d'):format(bunkerSiloIdx, i), imgPath, movingPlane.boxX, self.gui.lines[5], siloTable.boxWidth, 0);
					movingPlane.bshOverlayNonCompacted = Overlay:new(('silo_%d_box_%d_nonComp'):format(bunkerSiloIdx, i), imgPath, movingPlane.boxX, self.gui.lines[5], siloTable.boxWidth, 0);
					self:setOverlayColor(movingPlane.bshOverlay, 'default');
					self:setOverlayColor(movingPlane.bshOverlayNonCompacted, 'defaultNonCompacted');

					table.insert(siloTable.movingPlanes, movingPlane);
				end;

				local name = getUserAttribute(t.triggerId, 'bshName');
				if name ~= nil and self.mapI18n then
					siloTable.name = self.mapI18n:getText('BSH_' .. name);
					t.bshName = siloTable.name;
					t.bunkerSilo.bshName = siloTable.name;
				end;

				table.insert(self.silos, siloTable);
				self.siloTriggerIdToIdx[t.triggerId] = bunkerSiloIdx;

			-- BGA liquid manure tank(s)
			elseif t.bga ~= nil then
				local triggerId = t.bga.liquidManureSiloTrigger.triggerId;
				if not self.tanksIdentifiedById[triggerId] then
					liquidManureIdx = liquidManureIdx + 1;
					self.tempTankTriggers[liquidManureIdx] = v;

					local tankTable = {
						tankIdx = liquidManureIdx;
						tankNum = liquidManureIdx;
						triggerId = triggerId;
						isBGA = true;
						fillLevel = t.bga.liquidManureSiloTrigger.fillLevel;
						fillLevelFormatted = '0';
						fillLevelPct = 0;
						fillLevelPctFormatted = '0';
						capacity = t.bga.liquidManureSiloTrigger.capacity;
						capacityFormatted = self:formatNumber(t.bga.liquidManureSiloTrigger.capacity, 0);
						name = ('%s %i (%s)'):format(g_i18n:getText('BUNKERSILOS_TANK'), liquidManureIdx, g_i18n:getText('BUNKERSILOS_TANKTYPE_BGA'));
					};

					local name = getUserAttribute(triggerId, 'bshName');
					if name ~= nil and self.mapI18n then
						tankTable.name = self.mapI18n:getText('BSH_' .. name);
						t.bga.liquidManureSiloTrigger.bshName = tankTable.name;
					end;

					table.insert(self.tanks, tankTable);
					self.tanksIdentifiedById[triggerId] = true;
					-- print(string.format('# BSH: add LiquidManureTank %s (triggerId %s, name \'%s\') [BGA] / total: %s', tostring(liquidManureIdx), tostring(triggerId), tostring(tankTable.name), tostring(#self.tanks)));
				end;

			-- Farm (cows) liquid manure tank
			elseif t.animalHusbandry ~= nil and t.animalHusbandry.liquidManureTrigger ~= nil then
				local triggerId = t.animalHusbandry.liquidManureTrigger.triggerId;
				if not self.tanksIdentifiedById[triggerId] then
					liquidManureIdx = liquidManureIdx + 1;
					self.tempTankTriggers[liquidManureIdx] = v;
					local tankTable = {
						tipTriggersIdx = k;
						tankIdx = liquidManureIdx;
						tankNum = liquidManureIdx;
						triggerId = triggerId;
						isFarm = true;
						fillLevel = t.animalHusbandry.liquidManureTrigger.fillLevel;
						fillLevelFormatted = '0';
						fillLevelPct = 0;
						fillLevelPctFormatted = '0';
						capacity = t.animalHusbandry.liquidManureTrigger.capacity;
						capacityFormatted = self:formatNumber(t.animalHusbandry.liquidManureTrigger.capacity, 0);
						name = ('%s %i (%s)'):format(g_i18n:getText('BUNKERSILOS_TANK'), liquidManureIdx, g_i18n:getText('BUNKERSILOS_TANKTYPE_FARM'));
					};

					local name = getUserAttribute(triggerId, 'bshName');
					if name ~= nil and self.mapI18n then
						tankTable.name = self.mapI18n:getText('BSH_' .. name);
					end;

					table.insert(self.tanks, tankTable);
					self.tanksIdentifiedById[triggerId] = true;
					-- print(string.format('# BSH: add LiquidManureTank %s (triggerId %s, name \'%s\') [Farm] / total: %s', tostring(liquidManureIdx), tostring(triggerId), tostring(tankTable.name), tostring(#self.tanks)));
				end;
			end;
		end;
	end; --END for

	for i,onCreateObject in pairs(g_currentMission.onCreateLoadedObjects) do
		-- ManureLager tanks
		if onCreateObject.ManureLagerDirtyFlag ~= nil or Utils.endsWith(onCreateObject.className, 'ManureLager') then
			local triggerId = onCreateObject.triggerId;
			if not self.tanksIdentifiedById[triggerId] then
				liquidManureIdx = liquidManureIdx + 1;
				local tankTable = {
					tankIdx = liquidManureIdx;
					tankNum = liquidManureIdx;
					onCreateIndex = i;
					isManureLager = true;
					fillLevel = onCreateObject.fillLevel;
					fillLevelFormatted = '0';
					fillLevelPct = 0;
					fillLevelPctFormatted = '0';
					capacity = onCreateObject.capacity;
					capacityFormatted = self:formatNumber(onCreateObject.capacity, 0);
					name = ('%s %i (%s)'):format(g_i18n:getText('BUNKERSILOS_TANK'), liquidManureIdx, g_i18n:getText('BUNKERSILOS_TANKTYPE_MANURESTORAGE'));
				};

				local name = getUserAttribute(triggerId, 'bshName');
				if name ~= nil and self.mapI18n then
					tankTable.name = self.mapI18n:getText('BSH_' .. name);
					onCreateObject.bshName = tankTable.name;
				end;

				table.insert(self.tanks, tankTable);
				self.tanksIdentifiedById[triggerId] = true;
				-- print(string.format('# BSH: add LiquidManureTank %s (triggerId %s, name \'%s\') [ManureLager] / total: %s', tostring(liquidManureIdx), tostring(triggerId), tostring(tankTable.name), tostring(#self.tanks)));
			end;

		-- Pigs
		elseif onCreateObject.numSchweine ~= nil and onCreateObject.liquidManureSiloTrigger ~= nil then
			-- print(self:tableShow(onCreateObject, 'onCreateObject [pigs]'));
			local triggerId = onCreateObject.liquidManureSiloTrigger.triggerId;
			if triggerId and not self.tanksIdentifiedById[triggerId] then
				liquidManureIdx = liquidManureIdx + 1;
				local tankTable = {
					tankIdx = liquidManureIdx;
					tankNum = liquidManureIdx;
					onCreateIndex = i;
					isPigs = true;
					fillLevel = onCreateObject.liquidManureSiloTrigger.fillLevel;
					fillLevelFormatted = '0';
					fillLevelPct = 0;
					fillLevelPctFormatted = '0';
					capacity = onCreateObject.liquidManureSiloTrigger.capacity;
					capacityFormatted = self:formatNumber(onCreateObject.liquidManureSiloTrigger.capacity, 0);
					name = ('%s %i (%s)'):format(g_i18n:getText('BUNKERSILOS_TANK'), liquidManureIdx, g_i18n:getText('BUNKERSILOS_TANKTYPE_PIGS'));
				};

				local name = getUserAttribute(triggerId, 'bshName');
				if name ~= nil and self.mapI18n then
					tankTable.name = self.mapI18n:getText('BSH_' .. name);
					onCreateObject.bshName = tankTable.name;
				end;

				table.insert(self.tanks, tankTable);
				self.tanksIdentifiedById[triggerId] = true;
				-- print(string.format('# BSH: add LiquidManureTank %s (triggerId %s, name %q) [Pigs] / total: %s', tostring(liquidManureIdx), tostring(triggerId), tostring(tankTable.name), tostring(#self.tanks)));
			end;
		end;
	end;


	self.numSilos = #self.silos;
	if self.numSilos > 0 then
		-- SORT AND REINDEX SILOS
		table.sort(self.silos, function(a,b) return a.name:lower() < b.name:lower() end);
		for k,v in ipairs(self.silos) do
			v.bunkerSiloNum = k;
		end;

		self.activeSilo = 1;

		if self.numSilos > 1 then
			self.gui.changeSiloNegButton = self:registerButton('arrow.png', 'changeSilo', -1, self.gui.contentMinX, self.gui.lines[1], self.gui.arrowGfxWidth, self.gui.arrowGfxHeight, self.gui.arrowContentWidth, self.gui.arrowContentHeight);
			self.gui.changeSiloPosButton = self:registerButton('arrow.png', 'changeSilo', 1, self.gui.contentMaxX - self.gui.arrowContentWidth, self.gui.lines[1], self.gui.arrowGfxWidth, self.gui.arrowGfxHeight, self.gui.arrowContentWidth, self.gui.arrowContentHeight, true);
		end;
	else
		self.activeSilo = false;
	end;


	self.numTanks = #self.tanks;
	if self.numTanks > 0 then
		-- SORT AND REINDEX LIQUID MANURE TANKS
		table.sort(self.tanks, function(a,b) return a.name:lower() < b.name:lower() end);
		for k,v in ipairs(self.tanks) do
			v.tankNum = k;
		end;

		self.activeTank = 1;

		local imgPath = 'dataS2/menu/white.png';
		self.gui.tankBarBorderLeft = Overlay:new('tankBarBorderLeft', imgPath, self.gui.contentMinX, self.gui.lines[8], self.gui.tankBarBorderWidth, self.gui.tankBarHeight);
		self.gui.tankBarBorderRight = Overlay:new('tankBarBorderRight', imgPath, self.gui.contentMaxX - self.gui.tankBarBorderWidth, self.gui.lines[8], self.gui.tankBarBorderWidth, self.gui.tankBarHeight);
		self.gui.tankBar = Overlay:new('tankBar', imgPath, self.gui.tankBarMinX, self.gui.lines[8], 0, self.gui.tankBarHeight);
		self:setOverlayColor(self.gui.tankBarBorderLeft, 'default');
		self:setOverlayColor(self.gui.tankBarBorderRight, 'default');
		self:setOverlayColor(self.gui.tankBar, 'default');

		if self.numTanks > 1 then
			self.gui.changeTankNegButton = self:registerButton('arrow.png', 'changeTank', -1, self.gui.contentMinX, self.gui.lines[6], self.gui.arrowGfxWidth, self.gui.arrowGfxHeight, self.gui.arrowContentWidth, self.gui.arrowContentHeight);
			self.gui.changeTankPosButton = self:registerButton('arrow.png', 'changeTank', 1, self.gui.contentMaxX - self.gui.arrowContentWidth, self.gui.lines[6], self.gui.arrowGfxWidth, self.gui.arrowGfxHeight, self.gui.arrowContentWidth, self.gui.arrowContentHeight, true);
		end;
	else
		self.activeTank = false;
	end;

	-- stop vehicle/player camera movement if mouse active
	if not VehicleCamera.bshMouseInserted then
		VehicleCamera.mouseEvent = Utils.overwrittenFunction(VehicleCamera.mouseEvent, bsh.cancelMouseEvent);
		VehicleCamera.zoomSmoothly = Utils.overwrittenFunction(VehicleCamera.zoomSmoothly, bsh.cancelZoom);
		VehicleCamera.bshMouseInserted = true;
	end;
	if not Player.bshMouseInserted then
		Player.mouseEvent = Utils.overwrittenFunction(Player.mouseEvent, bsh.cancelMouseEvent);
		Player.bshMouseInserted = true;
	end;

	self.initialized = true;
	print(('## BunkerSilosHud v%s by %s loaded'):format(bsh.version, bsh.author));
end;

function bsh:cancelMouseEvent(superFunc, posX, posY, isDown, isUp, button)
	if bsh.hasMouseCursorActive then
		local x, y = InputBinding.mouseMovementX, InputBinding.mouseMovementY;
		InputBinding.mouseMovementX, InputBinding.mouseMovementY = 0, 0;
		superFunc(self, posX, posY, isDown, isUp, button);
		InputBinding.mouseMovementX, InputBinding.mouseMovementY = x, y;
	else
		superFunc(self, posX, posY, isDown, isUp, button);
	end;
end;
function bsh:cancelZoom(superFunc, offset)
	if bsh.hasMouseCursorActive and bsh.canScroll and (math.abs(offset) == 0.6 or math.abs(offset) == 0.75) then -- NOTE: make sure user zoomed with the mouse wheel: 0.6 (default), 0.75 (InteractiveControl)
		return;
	end;
	superFunc(self, offset);
end;

function bsh:getFullPixelX(n)
	return tonumber(('%d'):format(n * g_screenWidth)) / g_screenWidth;
end;

function bsh:getFullPixelY(n)
	return tonumber(('%d'):format(n * g_screenHeight)) / g_screenHeight;
end;

function bsh:registerButton(fileName, functionToCall, parameter, posX, posY, gfxWidth, gfxHeight, clickWidth, clickHeight, invertX)
	local filePath = Utils.getFilename(fileName, bsh.imgDir);
	local overlay = Overlay:new(functionToCall, filePath, posX, posY, gfxWidth, gfxHeight);
	overlay.a = 0.7; --set alpha
	if invertX then
		overlay:setInvertX(true);
	end;

	local button = {
		overlay = overlay,
		functionToCall = functionToCall,
		parameter = parameter,
		x = posX,
		x2 = (posX + clickWidth),
		y = posY,
		y2 = (posY + clickHeight)
	};

	table.insert(self.gui.buttons, button);

	return self.gui.buttons[#self.gui.buttons];
end;

function bsh:renderButtons()
	if bsh.hasMouseCursorActive then
		for _,button in ipairs(self.gui.buttons) do
			if button.isClicked and button.overlay.curColor ~= 'clicked' then
				self:setOverlayColor(button.overlay, 'clicked');
			elseif button.isHovered and button.overlay.curColor ~= 'defaultHover' then
				self:setOverlayColor(button.overlay, 'defaultHover');
			elseif not button.isClicked and not button.isHovered and button.overlay.curColor ~= 'default' then
				self:setOverlayColor(button.overlay, 'default');
			end;
			button.overlay:render();
		end;
	end;
end;

function bsh:deleteMap()
	self:deleteOverlay(self.gui.background);
	self:deleteOverlay(self.gui.effects);
	self:deleteOverlay(self.gui.mouseWheel);
	self:deleteOverlay(self.gui.warning);
	self:deleteOverlay(self.gui.tankBarBorderLeft);
	self:deleteOverlay(self.gui.tankBarBorderRight);
	self:deleteOverlay(self.gui.tankBar);

	for _,button in pairs(self.gui.buttons) do
		self:deleteOverlay(button.overlay);
	end;
	self.gui.buttons = {};

	for _,silo in ipairs(self.silos) do
		for _,mp in pairs(silo.movingPlanes) do
			if mp.bshOverlayNonCompacted ~= nil then
				self:deleteOverlay(mp.bshOverlayNonCompacted);
			end;
			if mp.bshOverlay ~= nil then
				self:deleteOverlay(mp.bshOverlay);
			end;
		end;
	end;

	self.initialized = false;
end;


function bsh:keyEvent(unicode, sym, modifier, isDown)
end;

function bsh:setHudState(state, dontHideMouseCursor)
	self.gui.hudState = state;
	bsh.hasMouseCursorActive = state == bsh.HUDSTATE_INTERACTIVE;

	-- open
	if state == bsh.HUDSTATE_INTERACTIVE then
		InputBinding.setShowMouseCursor(true);
		self.helpButtonText = g_i18n:getText('BUNKERSILOS_HIDEMOUSE');
		-- self.pdaLastState = g_currentMission.missionPDA.showPDA;
		-- g_currentMission.missionPDA.showPDA = false;

	-- hide mouse
	elseif state == bsh.HUDSTATE_ACTIVE then
		bsh.canScroll = false;
		InputBinding.setShowMouseCursor(false);
		self.helpButtonText = g_i18n:getText('BUNKERSILOS_HIDEHUD');

	-- close
	elseif state == bsh.HUDSTATE_CLOSED then
		for _,button in pairs(self.gui.buttons) do
			button.isClicked = false;
			button.isHovered = false;
		end;

		if not dontHideMouseCursor then
			InputBinding.setShowMouseCursor(false);
		end;
		bsh.canScroll = false;
		self.gui.warning.curBlinkTime = nil;
		self.helpButtonText = g_i18n:getText('BUNKERSILOS_SHOWHUD');
		-- g_currentMission.missionPDA.showPDA = self.pdaLastState;
	end;
end;

function bsh:changeSilo(move)
	self.activeSilo = self:loopNumberSequence(self.activeSilo + move, 1, self.numSilos);
end;

function bsh:changeTank(move)
	self.activeTank = self:loopNumberSequence(self.activeTank + move, 1, self.numTanks);
end;

function bsh:loopNumberSequence(n, minN, maxN)
	if n > maxN then
		return minN;
	elseif n < minN then
		return maxN;
	end;
	return n;
end;

function bsh:update(dt)
	if not bsh.hasAppendedSaveFn then
		self:appendSaveFn();
	end;

	if InputBinding.hasEvent(InputBinding.BUNKERSILOS_HUD) then
		self:setHudState(self:loopNumberSequence(self.gui.hudState + 1, bsh.HUDSTATE_INTERACTIVE, bsh.HUDSTATE_CLOSED));
	end;

	-- Deactivate HUD when PDA is activated
	-- if InputBinding.hasEvent(InputBinding.TOGGLE_PDA) and self.gui.hudState ~= bsh.HUDSTATE_CLOSED then
		-- self:setHudState(bsh.HUDSTATE_CLOSED, true);
	-- end;

	if self.gui.hudState ~= bsh.HUDSTATE_CLOSED then
		if (g_gui.currentGuiName ~= nil and g_gui.currentGuiName ~= '') then -- If player activates some GUI screen.
			-- Do not disable mousecursor here, due to the player might enter the real Farming Shop (Key P), and expects the mousecursor to be visible.
			self:setHudState(bsh.HUDSTATE_CLOSED, true);
        end;
	end;

	if g_currentMission.showHelpText and (self.input.hud.modifier == nil or Input.isKeyPressed(self.input.hud.modifier)) then
		g_currentMission:addHelpButtonText(self.helpButtonText, InputBinding.BUNKERSILOS_HUD);
	end;
end;


function bsh:updateTick(dt)
end;


function bsh:draw()
	if self.gui.hudState == bsh.HUDSTATE_CLOSED then return; end;
	local g = self.gui;

	g.background:render();
	if bsh.canScroll then
		self.gui.mouseWheel:render();
	end;
	setTextBold(false);
	setTextColor(unpack(g.colors.text));

	-- Silos
	if self.numSilos > 0 and self.activeSilo then
		self.displayWarning = false;

		local data = self.silos[self.activeSilo];

		-- UPDATE SILO DATA
		local t = g_currentMission.tipTriggers[ self.tempSiloTriggers[data.bunkerSiloIdx] ];

		data.state = tonumber(t.bunkerSilo.state);
		data.stateText = self.bunkerStates[data.state];
		if t.bunkerSilo.fillLevel ~= data.fillLevel then
			data.fillLevel = t.bunkerSilo.fillLevel;
			data.fillLevelFormatted = self:formatNumber(data.fillLevel, 0);
			data.fillLevelPct = data.fillLevel / (data.capacity + 0.00001);
			data.fillLevelPctFormatted = data.fillLevel == 0 and '0' or self:formatNumber(data.fillLevelPct * 100, 1);
			data.toFillFormatted = self:formatNumber(data.capacity - data.fillLevel, 0);
			-- print(string.format('updated fillLevel: fillLevelFormatted=%q, fillLevelPctFormatted=%s, toFillFormatted=%q', data.fillLevelFormatted, data.fillLevelPctFormatted, data.toFillFormatted));
		end;
		if t.bunkerSilo.compactedFillLevel ~= data.compactedFillLevel then
			data.compactedFillLevel = t.bunkerSilo.compactedFillLevel;
			data.compactPct = data.compactedFillLevel / (data.fillLevel + 0.00001);
			data.compactPctFormatted = data.fillLevel == 0 and '0' or self:formatNumber(data.compactPct * 100, 1);
			-- print(string.format('updated compactedFillLevel: compactPctFormatted=%q', data.compactPctFormatted));
		end;
		if t.bunkerSilo.state == 1 then
			local fermentingTime = t.bunkerSilo.fermentingTime;
			if fermentingTime ~= data.fermentingTime then
				-- print(string.format('Silo %s: update fermentingTime: old=%s, new=%s', data.name, tostring(data.fermentingTime), tostring(fermentingTime)));
				data.fermentingTime = fermentingTime;
				local fermentationPct = self:round(math.min(data.fermentingTime / (data.fermentingDuration + 0.00001), 1), 3);
				if fermentationPct ~= data.fermentationPct then
					-- print(string.format('Silo %s: update fermentationPct: old=%s, new=%s', data.name, tostring(data.fermentationPct), tostring(fermentationPct)));
					data.fermentationPct = fermentationPct;
					data.fermentationPctFormatted = data.fermentingTime == 0 and '0' or self:formatNumber(data.fermentationPct * 100, 1);
				end;
			end;
			self.displayWarning = data.fermentationPct >= 1;
		end;

		-- warning icon
		if self.displayWarning then
			self:setOverlayBlink(self.gui.warning);
			self.gui.warning:render();
		else
			self.gui.warning.curBlinkTime = nil;
		end;

		-- movingPlane boxes
		local rottenFillLevel = 0;
		if data.fillLevel > 0 then
			for i,mp in pairs(data.movingPlanes) do
				local origMp = t.bunkerSilo.movingPlanes[i];
				mp.compactFillLevel = origMp.compactFillLevel;
				mp.fillLevel = origMp.fillLevel;
				mp.isRotten = origMp.isRotten;
				if mp.fillLevel > 0 then
					if mp.isRotten then
						rottenFillLevel = rottenFillLevel + mp.fillLevel;

						if mp.bshOverlayNonCompacted.curColor ~= 'rottenNonCompacted' then
							self:setOverlayColor(mp.bshOverlayNonCompacted, 'rottenNonCompacted');
						end;
						if mp.bshOverlay.curColor ~= 'rotten' then
							self:setOverlayColor(mp.bshOverlay, 'rotten');
						end;
					else
						if mp.bshOverlayNonCompacted.curColor ~= 'defaultNonCompacted' then
							self:setOverlayColor(mp.bshOverlayNonCompactedCompacted, 'defaultNonCompacted');
						end;
						if mp.bshOverlay.curColor ~= 'default' then
							self:setOverlayColor(mp.bshOverlay, 'default');
						end;
					end;

					local nonCompactedHeight = (mp.fillLevel / mp.capacity) * g.boxMaxHeight;
					if mp.compactFillLevel then
						if data.state == 0 and mp.fillLevel ~= mp.compactFillLevel then
							mp.bshOverlayNonCompacted.height = nonCompactedHeight;
							mp.bshOverlayNonCompacted:render();

							mp.bshOverlay.height = (mp.compactFillLevel / mp.capacity) * g.boxMaxHeight;
						else
							mp.bshOverlay.height = nonCompactedHeight;
						end;
						mp.bshOverlay:render();

					else
						if data.state == 0 and data.compactPct < 1 then
							mp.bshOverlayNonCompacted.height = nonCompactedHeight;
							mp.bshOverlayNonCompacted:render();

							mp.bshOverlay.height = nonCompactedHeight * data.compactPct;
						else
							mp.bshOverlay.height = nonCompactedHeight;
						end;
						mp.bshOverlay:render();
					end;
				end; -- END if mp.fillLevel > 0
			end; -- END for mp in movingPlanes
		end; -- END if data.fillLevel > 0
		if rottenFillLevel ~= data.rottenFillLevel then
			data.rottenFillLevel = rottenFillLevel;
			data.rottenFillLevelPctFormatted = self:formatNumber((data.rottenFillLevel / (data.fillLevel + 0.00001)) * 100, 1);
			-- print(string.format('updated rottenFillLevel: rottenFillLevelPctFormatted=%q', data.rottenFillLevelPctFormatted));
		end;


		-- RENDER SILO DATA
		-- Line 1 ('Silo')
		setTextAlignment(RenderText.ALIGN_CENTER);
		renderText(g.baseX + (g.width/2), g.lines[1], g.fontSizeTitle, data.name);

		setTextAlignment(RenderText.ALIGN_LEFT);
		-- Line 2
		if data.state < 2 then
			renderText(g.textMinX, g.lines[2], g.fontSize, ('%s: %s (%s%% %s)'):format(g_i18n:getText('BUNKERSILOS_STATE'), data.stateText, data.compactPctFormatted, g_i18n:getText('BUNKERSILOS_COMPACT')));
		else
			if data.rottenFillLevel > 0 then
				renderText(g.textMinX, g.lines[2], g.fontSize, ('%s: %s (%s%% %s)'):format(g_i18n:getText('BUNKERSILOS_STATE'), data.stateText, data.rottenFillLevelPctFormatted, g_i18n:getText('BUNKERSILOS_ROTTEN')));
			else
				renderText(g.textMinX, g.lines[2], g.fontSize, ('%s: %s'):format(g_i18n:getText('BUNKERSILOS_STATE'), data.stateText));
			end;
		end;

		-- Line 3 -- game's 'fill_level' i18n is too long -> use own
		renderText(g.textMinX, g.lines[3], g.fontSize, ('%s: %s/%s (%s%%)'):format(g_i18n:getText('BUNKERSILOS_FILLLEVEL'), data.fillLevelFormatted, data.capacityFormatted, data.fillLevelPctFormatted));

		-- Line 4
		if data.state == 0 then
			renderText(g.textMinX, g.lines[4], g.fontSize, ('%s: %s'):format(g_i18n:getText('BUNKERSILOS_TOBEFILLED'), data.toFillFormatted));
		elseif data.state == 1 then
			renderText(g.textMinX, g.lines[4], g.fontSize, ('%s: %s%%'):format(g_i18n:getText('BUNKERSILOS_FERMENTATION_PROGRESS'), data.fermentationPctFormatted));
		end;

	-- no silos
	else
		setTextAlignment(RenderText.ALIGN_CENTER);
		renderText(g.baseX + (g.width/2), g.lines[2], g.fontSizeTitle, g_i18n:getText('BUNKERSILOS_NOSILOS'));
	end;


	--###########################################################################


	-- Liquid manure tanks
	if self.numTanks > 0 and self.activeTank then
		local data = self.tanks[self.activeTank];

		-- UPDATE TANK DATA
		local fillLevel = 0;
		if data.isBGA then
			fillLevel = g_currentMission.tipTriggers[ self.tempTankTriggers[data.tankIdx] ].bga.liquidManureSiloTrigger.fillLevel;
		elseif data.isFarm then
			fillLevel = g_currentMission.tipTriggers[ self.tempTankTriggers[data.tankIdx] ].animalHusbandry.liquidManureTrigger.fillLevel;
		elseif data.isManureLager then
			fillLevel = g_currentMission.onCreateLoadedObjects[data.onCreateIndex].fillLevel;
		elseif data.isPigs then
			fillLevel = g_currentMission.onCreateLoadedObjects[data.onCreateIndex].liquidManureSiloTrigger.fillLevel;
		end;

		if fillLevel ~= data.fillLevel then
			data.fillLevel = fillLevel;
			data.fillLevelFormatted = self:formatNumber(data.fillLevel, 0);
			data.fillLevelPct = data.fillLevel / data.capacity + 0.00001;
			data.fillLevelPctFormatted = self:formatNumber(data.fillLevelPct * 100, 1);
			-- print(string.format('BSH tank %q: updated fillLevel: fillLevelFormatted=%q, fillLevelPctFormatted=%q', data.name, data.fillLevelFormatted, data.fillLevelPctFormatted));
		end;

		-- RENDER TANK DATA
		-- Line 6 (title)
		setTextAlignment(RenderText.ALIGN_CENTER);
		renderText(g.baseX + (g.width/2), g.lines[6], g.fontSizeTitle, data.name);

		-- Line 7 (fill level) -- game's 'fill_level' i18n is too long -> use own
		setTextAlignment(RenderText.ALIGN_LEFT);
		renderText(g.textMinX, g.lines[7], g.fontSize, ('%s: %s/%s (%s%%)'):format(g_i18n:getText('BUNKERSILOS_FILLLEVEL'), data.fillLevelFormatted, data.capacityFormatted, data.fillLevelPctFormatted));

		-- Line 8 (Bar)
		g.tankBarBorderLeft:render();
		g.tankBarBorderRight:render();
		if data.fillLevel > 0 then
			g.tankBar.width = g.tankBarMaxWidth * data.fillLevelPct;
			g.tankBar:render();
		end;

	-- no tanks
	else
		setTextAlignment(RenderText.ALIGN_CENTER);
		renderText(g.baseX + (g.width/2), g.lines[2], g.fontSizeTitle, g_i18n:getText('BUNKERSILOS_NOTANKS'));
	end;


	self:renderButtons();
	g.effects:render(); -- see TODO #1


	-- Reset text settings for other mods
	setTextAlignment(RenderText.ALIGN_LEFT);
	setTextColor(1, 1, 1, 1);
	setTextBold(false);
end;


function bsh:mouseEvent(posX, posY, isDown, isUp, button)
	bsh.canScroll = false;
	if self.gui.hudState ~= bsh.HUDSTATE_INTERACTIVE or not self:mouseIsInArea(posX, posY, self.gui.baseX, self.gui.baseX + self.gui.gfxWidth, self.gui.baseY, self.gui.baseY + self.gui.gfxHeight) then
		return;
	end;

	-- CLICKING (up)
	if isUp and button == Input.MOUSE_BUTTON_LEFT then
		for _,button in pairs(self.gui.buttons) do
			button.isClicked = false;
			button.isHovered = false;
			if self:mouseIsOnButton(posX, posY, button) then
				button.isHovered = true;
				bsh[button.functionToCall](self, button.parameter);
				break; --no need to check the rest of the buttons
			end;
		end;

	-- CLICKING (down)
	elseif isDown and button == Input.MOUSE_BUTTON_LEFT then
		for _,button in pairs(self.gui.buttons) do
			button.isClicked = self:mouseIsOnButton(posX, posY, button);
			button.isHovered = false;
		end;

	-- HOVERING / SCROLLING
	elseif not isDown then -- NOTE: could give problems with Input.isMouseButtonPressed --- but doesn't seem to during tests
		for _,button in pairs(self.gui.buttons) do
			button.isClicked = false;
			button.isHovered = self:mouseIsOnButton(posX, posY, button);
		end;

		-- mouse is in silo area
		if self.numSilos > 1 and self:mouseIsInArea(posX, posY, unpack(self.gui.silosArea)) then
			bsh.canScroll = true;
			if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_UP) then
				self:changeSilo(1);
			elseif Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_DOWN) then
				self:changeSilo(-1);
			end;

		-- mouse is in tank area
		elseif self.numTanks > 1 and self:mouseIsInArea(posX, posY, unpack(self.gui.tankArea)) then
			bsh.canScroll = true;
			if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_UP) then
				self:changeTank(1);
			elseif Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_DOWN) then
				self:changeTank(-1);
			end;
		end;
	end;
end;

function bsh:mouseIsInArea(mouseX, mouseY, areaX1, areaX2, areaY1, areaY2)
	return mouseX >= areaX1 and mouseX <= areaX2 and mouseY >= areaY1 and mouseY <= areaY2;
end;

function bsh:mouseIsOnButton(mouseX, mouseY, button)
	return self:mouseIsInArea(mouseX, mouseY, button.x, button.x2, button.y, button.y2);
end;

function bsh:rgba(r, g, b, a)
	return { r/255, g/255, b/255, a };
end;

function bsh:round(n, precision)
	return math.floor((n * math.pow(10, precision)) + 0.5) / math.pow(10, precision);
end;

--[[
function bsh:formatNumber(number, precision)
	precision = math.min(precision, 2) or 0;

	local str = ('%.' .. precision .. 'f'):format(number):gsub('%.', self.numberDecimalSeparator);
	str = str:reverse():gsub('(%d%d%d)', '%1' .. self.numberSeparator):reverse();
	return str;
end;
]]

function bsh:formatNumber(number, precision)
	precision = precision or 0;

	local str = '';
	-- local firstDigit, rest, decimal = string.match(('%1.' .. precision .. 'f'):format(number), '^([^%d]*%d)(%d*).?(%d*)');
	local firstDigit, rest, decimal = ('%1.' .. precision .. 'f'):format(number):match('^([^%d]*%d)(%d*).?(%d*)');
	str = firstDigit .. rest:reverse():gsub('(%d%d%d)', '%1' .. self.numberSeparator):reverse();
	if decimal:len() > 0 then
		str = str .. self.numberDecimalSeparator .. decimal:sub(1, precision);
	end;
	return str;
end;


function bsh:getKeyIdOfAction(binding)
	if InputBinding.actions[binding] == nil then
		return nil;  -- Unknown input-binding.
	end;

	local n = #(InputBinding.actions[binding].keys1);
	if n == 0 then
		return nil; -- Input-binding has no keys
	elseif n == 1 then
		return InputBinding.actions[binding].keys1[1]; -- Input-binding has only one key -> return key
	elseif n == 2 and not Input.keyIdIsModifier[ InputBinding.actions[binding].keys1[2] ] then
		return InputBinding.actions[binding].keys1[2]; -- Input-binding has two keys -> return 2nd key
	end;

	return nil;
end;

--@src: Decker, compas.lua
function bsh:getKeyIdOfModifier(binding)
	if InputBinding.actions[binding] == nil then
		return nil;  -- Unknown input-binding.
	end;
	if #(InputBinding.actions[binding].keys1) <= 1 then
		return nil; -- Input-binding has only one or zero keys. (Well, in the keys1 - I'm not checking keys2)
	end;
	-- Check if first key in key-sequence is a modifier key (LSHIFT/RSHIFT/LCTRL/RCTRL/LALT/RALT)
	if Input.keyIdIsModifier[ InputBinding.actions[binding].keys1[1] ] then
		return InputBinding.actions[binding].keys1[1]; -- Return the keyId of the modifier key
	end;
	return nil;
end

function bsh:setOverlayColor(overlay, colorName)
	if overlay == nil or self.gui.colors[colorName] == nil or overlay.curColor == colorName then return; end;

	-- print(string.format('setOverlayColor(%d, %s) [previous curColor=%s]', overlay.overlayId, colorName, tostring(overlay.curColor)));
	overlay:setColor(unpack(self.gui.colors[colorName]));
	overlay.curColor = colorName;
end;

function bsh:setOverlayBlink(overlay, noFade)
	-- HARD BLINK
	if noFade then
		if overlay.a ~= 1 then
			overlay:setColor(overlay.r, overlay.g, overlay.b, 1);
		end;
		if overlay.curBlinkTime == nil then
			overlay.curBlinkTime = g_currentMission.time + self.blinkLength;
			overlay:setIsVisible(true);
		else
			if g_currentMission.time >= overlay.curBlinkTime then
				overlay.curBlinkTime = overlay.curBlinkTime + self.blinkLength;
				overlay:setIsVisible(not overlay.visible);
			end;
		end;
		return;
	end;

	-- FADE IN/OUT
	local alpha = 1;
	if overlay.curBlinkTime == nil then
		overlay.curBlinkTime = g_currentMission.time + self.blinkLength;
		overlay.fadeDir = 1;
		alpha = 0;
	else
		if g_currentMission.time < overlay.curBlinkTime then
			local alphaRatio = 1 - (overlay.curBlinkTime - g_currentMission.time) / self.blinkLength;
			if overlay.fadeDir == -1 then
				alpha = 1 - alphaRatio;
			else
				alpha = alphaRatio;
			end;
		else
			overlay.curBlinkTime = overlay.curBlinkTime + self.blinkLength;
			alpha = overlay.fadeDir == 1 and 1 or 0;
			overlay.fadeDir = -overlay.fadeDir;
		end;
	end;
	overlay:setColor(overlay.r, overlay.g, overlay.b, alpha);
end;

function bsh:deleteOverlay(overlay)
	if overlay and overlay.overlayId then
		delete(overlay);
		overlay = nil;
	end;
end;

addModEventListener(bsh);



-- ADD bshName TO SAVE ATTRIBUTES
function bsh:appendSaveFn()
	local oldBunkerSilogetSaveAttributesAndNodes = BunkerSilo.getSaveAttributesAndNodes;
	BunkerSilo.getSaveAttributesAndNodes = function(self, nodeIdent)
		local attributes, nodes = oldBunkerSilogetSaveAttributesAndNodes(self, nodeIdent);
		if self.bshName ~= nil then
			attributes = ('%s bshName=%q'):format(attributes, self.bshName);
			-- print(string.format('BunkerSilo save(): adding bshName %q to attributes', self.bshName));
			-- print(string.format('\t%s', attributes));
		end;
		return attributes, nodes;
	end;

	local oldBGAgetSaveAttributesAndNodes = Bga.getSaveAttributesAndNodes;
	Bga.getSaveAttributesAndNodes = function(self, nodeIdent)
		local attributes, nodes = oldBGAgetSaveAttributesAndNodes(self, nodeIdent);
		if self.liquidManureSiloTrigger ~= nil and self.liquidManureSiloTrigger.bshName ~= nil then
			attributes = ('%s bshName=%q'):format(attributes, self.liquidManureSiloTrigger.bshName);
			-- print(string.format('Bga save(): adding bshName %q to attributes', self.liquidManureSiloTrigger.bshName));
			-- print(string.format('\t%s', attributes));
		end;
		return attributes, nodes;
	end;

	bsh.hasAppendedSaveFn = true;
end;
