-- PalletSES class
-- Adds the possibility to define a sowing machine trigger and a dynamic mount trigger similar to the pallet and bale objects
-- Requires the Mountable specialization to be present (for dynamic mounting functionality)
--
-- @author  Stefan Geiger
-- @date  25/02/13
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.

PalletSES = {};

function PalletSES.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Mountable, specializations);
end;

function PalletSES:load(xmlFile)
    if self.isServer then
        self.dynamicMountTriggerId = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.dynamicMountTrigger#index"));
        if self.dynamicMountTriggerId ~= nil then
            addTrigger(self.dynamicMountTriggerId, "dynamicMountTriggerCallback", self);
        end

        self.dynamicMountTriggerCallback = PalletSES.dynamicMountTriggerCallback;
    end

    local sowingMachineFillTriggerId = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.sowingMachineFillTrigger#index"));
    if sowingMachineFillTriggerId ~= nil then
        self.sowingMachineFillTrigger = PalletSESFillTrigger:new(sowingMachineFillTriggerId);
    end
end;

function PalletSES:delete()
    if self.dynamicMountTriggerId ~= nil then
        removeTrigger(self.dynamicMountTriggerId);
    end
    if self.sowingMachineFillTrigger ~= nil then
        self.sowingMachineFillTrigger:delete();
    end
end;

function PalletSES:mouseEvent(posX, posY, isDown, isUp, button)
end;

function PalletSES:keyEvent(unicode, sym, modifier, isDown)
end;

function PalletSES:update(dt)
end;

function PalletSES:updateTick(dt)
end;

function PalletSES:draw()
end;

function PalletSES:dynamicMountTriggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
    if onEnter then
        local vehicle = g_currentMission.nodeToVehicle[otherActorId];
        if vehicle ~= nil and vehicle.dynamicMountAttacherNode ~= nil then
            if self.dynamicMountObjectActorId == nil then
                self:mountDynamic(vehicle, otherActorId, vehicle.dynamicMountAttacherNode, true, vehicle.dynamicMountAttacherForceLimitScale);
                self.dynamicMountObjectTriggerCount = 1;
            elseif otherActorId ~= self.dynamicMountObjectActorId and self.dynamicMountObjectTriggerCount == nil then
                -- we are already attached to another actor, but not from our mount trigger (e.g. a bale trailer)
                self:unmountDynamic();
                self:mountDynamic(vehicle, otherActorId, vehicle.dynamicMountAttacherNode, true, vehicle.dynamicMountAttacherForceLimitScale);
                self.dynamicMountObjectTriggerCount = 1;
            elseif otherActorId == self.dynamicMountObjectActorId then
                if self.dynamicMountObjectTriggerCount ~= nil then
                    self.dynamicMountObjectTriggerCount = self.dynamicMountObjectTriggerCount+1;
                end
            end
        end
    elseif onLeave then
        if otherActorId == self.dynamicMountObjectActorId and self.dynamicMountObjectTriggerCount ~= nil then
            self.dynamicMountObjectTriggerCount = self.dynamicMountObjectTriggerCount-1;
            if self.dynamicMountObjectTriggerCount == 0 then
                self:unmountDynamic();
                self.dynamicMountObjectTriggerCount = nil;
            end
        end
    end
end;

-- PalletSESFillTrigger class
-- Replacement for the original SowingMachineFillTrigger class which has a support lower price and supports dynamic deleting

PalletSESFillTrigger = {}

local PalletSESFillTrigger_mt = Class(PalletSESFillTrigger);

function PalletSESFillTrigger:new(nodeId, mt)
    if mt == nil then
        mt = PalletSESFillTrigger_mt;
    end
    local self = {};
    setmetatable(self, mt);

    self.triggerId = nodeId;
    addTrigger(nodeId, "triggerCallback", self);
	
	self.neededFruitType = getUserAttribute(nodeId, "neededFruitType"); 
    self.triggeredSowingMachines = {};

    return self;
end;

function PalletSESFillTrigger:delete()
    removeTrigger(self.triggerId);

    for sowingMachine,_ in pairs(self.triggeredSowingMachines) do
        if not sowingMachine.isDeleted then
            sowingMachine:removeSowingMachineFillTrigger(self);
        end
    end
end;

function PalletSESFillTrigger:triggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
    if onEnter or onLeave then
        local sowingMachine = g_currentMission.objectToTrailer[otherShapeId];

        if sowingMachine ~= nil and sowingMachine.addSowingMachineFillTrigger ~= nil and sowingMachine.removeSowingMachineFillTrigger ~= nil then
            if onEnter then
                if self.triggeredSowingMachines[sowingMachine] == nil then
					-- make sure the sowing machine supports the needed fruit type
					local fruitTypeDesc = FruitUtil.fruitTypes[self.neededFruitType];
					if sowingMachine.seeds ~= nil then
						for a=1, table.getn(sowingMachine.seeds) do
							if fruitTypeDesc.index == sowingMachine.seeds[a] then
								self.triggeredSowingMachines[sowingMachine] = sowingMachine;
								sowingMachine:addSowingMachineFillTrigger(self);
								break;
							end;
						end;
					end;
                end
            else -- onLeave
				if self.triggeredSowingMachines[sowingMachine] ~= nil then
					sowingMachine:removeSowingMachineFillTrigger(self);
					self.triggeredSowingMachines[sowingMachine] = nil;
				end;
            end;
        end;
    end;
end;

function PalletSESFillTrigger:fillSowingMachine(sowingMachine, delta)

    local fillType = Fillable.FILLTYPE_SEEDS;
    local priceScale = 0.9;
    if not sowingMachine:allowFillType(fillType, false) then
        return 0.0;
    end
    local oldFillLevel = sowingMachine:getFillLevel(fillType);
    sowingMachine:setFillLevel(oldFillLevel + delta, fillType, true);
    delta = sowingMachine:getFillLevel(fillType) - oldFillLevel;

    local fillTypeDesc = Fillable.fillTypeIndexToDesc[fillType]
    if fillTypeDesc ~= nil then

        local price = delta*fillTypeDesc.pricePerLiter*priceScale;
        g_currentMission.missionStats.expensesTotal = g_currentMission.missionStats.expensesTotal + price;
        g_currentMission.missionStats.expensesSession = g_currentMission.missionStats.expensesSession + price;

        g_currentMission:addSharedMoney(-price, "other");
    end

    return delta;
end

function PalletSESFillTrigger:getIsActivatable(fillable)
    if not fillable:allowFillType(Fillable.FILLTYPE_SEEDS, false) then
        return false;
    end;
    return true;
end;