--
-- PdaMod
--
-- @author  Decker_MMIV - fs-uk.com, forum.farming-simulator.com, modhoster.com
-- @date    2013-03-18
--
-- @history
--  2013-March
--      v0.01   - Initial experiment
--      v0.02   - Now with a 'PDA Config' screen, to make screens visible/hidden and in what order they should be in.
--              - Also made load/save of the ordering of the screens, in a PdaMod.XML file.
--  2013-April
--      v0.03   - Renamed to 'PdaMod'
--      v0.04   - Added global, so other mods can call modPDAMOD.registerPage() and modPDAMOD.drawTitle()
--      v0.05   - Misc. variable renamed.
--      v0.06   - Fix for mod-image spelling.
--  2013-June
--      v0.07   - Added modPDAMOD.drawPageOfPages()
--  2014-January
--      v0.08   - Changes to make PdaMod compatible with the PDAfix by Marhu.
--  2014-May
--      v0.10   - Added modPDAMOD.registerPlugin(...) so plugins can call that, instead of addModEventListener().
--  2014-June
--      v0.11   - Due to the NON-deterministic way that mods are loaded in multiplayer, the modPDAMOD.registerPlugin(...)
--                is removed in favor of a hopefully more robust "create object modPDAMODplugins, and add yourself to it" method.
--      v0.12   - Upper-case sort.
--      v0.13   - Added modPDAMOD.getFillTypePrice() in attempt to centralize price-per-quantity calculation, due to animals quantity-vs-price.
--      v0.14   - Minor tweaks
--      2.0.1   - Using http://semver.org/
--
--
-- Plugin API:
--      Please study the included plugin scripts, for how to use this.
--


PdaMod = {}
--
local modItem = ModsUtil.findModItemByModName(g_currentModName);
PdaMod.version = (modItem and modItem.version) and modItem.version or "?.?.?";
--
PdaMod.initStep = 0;
PdaMod.plugins = {}
PdaMod.screens = {}
PdaMod.screenOrder = {}
PdaMod.showConfig = false;
PdaMod.cfg = {
    atLeftCol = true
    ,leftColPos  = 0
    ,rightColPos = 0
    ,isScreenSelected = false
}
PdaMod.fileName = getUserProfileAppPath() .. "PdaModConfig.xml";
PdaMod.origUpdate = nil;
PdaMod.origDraw = nil;

PdaMod.pluginsMouseEvent = {}
PdaMod.pluginsKeyEvent = {}
PdaMod.pluginsUpdateFrame = {}
PdaMod.pluginsDraw = {}

--
addModEventListener(PdaMod);

-- Array for plugins to add themself to, which then registeres them.
getfenv(0)["modPDAMODplugins"] = getfenv(0)["modPDAMODplugins"] or {}

--
--
--

-- modPDAMOD.registerPage()
function PdaMod.registerPage(screenNameGeneric, screenNameI18N, drawFunc, drawTarget, drawParm, keyEventFunc, keyEventTarget, keyEventParm, insertIndex)
    if PdaMod.screens[screenNameGeneric] == nil then
        if screenNameI18N == nil then
            if g_i18n:hasText(screenNameGeneric) then
                screenNameI18N = g_i18n:getText(screenNameGeneric);
            else
                screenNameI18N = screenNameGeneric;
            end;
        end;
        PdaMod.screens[screenNameGeneric] = {nameGeneric=screenNameGeneric, nameI18N=screenNameI18N, numSubPages=0, subPages={}, showSubPage=1};
        table.insert(PdaMod.screenOrder, PdaMod.screens[screenNameGeneric]);
        print("PdaMod.registerPage: "..tostring(screenNameGeneric));
    end
    if screenNameGeneric ~= nil then
        table.insert(PdaMod.screens[screenNameGeneric].subPages, {drawFunc=drawFunc, drawTarget=drawTarget, drawParm=drawParm, keyEventFunc=keyEventFunc, keyEventTarget=keyEventTarget, keyEventParm=keyEventParm});
        PdaMod.screens[screenNameGeneric].numSubPages = table.getn(PdaMod.screens[screenNameGeneric].subPages);
    end
end

-- modPDAMOD.drawTitle()
function PdaMod.drawTitle(titleTxt)
    setTextAlignment(RenderText.ALIGN_LEFT);
    setTextBold(true);
    setTextColor(0, 0, 0, 1);
    renderText(g_currentMission.missionPDA.pdaTitleX, g_currentMission.missionPDA.pdaTitleY - 0.002, g_currentMission.missionPDA.pdaTitleTextSize, titleTxt);
    setTextColor(1, 1, 1, 1);
    renderText(g_currentMission.missionPDA.pdaTitleX, g_currentMission.missionPDA.pdaTitleY, g_currentMission.missionPDA.pdaTitleTextSize, titleTxt);
end

-- modPDAMOD.drawPageOfPages()
function PdaMod.drawPageOfPages(curPage,totalPages)
    setTextBold(false);
    setTextColor(1, 1, 1, 0.7);
    setTextAlignment(RenderText.ALIGN_RIGHT);
    renderText(g_currentMission.missionPDA.pdaPricesCol[4] + 0.025, g_currentMission.missionPDA.pdaHeadRow - g_currentMission.missionPDA.financeRowSpacing * 11, g_currentMission.missionPDA.pdaFontSize * 0.9, string.format("%d/%d", curPage, totalPages));
    setTextAlignment(RenderText.ALIGN_LEFT);
end

-- modPDAMOD.getFillTypePrice()
function PdaMod.getFillTypePrice(fillType, forUnitType, forQuantity)
    local fillDesc = Fillable.fillTypeIndexToDesc[fillType]
    if fillDesc and fillDesc.pricePerLiter ~= nil and fillDesc.pricePerLiter > 0 then
        -- If filldesc contains a mod-added 'showPriceForQuantity' - e.g. qty=1 for animals (pig,cow,sheep,horse,chicken,...), 
        -- else use vanilla default of qty=1000.
        forQuantity = Utils.getNoNil(forQuantity, Utils.getNoNil(fillDesc.showPriceForQuantity, 1000));
        -- TODO: Support for MoreRealistic
        -- forUnitType = Utils.getNoNil(forUnitType, "liter") -- default is "liter"
        --
        local difficultyFactor = (2 ^ (3 - g_currentMission.missionStats.difficulty))
        --
        return (fillDesc.pricePerLiter * forQuantity) * difficultyFactor
    end
    return nil
end


-- Make a global object, for all mods to see and use.
getfenv(0)["modPDAMOD"] = {
  version           = PdaMod.version
  ,registerPage     = PdaMod.registerPage       -- To call this from other mod-scripts, use:  modPDAMOD.registerPage(...)
  ,drawTitle        = PdaMod.drawTitle          -- To call this from other mod-scripts, use:  modPDAMOD.drawTitle(...)
  ,drawPageOfPages  = PdaMod.drawPageOfPages    -- To call this from other mod-scripts, use:  modPDAMOD.drawPageOfPages(...)
  ,getFillTypePrice = PdaMod.getFillTypePrice
};


--
--
--
function PdaMod:loadMap(name)
    PdaMod.initStep = 0;
    PdaMod.showScreen = 0;
    --
    PdaMod.plugins = getfenv(0)["modPDAMODplugins"]
    --
    for _,plugin in ipairs(PdaMod.plugins) do
        if plugin.loadMap ~= nil then
            plugin.loadMap(plugin, name);
        end
    end
end;

function PdaMod:deleteMap()
    PdaMod.screenOrder = {}
    PdaMod.screens = {}
    PdaMod.pluginsMouseEvent = {}
    PdaMod.pluginsKeyEvent = {}
    PdaMod.pluginsUpdateFrame = {}
    PdaMod.pluginsDraw = {}
    --
    for _,plugin in ipairs(PdaMod.plugins) do
        if plugin.deleteMap ~= nil then
            plugin.deleteMap(plugin);
        end
    end
end;

function PdaMod:mouseEvent(posX, posY, isDown, isUp, button)
    for _,plugin in ipairs(PdaMod.pluginsMouseEvent) do
        if plugin.mouseEvent ~= nil then
            plugin.mouseEvent(plugin, posX, posY, isDown, isUp, button);
        end
    end
end;

function PdaMod:keyEvent(unicode, sym, modifier, isDown)
    for _,plugin in ipairs(PdaMod.pluginsKeyEvent) do
        if plugin.keyEvent ~= nil then
            plugin.keyEvent(plugin, unicode, sym, modifier, isDown);
        end
    end
end;

function PdaMod:update(dt)
    -- Late initializing
    if PdaMod.initStep < 5 then
        PdaMod.initStep = PdaMod.initStep + 1;
        --
        -- The first 4 steps, is to allow other PDA-Mod plugins to register them self, before loading the screen-order from config file.
        --
        for _,plugin in ipairs(PdaMod.plugins) do
            if plugin.initialize ~= nil then
                plugin.initialize(plugin, PdaMod.initStep);
            end
            --
            if PdaMod.initStep == 1 then
                if plugin.mouseEvent  ~= nil then table.insert(PdaMod.pluginsMouseEvent,  plugin); end;
                if plugin.keyEvent    ~= nil then table.insert(PdaMod.pluginsKeyEvent,    plugin); end;
                if plugin.updateFrame ~= nil then table.insert(PdaMod.pluginsUpdateFrame, plugin); end;
                if plugin.draw        ~= nil then table.insert(PdaMod.pluginsDraw,        plugin); end;
            end
        end
        --
        if PdaMod.initStep == 5 then
            -- Register the default/vanilla game's PDA screens
            PdaMod.registerPage("defaultMap",       nil, PdaMod.subPageDraw_Original, g_currentMission.missionPDA, 1, PdaMod.subPageUpdate_Original, g_currentMission.missionPDA, 1, 0);
            PdaMod.registerPage("defaultWeather",   nil, PdaMod.subPageDraw_Original, g_currentMission.missionPDA, 3, PdaMod.subPageUpdate_Original, g_currentMission.missionPDA, 3, 0);
            PdaMod.registerPage("defaultPrices",    nil, PdaMod.subPageDraw_Original, g_currentMission.missionPDA, 4, PdaMod.subPageUpdate_Original, g_currentMission.missionPDA, 4, 0);
            PdaMod.registerPage("defaultFinances",  nil, PdaMod.subPageDraw_Original, g_currentMission.missionPDA, 5, PdaMod.subPageUpdate_Original, g_currentMission.missionPDA, 5, 0);
            PdaMod.registerPage("defaultStats",     nil, PdaMod.subPageDraw_Original, g_currentMission.missionPDA, 6, PdaMod.subPageUpdate_Original, g_currentMission.missionPDA, 6, 0);
            if g_currentMission.missionDynamicInfo.isMultiplayer then
                PdaMod.registerPage("defaultPlayers", nil, PdaMod.subPageDraw_Original, g_currentMission.missionPDA, 7, PdaMod.subPageUpdate_Original, g_currentMission.missionPDA, 7, 0);
            end;
            -- Final initialization.
            PdaMod.loadScreenOrder(self);
            -- Need the original MissionPDA's functions overwritten, to make PDA-Mod work correctly.
            PdaMod.origUpdate = g_currentMission.missionPDA.update;
            PdaMod.origDraw   = g_currentMission.missionPDA.draw;      
            --
            g_currentMission.missionPDA.update = PdaMod.missionPDA_update;
            g_currentMission.missionPDA.draw   = PdaMod.missionPDA_draw;            
        end;
    else
        for _,plugin in ipairs(PdaMod.pluginsUpdateFrame) do
            if plugin.updateFrame ~= nil then
                plugin.updateFrame(plugin, dt);
            end
        end
    end
end;

function PdaMod:draw()
    for _,plugin in pairs(PdaMod.pluginsDraw) do
        if plugin.draw ~= nil then
            plugin.draw(plugin);
        end
    end
end;


--
function PdaMod.subPageUpdate_Original(self, parm, _, dt)
    self.screen = parm;
    PdaMod.origUpdate(self, (dt~=nil and dt or 0));
end;
function PdaMod.subPageDraw_Original(self, parm, _)
    self.screen = parm;
    PdaMod.origDraw(self)
end

--
function PdaMod.loadScreenOrder(self)
    if g_dedicatedServerInfo ~= nil then
        print("** PdaMod is running on dedicated server, and will not load its configuration file.")
        return
    end

    print("** PdaMod is now trying to load its configuration file; " .. PdaMod.fileName);
    print("** Note that if the file does not yet exist, an error will be logged. This is normal behaviour.")
    local rootTag = "PdaMod";
    local xmlFile = loadXMLFile(rootTag, PdaMod.fileName);
    if xmlFile ~= nil then
        --
        local visibleScreenOrder = {};
        local knownScreens = {};
        local i=0;
        while true do
            local subTag = string.format("%s.pdaScreen(%d)", rootTag, i);
            local nameGeneric = getXMLString(xmlFile, subTag.."#name");
            local visible = getXMLBool(xmlFile, subTag.."#visible");
            if nameGeneric == nil or visible == nil then
                break;
            end;
            knownScreens[nameGeneric] = true;
            if visible then
                table.insert(visibleScreenOrder, nameGeneric);
            end;
            i = i + 1;
        end;
        --
        delete(xmlFile);
        xmlFile = nil;
        --
        PdaMod.screenOrder = {};
        for _,nameGeneric in pairs(visibleScreenOrder) do
            for _,screen in pairs(PdaMod.screens) do
                if screen.nameGeneric == nameGeneric then
                    table.insert(PdaMod.screenOrder, screen);
                    break;
                end;
            end;
        end;
        -- Any unknown pages, gets added too.
        for _,screen in pairs(PdaMod.screens) do
            if knownScreens[screen.nameGeneric] == nil then
                table.insert(PdaMod.screenOrder, screen);
            end;
        end;
    end;
end;

function PdaMod.saveScreenOrder(self)
    if g_dedicatedServerInfo ~= nil then
        print("** PdaMod is running on dedicated server, and will not save its configuration file.")
        return
    end
    
    local rootTag = "PdaMod";
    local xmlFile = createXMLFile(rootTag, PdaMod.fileName, rootTag);
    if xmlFile ~= nil then
        local i=0;
        for _,screenOrderPage in pairs(PdaMod.screenOrder) do
            local subTag = string.format("%s.pdaScreen(%d)", rootTag, i);
            setXMLString(xmlFile, subTag.."#name", screenOrderPage.nameGeneric);
            setXMLBool(xmlFile, subTag.."#visible", true);
            i=i+1
        end
  
        for _,screen in pairs(PdaMod.screens) do
            local foundAsVisible = false;
            for _,screenOrderPage in pairs(PdaMod.screenOrder) do
                if screen.nameGeneric == screenOrderPage.nameGeneric then
                    foundAsVisible = true;
                    break;
                end;
            end;
            if not foundAsVisible then
                local subTag = string.format("%s.pdaScreen(%d)", rootTag, i);
                setXMLString(xmlFile, subTag.."#name", screen.nameGeneric);
                setXMLBool(xmlFile, subTag.."#visible", false);
                i=i+1
            end;
        end;
        --
        saveXMLFile(xmlFile);
        --
        delete(xmlFile);
        xmlFile = nil;
    end;
end;


--
function PdaMod.missionPDA_update(self, dt)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.

    -- Checking for input events
    if self.enablePDA and g_gui.currentGui == nil then
        -- If in PDA-Mod configuration mode, special handling is needed
        if PdaMod.showConfig then
            
            if InputBinding.hasEvent(InputBinding.TOGGLE_PDA) then
                PdaMod.showConfig = false;
                playSample(self.pdaBeepSound, 1, 0.3, 0);
                PdaMod.saveScreenOrder(self);
            else
                g_currentMission:addHelpButtonText(g_i18n:getText("PdaModConfigClose"), InputBinding.TOGGLE_PDA);
                g_currentMission:addHelpButtonText(g_i18n:getText("PdaModConfigSelect"), InputBinding.MENU_ACCEPT);
                g_currentMission:addExtraPrintText(g_i18n:getText("PdaModConfigMove"));

                if InputBinding.hasEvent(InputBinding.MENU_UP) then
                    if PdaMod.cfg.atLeftCol then
                        if PdaMod.cfg.leftColPos > 1 then
                            local tempScreen = nil;
                            if PdaMod.cfg.isScreenSelected then
                                tempScreen = PdaMod.screenOrder[PdaMod.cfg.leftColPos];
                                table.remove(PdaMod.screenOrder, PdaMod.cfg.leftColPos);
                            end;
                            PdaMod.cfg.leftColPos = PdaMod.cfg.leftColPos - 1;
                            if tempScreen then
                                table.insert(PdaMod.screenOrder, PdaMod.cfg.leftColPos, tempScreen);
                            end;
                        end;
                    elseif not PdaMod.cfg.isScreenSelected then
                        PdaMod.cfg.rightColPos = Utils.clamp(PdaMod.cfg.rightColPos - 1, 1, table.getn(PdaMod.hiddenScreens));
                    end
                elseif InputBinding.hasEvent(InputBinding.MENU_DOWN) then
                    if PdaMod.cfg.atLeftCol then
                        if PdaMod.cfg.leftColPos < table.getn(PdaMod.screenOrder) then
                            local tempScreen = nil;
                            if PdaMod.cfg.isScreenSelected then
                                tempScreen = PdaMod.screenOrder[PdaMod.cfg.leftColPos];
                                table.remove(PdaMod.screenOrder, PdaMod.cfg.leftColPos);
                            end;
                            PdaMod.cfg.leftColPos = PdaMod.cfg.leftColPos + 1;
                            if tempScreen then
                                table.insert(PdaMod.screenOrder, PdaMod.cfg.leftColPos, tempScreen);
                            end;
                        end;
                    elseif not PdaMod.cfg.isScreenSelected then
                        PdaMod.cfg.rightColPos = Utils.clamp(PdaMod.cfg.rightColPos + 1, 1, table.getn(PdaMod.hiddenScreens));
                    end
                elseif InputBinding.hasEvent(InputBinding.MENU_LEFT) then
                    if not PdaMod.cfg.atLeftCol then
                        if PdaMod.cfg.isScreenSelected then
                            local tempScreen = PdaMod.hiddenScreens[PdaMod.cfg.rightColPos];
                            table.remove(PdaMod.hiddenScreens, PdaMod.cfg.rightColPos);
                            table.insert(PdaMod.screenOrder, PdaMod.cfg.leftColPos, tempScreen);
                        end;
                        if table.getn(PdaMod.screenOrder) > 0 then
                            PdaMod.cfg.atLeftCol = true;
                        end;
                    end;
                elseif InputBinding.hasEvent(InputBinding.MENU_RIGHT) then
                    if PdaMod.cfg.atLeftCol then
                        if PdaMod.cfg.isScreenSelected then
                            local tempScreen = PdaMod.screenOrder[PdaMod.cfg.leftColPos];
                            table.remove(PdaMod.screenOrder, PdaMod.cfg.leftColPos);
                            table.insert(PdaMod.hiddenScreens, tempScreen);
                            table.sort(PdaMod.hiddenScreens, function(l,r) return l.nameI18N < r.nameI18N; end);
                        end;
                        if table.getn(PdaMod.hiddenScreens) > 0 then
                            PdaMod.cfg.atLeftCol = false;
                        end;
                    end;
                elseif InputBinding.hasEvent(InputBinding.MENU_ACCEPT) then
                    PdaMod.cfg.isScreenSelected = not PdaMod.cfg.isScreenSelected;
                end;
            end;
            return; -- Do not process any other input events, while on PDA Config page.
        end;

        --
        if not self.showPDA and g_currentMission.controlPlayer then -- On foot
            g_currentMission:addHelpButtonText(g_i18n:getText("PdaModConfigOpen"), InputBinding.TOGGLE_PDA);
        end;
            
        --
        if InputBinding.hasEvent(InputBinding.TOGGLE_PDA_ZOOM) then
            if  PdaMod.showScreen > 0 and PdaMod.screenOrder[PdaMod.showScreen].numSubPages > 0 then
                local showSubPage = PdaMod.screenOrder[PdaMod.showScreen].showSubPage;
                local subPages    = PdaMod.screenOrder[PdaMod.showScreen].subPages;
                if subPages[showSubPage] ~= nil and subPages[showSubPage].keyEventFunc ~= nil then
                    -- This sub-page has its own special handling of the TOGGLE_PDA_ZOOM event
                    subPages[showSubPage].keyEventFunc(subPages[showSubPage].keyEventTarget, subPages[showSubPage].keyEventParm, nil, dt); --, Original_MissionPDA_update);
                else
                    PdaMod.screenOrder[PdaMod.showScreen].showSubPage = showSubPage + 1;
                    playSample(self.pdaBeepSound, 1, 0.15, 0);
                end;
                if PdaMod.screenOrder[PdaMod.showScreen].showSubPage > PdaMod.screenOrder[PdaMod.showScreen].numSubPages then
                    PdaMod.screenOrder[PdaMod.showScreen].showSubPage = 1;
                end;
            end;
        end
        
        -- Handling of 'press-and-hold' inputs vs 'press-and-release' inputs.
        local shortInput = nil; -- press-and-release (that is, less than 500 ms.)
        local longInput = nil;  -- press-and-hold (for more than 500 ms.)
        if PdaMod.longInput == nil then
            if InputBinding.isPressed(InputBinding.TOGGLE_PDA) then
                PdaMod.longInput = {};
                PdaMod.longInput.inputBinding = InputBinding.TOGGLE_PDA;
                PdaMod.longInput.timeout = g_currentMission.time + 500;
            end
        elseif PdaMod.longInput.timeout > 0 and PdaMod.longInput.timeout < g_currentMission.time then
            -- Activate longInput
            longInput = PdaMod.longInput.inputBinding;
            PdaMod.longInput.timeout = 0;
        elseif not InputBinding.isPressed(PdaMod.longInput.inputBinding) then
            if PdaMod.longInput.timeout > 0 then
                -- Activate shortInput
                shortInput = PdaMod.longInput.inputBinding;
            end;
            PdaMod.longInput = nil;
        end
        
        if shortInput ~= nil then
            if shortInput == InputBinding.TOGGLE_PDA then
                playSample(self.pdaBeepSound, 1, 0.3, 0);
                PdaMod.showScreen = (PdaMod.showScreen + 1) % (1 + table.getn(PdaMod.screenOrder));
                self.showPDA = (PdaMod.showScreen > 0);
            end;
        elseif longInput ~= nil then
            if longInput == InputBinding.TOGGLE_PDA then
                if not self.showPDA and g_currentMission.controlPlayer then -- On foot
                    -- Activate PdaMod-config screen
                    PdaMod.showConfig = true;
                    PdaMod.cfg = {
                        atLeftCol = table.getn(PdaMod.screenOrder) > 0
                        ,leftColPos  = 1
                        ,rightColPos = 1
                        ,isScreenSelected = false
                    };
                    playSample(self.pdaBeepSound, 1, 0.3, 0);
                else
                    -- Toggle PDA showing or not.
                    self.showPDA = not self.showPDA;
                    PdaMod.showScreen = 0;
                    playSample(self.pdaBeepSound, 1, 0.3, 0);
                end;
            end;
        end;
    end;
end

function PdaMod.missionPDA_draw(self)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.

    if self.showPDA and PdaMod.showScreen > 0 then
        local showSubPage = PdaMod.screenOrder[PdaMod.showScreen].showSubPage;
        local subPage     = PdaMod.screenOrder[PdaMod.showScreen].subPages[showSubPage];
        
        if subPage.keyEventFunc == nil then
            if PdaMod.showScreen < #PdaMod.screenOrder then
                g_currentMission:addHelpButtonText(g_i18n:getText("PDAShowNextScreen"), InputBinding.TOGGLE_PDA);
            else
                g_currentMission:addHelpButtonText(g_i18n:getText("PDAClose"), InputBinding.TOGGLE_PDA);
            end;
        end;
        
        if subPage.drawFunc ~= nil then
            subPage.drawFunc(subPage.drawTarget, subPage.drawParm); --, Original_MissionPDA_draw);
            --
            if PdaMod.screenOrder[PdaMod.showScreen].numSubPages > 1 and subPage.keyEventFunc == nil then
                g_currentMission:addHelpButtonText(g_i18n:getText("PdaModNextPage"), InputBinding.TOGGLE_PDA_ZOOM);
                PdaMod.drawPageOfPages(showSubPage, PdaMod.screenOrder[PdaMod.showScreen].numSubPages);
            end;
        end;
    elseif PdaMod.showConfig then
        PdaMod.hiddenScreens = {}
        for _,screen in pairs(PdaMod.screens) do
            local foundAsVisible = false;
            for _,screenOrderPage in pairs(PdaMod.screenOrder) do
                if screen.nameGeneric == screenOrderPage.nameGeneric then
                    foundAsVisible = true;
                    break;
                end;
            end;
            if not foundAsVisible then
                table.insert(PdaMod.hiddenScreens, screen);
            end;
        end;
        table.sort(PdaMod.hiddenScreens, function(l,r) return l.nameI18N < r.nameI18N; end);
        --
        self.hudPDABackgroundOverlay:render();
        --
        setTextAlignment(RenderText.ALIGN_LEFT);
       
        renderText(self.pdaCol1, self.pdaHeadRow - self.pdaRowSpacing * 0, self.pdaFontSize, g_i18n:getText("PdaModConfigVisible"));
        local row = 2;
        for i,screen in ipairs(PdaMod.screenOrder) do
            if PdaMod.cfg.atLeftCol and i == PdaMod.cfg.leftColPos then
                setTextColor(1,1,0,1);
                setTextBold(PdaMod.cfg.isScreenSelected);
            else
                setTextColor(1,1,1,1);
                setTextBold(false);
            end;
            renderText(self.pdaCol1, self.pdaHeadRow - self.pdaRowSpacing * row, self.pdaFontSize, tostring(i)..":"..screen.nameI18N);
            row = row + 1;
        end

        setTextColor(1,1,1,1);
        setTextBold(false);
        renderText(self.pdaCol2, self.pdaHeadRow - self.pdaRowSpacing * 0, self.pdaFontSize, g_i18n:getText("PdaModConfigHidden"));
        row = 2;
        for i,screen in ipairs(PdaMod.hiddenScreens) do
            if not PdaMod.cfg.atLeftCol and i == PdaMod.cfg.rightColPos then
                setTextColor(1,1,0,1);
                setTextBold(PdaMod.cfg.isScreenSelected);
            else
                setTextColor(1,1,1,1);
                setTextBold(false);
            end;
            renderText(self.pdaCol2, self.pdaHeadRow - self.pdaRowSpacing * row, self.pdaFontSize, screen.nameI18N);
            row = row + 1;
        end
        --
        self.hudPDAFrameOverlay:render();
        --
        PdaMod.drawTitle(g_i18n:getText("PdaModConfigTitle"));
    end;
end

--
print(string.format("Script loaded: PdaMod.LUA (v%s)", PdaMod.version));
