--
-- PdaPlugin_PriceHistory
--
-- @author  Decker_MMIV - fs-uk.com, forum.farming-simulator.com, modhoster.com
-- @date    2014-May
--
-- @history
--  2014-May
--      v0.01   - Initial experiment
--      v0.10   - Working with past 9 prices.
--  2014-June
--      v0.11   - Due to the NON-deterministic way that mods are loaded in multiplayer, the modPDAMOD.registerPlugin(...)
--                is replaced in favor of a hopefully more robust "create object modPDAMODplugins, and add yourself to it" method.
--      v0.12   - Fix for maps that contain multiple tip-triggers with same station-name, and yet different list of accepted fill-types.
--              - Upper-case sort of station & fill-type names.
--      2.0.0   - Using http://semver.org/
--      2.0.2   - Fix: Tested on HM13 map which caused some bugs regarding fruit-icons.
--


PdaPlugin_PriceHistory = {}
--
local modItem = ModsUtil.findModItemByModName(g_currentModName);
PdaPlugin_PriceHistory.version = (modItem and modItem.version) and modItem.version or "?.?.?";
--
PdaPlugin_PriceHistory.lastUpdatedDayHour = -1;

-- Register this PDAMod plugin
getfenv(0)["modPDAMODplugins"] = getfenv(0)["modPDAMODplugins"] or {}
table.insert(getfenv(0)["modPDAMODplugins"], PdaPlugin_PriceHistory);
--

function PdaPlugin_PriceHistory.initialize(self, initStep)
    if initStep == 4 then
        g_currentMission.environment:addHourChangeListener(PdaPlugin_PriceHistory);
        PdaPlugin_PriceHistory.pageNum = 1;
        --
        modPDAMOD.registerPage(
            "pricesHistory", nil,
            PdaPlugin_PriceHistory.subPageDraw,    g_currentMission.missionPDA, nil,
            PdaPlugin_PriceHistory.screenKeyEvent, g_currentMission.missionPDA, nil
        );
        -- Compute some column positions and widths...
        local pda = g_currentMission.missionPDA;
        -- Some other mod apparently has the brilliant idea (NOT!) to constantly modify 'pdaRowSpacing' in its update() function! WTF?
        -- I'm looking at you PDAExtensionForageStorage.LUA and HagenstedtModified2013-map! >:-(
        PdaPlugin_PriceHistory.pdaFontSize = pda.pdaHeight / 15
        PdaPlugin_PriceHistory.pdaRowSpacing = PdaPlugin_PriceHistory.pdaFontSize * 1.05
        PdaPlugin_PriceHistory.pdaColWidth = getTextWidth(PdaPlugin_PriceHistory.pdaFontSize, "888888");
        PdaPlugin_PriceHistory.pdaCol1 = pda.pdaCol1 - (PdaPlugin_PriceHistory.pdaFontSize / 8)

        -- Create some filltype overlays, to use behind the price history
        PdaPlugin_PriceHistory.FilltypeIcons = {}
        for _,ftDesc in pairs(Fillable.economyFillTypes) do
            local fillIcon = ftDesc.hudOverlayFilename;
            if fillIcon ~= nil and fillIcon ~= "" then
                local overlayId = createImageOverlay(fillIcon)
                if overlayId ~= nil and overlayId ~= 0 then
                    PdaPlugin_PriceHistory.FilltypeIcons[ftDesc.index] = overlayId;
                    setOverlayColor(PdaPlugin_PriceHistory.FilltypeIcons[ftDesc.index], 1,1,1, 0.1); -- 10% opaque
                end
            end;
        end;

        --
        PdaPlugin_PriceHistory.loadPrices(PdaPlugin_PriceHistory)
    end
end;

function PdaPlugin_PriceHistory:hourChanged()
    -- 10ms after, to give vanilla scripts time to update their values.
    addTimer(10, "refreshPrices", PdaPlugin_PriceHistory)
end

--
function PdaPlugin_PriceHistory.refreshPrices(self)
    if self.sellStations == nil then
        PdaPlugin_PriceHistory.loadPrices(self)
    end
    --
    local nowDayHour = g_currentMission.environment.currentDay*100 + g_currentMission.environment.currentHour;
    if self.lastUpdateDayHour ~= nowDayHour then
        self.lastUpdateDayHour = nowDayHour;
        --
        for _,sellStation in pairs(self.sellStations) do
            local foundFillTypes = {}
            for _,tipTrigger in pairs(sellStation.tipTriggers) do
                for fillType,accepted in pairs(tipTrigger.acceptedFillTypes) do
                    if not foundFillTypes[fillType] then
                        if accepted and not sellStation.fillTypesPrices[fillType] then
                            sellStation.fillTypesPrices[fillType] = {}
                        end
                        --
                        local price = 0;
                        local demandMultiplier = 1
                        if accepted then
                            local greatDemand = g_currentMission.economyManager:getCurrentGreatDemand(tipTrigger.stationName, fillType)
                            demandMultiplier = (greatDemand ~= nil) and greatDemand.demandMultiplier or demandMultiplier
                            price = modPDAMOD.getFillTypePrice(fillType) * tipTrigger.priceMultipliers[fillType] * demandMultiplier
                        end;
                        local entry = { px = math.ceil(price) }
                        if demandMultiplier > 1 then
                            entry.gd = true
                        end
                        table.insert(sellStation.fillTypesPrices[fillType], 1, entry);
                        foundFillTypes[fillType] = true
                        --
                        while (#sellStation.fillTypesPrices[fillType] > 72) do
                            table.remove(sellStation.fillTypesPrices[fillType], #sellStation.fillTypesPrices[fillType]);
                        end
                    end
                end
            end
        end
    end
    --
end

--
function PdaPlugin_PriceHistory.loadPrices(self)
    self.sellStations = {};
    self.sellStationsSorted = {}
    self.sellFillTypesSorted = {}
    --
    local getI18N = function(txt)
        if g_i18n:hasText(txt) then
            return g_i18n:getText(txt)
        end
        return txt;
    end
    if g_currentMission.missionInfo and g_currentMission.missionInfo.map and g_currentMission.missionInfo.map.customEnvironment then
        -- Try to find station-names from the map's own g_i18n object
        local env0 = getfenv(0)
        local mapMod = env0[g_currentMission.missionInfo.map.customEnvironment]
        if mapMod ~= nil and mapMod.g_i18n ~= nil then
            mapModI18N = mapMod.g_i18n;
            getI18N = function(txt)
                if g_i18n:hasText(txt) then
                    return g_i18n:getText(txt)
                elseif mapModI18N:hasText(txt) then
                    return mapModI18N:getText(txt)
                end
                return txt;
            end
        end
    end
    --
    for _,tipTrigger in pairs(g_currentMission.tipTriggers) do
        if tipTrigger.appearsOnPDA and tipTrigger.isEnabled and tipTrigger.stationName ~= nil and tipTrigger.stationName ~= ""
        and (tipTrigger.isExtendedTrigger == nil) -- AlternativeTipTrigger is excluded.
        then
            local fillTypesPrices = (self.sellStations[tipTrigger.stationName] ~= nil) and self.sellStations[tipTrigger.stationName].fillTypesPrices or {}
            for fillType,accepted in pairs(tipTrigger.acceptedFillTypes) do
                if accepted then
                    fillTypesPrices[fillType] = (fillTypesPrices[fillType] ~= nil) or {}
                    self.sellFillTypesSorted[fillType] = fillType;
                end
            end
            if not self.sellStations[tipTrigger.stationName] then
                table.insert(self.sellStationsSorted, tipTrigger.stationName)
            end

            local stationNameI18N = getI18N(tipTrigger.stationName);
            
            self.sellStations[tipTrigger.stationName] = {
                 tipTriggers      = (self.sellStations[tipTrigger.stationName] ~= nil) and self.sellStations[tipTrigger.stationName].tipTriggers or {}
                ,nameI18N        = stationNameI18N    --g_i18n:hasText(tipTrigger.stationName) and g_i18n:getText(tipTrigger.stationName) or tipTrigger.stationName
                ,fillTypesPrices = fillTypesPrices
            }
            table.insert(self.sellStations[tipTrigger.stationName].tipTriggers, tipTrigger) -- Fix for maps containing several tipTriggers with same station-name but different accepted-fill-types.
        end;
    end;
    table.sort(self.sellStationsSorted, function(a,b)
        return (self.sellStations[a].nameI18N):upper() < (self.sellStations[b].nameI18N):upper()
    end)
    --
    local sortArray = {}
    for fillType,_ in pairs(self.sellFillTypesSorted) do
        table.insert(sortArray, fillType)
    end
    table.sort(sortArray, function(a,b)
        local aName = Fillable.fillTypeIndexToDesc[a].nameI18N or Fillable.fillTypeIndexToDesc[a].name
        local bName = Fillable.fillTypeIndexToDesc[b].nameI18N or Fillable.fillTypeIndexToDesc[b].name
        return (aName):upper() < (bName):upper();
    end)
    self.sellFillTypesSorted = sortArray
    --
    print("** PdaModPlugin_PriceHistory is now trying to load its 'history' file.");
    print("** Note that if the file does not yet exist, an error will be logged. This is normal behaviour.")
    local rootTag = "priceHistory";
    local xmlFile = loadXMLFile(rootTag, g_currentMission.missionInfo.savegameDirectory.."/PdaModPlugin_PriceHistory.XML");
    if (xmlFile ~= nil and getXMLInt(xmlFile, rootTag.."#version") ~= nil) then
        self.lastUpdateDayHour = Utils.getNoNil(getXMLInt(xmlFile, rootTag.."#lastDayHour"), -1);
        local i=0;
        while true do
            local stationTag = rootTag..string.format(".station(%d)", i);
            i=i+1;
            local stationName = getXMLString(xmlFile, stationTag.."#name");
            if stationName == nil then
                break;
            end;
            --
            local found = self.sellStations[stationName]
            if found ~= nil then
                local fillTypes = {}
                for fillType,_ in pairs(found.fillTypesPrices) do
                    local fillTypeName = Fillable.fillTypeIntToName[fillType];
                    if fillTypeName ~= nil then
                        fillTypes[fillType] = fillTypeName;
                    end;
                end;
                --
                local h=0;
                while true do
                    local histTag = stationTag..string.format(".priceHist(%d)", h);
                    h=h+1
                    local histTime = getXMLInt(xmlFile, histTag.."#time");
                    if histTime == nil then
                        break;
                    end;
                    --
                    for fillType,fillTypeName in pairs(fillTypes) do
                        local priceAndGreatDemand = getXMLString(xmlFile, histTag.."#"..fillTypeName)
                        if priceAndGreatDemand then
                            priceAndGreatDemand = Utils.splitString(":", priceAndGreatDemand)
                            local entry = { 
                                px = math.ceil(tonumber(priceAndGreatDemand[1]) or 0) 
                               ,gd = (priceAndGreatDemand[2] ~= nil)
                            }
                            --if priceAndGreatDemand[2] then
                            --    entry.gd = true
                            --end
                            table.insert(found.fillTypesPrices[fillType], entry );
                        end
                    end;
                end;
            end;
        end;
        --
        delete(xmlFile);
        xmlFile = nil;
    end;
end

function PdaPlugin_PriceHistory.savePrices()
    if PdaPlugin_PriceHistory.sellStations ~= nil then
        local rootTag = "priceHistory";
        local xmlFile = createXMLFile(rootTag, g_currentMission.missionInfo.savegameDirectory.."/PdaModPlugin_PriceHistory.XML", rootTag);
        setXMLInt(xmlFile, rootTag.."#version", 2);
        setXMLInt(xmlFile, rootTag.."#lastDayHour", PdaPlugin_PriceHistory.lastUpdateDayHour);
        local i=0;
        for stationName,sellStation in pairs(PdaPlugin_PriceHistory.sellStations) do
            local stationTag = rootTag..string.format(".station(%d)", i);
            i=i+1;
            setXMLString(xmlFile, stationTag.."#name", stationName);
            --
            local h=0;
            while h ~= nil do
                local histTag = stationTag..string.format(".priceHist(%d)", h);
                h=h+1;
                for fillType,priceArray in pairs(sellStation.fillTypesPrices) do
                    if h > #priceArray then
                        h = nil;
                        break;
                    end;
                    local fillTypeName = Fillable.fillTypeIntToName[fillType];
                    if fillTypeName ~= nil and fillTypeName ~= "" then
                        local value = tostring(priceArray[h].px)
                        if priceArray[h].gd then
                            value = value ..":GD" -- GreatDemand
                        end
                        setXMLString(xmlFile, histTag.."#"..fillTypeName, value);
                    end
                end;
                if h ~= nil then
                    setXMLInt(xmlFile, histTag.."#time", h);
                end;
            end;
        end;
        saveXMLFile(xmlFile);
        delete(xmlFile);
        xmlFile = nil;
    end
end
CareerScreen.saveSelectedGame = Utils.appendedFunction(CareerScreen.saveSelectedGame, PdaPlugin_PriceHistory.savePrices);


function PdaPlugin_PriceHistory.screenKeyEvent(self, parm, origMissionPdaUpdateFunc)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.

    local maxTypes = table.getn(PdaPlugin_PriceHistory.sellFillTypesSorted)
    PdaPlugin_PriceHistory.pageNum = (PdaPlugin_PriceHistory.pageNum % maxTypes) + 1;
    playSample(self.pdaBeepSound, 1, 0.3, 0);
end;

--
function PdaPlugin_PriceHistory.subPageDraw(self, parm, origMissionPdaDrawFunc)
    -- Note: 'self' is in context of the g_currentMission.missionPDA object.

    local fillType = PdaPlugin_PriceHistory.sellFillTypesSorted[PdaPlugin_PriceHistory.pageNum]
    local fillName = Fillable.fillTypeIndexToDesc[fillType].nameI18N
    --
    self.hudPDABackgroundOverlay:render()
    --
    local topRow = self.pdaHeadRow - 0.01;
    if PdaPlugin_PriceHistory.FilltypeIcons[fillType] then
        local iconW = self.pdaHeight * 0.7;
        local iconH = iconW * (4/3);
        local iconX = self.pdaX + (self.pdaWidth - iconW);
        local iconY = topRow - iconH;
        renderOverlay(PdaPlugin_PriceHistory.FilltypeIcons[fillType], iconX, iconY, iconW, iconH);
    end
    --
    local histStart=1
    local y = topRow + PdaPlugin_PriceHistory.pdaRowSpacing
    --
    setTextBold(true);
    setTextColor(1,1,1,0.6);
    setTextAlignment(RenderText.ALIGN_CENTER);
    local x = PdaPlugin_PriceHistory.pdaCol1 + (PdaPlugin_PriceHistory.pdaColWidth / 2)
    local h = histStart
    while h < histStart+9 do
        renderText(x, y, PdaPlugin_PriceHistory.pdaFontSize, string.format("t%d", -(h-1)));
        h=h+1
        x = x + PdaPlugin_PriceHistory.pdaColWidth
    end
    setTextColor(1,1,0.7,0.6);
    renderText(x, y, PdaPlugin_PriceHistory.pdaFontSize, "Hi/Lo");
    y = y - PdaPlugin_PriceHistory.pdaRowSpacing
    --
    for _,stationName in pairs(PdaPlugin_PriceHistory.sellStationsSorted) do
        local sellStation = PdaPlugin_PriceHistory.sellStations[stationName]

        if sellStation and sellStation.fillTypesPrices[fillType] then
            setTextBold(true);
            setTextColor(1,1,1,0.8);
            setTextAlignment(RenderText.ALIGN_LEFT);
            renderText(PdaPlugin_PriceHistory.pdaCol1, y, PdaPlugin_PriceHistory.pdaFontSize, sellStation.nameI18N);
            y = y - PdaPlugin_PriceHistory.pdaRowSpacing

            setTextBold(false);
            setTextAlignment(RenderText.ALIGN_RIGHT);
            local x = PdaPlugin_PriceHistory.pdaCol1 + PdaPlugin_PriceHistory.pdaColWidth
            local h = 1
            local hi,lo
            while h < 73 do
                local entry = sellStation.fillTypesPrices[fillType][h]
                if not entry then
                    break
                end
                --
                hi = (hi ~= nil) and math.max(hi,entry.px) or entry.px
                lo = (lo ~= nil) and math.min(lo,entry.px) or entry.px
                --
                if h >= histStart and h < histStart+9 then
                    if entry.gd then
                        setTextColor(0.2,1,0.2,1); -- great demand
                    else
                        setTextColor(1,1,1,1);
                    end
                    renderText(x, y, PdaPlugin_PriceHistory.pdaFontSize, tostring(entry.px));
    
                    x=x+PdaPlugin_PriceHistory.pdaColWidth
                end
                h=h+1
            end
            -- Highest/Lowest price the past 72 in-game hours.
            if hi and lo then
                x = PdaPlugin_PriceHistory.pdaCol1 + (PdaPlugin_PriceHistory.pdaColWidth * 10)
                setTextColor(1,1,0.7,0.8);
                renderText(x, y + (PdaPlugin_PriceHistory.pdaRowSpacing * 0.9), PdaPlugin_PriceHistory.pdaFontSize*0.9, tostring(hi));
                renderText(x, y,                                                PdaPlugin_PriceHistory.pdaFontSize*0.9, tostring(lo));
            end
            --

            y = y - PdaPlugin_PriceHistory.pdaRowSpacing
        end
    end
    --
    self.hudPDAFrameOverlay:render()
    --
    modPDAMOD.drawTitle(string.format(g_i18n:getText("PriceHistory"), fillName));
    --
    local numPages = table.getn(PdaPlugin_PriceHistory.sellFillTypesSorted)
    if numPages > 1 then 
        g_currentMission:addHelpButtonText(g_i18n:getText("NextPage"), InputBinding.TOGGLE_PDA_ZOOM);
        modPDAMOD.drawPageOfPages(PdaPlugin_PriceHistory.pageNum, numPages)
    end;
end

print(string.format("Script loaded: PdaPlugin_PriceHistory.LUA (v%s)", PdaPlugin_PriceHistory.version));
