-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.


Kaercher = {};

function Kaercher.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Steerable, specializations);
end;

InitObjectClass(Kaercher, "Steerable");

function Kaercher:new(isServer, isClient, customMt)
    local mt = customMt;
    if mt == nil then
        mt = Kaercher_mt;
    end;
    local self = Kaercher:new(isServer, isClient, mt);
    registerObjectClassName(self, "Steerable");
    self.messageShown = false;
    return self;
end;

function Kaercher:load(xmlFilename, x,y,z, rx,ry,rz )

    if not Kaercher:superClass().load(self, xmlFilename, x,y,z, rx,ry,rz ) then
        return false;
    end;
	
    local xmlFile = loadXMLFile("TempXML", xmlFilename);
    self.playerInRangeDistance = Utils.getNoNil(getXMLFloat(xmlFile, "Steerable.playerInRangeDistance"), 6);
    self.washDistance = Utils.getNoNil(getXMLFloat(xmlFile, "Steerable.washDistance"), 20);
    self.actionRadius = Utils.getNoNil(getXMLFloat(xmlFile, "Steerable.actionRadius#distance"), 30);
    self.washMultiplier = Utils.getNoNil(getXMLFloat(xmlFile, "Steerable.washMultiplier"), 2);
    self.pricePerSecond = Utils.getNoNil(getXMLFloat(xmlFile, "Steerable.pricePerMinute"), 20) / 2000;
    self.lanceNode = Utils.indexToObject(self.nodeId, getXMLString(xmlFile, "Steerable.lance#index"));
    self.linkPosition = Utils.getVectorNFromString(Utils.getNoNil(getXMLString(xmlFile, "Steerable.lance#position"), "0 0 0"), 3);
    self.linkRotation = Utils.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, "Steerable.lance#rotation"), "0 0 0"), 3);
    self.lanceNodeParent = getParent(self.lanceNode);
    self.lanceRaycastNode = Utils.indexToObject(self.nodeId, getXMLString(xmlFile, "Steerable.lance#raycastNode"));
    if self.isClient then
        self.particleSystems = {};
        Utils.loadParticleSystem(xmlFile, self.particleSystems, "Steerable.particleSystem", self.nodeId, false, nil, g_currentMission.baseDirectory, nil);
        self.waterEffect = EffectManager:loadEffect(xmlFile, "Steerable.waterEffect", self.nodeId, self);
        self.sampleCompressor = Utils.loadSample(xmlFile, {}, "Steerable.compressorSound", nil, self.baseDirectory, self.nodeId);
        self.compressorPitchMin = Utils.getNoNil(getXMLFloat(xmlFile, "Steerable.compressorSound#pitchMin"), 0.5);
        self.sampleWashing = Utils.loadSample(xmlFile, {}, "Steerable.washingSound", nil, self.baseDirectory);
        self.sampleSwitch = Utils.loadSample(xmlFile, {}, "Steerable.switchSound", nil, self.baseDirectory);
        local filename = getXMLString(xmlFile, "Steerable.exhaust#filename");
        if filename ~= nil then
            local i3dNode = Utils.loadSharedI3DFile(filename, self.baseDirectory, false, false, false);
            if i3dNode ~= 0 then
                local linkNode = Utils.getNoNil(Utils.indexToObject(self.nodeId, getXMLString(xmlFile, "Steerable.exhaust#index")), self.nodeId);
                self.exhaustFilename = filename;
                self.exhaustNode = getChildAt(i3dNode, 0);
                link(linkNode, self.exhaustNode);
                setVisibility(self.exhaustNode, false);
                delete(i3dNode);
            end;
        end;
        self.targets = {};
        HandTool.loadTargets(xmlFile, "Steerable", self.nodeId, self.targets);
    end;
    delete(xmlFile);
    self.isPlayerInRange = false;
    self.isTurnedOn = false;
    self.activatable = KaercherActivatable:new(self);
    self.doWashing = false;
    self.lastInRangePosition = {0,0,0};
    self.isTurningOff = false;
    self.turnOffTime = 0;
    self.turnOffDuration = 500;
    return true;
end;

function Kaercher:delete()
    self:setIsTurnedOn(false, nil, false);
    if self.isClient then
        EffectManager:deleteEffect(self.waterEffect);
        if self.exhaustFilename ~= nil then
            Utils.releaseSharedI3DFile(self.exhaustFilename, self.baseDirectory, true);
        end;
        if self.particleSystems ~= nil then
            Utils.deleteParticleSystem(self.particleSystems);
        end;
        if self.sampleCompressor ~= nil then
            Utils.deleteSample(self.sampleCompressor);
        end;
        if self.sampleWashing ~= nil then
            Utils.deleteSample(self.sampleWashing);
        end;
        if self.sampleSwitch ~= nil then
            Utils.deleteSample(self.sampleSwitch);
        end;
    end;
    unregisterObjectClassName(self);
    g_currentMission:removeActivatableObject(self.activatable);
    Kaercher:superClass().delete(self);
end;

function Kaercher:readStream(streamId, connection)
    Kaercher:superClass().readStream(self, streamId, connection);
    if connection:getIsServer() then
        local isTurnedOn = streamReadBool(streamId);
        if isTurnedOn then
            local player = networkGetObject(streamReadInt32(streamId));
            if player ~= nil then
                self:setIsTurnedOn(isTurnedOn, player, true);
            end;
        end;
    end;
end;

function Kaercher:writeStream(streamId, connection)
    Kaercher:superClass().writeStream(self, streamId, connection);
    if not connection:getIsServer() then
        streamWriteBool(streamId, self.isTurnedOn);
        if self.isTurnedOn then
            streamWriteInt32(streamId, networkGetObjectId(self.currentPlayer));
        end;
    end;
end;

function Kaercher:activateHandtool(player)
    self:setIsTurnedOn(true, player, true);
end;

function Kaercher:update(dt)
    Kaercher:superClass().update(self, dt);
    if self.currentPlayer ~= nil then
        if isPlayerInRange then
            self.lastInRangePosition = {getTranslation(self.currentPlayer.rootNode)};
        else
            local kx, _, kz = getWorldTranslation(self.nodeId);
            local px, _, pz = getWorldTranslation(self.currentPlayer.rootNode);
            local len = Utils.vector2Length(px-kx, pz-kz);
            local x,y,z = unpack(self.lastInRangePosition);
            x = kx + ((px-kx) / len) * (self.actionRadius-0.00001*dt);
            z = kz + ((pz-kz) / len) * (self.actionRadius-0.00001*dt);
            y = math.max(y, getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 0, z));
            self.currentPlayer:moveToAbsoluteInternal(x, y, z);
            self.lastInRangePosition = {x,y,z};
            if not self.messageShown and self.currentPlayer == g_currentMission.player then
                g_currentMission:showBlinkingWarning(g_i18n:getText("KaercherRangeRestriction"), 6000);
                self.messageShown = true;
            end;
        end;
    end;
    if self.isServer then
        if self.isTurnedOn and self.doWashing then
            self.foundVehicle = nil;
            self:cleanVehicle(self.currentPlayer.cameraId, dt);
            if self.lanceRaycastNode ~= nil then
                self:cleanVehicle(self.lanceRaycastNode, dt);
            end;
            local price = self.pricePerSecond * (dt / 1000);
            g_currentMission.missionStats:updateStats("expenses", price);
            g_currentMission:addSharedMoney(-price, "vehicleRunningCost");
        end;
    end;
    if self.isTurningOff then
        if g_currentMission.time < self.turnOffTime then
            if self.sampleCompressor ~= nil then
                local pitch = Utils.lerp(self.compressorPitchMin, self.sampleCompressor.pitchOffset, Utils.clamp((self.turnOffTime - g_currentMission.time) / self.turnOffDuration, 0, 1));
                local volume = Utils.lerp(0, self.sampleCompressor.volume, Utils.clamp((self.turnOffTime - g_currentMission.time) / self.turnOffDuration, 0, 1));
                Utils.setSamplePitch(self.sampleCompressor, pitch);
                Utils.setSampleVolume(self.sampleCompressor, volume);
            end;
        else
            self.isTurningOff = false;
            if self.sampleCompressor ~= nil then
                Utils.stop3DSample(self.sampleCompressor);
            end;
        end;
    end;
end;

function Kaercher:cleanVehicle(node, dt)
    local x,y,z = getWorldTranslation(node);
    local dx, dy, dz = localDirectionToWorld(node, 0, 0, -1);
    local lastFoundVehicle = self.foundVehicle;
    --drawDebugLine(x, y, z, 1, 0, 0, x+dx*2, y+dy*2,z+dz*2, 0,0,1);
    raycastAll(x, y, z, dx, dy, dz, "washRaycastCallback", self.washDistance, self, 32+64+128+256+4096+8194);
    if self.foundVehicle ~= nil and lastFoundVehicle ~= self.foundVehicle then
        --drawDebugPoint(self.foundCoords[1], self.foundCoords[2], self.foundCoords[3], 1,1,1,1);
        self.foundVehicle:setDirtAmount(self.foundVehicle:getDirtAmount() - self.washMultiplier*dt/self.foundVehicle.washDuration);
    end;
end;

function Kaercher:updateTick(dt)
    Kaercher:superClass().updateTick(self, dt);
    local isPlayerInRange, player = self:getIsPlayerInRange(self.playerInRangeDistance);
    if isPlayerInRange then
        self.playerInRange = player;
        self.isPlayerInRange = true;
        g_currentMission:addActivatableObject(self.activatable);
    else
        self.playerInRange = nil;
        self.isPlayerInRange = false;
        g_currentMission:removeActivatableObject(self.activatable);
    end;
end;

function Kaercher:setIsWashing(doWashing, force, noEventSend)
    KaercherStateEvent.sendEvent(self, doWashing, noEventSend);
    if self.doWashing ~= doWashing then
        if self.isClient then
            Utils.setEmittingState(self.particleSystems, doWashing and self:getIsActiveForInput());
            if doWashing then
                EffectManager:startEffect(self.waterEffect)
                if self.sampleWashing ~= nil and self.currentPlayer == g_currentMission.player  then
                    if self:getIsActiveForSound() then
                        Utils.playSample(self.sampleWashing, 0, 0, 1);
                    end;
                end;
            else
                if force then
                    EffectManager:resetEffect(self.waterEffect);
                else
                    EffectManager:stopEffect(self.waterEffect);
                end;
                if self.sampleWashing ~= nil then
                    Utils.stopSample(self.sampleWashing, true);
                end;
            end;
        end;
        self.doWashing = doWashing;
    end;
end;

function Kaercher:setIsTurnedOn(isTurnedOn, player, noEventSend)
    KaercherTurnOnEvent.sendEvent(self, isTurnedOn, player, noEventSend);
    if self.isTurnedOn ~= isTurnedOn then
        if isTurnedOn then
            self.isTurnedOn = isTurnedOn;
            self.currentPlayer = player;
            -- player stuff
            local tool = {};
            tool.node = self.lanceNode;
            link(player.toolsRootNode, tool.node);
            setVisibility(tool.node, false);
            setTranslation(tool.node, unpack(self.linkPosition));
            setRotation(tool.node, unpack(self.linkRotation));
            tool.update = Kaercher.updateLance;
            tool.updateTick = Kaercher.updateTickLance;
            tool.delete = Kaercher.deleteLance;
            tool.draw = Kaercher.drawLance;
            tool.onActivate = Kaercher.activateLance;
            tool.onDeactivate = Kaercher.deactivateLance;
            tool.targets = self.targets;
            tool.owner = self;
            tool.static = false;
            self.tool = tool;
            self.currentPlayer:setTool(tool);
            self.currentPlayer.hasHPWLance = true;
            if self.isClient then
                if self.sampleSwitch ~= nil and self:getIsActiveForSound() then
                    Utils.playSample(self.sampleSwitch, 1, 0, nil);
                end;
                if self.sampleCompressor ~= nil then
                    Utils.setSamplePitch(self.sampleCompressor, self.sampleCompressor.pitchOffset);
                    Utils.setSampleVolume(self.sampleCompressor, self.sampleCompressor.volume);
                    Utils.play3DSample(self.sampleCompressor);
                end;
                if self.isTurningOff then
                    self.isTurningOff = false;
                end;
                setVisibility(self.lanceNode, g_currentMission.player == player);
            end;
        else
            self:onDeactivate();
        end;
        if self.exhaustNode ~= nil then
            setVisibility(self.exhaustNode, isTurnedOn);
        end;
    end;
end;

function Kaercher:onDeactivate()
    if self.isClient then
        if self.sampleSwitch ~= nil and self:getIsActiveForSound() then
            Utils.playSample(self.sampleSwitch, 1, 0, nil);
        end;
    end;
    self.isTurnedOn = false;
    setVisibility(self.lanceNode, true);
    self:setIsWashing(false, true, true);
    if self.currentPlayer ~= nil then
        self.currentPlayer:setToolById(0, true);
        self.currentPlayer.hasHPWLance = false;
    end;
    if self.isClient then
        if self.sampleWashing ~= nil then
            Utils.stopSample(self.sampleWashing, true);
        end;
        self.isTurningOff = true;
        self.turnOffTime = g_currentMission.time + self.turnOffDuration;
        link(self.lanceNodeParent, self.lanceNode);
        setTranslation(self.lanceNode, 0,0,0);
        setRotation(self.lanceNode, 0,0,0);
    end;
    self.currentPlayer = nil;
end;

function Kaercher:getIsActiveForInput()
    if self.isTurnedOn and self.currentPlayer == g_currentMission.player and g_gui.currentGui == nil then
        return true;
    end;
    return false;
end;

function Kaercher:getIsActiveForSound()
    return self:getIsActiveForInput();
end;

function Kaercher:washRaycastCallback(hitObjectId, x, y, z, distance)
    local vehicle = g_currentMission.nodeToVehicle[hitObjectId];
    if vehicle ~= nil and vehicle.getDirtAmount ~= nil and vehicle.setDirtAmount ~= nil and vehicle.washDuration ~= nil then
        self.foundCoords = {x,y,z};
        self.foundVehicle = vehicle;
        return false;
    end;
    return true;
end;

function Kaercher.activateLance(tool)
    setVisibility(tool.node, true);
end

function Kaercher.deactivateLance(tool)
    tool.owner:setIsTurnedOn(false, nil);
end

function Kaercher.deleteLance(tool)
    tool.owner:setIsTurnedOn(false, nil);
end

function Kaercher.drawLance(tool)
    if tool.owner.currentPlayer == g_currentMission.player then
        g_currentMission:addHelpButtonText(g_i18n:getText("ACTIVATE_HANDTOOL"), InputBinding.ACTIVATE_HANDTOOL);
    end;
end

function Kaercher.updateLance(tool, dt, allowInput)
    if allowInput then
        tool.owner:setIsWashing(InputBinding.isPressed(InputBinding.ACTIVATE_HANDTOOL), false, false);
    end;
end;

function Kaercher.updateTickLance(tool, dt, allowInput)
end;

KaercherActivatable = {}
local KaercherActivatable_mt = Class(KaercherActivatable);

function KaercherActivatable:new(Kaercher)
    local self = {};
    setmetatable(self, KaercherActivatable_mt);
    self.Kaercher = Kaercher;
    self.activateText = "unknown";
    return self;
end;
function KaercherActivatable:getIsActivatable()
    if not self.Kaercher.isPlayerInRange then
        return false;
    end;
    if self.Kaercher.playerInRange ~= g_currentMission.player then
        return false;
    end;
    if not self.Kaercher.playerInRange.isControlled then
        return false;
    end;
    if self.Kaercher.isTurnedOn and self.Kaercher.currentPlayer ~= g_currentMission.player then
        return false;
    end;
    if not self.Kaercher.isTurnedOn and g_currentMission.player.hasHPWLance == true then
        return false;
    end;
    if self.Kaercher.isDeleted then
        return false;
    end;
    self:updateActivateText();
    return true;
end;
function KaercherActivatable:onActivateObject()
    self.Kaercher:setIsTurnedOn(not self.Kaercher.isTurnedOn, g_currentMission.player);
    self:updateActivateText();
    g_currentMission:addActivatableObject(self);
end;
function KaercherActivatable:drawActivate()
end;
function KaercherActivatable:updateActivateText()
    if self.Kaercher.isTurnedOn then
        self.activateText = string.format(g_i18n:getText("turn_off_OBJECT"), g_i18n:getText("TypeDesc_Kaercher"));
    else
        self.activateText = string.format(g_i18n:getText("turn_on_OBJECT"), g_i18n:getText("TypeDesc_Kaercher"));
    end;
end;