--
-- AdditionalFruitTypes
--
-- @author: Jakob Tischler
-- @version: 0.5
-- @date: 06 Aug 2014
-- @history: 0.4 (01 Jul 2013): initial implementation
--           0.5 (05 Dec 2014): conversion to FS15
--
-- Copyright (C) 2014 Jakob Tischler
-- 
-- @usage:
--[[
	0)	reference this .lua file in the map's <extraSourceFiles>

	1)	add to map's .lua file, :new() function, at the beginning:

		-- AdditionalFruitTypes # START #
		local additionalFruitTypesPath =  Utils.getFilename('additionalFruitTypes.lua', baseDirectory);
		if fileExists(additionalFruitTypesPath) then
			source(additionalFruitTypesPath);
			AdditionalFruitTypes:run(baseDirectory);
		end;
		-- AdditionalFruitTypes #  END  #

	2)	add a directory to your map where the fruit overlays will be stored, e.g. "fruitHuds"
		all fruit type overlays have to be named in the format "hud_fruit_fruitName.dds", where "fruitName" is your fruit's name
		also add small icons(64x64) in the format "hud_fruit_fruitName_small.dds". otherwise you will get a notification in the log and the large icon will be used

	3)	add to map's modDesc.xml, l10n section:

		<!-- MAP FRUITS -->
		<text name="newFruitName">
			<de>Fruchtname</de>
			<en>fruit name</en>
		</text>
		<text name="gerstenschrot">
			<de>Gerstenschrot</de>
			<en>Barley grout</en>
		</text>
		<text name="weizenmehl">
			<de>Weizenmehl</de>
			<en>Wheat flour</en>
		</text>

	4)	add to map's modDesc.xml:
		
		<AdditionalFruitTypes hudDirectory="fruitHuds/">
			<fruitType name="luzerne" needsSeeding="true" allowsSeeding="true" useSeedingWidth="false" directionSnapAngle="0" minHarvestingGrowthState="4" maxHarvestingGrowthState="6" cutState="1" allowsPartialGrowthState="false" pricePerLiter="0.15" literPerSqm="4" massPerLiter="0.0003" seedUsagePerSqm="0.1" partOfEconomy="false" fruitMapColor="0.4863, 0.5686, 1, 1" hasWindrow="true" windrowPricePerLiter="0.04" windrowLiterPerSqm="5" windrowMassPerLiter="0.0003" forageWagonConversion="grass"/>		
			<!--<fillType name="gerstenschrot" pricePerLiter="0.8" partOfEconomy="true" />-->		
		</AdditionalFruitTypes>

		Adjust the "hudDirectory" tag to your preference.
--]]

-- ##################################################


AdditionalFruitTypes = {};
AdditionalFruitTypes.version = '0.5';
AdditionalFruitTypes.author = 'Jakob Tischler+upsidedown';

function AdditionalFruitTypes:run(baseDirectory)

	if self.initialized then return end;

	local xmlFilePath =  Utils.getFilename('modDesc.xml', baseDirectory);
	if fileExists(xmlFilePath) then
		local xmlFile = loadXMLFile('modDescXML', xmlFilePath);
		local key = 'modDesc.AdditionalFruitTypes';

		if hasXMLProperty(xmlFile, key) then
			local hudDirectory = getXMLString(xmlFile, key..'#hudDirectory');
			if hudDirectory ~= nil then
				if hudDirectory:sub(-1) ~= '/' then
					hudDirectory = hudDirectory .. '/';
				end;
				hudDirectory = baseDirectory .. hudDirectory;

				print(('AdditionalFruitTypes v%s by %s loaded'):format(AdditionalFruitTypes.version, AdditionalFruitTypes.author));
				self:registerFruitTypes(xmlFile, key, hudDirectory);
			else
				print('Error: AdditionalFruitTypes could not find hud directory.');
			end; -- END hudDirectory ~= nil
		end; -- END hasXMLProperty(xmlFile, key)
		delete(xmlFile);
	end; -- END fileExists(xmlFilePath)

	self.initialized = true;
end;

function AdditionalFruitTypes:registerFruitTypes(xmlFile, key, hudDirectory)
	-- FRUIT TYPES
	local a = 0;
	while true do
		local fruitTypeKey = key .. ('.fruitType(%d)'):format(a);
		if not hasXMLProperty(xmlFile, fruitTypeKey) then
			break;
		end;
		
		local name = getXMLString(xmlFile, fruitTypeKey..'#name');
		if name == nil then
			print(('Error: missing "name" attribute for fruitType #%d in "AdditionalFruitTypes". Adding fruitTypes aborted.'):format(a));
			break;
		end;

		local gameKey = 'FRUITTYPE_' .. name:upper();
		if FruitUtil[gameKey] == nil --[[and FruitUtil.NUM_FRUITTYPES < 64]] then
			local fruitType = {
				name = name,
				needsSeeding =             Utils.getNoNil(getXMLBool(  xmlFile, fruitTypeKey..'#needsSeeding'), true),
				allowsSeeding =            Utils.getNoNil(getXMLBool(  xmlFile, fruitTypeKey..'#allowsSeeding'), true),
				useSeedingWidth =          Utils.getNoNil(getXMLBool(  xmlFile, fruitTypeKey..'#useSeedingWidth'), false),
				directionSnapAngle =       Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#directionSnapAngle'), 0),
				minHarvestingGrowthState = Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#minHarvestingGrowthState'), 4),
				maxHarvestingGrowthState = Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#maxHarvestingGrowthState'), 6),
				cutState =                 Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#cutState'), 8),
				allowsPartialGrowthState = Utils.getNoNil(getXMLBool(  xmlFile, fruitTypeKey..'#allowsPartialGrowthState'), false),
				pricePerLiter =            Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#pricePerLiter'), 0.29),
				literPerSqm =              Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#literPerSqm'), 1.2),
				seedUsagePerSqm =          Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#seedUsagePerSqm'), 0.1),
				partOfEconomy =            Utils.getNoNil(getXMLBool(  xmlFile, fruitTypeKey..'#partOfEconomy'), true),
				massPerLiter = 			   Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#massPerLiter'), 0.0008),
				fruitMapColor =                           getXMLString(xmlFile, fruitTypeKey..'#fruitMapColor'),
				maxPhysicalSurfaceAngle = 				  getXMLFloat( xmlFile, fruitTypeKey..'#maxPhysicalSurfaceAngle'),
				hasWindrow = 			   Utils.getNoNil(getXMLBool(  xmlFile, fruitTypeKey..'#hasWindrow'), false),
				windrowPricePerLiter =     Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#windrowPricePerLiter'), 0.4),
				windrowLiterPerSqm =       Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#windrowLiterPerSqm'), 5),
				windrowMassPerLiter =      Utils.getNoNil(getXMLFloat( xmlFile, fruitTypeKey..'#windrowMassPerLiter'), 0.0003),
				forageWagonConversion =    getXMLString( xmlFile, fruitTypeKey..'#forageWagonConversion')
			};

			if fruitType.fruitMapColor ~= nil then
				fruitType.fruitMapColor = self:tableMap(Utils.splitString(',', fruitType.fruitMapColor), tonumber);
				if table.getn(fruitType.fruitMapColor) ~= 4 then
					fruitType.fruitMapColor = { 1, 0, 0, 1 };
				else
					for k=1,3 do
						fruitType.fruitMapColor[k] = fruitType.fruitMapColor[k]/255;
					end;
				end;
			else
				fruitType.fruitMapColor = { 1, 0, 0, 1 };
			end;
			
			local localName = fruitType.name;
			if g_i18n:hasText(fruitType.name) then
				localName = g_i18n:getText(fruitType.name);
				g_i18n.globalI18N.texts[fruitType.name] = localName;
			end;

			local hudFile = ('%shud_fruit_%s.dds'):format(hudDirectory, fruitType.name);
			local hudFile_small = ('%shud_fruit_%s_small.dds'):format(hudDirectory, fruitType.name);
			
			if not fileExists(Utils.getFilename(hudFile_small, baseDirectory)) then				
				print(hudFile_small.." not found. This is *not* a problem, but its not 100% perfect")
				hudFile_small = hudFile;
			end;			
			
			local key = FruitUtil.registerFruitType(fruitType.name, localName, fruitType.needsSeeding, fruitType.allowsSeeding, fruitType.useSeedingWidth, fruitType.directionSnapAngle, fruitType.minHarvestingGrowthState, fruitType.maxHarvestingGrowthState, fruitType.cutState, fruitType.allowsPartialGrowthState, fruitType.pricePerLiter, fruitType.literPerSqm, fruitType.seedUsagePerSqm, fruitType.partOfEconomy, hudFile, hudFile_small, fruitType.fruitMapColor,fruitType.massPerLiter,fruitType.maxPhysicalSurfaceAngle);
			if key ~= nil then
				print(('\\__ Register fruitType: %s (%s) [key %s]'):format(localName, fruitType.name, tostring(key)));
			end;
			
			if fruitType.hasWindrow and key ~= nil then
				local windrowName = fruitType.name.."_windrow";
				local localName = windrowName;
				if g_i18n:hasText(g_i18n:hasText(windrowName)) then
					localName = g_i18n:hasText(windrowName);
				end;
				local hudFile = ('%shud_fruit_%s.dds'):format(hudDirectory, windrowName);
				if not fileExists(Utils.getFilename(hudFile_small, baseDirectory)) then
					print(hudFile_small.." not found. This is *not* a problem, but its not 100% perfect")
					hudFile_small = hudFile;
				end;
				-- FruitUtil.registerFruitTypeWindrow(key, windrowName, localName, fruitType.windrowPricePerLiter, fruitType.windrowLiterPerSqm, fruitType.partOfEconomy, hudFile, hudFile_small, fruitType.windrowMassPerLiter);
				FruitUtil.registerFruitTypeWindrow(key, windrowName, localName, fruitType.windrowPricePerLiter, fruitType.windrowLiterPerSqm, false, hudFile, hudFile_small, fruitType.windrowMassPerLiter);
				if FruitUtil.fruitTypeToWindrowFillType[key] ~= nil then
					print(('\\____ Register windrowType: %s (%s)'):format(localName, windrowName));
					if fruitType.forageWagonConversion ~= nil then
						local target = "FRUITTYPE_"..string.upper(fruitType.forageWagonConversion);
						if FruitUtil[target] ~= nil then
							FruitUtil.registerFruitTypeWindrowForageWagonConversion(key, FruitUtil[target]);
							print(('\\______ ForageWagonConversion registered: %s -> %s'):format(windrowName, fruitType.forageWagonConversion));
						else
							print("Error: incorrect target for ForageWagonConversion: "..target)
						end;
					end;					
				end;
				
				
				
			end;

			a = a + 1;
		else
			a = a + 1;
			print(('fruit type %q already exists. "AdditionalFruitTypes" will skip its registration.'):format(name));
		end;
	end;


	-- FILL TYPES
	local d = 0;
	while true do
		local fillTypeKey = key .. ('.fillType(%d)'):format(d);
		if not hasXMLProperty(xmlFile, fillTypeKey) then
			break;
		end;

		local name = getXMLString(xmlFile, fillTypeKey..'#name');
		if name == nil then
			print(('Error: missing "name" attribute for fillType #%d in "AdditionalFruitTypes". Adding fillTypes aborted.'):format(b));
			break;
		end;

		local gameKey = 'FILLTYPE_' .. name:upper();
		if Fillable[gameKey] == nil --[[and Fillable.NUM_FILLTYPES < 64]] then
			local fillType = {
				name = name,
				pricePerLiter = Utils.getNoNil(getXMLFloat(xmlFile, fillTypeKey .. '#pricePerLiter'), 0.8),
				partOfEconomy = Utils.getNoNil(getXMLBool( xmlFile, fillTypeKey .. '#partOfEconomy'), false)
			};

			local localName = fillType.name;
			if g_i18n:hasText(fillType.name) then
				localName = g_i18n:getText(fillType.name);
				g_i18n.globalI18N.texts[fillType.name] = localName;
			end;

			local key = Fillable.registerFillType(fillType.name, localName, fillType.pricePerLiter, fillType.partOfEconomy, ('%shud_fruit_%s.dds'):format(hudDirectory, fillType.name));
			if key ~= nil then
				print(('\\__ Register fillType: %s (%s) [key %s]'):format(localName, fillType.name, tostring(key)));
			end;
			d = d + 1;
		else
			d = d + 1;
			print(('Error: fill type %q already exists. "AdditionalFruitTypes" will skip its registration.'):format(name));
		end;
	end;	
end;

function AdditionalFruitTypes:tableMap(table, func)
	local newArray = {};
	for i,v in ipairs(table) do
		newArray[i] = func(v);
	end;
	return newArray;
end;