--
-- InteractiveButtons
-- Specialization for an interactive control button
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.1
-- @date  		29/08/11
-- @history:	v1.0 - Initial version
--				v2.0 - converted to ls2011
--				v2.1 - improvements
--
-- free for noncommerical-usage
--

InteractiveButtons = {};

function InteractiveButtons.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(InteractiveControl, specializations);
end;

function InteractiveButtons:load(xmlFile)

	local i=0;
	while true do
		local buttonName = string.format("vehicle.interactiveComponents.buttons.button(%d)", i);	
		if not hasXMLProperty(xmlFile, buttonName) then
			break;
		end;
		local name = Utils.getNoNil(g_i18n:getText(getXMLString(xmlFile, buttonName .. "#name")), "ERROR");
		local mark = Utils.indexToObject(self.components, getXMLString(xmlFile, buttonName .. "#mark"));
		local highlight = getChildAt(mark, 0);
		local size = Utils.getNoNil(getXMLFloat(xmlFile, buttonName .. "#size"), 0.1);
		local event = getXMLString(xmlFile, buttonName .. "#event");
		local onMessage = g_i18n:getText(Utils.getNoNil(getXMLString(xmlFile, buttonName .. "#onMessage"), "ic_button_on"));
		local offMessage =  g_i18n:getText(Utils.getNoNil(getXMLString(xmlFile, buttonName .. "#offMessage") , "ic_button_off"));
	
		local button = Button:new(nil, highlight, name, mark, size, event, self, onMessage, offMessage, self.infoBar);
		
		button.synch = Utils.getNoNil(getXMLBool(xmlFile, buttonName .. "#synch"), true);

		table.insert(self.interactiveObjects, button);
		i = i + 1;
	end;
end;

function InteractiveButtons:delete()
end;

function InteractiveButtons:mouseEvent(posX, posY, isDown, isUp, button)
end;

function InteractiveButtons:keyEvent(unicode, sym, modifier, isDown)
end;

function InteractiveButtons:update(dt)	
end;

function InteractiveButtons:draw()
end;



--
-- Button Class
-- Specifies an interactive Button
--
-- SFM-Modding
-- @author  Manuel Leithner
-- @date  29/08/11
--

Button = {};

function Button:new(node, highlight, name, mark, size, event, vehicle, onMessage, offMessage, infobar)

	local Button_mt = Class(Button, InteractiveComponentInterface);	
    local instance = InteractiveComponentInterface:new(node, highlight, name, mark, size, onMessage, offMessage, infobar, Button_mt);

	instance.vehicle = vehicle;
	instance.event = event;
	
	return instance;	
end;

function Button:delete()
	InteractiveComponentInterface.delete(self);
end;

function Button:mouseEvent(posX, posY, isDown, isUp, button)
	InteractiveComponentInterface.mouseEvent(self, posX, posY, isDown, isUp, button);
end;

function Button:keyEvent(unicode, sym, modifier, isDown)
	InteractiveComponentInterface.keyEvent(self, unicode, sym, modifier, isDown);
end;

function Button:update(dt)
	if self.vehicle ~= nil then
		if self.event == "togglePipe" then
			self.isOpen = self.vehicle.targetPipeState ~= 1;
		elseif self.event == "toggleChopper" then
			self.isOpen = not self.vehicle.isStrawEnabled;
		elseif self.event == "toggleOnOff" then
			self.isOpen = self.vehicle.isTurnedOn;
		elseif self.event == "toggleFolding" then
			if self.vehicle:getToggledFoldDirection() == self.vehicle.turnOnFoldDirection then
				self.isOpen = false;
			else
				self.isOpen = true;
			end;
		end;
	end;
	InteractiveComponentInterface.update(self, dt);
end;

function Button:draw()
	InteractiveComponentInterface.draw(self);
end;

function Button:doAction(noEventSend, forceAction)
	if self.vehicle ~= nil then
		if self.event == "togglePipe" then
			if forceAction == nil then
				local nextState = self.vehicle.targetPipeState+1;
				if nextState > self.vehicle.numPipeStates then
					nextState = 1;
				end;
				self.vehicle:setPipeState(nextState, true);
			end;
		elseif self.event == "toggleChopper" then
			local newValue = not self.vehicle.isStrawEnabled;
			if forceAction ~= nil then
				newValue = not forceAction;
			end;
			if self.vehicle.lastValidInputFruitType ~= FruitUtil.FRUITTYPE_UNKNOWN then
				local fruitDesc = FruitUtil.fruitIndexToDesc[self.vehicle.lastValidInputFruitType];
				if fruitDesc.hasWindrow then
					--self.vehicle.isStrawEnabled = not self.vehicle.isStrawEnabled;
					self.vehicle:setIsStrawEnabled(newValue);
				end;
			else
				--self.vehicle.isStrawEnabled = not self.vehicle.isStrawEnabled;
				self.vehicle:setIsStrawEnabled(newValue);
			end;
		elseif self.event == "toggleOnOff" then
			if self.vehicle:getIsTurnedOnAllowed(not self.vehicle.isTurnedOn)  then
				self.vehicle:setIsTurnedOn(not self.vehicle.isTurnedOn);
			end;
		elseif self.event == "toggleFolding" then
			if self.vehicle:getIsFoldAllowed() then
				if self.vehicle:getToggledFoldDirection() == self.vehicle.turnOnFoldDirection then
					self.vehicle:setFoldState(self.vehicle:getToggledFoldDirection(), true);
				else
					self.vehicle:setFoldState(self.vehicle:getToggledFoldDirection(), false);
				end;
			end;
		elseif self.event == "toggledrl" then	
			self.vehicle:setState("work:3", not self.vehicle.B3["work"][3].a);
			print(table.getn(self.vehicle.B3["work"]));
		elseif self.event == "toggleBeacon" then
			self.vehicle:setBeaconLightsVisibility(not self.vehicle.beaconLightsActive);
		elseif self.event == "frontwork" then
			self.vehicle:setState("work:1", not self.vehicle.B3["work"][1].a);
		elseif self.event == "rearwork" then	
			self.vehicle:setState("work:2", not self.vehicle.B3["work"][2].a);
		elseif self.event == "lowerBack" then
			if table.getn(self.vehicle.attachedImplements) > 0 then
				local object = self.vehicle.attachedImplements[1].object;
				local jointDesc = object.attacherVehicle.attacherJoints[self.vehicle.attachedImplements[1].jointDescIndex];
				if object.attacherJoint.allowsLowering and jointDesc.allowsLowering then
					object.attacherVehicle:setJointMoveDown(self.vehicle.attachedImplements[1].jointDescIndex, not jointDesc.moveDown, false);
				end;
			end;
		end;
	end;
end;

function Button:onEnter()
	InteractiveComponentInterface.onEnter(self);
end;

function Button:onExit()
	InteractiveComponentInterface.onExit(self);
end;

function Button:setActive()
	InteractiveComponentInterface.setActive(self, isActive);
end;

function Button:setVisible(isVisible)
	InteractiveComponentInterface.setVisible(self, isVisible);
end;