--
-- InteractiveBackupCam
-- Specialization for InteractiveBackupCam
--
-- @author:  	Manuel Leithner
-- @date:		03/03/2011
-- @version:	v1.0
-- @history:	v1.0 - initial implementation
--
-- free for noncommerical-usage
--

InteractiveBackupCam = {};

function InteractiveBackupCam.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(InteractiveControl, specializations);
end;

function InteractiveBackupCam:load(xmlFile)
	
	source(Utils.getFilename("scripts/InteractiveComponentInterface.lua", self.baseDirectory));
	self.setBackupCam = SpecializationUtil.callSpecializationsFunction("setBackupCam");
	self.addBackupCam = SpecializationUtil.callSpecializationsFunction("addBackupCam");
	self.removeBackupCam = SpecializationUtil.callSpecializationsFunction("removeBackupCam");
	
	self.backupCams = {};
	self.currentCam = 0;
	self.maxBackupCams = 5;
	
	local i = 0;
	while true do		
		local key = string.format("vehicle.backupCams.cam(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;
		local cam = VehicleCamera:new(self);
		cam:loadFromXML(xmlFile, key);
		cam.overlayId = createRenderOverlay(cam.cameraNode, getScreenAspectRatio(), 256, 256);
		self:addBackupCam(cam);
		i = i + 1;
	end;
	
	self.mark = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.backupCamPanel#highlight"));
	self.highlight = getChildAt(self.mark,0);

	local overlays = {};
	local buttons = {};	
		
	local bgX = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.backupCamPanel#xPos"), 0.728);
	local bgY = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.backupCamPanel#yPos"), 0.575);
	local size = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.backupCamPanel#size"), 0.1);	
		
    self.panelSound = createSample("panelSample");
    loadSample(self.panelSound, Utils.getFilename("ControlPanel/peeptone.wav", self.baseDirectory), false);		

	-- add main background overlay	
	table.insert(overlays, Overlay:new("controlPanelBG", Utils.getFilename("ControlPanel/panel_bg_diffuse.dds", self.baseDirectory), bgX + 0.012, bgY+0.085, 0.245, 0.21));
	table.insert(overlays, Overlay:new("controlPanel", Utils.getFilename("ControlPanel/panel_diffuse.dds", self.baseDirectory), bgX, bgY, 0.265, 0.3533));
	
	
	table.insert(buttons, Overlay:new("controlButtonClose", Utils.getFilename("ControlPanel/panel_button_onoff_diffuse.dds", self.baseDirectory), 		bgX+0.164, bgY+0.052, 0.02025, 0.027));	
	table.insert(buttons, Overlay:new("controlButton1",    	Utils.getFilename("ControlPanel/panel_button1_diffuse.dds", self.baseDirectory), 			bgX+0.0445,  bgY+0.053, 0.018, 0.024));
	table.insert(buttons, Overlay:new("controlButton2",    	Utils.getFilename("ControlPanel/panel_button2_diffuse.dds", self.baseDirectory), 			bgX+0.0445+1*0.022, bgY+0.053, 0.018, 0.024));
	table.insert(buttons, Overlay:new("controlButton3",    	Utils.getFilename("ControlPanel/panel_button3_diffuse.dds", self.baseDirectory), 			bgX+0.0445+2*0.022, bgY+0.053, 0.018, 0.024));
	table.insert(buttons, Overlay:new("controlButton4",    	Utils.getFilename("ControlPanel/panel_button4_diffuse.dds", self.baseDirectory), 			bgX+0.0445+3*0.022, bgY+0.053, 0.018, 0.024));
	table.insert(buttons, Overlay:new("controlButton5",    	Utils.getFilename("ControlPanel/panel_button5_diffuse.dds", self.baseDirectory), 			bgX+0.0445+4*0.022, bgY+0.053, 0.018, 0.024));
	
	-- create new control panel instance
	self.panel = BackupCamPanel:new(self.controlPanel, self.highlight, self.mark, g_i18n:getText("BackupUpCam"), size, overlays, buttons, self.panelSound, self, bgX, bgY, self.infoBar, self);
	
	self.lastCam = 0;
	self.viewportId = nil;
	
	self.printCannotEnableError = 0;
	
	table.insert(self.interactiveObjects, self.panel);	
end;

function InteractiveBackupCam:delete()
	if self.viewportId ~= nil then
		self:setBackupCam(0);
	end;
end;

function InteractiveBackupCam:mouseEvent(posX, posY, isDown, isUp, button)
end;

function InteractiveBackupCam:keyEvent(unicode, sym, modifier, isDown)
end;

function InteractiveBackupCam:update(dt)
	if g_gui.currentGui ~= nil then
		if self.currentCam ~= 0 then
			self.lastCam = self.currentCam;
			self:setBackupCam(0);
		end;
	else
		if self.lastCam ~= 0 then
			self:setBackupCam(self.lastCam);
			self.lastCam = 0;
		end;
	end;
end;

function InteractiveBackupCam:draw()
	if self.printCannotEnableError > self.time then
		g_currentMission:addWarning(g_i18n:getText("CANNOT_ENABLE_CAMS"), 0.018, 0.033);
	end;
end;

function InteractiveBackupCam:onLeave()
	if self.viewportId ~= nil then
		self:setBackupCam(0);
	end;
end;

function InteractiveBackupCam:addBackupCam(cam)
	if table.getn(self.backupCams) < self.maxBackupCams then
		table.insert(self.backupCams, cam);
	else
		print("cam maximum reached");
	end;
end;

function InteractiveBackupCam:removeBackupCam(cam)
	for k, backupCam in pairs(self.backupCams) do
		if cam == backupCam then
			table.remove(self.backupCams, k);
			break;
		end;
	end;
	local camCount = table.getn(self.backupCams);
	if self.currentCam > camCount then
		self:setBackupCam(camCount);
	end;
end;

function InteractiveBackupCam:setBackupCam(camId)
	--[[ if camId == 0 or self.specialJoint.attachable == nil or not self.specialJoint.attachable.isAICutterHelperActive  then
		local cam = self.backupCams[camId];	
		if cam ~= nil then	
			self.currentCam = camId;
			setRootNode(getRootNode(), 1);
			setCamera(cam.cameraNode, 1);
			setViewport(1, 0.74, 0.13, 0.245, 0.21);
			self.viewportId = 1;
		else
			self.currentCam = 0;
			setCamera(0, 1);
			self.viewportId = nil;
		end;	
	else
		self.currentCam = 0;
		setCamera(0, 1);
		self.viewportId = nil;
		self.printCannotEnableError = self.time + 1800;
	end ]]--
	if camId == 0 then
		local cam = self.backupCams[camId];	
		self.currentCam = 0;
		setCamera(0, 1);
		self.viewportId = nil;
	else
		local cam = self.backupCams[camId];	
		if cam ~= nil then	
		--[[	self.currentCam = camId;
			setRootNode(getRootNode(), 1);
			setCamera(cam.cameraNode, 1);
			setViewport(1, 0.74, 0.13, 0.245, 0.21);
			self.viewportId = 1; ]]
			self.viewportId = 1;
			self.currentCam = camId;
		end;
	end; 		
end;


--
-- BackupCamPanel Class
-- Specialization for BackupCamPanel
--
-- @author:  	Manuel Leithner
-- @date:		03/03/11
-- @version:	v1.0
-- @history:	v1.0 - initial implementation
--

BackupCamPanel = {};
function BackupCamPanel:new(node, highlight, mark, name, size, overlays, buttons, sound, controller, xPos, yPos, infoBar, parent)

	local BackupCamPanel_mt = Class(BackupCamPanel, InteractiveComponentInterface);	
    local instance = InteractiveComponentInterface:new(node, highlight, name, mark, size, nil, nil, infoBar, BackupCamPanel_mt);
	
	instance.onMessage = g_i18n:getText("ic_component_open");
	instance.offMessage = g_i18n:getText("ic_component_close");	
	instance.noCam = g_i18n:getText("ic_component_nocam");	
	instance.controller = controller;
	instance.sound = sound;
	instance.overlays = overlays;
	instance.xPos = xPos;
	instance.yPos = yPos;
	instance.isLocalOnly = true;
	instance.parent = parent;
	if instance.overlays == nil then
		instance.overlays = {};
	end;
	
	-- add buttons
	local actionEvent = instance.onPanelAction;
	for k,v in pairs(buttons) do		
		table.insert(instance.overlays, OverlayNumberedButton:new(v, k, actionEvent, instance, {1, 1, 1, 0.8}));		
	end;	
	instance.mainBackground = instance.overlays[1];
	
	instance.bgX = instance.mainBackground.x;
	instance.bgY = instance.mainBackground.y;
	
	return instance;	
end;


function BackupCamPanel:delete()
	for k,v in pairs(self.overlays) do
		v:delete();
	end;
	InteractiveComponentInterface.delete(self);
end;

function BackupCamPanel:mouseEvent(posX, posY, isDown, isUp, button)
	if self.isOpen then
		for k,v in pairs(self.overlays) do
			if v.mouseEvent ~= nil then
				v:mouseEvent(posX, posY, isDown, isUp, button);
			end;
		end;
	end;
	InteractiveComponentInterface.mouseEvent(self, posX, posY, isDown, isUp, button);
end;

function BackupCamPanel:keyEvent(unicode, sym, modifier, isDown)
	InteractiveComponentInterface.keyEvent(self, unicode, sym, modifier, isDown);
end;

function BackupCamPanel:update(dt)
	InteractiveComponentInterface.update(self, dt);
end;

function BackupCamPanel:draw()
	if self.isOpen then
		if self.parent.currentCam > 0 then
			local cam = self.parent.backupCams[self.parent.currentCam];
			updateRenderOverlay(cam.overlayId);
			renderOverlay(cam.overlayId, 0.74, 0.655, 0.245, 0.21);	
		end;	
		for k, overlay in pairs(self.overlays) do
			if k == 1 then
				if self.controller.currentCam == 0 or self.controller.currentCam > table.getn(self.controller.backupCams) then
					overlay:render();
					setTextColor(1,1,1,0.8);
					setTextBold(true);
					setTextAlignment(RenderText.ALIGN_CENTER);
					renderText(self.xPos + 0.126, self.yPos + 0.18, 0.02, self.noCam);
					setTextAlignment(RenderText.ALIGN_LEFT);
				end;
			else
				overlay:render();
			end;
		end;
	end;
	InteractiveComponentInterface.draw(self);
end;

function BackupCamPanel:doAction(noEventSend, forceAction)	
	InteractiveComponentInterface.doAction(self, forceAction);
	if self.isOpen then
		self.controller:setPanelOverlay(self);
		self.controller:setBackupCam(self.controller.currentCam);
	else
		self.controller:setPanelOverlay(nil);
		self.controller:setBackupCam(0);
	end;
end;

function BackupCamPanel:onEnter(dt)
	InteractiveComponentInterface.onEnter(self, dt);
end;

function BackupCamPanel:onExit(dt)
	InteractiveComponentInterface.onExit(self, dt);
end;

function BackupCamPanel:setActive()
	InteractiveComponentInterface.setActive(self, isActive);
end;

function BackupCamPanel:setVisible(isVisible)
	InteractiveComponentInterface.setVisible(self, isVisible);
end;

function BackupCamPanel:onPanelAction(id)
	playSample(self.sound, 1, 0.1,0);	
	if id == 1 then
		self:doAction();
	else
		self.controller:setBackupCam(id - 1);
	end;
end;

-- Numbered Button
OverlayNumberedButton = {};
local OverlayNumberedButton_mt = Class(OverlayNumberedButton);

function OverlayNumberedButton:new(overlay, number, onClick, target, disabledColor)
    local disabledColor = disabledColor;
    if disabledColor == nil then
        disabledColor = {0.8, 0.8, 0.8, 0.4};
    end;
    return setmetatable( {overlay=overlay, number=number, onClick=onClick, target=target, isDisabled=false, disabledColor=disabledColor}, OverlayNumberedButton_mt);
end;

function OverlayNumberedButton:delete()
    self.overlay:delete();
end;

function OverlayNumberedButton:mouseEvent(posX, posY, isDown, isUp, button)
	local isOver = posX >= self.overlay.x and posX <= self.overlay.x+self.overlay.width and posY >= self.overlay.y and posY <= self.overlay.y+self.overlay.height;
    if not self.isDisabled and isOver then
        self.overlay:setColor(1.0, 1.0, 1.0, 1.0);

        if isDown and button == Input.MOUSE_BUTTON_LEFT and self.onClick ~= nil then
            if self.target ~= nil then
                self.onClick(self.target, self.number);
            else
                self.onClick(self.number);
            end;
        end;

    else
        self:reset();
    end;
end;

function OverlayNumberedButton:render()
    self.overlay:render();
end;

function OverlayNumberedButton:setIsDisabled(isDisabled)
    if self.isDisabled ~= isDisabled then
        self.isDisabled = isDisabled;
        self:reset();
    end;
end;

function OverlayNumberedButton:reset()
    if self.isDisabled then
        self.overlay:setColor(unpack(self.disabledColor));
    else
        self.overlay:setColor(1.0, 1.0, 1.0, 0.8);
    end;
end;